<?php
/**
 * The checkout functionality of the plugin.
 *
 * @link       https://example.com
 * @since      1.0.0
 *
 * @package    aicoso_pickup_delivery
 */

if ( ! defined( 'WPINC' ) ) {
	die;
}

/**
 * The checkout class.
 *
 * Handles the integration with WooCommerce checkout to display and process
 * delivery/pickup options.
 *
 * @since      1.0.0
 * @package    aicoso_pickup_delivery
 * @author     Your Name
 */
class AICOSO_Pickup_Delivery_Checkout {

	/**
	 * The ID of this plugin.
	 *
	 * @since    1.0.0
	 * @access   private
	 * @var      string    $plugin_name    The ID of this plugin.
	 */
	private $plugin_name;

	/**
	 * The version of this plugin.
	 *
	 * @since    1.0.0
	 * @access   private
	 * @var      string    $version    The current version of this plugin.
	 */
	private $version;

	/**
	 * Initialize the class and set its properties.
	 *
	 * @since    1.0.0
	 * @param    string $plugin_name       The name of this plugin.
	 * @param    string $version           The version of this plugin.
	 */
	public function __construct( $plugin_name, $version ) {
		$this->plugin_name = $plugin_name;
		$this->version     = $version;

		// Add checkout fields.
		add_filter( 'woocommerce_checkout_fields', array( $this, 'add_checkout_fields' ) );

		// Add AJAX handlers for dynamic fields.
		add_action( 'wp_ajax_aicoso_pickup_delivery_get_dates', array( $this, 'ajax_get_dates' ) );
		add_action( 'wp_ajax_nopriv_aicoso_pickup_delivery_get_dates', array( $this, 'ajax_get_dates' ) );

		add_action( 'wp_ajax_aicoso_pickup_delivery_get_time_slots', array( $this, 'ajax_get_time_slots' ) );
		add_action( 'wp_ajax_nopriv_aicoso_pickup_delivery_get_time_slots', array( $this, 'ajax_get_time_slots' ) );

		add_action( 'wp_ajax_aicoso_pickup_delivery_get_locations', array( $this, 'ajax_get_locations' ) );
		add_action( 'wp_ajax_nopriv_aicoso_pickup_delivery_get_locations', array( $this, 'ajax_get_locations' ) );
	}

	/**
	 * Add delivery/pickup selector to the checkout form
	 *
	 * @since    1.0.0
	 */
	public function add_delivery_pickup_selector() {
		// Prevent duplicate display - check if we've already added the selector.
		static $selector_added = false;
		if ( $selector_added ) {
			return;
		}

		// Get settings.
		$settings         = AICOSO_Pickup_Delivery_Data::get_settings();
		$enable_selection = isset( $settings['enable_order_type_selection'] ) ? $settings['enable_order_type_selection'] : 'yes';
		$default_type     = isset( $settings['default_order_type'] ) ? $settings['default_order_type'] : 'delivery';

		if ( 'yes' !== $enable_selection ) {
			// When selection is disabled, add hidden fields with default type.
			// This ensures order type is saved for admin emails.
			echo '<input type="hidden" id="aicoso_pickup_delivery_type" name="aicoso_pickup_delivery_type" value="' . esc_attr( $default_type ) . '" />';
			echo '<input type="hidden" id="aicoso_pickup_delivery_type_selector" name="aicoso_pickup_delivery_type_selector" value="' . esc_attr( $default_type ) . '" />';
			$selector_added = true;
			return;
		}

		// Include template.
		wc_get_template(
			'checkout/delivery-pickup-selector.php',
			array(
				'default_type' => $default_type,
			),
			'aicoso-pickup-and-delivery-for-woocommerce/',
			AICOSO_PICKUP_DELIVERY_PLUGIN_PATH . 'templates/'
		);

		$selector_added = true;
	}

	/**
	 * Add custom fields to the checkout form
	 *
	 * @since    1.0.0
	 * @param    array $fields    Checkout fields.
	 * @return   array               Modified checkout fields
	 */
	public function add_checkout_fields( $fields ) {
		// Get settings.
		$settings         = AICOSO_Pickup_Delivery_Data::get_settings();
		$enable_selection = isset( $settings['enable_order_type_selection'] ) ? $settings['enable_order_type_selection'] : 'yes';
		$default_type     = isset( $settings['default_order_type'] ) ? $settings['default_order_type'] : 'delivery';

		// Only add a hidden field to store the selected type value.
		// All visible fields are handled by the template to avoid duplication.
		$fields['order']['aicoso_pickup_delivery_type'] = array(
			'type'     => 'hidden',
			'default'  => $default_type,
			'class'    => array( 'aicoso-pickup-delivery-type-hidden' ),
			'required' => false,
			'priority' => 10,
		);

		// Note: All other fields (date, time, location) are handled by the template.
		// to prevent duplication on shortcode-based checkout pages.

		return $fields;
	}

	/**
	 * Validate checkout fields
	 *
	 * @since    1.0.0
	 */
	public function validate_checkout_fields() {
		// Get settings to check if order type selection is enabled.
		$settings         = AICOSO_Pickup_Delivery_Data::get_settings();
		$enable_selection = isset( $settings['enable_order_type_selection'] ) ? $settings['enable_order_type_selection'] : 'yes';

		// If order type selection is disabled, skip validation entirely.
		if ( 'yes' !== $enable_selection ) {
			return;
		}

		// Get order type - check both field names for compatibility.
		$order_type = '';

		// First check the selector field.
		// phpcs:ignore WordPress.Security.NonceVerification.Missing -- Nonce handled by WooCommerce
		if ( isset( $_POST['aicoso_pickup_delivery_type_selector'] ) ) {
			// phpcs:ignore WordPress.Security.NonceVerification.Missing -- Nonce handled by WooCommerce
			$order_type = sanitize_text_field( wp_unslash( $_POST['aicoso_pickup_delivery_type_selector'] ) );
		// phpcs:ignore WordPress.Security.NonceVerification.Missing -- Nonce handled by WooCommerce
		} elseif ( isset( $_POST['aicoso_pickup_delivery_type'] ) ) {
			// Then check the hidden field (fallback).
			// phpcs:ignore WordPress.Security.NonceVerification.Missing -- Nonce handled by WooCommerce
			$order_type = sanitize_text_field( wp_unslash( $_POST['aicoso_pickup_delivery_type'] ) );
		}

		// Validate order type.
		if ( ! in_array( $order_type, array( 'delivery', 'pickup' ), true ) ) {
			wc_add_notice( __( 'Please select a valid order type.', 'aicoso-pickup-and-delivery-for-woocommerce' ), 'error' );
			return;
		}

		// Validate delivery fields.
		if ( 'delivery' === $order_type ) {
			// Get settings.
			$settings                   = AICOSO_Pickup_Delivery_Data::get_settings();
			$enable_delivery_date       = isset( $settings['enable_delivery_date'] ) ? $settings['enable_delivery_date'] : 'yes';
			$enable_delivery_time_slots = isset( $settings['enable_delivery_time_slots'] ) ? $settings['enable_delivery_time_slots'] : 'yes';

			if ( 'yes' === $enable_delivery_date ) {
	// phpcs:ignore WordPress.Security.NonceVerification.Missing -- Nonce handled by WooCommerce checkout
				$delivery_date = isset( $_POST['aicoso_pickup_delivery_delivery_date'] ) ? sanitize_text_field( wp_unslash( $_POST['aicoso_pickup_delivery_delivery_date'] ) ) : '';

				if ( empty( $delivery_date ) ) {
					wc_add_notice( __( 'Please select a delivery date.', 'aicoso-pickup-and-delivery-for-woocommerce' ), 'error' );
					return;
				}

				// Validate date format.
				if ( ! AICOSO_Pickup_Delivery_Date_Manager::is_valid_date_format( $delivery_date ) ) {
					wc_add_notice( __( 'Invalid delivery date format. Please use the date picker.', 'aicoso-pickup-and-delivery-for-woocommerce' ), 'error' );
					return;
				}

				// Check if date is available.
				if ( ! AICOSO_Pickup_Delivery_Date_Manager::is_date_available( $delivery_date, 'delivery' ) ) {
					wc_add_notice( __( 'The selected delivery date is not available. Please choose another date.', 'aicoso-pickup-and-delivery-for-woocommerce' ), 'error' );
					return;
				}
			}

			if ( 'yes' === $enable_delivery_time_slots ) {
	// phpcs:ignore WordPress.Security.NonceVerification.Missing -- Nonce handled by WooCommerce checkout
				$delivery_time_slot = isset( $_POST['aicoso_pickup_delivery_delivery_time_slot'] ) ? sanitize_text_field( wp_unslash( $_POST['aicoso_pickup_delivery_delivery_time_slot'] ) ) : '';

				if ( empty( $delivery_time_slot ) ) {
					wc_add_notice( __( 'Please select a delivery time.', 'aicoso-pickup-and-delivery-for-woocommerce' ), 'error' );
					return;
				}

				// Check if time slot is available (if date was provided and valid).
				if ( ! empty( $delivery_date ) && AICOSO_Pickup_Delivery_Date_Manager::is_valid_date_format( $delivery_date ) ) {
					if ( ! AICOSO_Pickup_Delivery_Timeslot_Manager::is_time_slot_available( $delivery_date, $delivery_time_slot, 'delivery' ) ) {
						wc_add_notice( __( 'The selected delivery time is not available. Please choose another time.', 'aicoso-pickup-and-delivery-for-woocommerce' ), 'error' );
						return;
					}
				}
			}
		}

		// Validate pickup fields.
		if ( 'pickup' === $order_type ) {
			// Get settings.
			$settings                 = AICOSO_Pickup_Delivery_Data::get_settings();
			$enable_pickup_date       = isset( $settings['enable_pickup_date'] ) ? $settings['enable_pickup_date'] : 'yes';
			$enable_pickup_time_slots = isset( $settings['enable_pickup_time_slots'] ) ? $settings['enable_pickup_time_slots'] : 'yes';

			if ( 'yes' === $enable_pickup_date ) {
	// phpcs:ignore WordPress.Security.NonceVerification.Missing -- Nonce handled by WooCommerce checkout
				$pickup_date = isset( $_POST['aicoso_pickup_delivery_pickup_date'] ) ? sanitize_text_field( wp_unslash( $_POST['aicoso_pickup_delivery_pickup_date'] ) ) : '';

				if ( empty( $pickup_date ) ) {
					wc_add_notice( __( 'Please select a pickup date.', 'aicoso-pickup-and-delivery-for-woocommerce' ), 'error' );
					return;
				}

				// Validate date format.
				if ( ! AICOSO_Pickup_Delivery_Date_Manager::is_valid_date_format( $pickup_date ) ) {
					wc_add_notice( __( 'Invalid pickup date format. Please use the date picker.', 'aicoso-pickup-and-delivery-for-woocommerce' ), 'error' );
					return;
				}

				// Check if date is available.
				if ( ! AICOSO_Pickup_Delivery_Date_Manager::is_date_available( $pickup_date, 'pickup' ) ) {
					wc_add_notice( __( 'The selected pickup date is not available. Please choose another date.', 'aicoso-pickup-and-delivery-for-woocommerce' ), 'error' );
					return;
				}
			}

			if ( 'yes' === $enable_pickup_time_slots ) {
	// phpcs:ignore WordPress.Security.NonceVerification.Missing -- Nonce handled by WooCommerce checkout
				$pickup_time_slot = isset( $_POST['aicoso_pickup_delivery_pickup_time_slot'] ) ? sanitize_text_field( wp_unslash( $_POST['aicoso_pickup_delivery_pickup_time_slot'] ) ) : '';

				if ( empty( $pickup_time_slot ) ) {
					wc_add_notice( __( 'Please select a pickup time.', 'aicoso-pickup-and-delivery-for-woocommerce' ), 'error' );
					return;
				}

				// Check if time slot is available (if date was provided and valid).
				if ( ! empty( $pickup_date ) && AICOSO_Pickup_Delivery_Date_Manager::is_valid_date_format( $pickup_date ) ) {
					if ( ! AICOSO_Pickup_Delivery_Timeslot_Manager::is_time_slot_available( $pickup_date, $pickup_time_slot, 'pickup' ) ) {
						wc_add_notice( __( 'The selected pickup time is not available. Please choose another time.', 'aicoso-pickup-and-delivery-for-woocommerce' ), 'error' );
						return;
					}
				}
			}

			// Validate pickup location.
// phpcs:ignore WordPress.Security.NonceVerification.Missing -- Nonce handled by WooCommerce checkout
			$pickup_location = isset( $_POST['aicoso_pickup_delivery_pickup_location'] ) ? sanitize_text_field( wp_unslash( $_POST['aicoso_pickup_delivery_pickup_location'] ) ) : '';

			if ( empty( $pickup_location ) ) {
				wc_add_notice( __( 'Please select a pickup location.', 'aicoso-pickup-and-delivery-for-woocommerce' ), 'error' );
				return;
			}

			// Check if location is available (if date was provided and valid).
			if ( ! empty( $pickup_date ) && AICOSO_Pickup_Delivery_Date_Manager::is_valid_date_format( $pickup_date ) ) {
				if ( ! AICOSO_Pickup_Delivery_Location_Manager::is_location_available( $pickup_date, $pickup_location ) ) {
					wc_add_notice( __( 'The selected pickup location is not available. Please choose another location.', 'aicoso-pickup-and-delivery-for-woocommerce' ), 'error' );
					return;
				}
			}
		}
	}

	/**
	 * Save delivery/pickup details early during order creation
	 *
	 * @since     1.0.0
	 * @param     WC_Order $order    The order object.
	 * @param     array    $data     The checkout data.
	 */
	public function save_delivery_pickup_details_early( $order, $data ) { // phpcs:ignore Generic.CodeAnalysis.UnusedFunctionParameter.FoundAfterLastUsed
		// Get settings.
		$settings         = AICOSO_Pickup_Delivery_Data::get_settings();
		$enable_selection = isset( $settings['enable_order_type_selection'] ) ? $settings['enable_order_type_selection'] : 'yes';

		// Get order type - check both field names for compatibility.
		$order_type = '';

		// If selection is disabled, use default order type.
		if ( 'no' === $enable_selection ) {
			// Use default order type when selection is disabled.
			// This ensures admin emails still work.
			$order_type = isset( $settings['default_order_type'] ) ? $settings['default_order_type'] : 'delivery';
		} else {
			// First check the selector field.
			if ( isset( $_POST['aicoso_pickup_delivery_order_type'] ) ) { // phpcs:ignore WordPress.Security.NonceVerification.Missing -- Nonce handled by WooCommerce checkout.
				$order_type = sanitize_text_field( wp_unslash( $_POST['aicoso_pickup_delivery_order_type'] ) ); // phpcs:ignore WordPress.Security.NonceVerification.Missing
			}
			// Check for delivery type specific field.
			if ( empty( $order_type ) && isset( $_POST['aicoso_pickup_delivery_type'] ) ) { // phpcs:ignore WordPress.Security.NonceVerification.Missing -- Nonce handled by WooCommerce checkout.
				$order_type = sanitize_text_field( wp_unslash( $_POST['aicoso_pickup_delivery_type'] ) ); // phpcs:ignore WordPress.Security.NonceVerification.Missing
			}

			// If still empty, use default.
			if ( empty( $order_type ) ) {
				$order_type = isset( $settings['default_order_type'] ) ? $settings['default_order_type'] : 'delivery';
			}
		}

		// Save order type using update_meta_data method.
		$order->update_meta_data( '_aicoso_pickup_delivery_type', $order_type );

		// Save delivery details.
		if ( 'delivery' === $order_type ) {
			if ( isset( $_POST['aicoso_pickup_delivery_delivery_date'] ) ) { // phpcs:ignore WordPress.Security.NonceVerification.Missing -- Nonce handled by WooCommerce checkout.
				$order->update_meta_data( '_aicoso_pickup_delivery_delivery_date', sanitize_text_field( wp_unslash( $_POST['aicoso_pickup_delivery_delivery_date'] ) ) ); // phpcs:ignore WordPress.Security.NonceVerification.Missing
			}

			if ( isset( $_POST['aicoso_pickup_delivery_delivery_time_slot'] ) ) { // phpcs:ignore WordPress.Security.NonceVerification.Missing -- Nonce handled by WooCommerce checkout.
				$order->update_meta_data( '_aicoso_pickup_delivery_delivery_time_slot', sanitize_text_field( wp_unslash( $_POST['aicoso_pickup_delivery_delivery_time_slot'] ) ) ); // phpcs:ignore WordPress.Security.NonceVerification.Missing
			}
		}

		// Save pickup details.
		if ( 'pickup' === $order_type ) {
			if ( isset( $_POST['aicoso_pickup_delivery_pickup_date'] ) ) { // phpcs:ignore WordPress.Security.NonceVerification.Missing -- Nonce handled by WooCommerce checkout.
				$order->update_meta_data( '_aicoso_pickup_delivery_pickup_date', sanitize_text_field( wp_unslash( $_POST['aicoso_pickup_delivery_pickup_date'] ) ) ); // phpcs:ignore WordPress.Security.NonceVerification.Missing
			}

			if ( isset( $_POST['aicoso_pickup_delivery_pickup_time_slot'] ) ) { // phpcs:ignore WordPress.Security.NonceVerification.Missing -- Nonce handled by WooCommerce checkout.
				$order->update_meta_data( '_aicoso_pickup_delivery_pickup_time_slot', sanitize_text_field( wp_unslash( $_POST['aicoso_pickup_delivery_pickup_time_slot'] ) ) ); // phpcs:ignore WordPress.Security.NonceVerification.Missing
			}

			if ( isset( $_POST['aicoso_pickup_delivery_pickup_location'] ) ) { // phpcs:ignore WordPress.Security.NonceVerification.Missing -- Nonce handled by WooCommerce checkout.
				$order->update_meta_data( '_aicoso_pickup_delivery_pickup_location', sanitize_text_field( wp_unslash( $_POST['aicoso_pickup_delivery_pickup_location'] ) ) ); // phpcs:ignore WordPress.Security.NonceVerification.Missing
			}
		}
	}

	/**
	 * Save delivery/pickup details to order meta
	 *
	 * @since    1.0.0
	 * @param    int $order_id    Order ID.
	 */
	public function save_delivery_pickup_details( $order_id ) {
		if ( ! $order_id ) {
			return;
		}

		// Get order object - compatible with HPOS.
		$order = wc_get_order( $order_id );
		if ( ! $order ) {
			return;
		}

		// Get settings.
		$settings         = AICOSO_Pickup_Delivery_Data::get_settings();
		$enable_selection = isset( $settings['enable_order_type_selection'] ) ? $settings['enable_order_type_selection'] : 'yes';

		// Get order type - check both field names for compatibility.
		$order_type = '';

		// If selection is disabled, use default order type.
		if ( 'no' === $enable_selection ) {
			// Use default order type when selection is disabled.
			// This ensures admin emails still work.
			$order_type = isset( $settings['default_order_type'] ) ? $settings['default_order_type'] : 'delivery';
		} else {
			// First check the selector field.
			// phpcs:ignore WordPress.Security.NonceVerification.Missing -- Nonce handled by WooCommerce checkout
			if ( isset( $_POST['aicoso_pickup_delivery_type_selector'] ) ) {
				// phpcs:ignore WordPress.Security.NonceVerification.Missing -- Nonce handled by WooCommerce checkout
				$order_type = sanitize_text_field( wp_unslash( $_POST['aicoso_pickup_delivery_type_selector'] ) );
			// phpcs:ignore WordPress.Security.NonceVerification.Missing -- Nonce handled by WooCommerce checkout
			} elseif ( isset( $_POST['aicoso_pickup_delivery_type'] ) ) {
				// Then check the hidden field (fallback).
				// phpcs:ignore WordPress.Security.NonceVerification.Missing -- Nonce handled by WooCommerce checkout
				$order_type = sanitize_text_field( wp_unslash( $_POST['aicoso_pickup_delivery_type'] ) );
			}

			// If order type is still empty or invalid, use default.
			if ( empty( $order_type ) || ! in_array( $order_type, array( 'delivery', 'pickup' ), true ) ) {
				$order_type = isset( $settings['default_order_type'] ) ? $settings['default_order_type'] : 'delivery';
			}
		}

		// Save order type using update_meta_data method.
		$order->update_meta_data( '_aicoso_pickup_delivery_type', $order_type );

		// Save delivery details.
		if ( 'delivery' === $order_type ) {
			if ( isset( $_POST['aicoso_pickup_delivery_delivery_date'] ) ) { // phpcs:ignore WordPress.Security.NonceVerification.Missing -- Nonce handled by WooCommerce checkout.
				$order->update_meta_data( '_aicoso_pickup_delivery_delivery_date', sanitize_text_field( wp_unslash( $_POST['aicoso_pickup_delivery_delivery_date'] ) ) ); // phpcs:ignore WordPress.Security.NonceVerification.Missing
			}

			if ( isset( $_POST['aicoso_pickup_delivery_delivery_time_slot'] ) ) { // phpcs:ignore WordPress.Security.NonceVerification.Missing -- Nonce handled by WooCommerce checkout.
				$order->update_meta_data( '_aicoso_pickup_delivery_delivery_time_slot', sanitize_text_field( wp_unslash( $_POST['aicoso_pickup_delivery_delivery_time_slot'] ) ) ); // phpcs:ignore WordPress.Security.NonceVerification.Missing
			}
		}

		// Save pickup details.
		if ( 'pickup' === $order_type ) {
			if ( isset( $_POST['aicoso_pickup_delivery_pickup_date'] ) ) { // phpcs:ignore WordPress.Security.NonceVerification.Missing -- Nonce handled by WooCommerce checkout.
				$order->update_meta_data( '_aicoso_pickup_delivery_pickup_date', sanitize_text_field( wp_unslash( $_POST['aicoso_pickup_delivery_pickup_date'] ) ) ); // phpcs:ignore WordPress.Security.NonceVerification.Missing
			}

			if ( isset( $_POST['aicoso_pickup_delivery_pickup_time_slot'] ) ) { // phpcs:ignore WordPress.Security.NonceVerification.Missing -- Nonce handled by WooCommerce checkout.
				$order->update_meta_data( '_aicoso_pickup_delivery_pickup_time_slot', sanitize_text_field( wp_unslash( $_POST['aicoso_pickup_delivery_pickup_time_slot'] ) ) ); // phpcs:ignore WordPress.Security.NonceVerification.Missing
			}

			if ( isset( $_POST['aicoso_pickup_delivery_pickup_location'] ) ) { // phpcs:ignore WordPress.Security.NonceVerification.Missing -- Nonce handled by WooCommerce checkout.
				$order->update_meta_data( '_aicoso_pickup_delivery_pickup_location', sanitize_text_field( wp_unslash( $_POST['aicoso_pickup_delivery_pickup_location'] ) ) ); // phpcs:ignore WordPress.Security.NonceVerification.Missing
			}
		}

		// Save the order.
		$order->save();
	}

	/**
	 * AJAX handler for getting available dates
	 *
	 * @since    1.0.0
	 */
	public function ajax_get_dates() {
		// Check nonce.
		check_ajax_referer( 'aicoso_pickup_delivery_nonce', 'nonce' );

		$type = isset( $_POST['type'] ) ? sanitize_text_field( wp_unslash( $_POST['type'] ) ) : 'delivery';

		if ( ! in_array( $type, array( 'delivery', 'pickup' ), true ) ) {
			wp_send_json_error( array( 'message' => __( 'Invalid order type', 'aicoso-pickup-and-delivery-for-woocommerce' ) ) );
			return;
		}

		$available_dates = AICOSO_Pickup_Delivery_Date_Manager::get_available_dates( $type );

		wp_send_json_success(
			array(
				'dates'           => $available_dates,
				'datepicker_data' => AICOSO_Pickup_Delivery_Date_Manager::get_datepicker_data( $type ),
			)
		);
	}

	/**
	 * AJAX handler for getting available time slots
	 *
	 * @since    1.0.0
	 */
	public function ajax_get_time_slots() {
		// Check nonce.
		check_ajax_referer( 'aicoso_pickup_delivery_nonce', 'nonce' );

		$type = isset( $_POST['type'] ) ? sanitize_text_field( wp_unslash( $_POST['type'] ) ) : 'delivery';
		$date = isset( $_POST['date'] ) ? sanitize_text_field( wp_unslash( $_POST['date'] ) ) : '';

		if ( ! in_array( $type, array( 'delivery', 'pickup' ), true ) ) {
			wp_send_json_error( array( 'message' => __( 'Invalid order type', 'aicoso-pickup-and-delivery-for-woocommerce' ) ) );
			return;
		}

		if ( empty( $date ) || ! AICOSO_Pickup_Delivery_Date_Manager::is_valid_date_format( $date ) ) {
			wp_send_json_error( array( 'message' => __( 'Invalid date format', 'aicoso-pickup-and-delivery-for-woocommerce' ) ) );
			return;
		}

		$available_time_slots = AICOSO_Pickup_Delivery_Timeslot_Manager::get_available_time_slots( $date, $type );

		wp_send_json_success( array( 'time_slots' => $available_time_slots ) );
	}

	/**
	 * AJAX handler for getting available pickup locations
	 *
	 * @since    1.0.0
	 */
	public function ajax_get_locations() {
		// Check nonce.
		check_ajax_referer( 'aicoso_pickup_delivery_nonce', 'nonce' );

		$date = isset( $_POST['date'] ) ? sanitize_text_field( wp_unslash( $_POST['date'] ) ) : '';

		if ( empty( $date ) || ! AICOSO_Pickup_Delivery_Date_Manager::is_valid_date_format( $date ) ) {
			wp_send_json_error( array( 'message' => __( 'Invalid date format', 'aicoso-pickup-and-delivery-for-woocommerce' ) ) );
			return;
		}

		$available_locations = AICOSO_Pickup_Delivery_Location_Manager::get_available_locations( $date );

		wp_send_json_success( array( 'locations' => $available_locations ) );
	}
}
