<?php
/**
 * The email management functionality of the plugin.
 *
 * @link       https://example.com
 * @since      1.0.0
 *
 * @package    aicoso_pickup_delivery
 */

if ( ! defined( 'WPINC' ) ) {
	die;
}

/**
 * The email manager class.
 *
 * Handles the registration and management of custom email notifications
 * for delivery and pickup orders.
 *
 * @since      1.0.0
 * @package    aicoso_pickup_delivery
 * @author     Your Name
 */
class AICOSO_Pickup_Delivery_Email_Manager {

	/**
	 * The ID of this plugin.
	 *
	 * @since    1.0.0
	 * @access   private
	 * @var      string    $plugin_name    The ID of this plugin.
	 */
	private $plugin_name;

	/**
	 * The version of this plugin.
	 *
	 * @since    1.0.0
	 * @access   private
	 * @var      string    $version    The current version of this plugin.
	 */
	private $version;

	/**
	 * Initialize the class and set its properties.
	 *
	 * @since    1.0.0
	 * @param    string $plugin_name       The name of this plugin.
	 * @param    string $version           The version of this plugin.
	 */
	public function __construct( $plugin_name, $version ) {
		$this->plugin_name = $plugin_name;
		$this->version     = $version;
	}

	/**
	 * Register custom email classes with WooCommerce
	 *
	 * @since     1.0.0
	 * @param     array $email_classes    WooCommerce email classes.
	 * @return    array                      Modified email classes
	 */
	public function register_emails( $email_classes ) {

		// Only proceed if WC_Email class exists.
		if ( ! class_exists( 'WC_Email' ) ) {
			// Try to load WooCommerce email class if not loaded yet.
			if ( defined( 'WC_ABSPATH' ) ) {
				include_once WC_ABSPATH . 'includes/emails/class-wc-email.php';
			}

			// Check again after trying to load.
			if ( ! class_exists( 'WC_Email' ) ) {
				return $email_classes;
			}
		}

		// Include email classes if not already loaded.
		if ( ! class_exists( 'AICOSO_Email_Delivery_Confirmation' ) ) {
			require_once AICOSO_PICKUP_DELIVERY_PLUGIN_PATH . 'includes/emails/class-aicoso-email-delivery-confirmation.php';
		}
		if ( ! class_exists( 'AICOSO_Email_Pickup_Confirmation' ) ) {
			require_once AICOSO_PICKUP_DELIVERY_PLUGIN_PATH . 'includes/emails/class-aicoso-email-pickup-confirmation.php';
		}
		if ( ! class_exists( 'AICOSO_Email_Admin_New_Pickup_Delivery_Order' ) ) {
			$admin_email_file = AICOSO_PICKUP_DELIVERY_PLUGIN_PATH . 'includes/emails/class-aicoso-email-admin-new-pickup-delivery-order.php';
			if ( file_exists( $admin_email_file ) ) {
				require_once $admin_email_file;
			}
		}

		// Check if emails are already registered to avoid duplicates.
		$pickup_exists   = false;
		$delivery_exists = false;
		$admin_exists    = false;

		foreach ( $email_classes as $key => $email ) {
			if ( 'AICOSO_Email_Pickup_Confirmation' === $key || ( is_object( $email ) && 'AICOSO_Email_Pickup_Confirmation' === get_class( $email ) ) ) {
				$pickup_exists = true;
			}
			if ( 'AICOSO_Email_Delivery_Confirmation' === $key || ( is_object( $email ) && 'AICOSO_Email_Delivery_Confirmation' === get_class( $email ) ) ) {
				$delivery_exists = true;
			}
			if ( 'AICOSO_Email_Admin_New_Pickup_Delivery_Order' === $key || ( is_object( $email ) && 'AICOSO_Email_Admin_New_Pickup_Delivery_Order' === get_class( $email ) ) ) {
				$admin_exists = true;
			}
		}

		// Register email classes only if not already registered.
		if ( ! $delivery_exists ) {
			$email_classes['AICOSO_Email_Delivery_Confirmation'] = new AICOSO_Email_Delivery_Confirmation();
		}
		if ( ! $pickup_exists ) {
			$email_classes['AICOSO_Email_Pickup_Confirmation'] = new AICOSO_Email_Pickup_Confirmation();
		}
		if ( ! $admin_exists && class_exists( 'AICOSO_Email_Admin_New_Pickup_Delivery_Order' ) ) {
			$email_classes['AICOSO_Email_Admin_New_Pickup_Delivery_Order'] = new AICOSO_Email_Admin_New_Pickup_Delivery_Order();
		}

		return $email_classes;
	}

	/**
	 * Add delivery/pickup details to order emails
	 *
	 * @since     1.0.0
	 * @param     WC_Order $order            The order object.
	 * @param     bool     $sent_to_admin    Whether the email is being sent to admin.
	 * @param     bool     $plain_text       Whether the email is plain text.
	 */
	public function add_delivery_pickup_details_to_emails( $order, $sent_to_admin, $plain_text ) {
		if ( ! $order instanceof WC_Order ) {
			return;
		}

		$order_type = AICOSO_Pickup_Delivery_Data::get_order_type( $order );

		// Only proceed if this is a delivery or pickup order.
		if ( ! in_array( $order_type, array( 'delivery', 'pickup' ), true ) ) {
			return;
		}

		// Load the appropriate template based on order type.
		if ( 'delivery' === $order_type ) {
			$template = $plain_text ? 'emails/plain/delivery-details.php' : 'emails/delivery-details.php';
		} else {
			$template = $plain_text ? 'emails/plain/pickup-details.php' : 'emails/pickup-details.php';
		}

		wc_get_template(
			$template,
			array( 'order' => $order ),
			'aicoso-pickup-and-delivery-for-woocommerce/',
			AICOSO_PICKUP_DELIVERY_PLUGIN_PATH . 'templates/'
		);
	}

	/**
	 * Send notification when order details are updated
	 *
	 * @since     1.0.0
	 * @param     int    $order_id        The order ID.
	 * @param     string $old_order_type  The previous order type.
	 * @param     string $new_order_type  The new order type.
	 */
	public function send_update_notification( $order_id, $old_order_type, $new_order_type ) {
		$order = wc_get_order( $order_id );

		if ( ! $order ) {
			return;
		}

		// Get settings.
		$settings = AICOSO_Pickup_Delivery_Data::get_settings();

		if ( 'delivery' === $new_order_type ) {
			// Check if delivery confirmation email is enabled.
			$enable_email = isset( $settings['enable_delivery_confirmation_email'] ) ? $settings['enable_delivery_confirmation_email'] : 'yes';

			if ( 'yes' === $enable_email ) {

				// Trigger delivery confirmation email.
				do_action( 'aicoso_pickup_delivery_delivery_details_updated', $order_id, $order );
			}
		} elseif ( 'pickup' === $new_order_type ) {
			// Check if pickup confirmation email is enabled.
			$enable_email = isset( $settings['enable_pickup_confirmation_email'] ) ? $settings['enable_pickup_confirmation_email'] : 'yes';

			if ( 'yes' === $enable_email ) {
				// Trigger pickup confirmation email.
				do_action( 'aicoso_pickup_delivery_pickup_details_updated', $order_id, $order );
			}
		}

		// Send admin notification if enabled.
		$enable_admin_email = isset( $settings['enable_admin_notification_email'] ) ? $settings['enable_admin_notification_email'] : 'yes';

		if ( 'yes' === $enable_admin_email ) {
			$this->send_admin_update_notification( $order_id, $old_order_type, $new_order_type );
		}
	}

	/**
	 * Send notification to admin for new pickup/delivery orders
	 *
	 * @deprecated This method is deprecated. Admin notifications are now handled by AICOSO_Email_Admin_New_Pickup_Delivery_Order class.
	 * @since     1.0.0
	 * @param     int      $order_id    The order ID.
	 * @param     WC_Order $order       The order object.
	 */
	public function send_admin_new_order_notification( $order_id, $order = null ) {
		// phpcs:ignore Generic.CodeAnalysis.UnusedFunctionParameter.FoundAfterLastUsed -- Deprecated method kept for backwards compatibility.
		// This method is deprecated and no longer used.
		// Admin notifications are now handled by AICOSO_Email_Admin_New_Pickup_Delivery_Order class.
	}

	/**
	 * Send notification to admin when order details are updated
	 *
	 * @since     1.0.0
	 * @param     int    $order_id        The order ID.
	 * @param     string $old_order_type  The previous order type.
	 * @param     string $new_order_type  The new order type.
	 */
	private function send_admin_update_notification( $order_id, $old_order_type, $new_order_type ) {
		$order = wc_get_order( $order_id );

		if ( ! $order ) {
			return;
		}

		// Get admin email.
		$admin_email = get_option( 'admin_email' );

		// If this is a pickup order, check if there's a location-specific email.
		if ( 'pickup' === $new_order_type ) {
			$pickup_location = $order->get_meta( '_aicoso_pickup_delivery_pickup_location', true );

			if ( ! empty( $pickup_location ) && class_exists( 'AICOSO_Pickup_Delivery_Location_Manager' ) ) {
				$location_email = AICOSO_Pickup_Delivery_Location_Manager::get_location_email( $pickup_location );

				if ( ! empty( $location_email ) ) {
					$admin_email = $location_email;
				}
			}
		}

		// Create email subject.
		$subject = sprintf(
			/* translators: 1: Site name, 2: Order number, 3: Old order type, 4: New order type */
			__( '[%1$s] Order #%2$s - Changed from %3$s to %4$s', 'aicoso-pickup-and-delivery-for-woocommerce' ),
			get_bloginfo( 'name' ),
			$order->get_order_number(),
			'delivery' === $old_order_type ? __( 'Delivery', 'aicoso-pickup-and-delivery-for-woocommerce' ) : __( 'Pickup', 'aicoso-pickup-and-delivery-for-woocommerce' ),
			'delivery' === $new_order_type ? __( 'Delivery', 'aicoso-pickup-and-delivery-for-woocommerce' ) : __( 'Pickup', 'aicoso-pickup-and-delivery-for-woocommerce' )
		);

		$message = sprintf(
			/* translators: 1: Order number, 2: Old order type, 3: New order type */
			__( 'Order #%1$s has been changed from %2$s to %3$s.', 'aicoso-pickup-and-delivery-for-woocommerce' ),
			$order->get_order_number(),
			'delivery' === $old_order_type ? __( 'Delivery', 'aicoso-pickup-and-delivery-for-woocommerce' ) : __( 'Pickup', 'aicoso-pickup-and-delivery-for-woocommerce' ),
			'delivery' === $new_order_type ? __( 'Delivery', 'aicoso-pickup-and-delivery-for-woocommerce' ) : __( 'Pickup', 'aicoso-pickup-and-delivery-for-woocommerce' )
		) . "\n\n";

		// Add details based on new order type.
		if ( 'delivery' === $new_order_type ) {
			$delivery_date = $order->get_meta( '_aicoso_pickup_delivery_delivery_date', true );
			$delivery_time = $order->get_meta( '_aicoso_pickup_delivery_delivery_time_slot', true );

			if ( $delivery_date ) {
				$message .= __( 'Delivery Date:', 'aicoso-pickup-and-delivery-for-woocommerce' ) . ' ' . AICOSO_Pickup_Delivery_Data::format_date( $delivery_date ) . "\n";
			}

			if ( $delivery_time ) {
				$message .= __( 'Delivery Time:', 'aicoso-pickup-and-delivery-for-woocommerce' ) . ' ' . $delivery_time . "\n";
			}
		} elseif ( 'pickup' === $new_order_type ) {
			$pickup_date     = $order->get_meta( '_aicoso_pickup_delivery_pickup_date', true );
			$pickup_time     = $order->get_meta( '_aicoso_pickup_delivery_pickup_time_slot', true );
			$pickup_location = $order->get_meta( '_aicoso_pickup_delivery_pickup_location', true );

			if ( $pickup_date ) {
				$message .= __( 'Pickup Date:', 'aicoso-pickup-and-delivery-for-woocommerce' ) . ' ' . AICOSO_Pickup_Delivery_Data::format_date( $pickup_date ) . "\n";
			}

			if ( $pickup_time ) {
				$message .= __( 'Pickup Time:', 'aicoso-pickup-and-delivery-for-woocommerce' ) . ' ' . $pickup_time . "\n";
			}

			if ( $pickup_location ) {
				$message .= __( 'Pickup Location:', 'aicoso-pickup-and-delivery-for-woocommerce' ) . ' ' . $pickup_location . "\n";
			}
		}

		// Add order link.
		$message .= "\n" . __( 'View Order:', 'aicoso-pickup-and-delivery-for-woocommerce' ) . ' ' . admin_url( 'post.php?post=' . $order_id . '&action=edit' );

		// Send email.
		wp_mail( $admin_email, $subject, $message );
	}
}
