<?php
/**
 * Class AICOSO_Email_Pickup_Confirmation
 *
 * @package    aicoso_pickup_delivery
 * @version    1.0.0
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

// Make sure WooCommerce is active and WC_Email class exists before proceeding.
if ( ! class_exists( 'WC_Email' ) ) {
	return;
}

/**
 * Pickup Confirmation Email.
 *
 * An email sent to the customer when a pickup order is processed.
 *
 * @class       AICOSO_Email_Pickup_Confirmation
 * @extends     WC_Email
 */
class AICOSO_Email_Pickup_Confirmation extends WC_Email {

	/**
	 * Track sent emails to prevent duplicates
	 *
	 * @var array
	 */
	private static $sent_emails = array();

	/**
	 * Constructor.
	 */
	public function __construct() {
		$this->id             = 'pickup_confirmation';
		$this->customer_email = true;
		$this->title          = __( 'Pickup Confirmation', 'aicoso-pickup-and-delivery-for-woocommerce' );
		$this->description    = __( 'This email is sent to the customer when a pickup order is processed.', 'aicoso-pickup-and-delivery-for-woocommerce' );
		$this->template_html  = 'emails/pickup-confirmation.php';
		$this->template_plain = 'emails/plain/pickup-confirmation.php';
		$this->template_base  = AICOSO_PICKUP_DELIVERY_PLUGIN_PATH . 'templates/';
		$this->placeholders   = array(
			'{site_title}'   => $this->get_blogname(),
			'{order_date}'   => '',
			'{order_number}' => '',
		);

		// Call parent constructor.
		parent::__construct();

		// Triggers for this email - removed status change hooks to prevent duplicates.
		// Only trigger on custom plugin action and new order creation.

		// Custom hook for our plugin - for manual triggering or updates.
		add_action( 'aicoso_pickup_delivery_pickup_details_updated', array( $this, 'trigger' ), 10, 2 );

		// Hook into new order creation - only use one hook to prevent duplicates.
		add_action( 'woocommerce_checkout_order_processed', array( $this, 'maybe_trigger_for_new_order' ), 10, 3 );
	}

	/**
	 * Get email subject.
	 *
	 * @since  1.0.0
	 * @return string
	 */
	public function get_default_subject() {
		return __( 'Your {site_title} pickup order is confirmed - Order #{order_number}', 'aicoso-pickup-and-delivery-for-woocommerce' );
	}

	/**
	 * Get email heading.
	 *
	 * @since  1.0.0
	 * @return string
	 */
	public function get_default_heading() {
		return __( 'Your pickup order is confirmed', 'aicoso-pickup-and-delivery-for-woocommerce' );
	}

	/**
	 * Check if this is a pickup order and trigger email
	 *
	 * @param int      $order_id Order ID.
	 * @param WC_Order $order    Order object.
	 */
	public function check_and_trigger( $order_id, $order = null ) {
		if ( ! $order ) {
			$order = wc_get_order( $order_id );
		}

		if ( ! $order ) {
			return;
		}

		// Check if this is a pickup order.
		$order_type = $order->get_meta( '_aicoso_pickup_delivery_type', true );

		if ( 'pickup' === $order_type ) {
			// Check plugin settings before triggering.
			$plugin_settings = get_option( 'aicoso_pickup_delivery_settings', array() );

			// Check if order type selection is enabled.
			$order_type_selection_enabled = isset( $plugin_settings['enable_order_type_selection'] ) ? $plugin_settings['enable_order_type_selection'] : 'yes';

			// Only send email if order type selection is enabled.
			if ( 'yes' !== $order_type_selection_enabled ) {
				return;
			}

			// Check if pickup confirmation email is enabled.
			$plugin_enabled = isset( $plugin_settings['enable_pickup_confirmation_email'] ) ? $plugin_settings['enable_pickup_confirmation_email'] : 'yes';

			if ( 'yes' === $plugin_enabled ) {
				$this->trigger( $order_id, $order );
			}
		}
	}

	/**
	 * Maybe trigger for new order
	 *
	 * @param int $order_id Order ID.
	 */
	public function maybe_trigger_for_new_order_id( $order_id ) {
		$order = wc_get_order( $order_id );
		if ( $order ) {
			$this->maybe_trigger_for_new_order( $order_id, array(), $order );
		}
	}

	/**
	 * Maybe trigger email for new orders
	 *
	 * @param int      $order_id Order ID.
	 * @param array    $posted_data Posted data.
	 * @param WC_Order $order Order object.
	 */
	public function maybe_trigger_for_new_order( $order_id, $posted_data = array(), $order = null ) {
		if ( ! $order ) {
			$order = wc_get_order( $order_id );
		}

		if ( ! $order ) {
			return;
		}

		// Check if this is a pickup order.
		$order_type = $order->get_meta( '_aicoso_pickup_delivery_type', true );

		// Get plugin settings.
		$settings = get_option( 'aicoso_pickup_delivery_settings', array() );

		// Check if order type selection is enabled.
		$order_type_selection_enabled = isset( $settings['enable_order_type_selection'] ) ? $settings['enable_order_type_selection'] : 'yes';

		// Only send email if order type selection is enabled.
		if ( 'yes' !== $order_type_selection_enabled ) {
			return;
		}

		$send_on_new_order = isset( $settings['send_pickup_email_on_new_order'] ) ? $settings['send_pickup_email_on_new_order'] : 'yes';

		if ( 'pickup' === $order_type && 'yes' === $send_on_new_order ) {
			$this->trigger( $order_id, $order );
		}
	}

	/**
	 * Trigger the sending of this email.
	 *
	 * @param int      $order_id The order ID.
	 * @param WC_Order $order    Order object.
	 */
	public function trigger( $order_id, $order = false ) {
		// Prevent duplicate emails.
		$email_key = 'pickup_' . $order_id;
		if ( isset( self::$sent_emails[ $email_key ] ) ) {
			return;
		}

		$this->setup_locale();

		if ( $order_id && ! is_a( $order, 'WC_Order' ) ) {
			$order = wc_get_order( $order_id );
		}

		if ( is_a( $order, 'WC_Order' ) ) {
			$this->object                         = $order;
			$this->recipient                      = $this->object->get_billing_email();
			$this->placeholders['{order_date}']   = wc_format_datetime( $this->object->get_date_created() );
			$this->placeholders['{order_number}'] = $this->object->get_order_number();

			// Get pickup details.
			$pickup_date     = $order->get_meta( '_aicoso_pickup_delivery_pickup_date', true );
			$pickup_time     = $order->get_meta( '_aicoso_pickup_delivery_pickup_time_slot', true );
			$pickup_location = $order->get_meta( '_aicoso_pickup_delivery_pickup_location', true );

			// Add custom placeholders.
			$this->placeholders['{pickup_date}']     = $pickup_date ? $pickup_date : __( 'To be confirmed', 'aicoso-pickup-and-delivery-for-woocommerce' );
			$this->placeholders['{pickup_time}']     = $pickup_time ? $pickup_time : __( 'To be confirmed', 'aicoso-pickup-and-delivery-for-woocommerce' );
			$this->placeholders['{pickup_location}'] = $pickup_location ? $pickup_location : __( 'To be confirmed', 'aicoso-pickup-and-delivery-for-woocommerce' );
		}

		// Check both WooCommerce email setting and plugin setting.
		$plugin_settings = get_option( 'aicoso_pickup_delivery_settings', array() );
		$plugin_enabled  = isset( $plugin_settings['enable_pickup_confirmation_email'] ) ? $plugin_settings['enable_pickup_confirmation_email'] : 'yes';

		if ( $this->is_enabled() && 'yes' === $plugin_enabled && $this->get_recipient() ) {
			$this->send( $this->get_recipient(), $this->get_subject(), $this->get_content(), $this->get_headers(), $this->get_attachments() );

			// Mark email as sent to prevent duplicates.
			self::$sent_emails[ $email_key ] = true;
		}

		$this->restore_locale();
	}

	/**
	 * Get content html.
	 *
	 * @return string
	 */
	public function get_content_html() {
		return wc_get_template_html(
			$this->template_html,
			array(
				'order'              => $this->object,
				'email_heading'      => $this->get_heading(),
				'additional_content' => $this->get_additional_content(),
				'sent_to_admin'      => false,
				'plain_text'         => false,
				'email'              => $this,
			),
			'',
			$this->template_base
		);
	}

	/**
	 * Get content plain.
	 *
	 * @return string
	 */
	public function get_content_plain() {
		return wc_get_template_html(
			$this->template_plain,
			array(
				'order'              => $this->object,
				'email_heading'      => $this->get_heading(),
				'additional_content' => $this->get_additional_content(),
				'sent_to_admin'      => false,
				'plain_text'         => true,
				'email'              => $this,
			),
			'',
			$this->template_base
		);
	}

	/**
	 * Get additional content.
	 *
	 * @return string
	 */
	public function get_additional_content() {
		$content = $this->get_option( 'additional_content', '' );

		if ( $content ) {
			$content = $this->format_string( $content );
		}

		return $content;
	}

	/**
	 * Initialize settings form fields.
	 */
	public function init_form_fields() {
		$this->form_fields = array(
			'enabled'            => array(
				'title'   => __( 'Enable/Disable', 'aicoso-pickup-and-delivery-for-woocommerce' ),
				'type'    => 'checkbox',
				'label'   => __( 'Enable this email notification', 'aicoso-pickup-and-delivery-for-woocommerce' ),
				'default' => 'yes',
			),
			'subject'            => array(
				'title'       => __( 'Subject', 'aicoso-pickup-and-delivery-for-woocommerce' ),
				'type'        => 'text',
				'desc_tip'    => true,
				/* translators: %1$s: string placeholder */
				'description' => sprintf( __( 'Available placeholders: %s', 'aicoso-pickup-and-delivery-for-woocommerce' ), '<code>{site_title}, {order_date}, {order_number}, {pickup_date}, {pickup_time}, {pickup_location}</code>' ),
				'placeholder' => $this->get_default_subject(),
				'default'     => $this->get_default_subject(),
			),
			'heading'            => array(
				'title'       => __( 'Email heading', 'aicoso-pickup-and-delivery-for-woocommerce' ),
				'type'        => 'text',
				'desc_tip'    => true,
				/* translators: %1$s: string placeholder */
				'description' => sprintf( __( 'Available placeholders: %s', 'aicoso-pickup-and-delivery-for-woocommerce' ), '<code>{site_title}, {order_date}, {order_number}, {pickup_date}, {pickup_time}, {pickup_location}</code>' ),
				'placeholder' => $this->get_default_heading(),
				'default'     => $this->get_default_heading(),
			),
			'additional_content' => array(
				'title'       => __( 'Additional content', 'aicoso-pickup-and-delivery-for-woocommerce' ),
				/* translators: %1$s: string placeholder */
				'description' => __( 'Text to appear below the main email content.', 'aicoso-pickup-and-delivery-for-woocommerce' ) . ' ' . sprintf( __( 'Available placeholders: %s', 'aicoso-pickup-and-delivery-for-woocommerce' ), '<code>{site_title}, {order_date}, {order_number}, {pickup_date}, {pickup_time}, {pickup_location}</code>' ),
				'css'         => 'width:400px; height: 75px;',
				'placeholder' => __( 'N/A', 'aicoso-pickup-and-delivery-for-woocommerce' ),
				'type'        => 'textarea',
				'default'     => '',
				'desc_tip'    => true,
			),
			'email_type'         => array(
				'title'       => __( 'Email type', 'aicoso-pickup-and-delivery-for-woocommerce' ),
				'type'        => 'select',
				'description' => __( 'Choose which format of email to send.', 'aicoso-pickup-and-delivery-for-woocommerce' ),
				'default'     => 'html',
				'class'       => 'email_type wc-enhanced-select',
				'options'     => $this->get_email_type_options(),
				'desc_tip'    => true,
			),
		);
	}
}
