<?php
/**
 * Admin New Pickup/Delivery Order Email
 *
 * @package Aicoso_Pickup_Delivery_For_Woocommerce
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Admin New Pickup/Delivery Order Email class
 */
class AICOSO_Email_Admin_New_Pickup_Delivery_Order extends WC_Email {

	/**
	 * Constructor
	 */
	public function __construct() {
		$this->id             = 'admin_new_pickup_delivery_order';
		$this->title          = __( 'Admin New Pickup/Delivery Order', 'aicoso-pickup-and-delivery-for-woocommerce' );
		$this->description    = __( 'Admin new order emails are sent to chosen recipient(s) when a new pickup or delivery order is received.', 'aicoso-pickup-and-delivery-for-woocommerce' );
		$this->template_html  = 'emails/admin-new-pickup-delivery-order.php';
		$this->template_plain = 'emails/plain/admin-new-pickup-delivery-order.php';
		$this->template_base  = AICOSO_PICKUP_DELIVERY_PLUGIN_PATH . 'templates/';
		$this->placeholders   = array(
			'{order_date}'   => '',
			'{order_number}' => '',
			'{order_type}'   => '',
		);

		// Enable by default.
		$this->enabled = 'yes';

		// Triggers for new orders.
		add_action( 'woocommerce_order_status_pending_to_processing_notification', array( $this, 'trigger' ), 10, 2 );
		add_action( 'woocommerce_order_status_pending_to_completed_notification', array( $this, 'trigger' ), 10, 2 );
		add_action( 'woocommerce_order_status_pending_to_on-hold_notification', array( $this, 'trigger' ), 10, 2 );
		add_action( 'woocommerce_order_status_failed_to_processing_notification', array( $this, 'trigger' ), 10, 2 );
		add_action( 'woocommerce_order_status_failed_to_completed_notification', array( $this, 'trigger' ), 10, 2 );
		add_action( 'woocommerce_order_status_failed_to_on-hold_notification', array( $this, 'trigger' ), 10, 2 );
		add_action( 'woocommerce_order_status_cancelled_to_processing_notification', array( $this, 'trigger' ), 10, 2 );
		add_action( 'woocommerce_order_status_cancelled_to_completed_notification', array( $this, 'trigger' ), 10, 2 );
		add_action( 'woocommerce_order_status_cancelled_to_on-hold_notification', array( $this, 'trigger' ), 10, 2 );

		// Call parent constructor.
		parent::__construct();

		// Force enable if not explicitly set (for backwards compatibility).
		if ( empty( $this->enabled ) ) {
			$this->enabled = 'yes';
		}

		// Other settings.
		$this->recipient = $this->get_option( 'recipient', get_option( 'admin_email' ) );
	}

	/**
	 * Get email subject.
	 *
	 * @return string
	 */
	public function get_default_subject() {
		return __( '[{site_title}]: New {order_type} order #{order_number}', 'aicoso-pickup-and-delivery-for-woocommerce' );
	}

	/**
	 * Get email heading.
	 *
	 * @return string
	 */
	public function get_default_heading() {
		return __( 'New {order_type} Order', 'aicoso-pickup-and-delivery-for-woocommerce' );
	}

	/**
	 * Trigger the sending of this email.
	 *
	 * @param int      $order_id Order ID.
	 * @param WC_Order $order Order object.
	 */
	public function trigger( $order_id, $order = false ) {
		$this->setup_locale();

		if ( $order_id && ! is_a( $order, 'WC_Order' ) ) {
			$order = wc_get_order( $order_id );
		}

		if ( is_a( $order, 'WC_Order' ) ) {
			$this->object = $order;

			// Check if order has pickup/delivery type.
			$order_type = $order->get_meta( '_aicoso_pickup_delivery_type', true );

			// If no order type is set, check settings for default.
			if ( empty( $order_type ) ) {
				$settings   = get_option( 'aicoso_pickup_delivery_settings', array() );
				$order_type = isset( $settings['default_order_type'] ) ? $settings['default_order_type'] : 'delivery';

				// Save the order type for future reference.
				$order->update_meta_data( '_aicoso_pickup_delivery_type', $order_type );
				$order->save();
			}

			// Only send email for valid pickup/delivery orders.
			if ( ! in_array( $order_type, array( 'pickup', 'delivery' ), true ) ) {
				$this->restore_locale();
				return;
			}

			// Check backend settings to see if admin notification is enabled.
			$settings                  = get_option( 'aicoso_pickup_delivery_settings', array() );
			$enable_admin_notification = isset( $settings['enable_admin_notification_email'] ) ? $settings['enable_admin_notification_email'] : 'yes';

			// Only check plugin settings if explicitly disabled.
			// This allows the email to work by default even if settings haven't been configured.
			if ( 'no' === $enable_admin_notification ) {
				$this->restore_locale();
				return;
			}

			$this->placeholders['{order_date}']   = wc_format_datetime( $order->get_date_created() );
			$this->placeholders['{order_number}'] = $order->get_order_number();
			$this->placeholders['{order_type}']   = ucfirst( $order_type );
		}

		// For admin notifications, send if we have a recipient.
		// The plugin settings control whether to send, not the WooCommerce email settings.
		if ( $this->get_recipient() ) {
			$this->send( $this->get_recipient(), $this->get_subject(), $this->get_content(), $this->get_headers(), $this->get_attachments() );
		}

		$this->restore_locale();
	}

	/**
	 * Get content html.
	 *
	 * @return string
	 */
	public function get_content_html() {
		return wc_get_template_html(
			$this->template_html,
			array(
				'order'              => $this->object,
				'email_heading'      => $this->get_heading(),
				'additional_content' => $this->get_additional_content(),
				'sent_to_admin'      => true,
				'plain_text'         => false,
				'email'              => $this,
			),
			'',
			$this->template_base
		);
	}

	/**
	 * Get content plain.
	 *
	 * @return string
	 */
	public function get_content_plain() {
		return wc_get_template_html(
			$this->template_plain,
			array(
				'order'              => $this->object,
				'email_heading'      => $this->get_heading(),
				'additional_content' => $this->get_additional_content(),
				'sent_to_admin'      => true,
				'plain_text'         => true,
				'email'              => $this,
			),
			'',
			$this->template_base
		);
	}

	/**
	 * Default content to show below main email content.
	 *
	 * @return string
	 */
	public function get_default_additional_content() {
		return __( 'Please process this order as soon as possible.', 'aicoso-pickup-and-delivery-for-woocommerce' );
	}

	/**
	 * Initialize settings form fields.
	 */
	public function init_form_fields() {
		/* translators: %s: List of available placeholder codes */
		$placeholder_text = sprintf( __( 'Available placeholders: %s', 'aicoso-pickup-and-delivery-for-woocommerce' ), '<code>' . esc_html( implode( '</code>, <code>', array_keys( $this->placeholders ) ) ) . '</code>' );

		$this->form_fields = array(
			'enabled'            => array(
				'title'   => __( 'Enable/Disable', 'aicoso-pickup-and-delivery-for-woocommerce' ),
				'type'    => 'checkbox',
				'label'   => __( 'Enable this email notification', 'aicoso-pickup-and-delivery-for-woocommerce' ),
				'default' => 'yes',
			),
			'recipient'          => array(
				'title'       => __( 'Recipient(s)', 'aicoso-pickup-and-delivery-for-woocommerce' ),
				'type'        => 'text',
				/* translators: %s: Admin email address */
				'description' => sprintf( __( 'Enter recipients (comma separated) for this email. Defaults to %s.', 'aicoso-pickup-and-delivery-for-woocommerce' ), '<code>' . esc_attr( get_option( 'admin_email' ) ) . '</code>' ),
				'placeholder' => '',
				'default'     => '',
				'desc_tip'    => true,
			),
			'subject'            => array(
				'title'       => __( 'Subject', 'aicoso-pickup-and-delivery-for-woocommerce' ),
				'type'        => 'text',
				'desc_tip'    => true,
				'description' => $placeholder_text,
				'placeholder' => $this->get_default_subject(),
				'default'     => '',
			),
			'heading'            => array(
				'title'       => __( 'Email heading', 'aicoso-pickup-and-delivery-for-woocommerce' ),
				'type'        => 'text',
				'desc_tip'    => true,
				'description' => $placeholder_text,
				'placeholder' => $this->get_default_heading(),
				'default'     => '',
			),
			'additional_content' => array(
				'title'       => __( 'Additional content', 'aicoso-pickup-and-delivery-for-woocommerce' ),
				'description' => __( 'Text to appear below the main email content.', 'aicoso-pickup-and-delivery-for-woocommerce' ) . ' ' . $placeholder_text,
				'css'         => 'width:400px; height: 75px;',
				'placeholder' => __( 'N/A', 'aicoso-pickup-and-delivery-for-woocommerce' ),
				'type'        => 'textarea',
				'default'     => $this->get_default_additional_content(),
				'desc_tip'    => true,
			),
			'email_type'         => array(
				'title'       => __( 'Email type', 'aicoso-pickup-and-delivery-for-woocommerce' ),
				'type'        => 'select',
				'description' => __( 'Choose which format of email to send.', 'aicoso-pickup-and-delivery-for-woocommerce' ),
				'default'     => 'html',
				'class'       => 'email_type wc-enhanced-select',
				'options'     => $this->get_email_type_options(),
				'desc_tip'    => true,
			),
		);
	}
}
