<?php
/**
 * The time slot management functionality of the plugin.
 *
 * @link       https://example.com
 * @since      1.0.0
 *
 * @package    aicoso_pickup_delivery
 */

if ( ! defined( 'WPINC' ) ) {
	die;
}

/**
 * The time slot management class.
 *
 * Handles time slot availability, validation, and time slot operations.
 *
 * @since      1.0.0
 * @package    aicoso_pickup_delivery
 * @author     Your Name
 */
class AICOSO_Pickup_Delivery_Timeslot_Manager {

	/**
	 * Get available time slots for a specific date and service type
	 *
	 * @since     1.0.0
	 * @param     string $date           Date in Y-m-d format.
	 * @param     string $type           'delivery' or 'pickup'.
	 * @return    array                     Array of available time slots
	 */
	public static function get_available_time_slots( $date, $type = 'delivery' ) {
		if ( ! AICOSO_Pickup_Delivery_Date_Manager::is_valid_date_format( $date ) ) {
			return array();
		}

		// Get settings.
		$settings          = AICOSO_Pickup_Delivery_Data::get_settings();
		$enable_time_slots = isset( $settings[ 'enable_' . $type . '_time_slots' ] ) ? $settings[ 'enable_' . $type . '_time_slots' ] : 'yes';

		// If time slots are disabled, return empty array.
		if ( 'yes' !== $enable_time_slots ) {
			return array();
		}

		// Get all time slots for this type.
		$all_time_slots = ( 'delivery' === $type )
			? AICOSO_Pickup_Delivery_Data::get_delivery_time_slots()
			: AICOSO_Pickup_Delivery_Data::get_pickup_time_slots();

		$available_slots = array();

		foreach ( $all_time_slots as $slot ) {
			$slot_capacity = isset( $slot['max_orders'] ) ? absint( $slot['max_orders'] ) : 10;
			$slot_name     = isset( $slot['slot_name'] ) ? sanitize_text_field( $slot['slot_name'] ) : '';

			// Skip if no slot name.
			if ( empty( $slot_name ) ) {
				continue;
			}

			// Check if we have reached the maximum orders for this time slot.
			$orders_count = AICOSO_Pickup_Delivery_Data::count_orders_for_time_slot( $date, $type, $slot_name );

			if ( $orders_count >= $slot_capacity ) {
				continue;
			}

			// Time slot is available.
			$available_slots[] = array(
				'name'               => $slot_name,
				'start_time'         => isset( $slot['start_time'] ) ? $slot['start_time'] : '',
				'end_time'           => isset( $slot['end_time'] ) ? $slot['end_time'] : '',
				'formatted_time'     => self::format_time_slot( $slot ),
				'remaining_capacity' => $slot_capacity - $orders_count,
			);
		}

		return $available_slots;
	}

	/**
	 * Check if a specific time slot is available for a date and service type
	 *
	 * @since     1.0.0
	 * @param     string $date           Date in Y-m-d format.
	 * @param     string $time_slot      Time slot name.
	 * @param     string $type           'delivery' or 'pickup'.
	 * @return    boolean                   Whether the time slot is available
	 */
	public static function is_time_slot_available( $date, $time_slot, $type = 'delivery' ) {
		// Check if the date is available first.
		if ( ! AICOSO_Pickup_Delivery_Date_Manager::is_date_available( $date, $type ) ) {
			return false;
		}

		// Get settings.
		$settings          = AICOSO_Pickup_Delivery_Data::get_settings();
		$enable_time_slots = isset( $settings[ 'enable_' . $type . '_time_slots' ] ) ? $settings[ 'enable_' . $type . '_time_slots' ] : 'yes';

		// If time slots are disabled.
		if ( 'yes' !== $enable_time_slots ) {
			return false;
		}

		// Get time slot data.
		$time_slot_data = self::get_time_slot_by_name( $time_slot, $type );

		if ( empty( $time_slot_data ) ) {
			return false;
		}

		$slot_capacity = isset( $time_slot_data['max_orders'] ) ? absint( $time_slot_data['max_orders'] ) : 10;

		// Check if we've reached the maximum number of orders for this time slot.
		$orders_count = AICOSO_Pickup_Delivery_Data::count_orders_for_time_slot( $date, $type, $time_slot );

		if ( $orders_count >= $slot_capacity ) {
			return false;
		}

		return true;
	}

	/**
	 * Get a time slot by its name
	 *
	 * @since     1.0.0
	 * @param     string $slot_name      Time slot name.
	 * @param     string $type           'delivery' or 'pickup'.
	 * @return    array                     Time slot data or empty array if not found
	 */
	public static function get_time_slot_by_name( $slot_name, $type = 'delivery' ) {
		// Get all time slots for this type.
		$all_time_slots = ( 'delivery' === $type )
			? AICOSO_Pickup_Delivery_Data::get_delivery_time_slots()
			: AICOSO_Pickup_Delivery_Data::get_pickup_time_slots();

		foreach ( $all_time_slots as $slot ) {
			if ( isset( $slot['slot_name'] ) && $slot_name === $slot['slot_name'] ) {
				return $slot;
			}
		}

		return array();
	}

	/**
	 * Format a time slot for display
	 *
	 * @since     1.0.0
	 * @param     array $slot           Time slot data.
	 * @return    string                    Formatted time slot text
	 */
	public static function format_time_slot( $slot ) {
		if ( ! isset( $slot['slot_name'] ) ) {
			return '';
		}

		// If the slot name already includes formatted times, just return it.
		if ( isset( $slot['slot_name'] ) && ! empty( $slot['slot_name'] ) ) {
			return $slot['slot_name'];
		}

		// Otherwise, format the start and end times.
		$start_time = isset( $slot['start_time'] ) ? $slot['start_time'] : '';
		$end_time   = isset( $slot['end_time'] ) ? $slot['end_time'] : '';

		if ( empty( $start_time ) || empty( $end_time ) ) {
			return '';
		}

		$formatted_start = AICOSO_Pickup_Delivery_Data::format_time( $start_time );
		$formatted_end   = AICOSO_Pickup_Delivery_Data::format_time( $end_time );

		return sprintf( /* translators: %s: time slot name */
			__( '%1$s - %2$s', 'aicoso-pickup-and-delivery-for-woocommerce' ),
			$formatted_start,
			$formatted_end
		);
	}

	/**
	 * Add a new time slot
	 *
	 * @since     1.0.0
	 * @param     array  $slot_data      Time slot data.
	 * @param     string $type           'delivery' or 'pickup'.
	 * @return    boolean                   Success or failure
	 */
	public static function add_time_slot( $slot_data, $type = 'delivery' ) {
		// Validate required fields.
		if ( ! isset( $slot_data['slot_name'] ) || empty( $slot_data['slot_name'] ) ) {
			return false;
		}

		if ( ! isset( $slot_data['start_time'] ) || empty( $slot_data['start_time'] ) ) {
			return false;
		}

		if ( ! isset( $slot_data['end_time'] ) || empty( $slot_data['end_time'] ) ) {
			return false;
		}

		// Get all time slots for this type.
		$time_slots = ( 'delivery' === $type )
			? AICOSO_Pickup_Delivery_Data::get_delivery_time_slots()
			: AICOSO_Pickup_Delivery_Data::get_pickup_time_slots();

		// Check if slot name already exists.
		foreach ( $time_slots as $slot ) {
			if ( $slot_data['slot_name'] === $slot['slot_name'] ) {
				return false;
			}
		}

		// Add new slot.
		$time_slots[] = array(
			'slot_name'  => sanitize_text_field( $slot_data['slot_name'] ),
			'start_time' => sanitize_text_field( $slot_data['start_time'] ),
			'end_time'   => sanitize_text_field( $slot_data['end_time'] ),
			'max_orders' => isset( $slot_data['max_orders'] ) ? absint( $slot_data['max_orders'] ) : 10,
		);

		// Sort time slots by start time.
		usort( $time_slots, array( __CLASS__, 'sort_time_slots' ) );

		// Save time slots.
		$option_name = ( 'delivery' === $type )
			? 'aicoso_pickup_delivery_delivery_time_slots'
			: 'aicoso_pickup_delivery_pickup_time_slots';

		return update_option( $option_name, $time_slots );
	}

	/**
	 * Update an existing time slot
	 *
	 * @since     1.0.0
	 * @param     string $slot_name      Original time slot name to update.
	 * @param     array  $slot_data      New time slot data.
	 * @param     string $type           'delivery' or 'pickup'.
	 * @return    boolean                   Success or failure
	 */
	public static function update_time_slot( $slot_name, $slot_data, $type = 'delivery' ) {
		// Validate required fields.
		if ( empty( $slot_name ) ) {
			return false;
		}

		if ( ! isset( $slot_data['slot_name'] ) || empty( $slot_data['slot_name'] ) ) {
			return false;
		}

		if ( ! isset( $slot_data['start_time'] ) || empty( $slot_data['start_time'] ) ) {
			return false;
		}

		if ( ! isset( $slot_data['end_time'] ) || empty( $slot_data['end_time'] ) ) {
			return false;
		}

		// Get all time slots for this type.
		$time_slots = ( 'delivery' === $type )
			? AICOSO_Pickup_Delivery_Data::get_delivery_time_slots()
			: AICOSO_Pickup_Delivery_Data::get_pickup_time_slots();

		$found = false;

		// Update slot.
		foreach ( $time_slots as $key => $slot ) {
			if ( $slot_name === $slot['slot_name'] ) {
				$time_slots[ $key ] = array(
					'slot_name'  => sanitize_text_field( $slot_data['slot_name'] ),
					'start_time' => sanitize_text_field( $slot_data['start_time'] ),
					'end_time'   => sanitize_text_field( $slot_data['end_time'] ),
					'max_orders' => isset( $slot_data['max_orders'] ) ? absint( $slot_data['max_orders'] ) : 10,
				);
				$found              = true;
				break;
			}
		}

		if ( ! $found ) {
			return false;
		}

		// Sort time slots by start time.
		usort( $time_slots, array( __CLASS__, 'sort_time_slots' ) );

		// Save time slots.
		$option_name = ( 'delivery' === $type )
			? 'aicoso_pickup_delivery_delivery_time_slots'
			: 'aicoso_pickup_delivery_pickup_time_slots';

		return update_option( $option_name, $time_slots );
	}

	/**
	 * Delete a time slot
	 *
	 * @since     1.0.0
	 * @param     string $slot_name      Time slot name to delete.
	 * @param     string $type           'delivery' or 'pickup'.
	 * @return    boolean                   Success or failure
	 */
	public static function delete_time_slot( $slot_name, $type = 'delivery' ) {
		if ( empty( $slot_name ) ) {
			return false;
		}

		// Get all time slots for this type.
		$time_slots = ( 'delivery' === $type )
			? AICOSO_Pickup_Delivery_Data::get_delivery_time_slots()
			: AICOSO_Pickup_Delivery_Data::get_pickup_time_slots();

		$found = false;

		// Find and remove slot.
		foreach ( $time_slots as $key => $slot ) {
			if ( $slot_name === $slot['slot_name'] ) {
				unset( $time_slots[ $key ] );
				$found = true;
				break;
			}
		}

		if ( ! $found ) {
			return false;
		}

		// Reindex array.
		$time_slots = array_values( $time_slots );

		// Save time slots.
		$option_name = ( 'delivery' === $type )
			? 'aicoso_pickup_delivery_delivery_time_slots'
			: 'aicoso_pickup_delivery_pickup_time_slots';

		return update_option( $option_name, $time_slots );
	}

	/**
	 * Sort time slots by start time
	 *
	 * @since     1.0.0
	 * @param     array $a              First time slot.
	 * @param     array $b              Second time slot.
	 * @return    int                       Sort result
	 */
	public static function sort_time_slots( $a, $b ) {
		$time_a = isset( $a['start_time'] ) ? strtotime( 'today ' . $a['start_time'] ) : 0;
		$time_b = isset( $b['start_time'] ) ? strtotime( 'today ' . $b['start_time'] ) : 0;

		if ( $time_b === $time_a ) {
			return 0;
		}

		return ( $time_a < $time_b ) ? -1 : 1;
	}
}
