<?php
/**
 * The pickup location management functionality of the plugin.
 *
 * @link       https://example.com
 * @since      1.0.0
 *
 * @package    aicoso_pickup_delivery
 */

if ( ! defined( 'WPINC' ) ) {
	die;
}

/**
 * The pickup location management class.
 *
 * Handles pickup locations, availability, and location operations.
 *
 * @since      1.0.0
 * @package    aicoso_pickup_delivery
 * @author     Your Name
 */
class AICOSO_Pickup_Delivery_Location_Manager {

	/**
	 * Get available pickup locations for a specific date
	 *
	 * @since     1.0.0
	 * @param     string $date           Date in Y-m-d format.
	 * @return    array                     Array of available pickup locations
	 */
	public static function get_available_locations( $date ) {
		if ( ! AICOSO_Pickup_Delivery_Date_Manager::is_valid_date_format( $date ) ) {
			return array();
		}

		// Check if the date is available first.
		if ( ! AICOSO_Pickup_Delivery_Date_Manager::is_date_available( $date, 'pickup' ) ) {
			return array();
		}

		// Get all pickup locations.
		$all_locations = AICOSO_Pickup_Delivery_Data::get_pickup_locations();

		$available_locations = array();

		foreach ( $all_locations as $location ) {
			$location_name     = isset( $location['location_name'] ) ? sanitize_text_field( $location['location_name'] ) : '';
			$location_capacity = isset( $location['max_orders'] ) ? absint( $location['max_orders'] ) : 25;

			// Skip if no location name.
			if ( empty( $location_name ) ) {
				continue;
			}

			// Check if we've reached the maximum orders for this pickup location.
			$orders_count = AICOSO_Pickup_Delivery_Data::count_orders_for_pickup_location( $date, $location_name );

			if ( $orders_count >= $location_capacity ) {
				continue;
			}

			// Location is available.
			$available_locations[] = array(
				'name'               => $location_name,
				'address'            => isset( $location['location_address'] ) ? $location['location_address'] : '',
				'map_url'            => isset( $location['location_map_url'] ) ? $location['location_map_url'] : '',
				'remaining_capacity' => $location_capacity - $orders_count,
			);
		}

		return $available_locations;
	}

	/**
	 * Check if a specific pickup location is available for a date
	 *
	 * @since     1.0.0
	 * @param     string $date           Date in Y-m-d format.
	 * @param     string $location_name  Pickup location name.
	 * @return    boolean                   Whether the pickup location is available
	 */
	public static function is_location_available( $date, $location_name ) {
		// Check if the date is available first.
		if ( ! AICOSO_Pickup_Delivery_Date_Manager::is_date_available( $date, 'pickup' ) ) {
			return false;
		}

		// Get location data.
		$location_data = self::get_location_by_name( $location_name );

		if ( empty( $location_data ) ) {
			return false;
		}

		$location_capacity = isset( $location_data['max_orders'] ) ? absint( $location_data['max_orders'] ) : 25;

		// Check if we've reached the maximum number of orders for this pickup location.
		$orders_count = AICOSO_Pickup_Delivery_Data::count_orders_for_pickup_location( $date, $location_name );

		if ( $orders_count >= $location_capacity ) {
			return false;
		}

		return true;
	}

	/**
	 * Get a pickup location by its name
	 *
	 * @since     1.0.0
	 * @param     string $location_name  Pickup location name.
	 * @return    array                     Location data or empty array if not found
	 */
	public static function get_location_by_name( $location_name ) {
		// Get all pickup locations.
		$all_locations = AICOSO_Pickup_Delivery_Data::get_pickup_locations();

		foreach ( $all_locations as $location ) {
			if ( isset( $location['location_name'] ) && $location['location_name'] === $location_name ) {
				return $location;
			}
		}

		return array();
	}

	/**
	 * Add a new pickup location
	 *
	 * @since     1.0.0
	 * @param     array $location_data  Pickup location data.
	 * @return    boolean                   Success or failure
	 */
	public static function add_location( $location_data ) {
		// Validate required fields.
		if ( ! isset( $location_data['location_name'] ) || empty( $location_data['location_name'] ) ) {
			return false;
		}

		// Get all pickup locations.
		$pickup_locations = AICOSO_Pickup_Delivery_Data::get_pickup_locations();

		// Check if location name already exists.
		foreach ( $pickup_locations as $location ) {
			if ( $location['location_name'] === $location_data['location_name'] ) {
				return false;
			}
		}

		// Add new location.
		$pickup_locations[] = array(
			'location_name'    => sanitize_text_field( $location_data['location_name'] ),
			'location_address' => isset( $location_data['location_address'] ) ? sanitize_text_field( $location_data['location_address'] ) : '',
			'location_map_url' => isset( $location_data['location_map_url'] ) ? esc_url_raw( $location_data['location_map_url'] ) : '',
			'location_email'   => isset( $location_data['location_email'] ) ? sanitize_email( $location_data['location_email'] ) : '',
			'max_orders'       => isset( $location_data['max_orders'] ) ? absint( $location_data['max_orders'] ) : 25,
		);

		// Sort locations alphabetically by name.
		usort( $pickup_locations, array( __CLASS__, 'sort_locations' ) );

		return update_option( 'aicoso_pickup_delivery_pickup_locations', $pickup_locations );
	}

	/**
	 * Update an existing pickup location
	 *
	 * @since     1.0.0
	 * @param     string $location_name  Original pickup location name to update.
	 * @param     array  $location_data  New pickup location data.
	 * @return    boolean                   Success or failure
	 */
	public static function update_location( $location_name, $location_data ) {
		// Validate required fields.
		if ( empty( $location_name ) ) {
			return false;
		}

		if ( ! isset( $location_data['location_name'] ) || empty( $location_data['location_name'] ) ) {
			return false;
		}

		// Get all pickup locations.
		$pickup_locations = AICOSO_Pickup_Delivery_Data::get_pickup_locations();

		$found = false;

		// Update location.
		foreach ( $pickup_locations as $key => $location ) {
			if ( $location['location_name'] === $location_name ) {
				$pickup_locations[ $key ] = array(
					'location_name'    => sanitize_text_field( $location_data['location_name'] ),
					'location_address' => isset( $location_data['location_address'] ) ? sanitize_text_field( $location_data['location_address'] ) : '',
					'location_map_url' => isset( $location_data['location_map_url'] ) ? esc_url_raw( $location_data['location_map_url'] ) : '',
					'location_email'   => isset( $location_data['location_email'] ) ? sanitize_email( $location_data['location_email'] ) : '',
					'max_orders'       => isset( $location_data['max_orders'] ) ? absint( $location_data['max_orders'] ) : 25,
				);
				$found                    = true;
				break;
			}
		}

		if ( ! $found ) {
			return false;
		}

		// Sort locations alphabetically by name.
		usort( $pickup_locations, array( __CLASS__, 'sort_locations' ) );

		return update_option( 'aicoso_pickup_delivery_pickup_locations', $pickup_locations );
	}

	/**
	 * Delete a pickup location
	 *
	 * @since     1.0.0
	 * @param     string $location_name  Pickup location name to delete.
	 * @return    boolean                   Success or failure
	 */
	public static function delete_location( $location_name ) {
		if ( empty( $location_name ) ) {
			return false;
		}

		// Get all pickup locations.
		$pickup_locations = AICOSO_Pickup_Delivery_Data::get_pickup_locations();

		$found = false;

		// Find and remove location.
		foreach ( $pickup_locations as $key => $location ) {
			if ( $location['location_name'] === $location_name ) {
				unset( $pickup_locations[ $key ] );
				$found = true;
				break;
			}
		}

		if ( ! $found ) {
			return false;
		}

		// Reindex array.
		$pickup_locations = array_values( $pickup_locations );

		// Do not allow empty locations array - keep at least one.
		if ( empty( $pickup_locations ) ) {
			$pickup_locations[] = array(
				'location_name'    => __( 'Main Store', 'aicoso-pickup-and-delivery-for-woocommerce' ),
				'location_address' => '',
				'location_map_url' => '',
				'location_email'   => get_option( 'admin_email' ),
				'max_orders'       => 25,
			);
		}

		return update_option( 'aicoso_pickup_delivery_pickup_locations', $pickup_locations );
	}

	/**
	 * Sort locations alphabetically by name
	 *
	 * @since     1.0.0
	 * @param     array $a              First location.
	 * @param     array $b              Second location.
	 * @return    int                       Sort result
	 */
	public static function sort_locations( $a, $b ) {
		$name_a = isset( $a['location_name'] ) ? $a['location_name'] : '';
		$name_b = isset( $b['location_name'] ) ? $b['location_name'] : '';

		return strcasecmp( $name_a, $name_b );
	}

	/**
	 * Get location admin email
	 *
	 * @since     1.0.0
	 * @param     string $location_name  Pickup location name.
	 * @return    string                    Admin email or site admin email if not set
	 */
	public static function get_location_email( $location_name ) {
		$location_data = self::get_location_by_name( $location_name );

		if ( ! empty( $location_data ) && isset( $location_data['location_email'] ) && ! empty( $location_data['location_email'] ) ) {
			return $location_data['location_email'];
		}

		// Return site admin email as fallback.
		return get_option( 'admin_email' );
	}

	/**
	 * Get Google Maps URL for a location
	 *
	 * @since     1.0.0
	 * @param     string $location_name  Pickup location name.
	 * @return    string                    Google Maps URL or empty string
	 */
	public static function get_location_map_url( $location_name ) {
		$location_data = self::get_location_by_name( $location_name );

		if ( ! empty( $location_data ) && isset( $location_data['location_map_url'] ) && ! empty( $location_data['location_map_url'] ) ) {
			return $location_data['location_map_url'];
		}

		return '';
	}
}
