<?php
/**
 * The date management functionality of the plugin.
 *
 * @link       https://example.com
 * @since      1.0.0
 *
 * @package    aicoso_pickup_delivery
 */

if ( ! defined( 'WPINC' ) ) {
	die;
}

/**
 * The date management class.
 *
 * Handles date availability, blackout dates, and date validation.
 *
 * @since      1.0.0
 * @package    aicoso_pickup_delivery
 * @author     Your Name
 */
class AICOSO_Pickup_Delivery_Date_Manager {

	/**
	 * Get available dates for a specific service type
	 *
	 * @since     1.0.0
	 * @param     string $type           'delivery' or 'pickup'.
	 * @param     int    $days_ahead     How many days ahead to calculate.
	 * @return    array                     Array of available dates
	 */
	public static function get_available_dates( $type = 'delivery', $days_ahead = 30 ) {
		$available_dates = array();
		$blackout_dates  = AICOSO_Pickup_Delivery_Data::get_blackout_dates( $type );

		// Get settings.
		$settings       = AICOSO_Pickup_Delivery_Data::get_settings();
		$lead_time      = isset( $settings[ $type . '_lead_time' ] ) ? absint( $settings[ $type . '_lead_time' ] ) : 1;
		$available_days = isset( $settings[ $type . '_days' ] ) ? $settings[ $type . '_days' ] : array( 1, 2, 3, 4, 5 ); // Default Monday to Friday.
		$max_per_day    = isset( $settings[ 'max_' . $type . '_per_day' ] ) ? absint( $settings[ 'max_' . $type . '_per_day' ] ) : 25;

		// Get current date considering WP timezone.
		$timezone     = new DateTimeZone( wp_timezone_string() );
		$current_date = new DateTime( 'now', $timezone );

		// Add lead time.
		$start_date = clone $current_date;
		if ( $lead_time > 0 ) {
			$start_date->modify( '+' . $lead_time . ' day' );
		}

		// Loop through days ahead to find available dates.
		for ( $i = 0; $i < $days_ahead; $i++ ) {
			$check_date = clone $start_date;
			$check_date->modify( '+' . $i . ' day' );

			$date_string = $check_date->format( 'Y-m-d' );
			// Get weekday: 1 (Monday) to 7 (Sunday).
			$weekday = (int) $check_date->format( 'N' );

			// Skip if this weekday is not in the available days.
			if ( ! in_array( $weekday, $available_days, true ) ) {
				continue;
			}

			// Skip if this date is in the blackout dates.
			if ( in_array( $date_string, $blackout_dates, true ) ) {
				continue;
			}

			// Check if we've reached the maximum number of orders for this date.
			$orders_count = AICOSO_Pickup_Delivery_Data::count_orders_for_date( $date_string, $type );
			if ( $orders_count >= $max_per_day ) {
				continue;
			}

			// This date is available.
			$available_dates[] = array(
				'date'               => $date_string,
				'formatted_date'     => AICOSO_Pickup_Delivery_Data::format_date( $date_string ),
				'remaining_capacity' => $max_per_day - $orders_count,
			);
		}

		return $available_dates;
	}

	/**
	 * Check if a specific date is available for a service type
	 *
	 * @since     1.0.0
	 * @param     string $date          Date in Y-m-d format.
	 * @param     string $type          'delivery' or 'pickup'.
	 * @return    boolean                  Whether the date is available
	 */
	public static function is_date_available( $date, $type = 'delivery' ) {
		// Check if it's a valid date.
		if ( ! self::is_valid_date_format( $date ) ) {
			return false;
		}

		// Get blackout dates for the specific type and normalize them to Y-m-d.
		$blackout_dates = AICOSO_Pickup_Delivery_Data::get_blackout_dates( $type );
		$blackout_dates = array_values(
			array_filter(
				array_map(
					function ( $d ) {
						if ( ! is_string( $d ) ) {
								return '';
						}
						$dt = DateTime::createFromFormat( 'Y-m-d', $d );
						return $dt ? $dt->format( 'Y-m-d' ) : '';
					},
					$blackout_dates
				)
			)
		);

		if ( in_array( $date, $blackout_dates, true ) ) {
			return false;
		}

		// Get settings.
		$settings       = AICOSO_Pickup_Delivery_Data::get_settings();
		$available_days = isset( $settings[ $type . '_days' ] ) ? $settings[ $type . '_days' ] : array( 1, 2, 3, 4, 5 ); // Default Monday to Friday.
		$available_days = array_map( 'intval', (array) $available_days );
		$max_per_day    = isset( $settings[ 'max_' . $type . '_per_day' ] ) ? absint( $settings[ 'max_' . $type . '_per_day' ] ) : 25;
		$lead_time      = isset( $settings[ $type . '_lead_time' ] ) ? absint( $settings[ $type . '_lead_time' ] ) : 1;

		// Create date object using site timezone and check weekday.
		$timezone = new DateTimeZone( wp_timezone_string() );
		$date_obj = DateTime::createFromFormat( 'Y-m-d', $date, $timezone );
		if ( ! $date_obj ) {
			return false;
		}
		// Get weekday: 1 (Monday) to 7 (Sunday).
		$weekday = (int) $date_obj->format( 'N' );
		if ( ! in_array( $weekday, $available_days, true ) ) {
			return false;
		}

		// Check lead time.
		$timezone     = new DateTimeZone( wp_timezone_string() );
		$current_date = new DateTime( 'now', $timezone );
		$min_date     = clone $current_date;
		if ( $lead_time > 0 ) {
			$min_date->modify( '+' . $lead_time . ' day' );
		}

		// Compare using same timezone-aware DateTime objects.
		if ( $date_obj < $min_date ) {
			return false;
		}

		// Check if we've reached the maximum number of orders for this date.
		$orders_count = AICOSO_Pickup_Delivery_Data::count_orders_for_date( $date, $type );
		if ( $orders_count >= $max_per_day ) {
			return false;
		}

		// Check if we've reached the maximum total orders per day.
		$max_total_per_day = isset( $settings['max_orders_per_day'] ) ? absint( $settings['max_orders_per_day'] ) : 50;
		$total_orders      = AICOSO_Pickup_Delivery_Data::count_orders_for_date( $date, 'delivery' ) +
						AICOSO_Pickup_Delivery_Data::count_orders_for_date( $date, 'pickup' );

		if ( $total_orders >= $max_total_per_day ) {
			return false;
		}

		return true;
	}

	/**
	 * Get the next available date for a service type
	 *
	 * @since     1.0.0
	 * @param     string $type          'delivery' or 'pickup'.
	 * @return    string                   The next available date (Y-m-d) or empty if none found
	 */
	public static function get_next_available_date( $type = 'delivery' ) {
		$available_dates = self::get_available_dates( $type, 60 ); // Look 60 days ahead.

		if ( ! empty( $available_dates ) ) {
			return $available_dates[0]['date'];
		}

		return '';
	}

	/**
	 * Add a blackout date
	 *
	 * @since     1.0.0
	 * @param     string $date          Date in Y-m-d format.
	 * @param     string $type          'delivery' or 'pickup' (optional, defaults to 'delivery').
	 * @return    boolean                  Success or failure
	 */
	public static function add_blackout_date( $date, $type = 'delivery' ) {
		if ( ! self::is_valid_date_format( $date ) ) {
			return false;
		}

		$blackout_dates = AICOSO_Pickup_Delivery_Data::get_blackout_dates( $type );

		if ( ! in_array( $date, $blackout_dates, true ) ) {
			$blackout_dates[] = $date;

			// Keep array sorted.
			sort( $blackout_dates );

			// Use the appropriate option key based on type.
			$option_key = ( 'pickup' === $type ) ? 'aicoso_pickup_delivery_pickup_blackout_dates' : 'aicoso_pickup_delivery_delivery_blackout_dates';
			update_option( $option_key, $blackout_dates );
			return true;
		}

		return false;
	}

	/**
	 * Remove a blackout date
	 *
	 * @since     1.0.0
	 * @param     string $date          Date in Y-m-d format.
	 * @param     string $type          'delivery' or 'pickup' (optional, defaults to 'delivery').
	 * @return    boolean                  Success or failure
	 */
	public static function remove_blackout_date( $date, $type = 'delivery' ) {
		$blackout_dates = AICOSO_Pickup_Delivery_Data::get_blackout_dates( $type );

		$key = array_search( $date, $blackout_dates, true );
		if ( false !== $key ) {
			unset( $blackout_dates[ $key ] );
			$blackout_dates = array_values( $blackout_dates ); // Reindex array.

			// Use the appropriate option key based on type.
			$option_key = ( 'pickup' === $type ) ? 'aicoso_pickup_delivery_pickup_blackout_dates' : 'aicoso_pickup_delivery_delivery_blackout_dates';
			update_option( $option_key, $blackout_dates );
			return true;
		}

		return false;
	}

	/**
	 * Check if a string is a valid Y-m-d date format
	 *
	 * @since     1.0.0
	 * @param     string $date          Date string to validate.
	 * @return    boolean                  Whether it's a valid date format
	 */
	public static function is_valid_date_format( $date ) {
		if ( ! is_string( $date ) ) {
			return false;
		}

		$d = DateTime::createFromFormat( 'Y-m-d', $date );
		return $d && $d->format( 'Y-m-d' ) === $date;
	}

	/**
	 * Get dates for the datepicker (for frontend JavaScript)
	 *
	 * @since     1.0.0
	 * @param     string $type           'delivery' or 'pickup'.
	 * @return    array                     Date information for the datepicker
	 */
	public static function get_datepicker_data( $type = 'delivery' ) {
		// Get settings.
		$settings       = AICOSO_Pickup_Delivery_Data::get_settings();
		$available_days = isset( $settings[ $type . '_days' ] ) ? $settings[ $type . '_days' ] : array( 1, 2, 3, 4, 5 ); // Default Monday to Friday.

		// Map from numerical days (1-7) to datepicker days (0-6, Sunday-Saturday).
		$datepicker_days = array();
		foreach ( $available_days as $day ) {
			// Convert from ISO-8601 (1 = Monday) to JavaScript (0 = Sunday).
			$datepicker_day    = ( $day % 7 ); // Sunday = 0, Monday = 1, ..., Saturday = 6.
			$datepicker_days[] = $datepicker_day;
		}

		// Get available dates.
		$available_dates = self::get_available_dates( $type, 60 ); // Look 60 days ahead.

		// Get blackout dates for the specific type and normalize them.
		$blackout_dates = AICOSO_Pickup_Delivery_Data::get_blackout_dates( $type );
		$blackout_dates = array_values(
			array_filter(
				array_map(
					function ( $d ) {
						if ( ! is_string( $d ) ) {
								return '';
						}
						$dt = DateTime::createFromFormat( 'Y-m-d', $d );
						return $dt ? $dt->format( 'Y-m-d' ) : '';
					},
					(array) $blackout_dates
				)
			)
		);

		// Calculate min date (with lead time).
		$lead_time    = isset( $settings[ $type . '_lead_time' ] ) ? absint( $settings[ $type . '_lead_time' ] ) : 1;
		$timezone     = new DateTimeZone( wp_timezone_string() );
		$current_date = new DateTime( 'now', $timezone );
		$min_date     = clone $current_date;
		if ( $lead_time > 0 ) {
			$min_date->modify( '+' . $lead_time . ' day' );
		}

		return array(
			'available_days'  => $datepicker_days,
			'available_dates' => wp_list_pluck( $available_dates, 'date' ),
			'blackout_dates'  => $blackout_dates,
			'min_date'        => $min_date->format( 'Y-m-d' ),
		);
	}
}
