<?php
/**
 * The file that defines the core plugin class
 *
 * A class definition that includes attributes and functions used across both the
 * public-facing side of the site and the admin area.
 *
 * @link       https://example.com
 * @since      1.0.0
 *
 * @package    aicoso_pickup_delivery
 */

if ( ! defined( 'WPINC' ) ) {
	die;
}

/**
 * The core plugin class.
 *
 * This is used to define internationalization, admin-specific hooks, and
 * public-facing site hooks.
 *
 * Also maintains the unique identifier of this plugin as well as the current
 * version of the plugin.
 *
 * @since      1.0.0
 * @package    aicoso_pickup_delivery
 * @author     Your Name
 */
class AICOSO_Pickup_Delivery {

	/**
	 * The loader that's responsible for maintaining and registering all hooks that power
	 * the plugin.
	 *
	 * @since    1.0.0
	 * @access   protected
	 * @var      AICOSO_Pickup_Delivery_Loader    $loader    Maintains and registers all hooks for the plugin.
	 */
	protected $loader;

	/**
	 * The unique identifier of this plugin.
	 *
	 * @since    1.0.0
	 * @access   protected
	 * @var      string    $plugin_name    The string used to uniquely identify this plugin.
	 */
	protected $plugin_name;

	/**
	 * The current version of the plugin.
	 *
	 * @since    1.0.0
	 * @access   protected
	 * @var      string    $version    The current version of this plugin.
	 */
	protected $version;

	/**
	 * Define the core functionality of the plugin.
	 *
	 * Set the plugin name and the plugin version that can be used throughout the plugin.
	 * Load the dependencies, define the locale, and set the hooks for the admin area and
	 * the public-facing side of the site.
	 *
	 * @since    1.0.0
	 */
	public function __construct() {
		if ( defined( 'AICOSO_PICKUP_DELIVERY_VERSION' ) ) {
			$this->version = AICOSO_PICKUP_DELIVERY_VERSION;
		} else {
			$this->version = '1.0.0';
		}
		$this->plugin_name = 'aicoso-pickup-and-delivery-for-woocommerce';

		$this->load_dependencies();
		$this->set_locale();
		$this->define_admin_hooks();
		$this->define_frontend_hooks();

		// Register email hooks with multiple fallbacks to ensure they load.
		add_action( 'woocommerce_init', array( $this, 'define_email_hooks' ), 5 );
		add_action( 'woocommerce_email_classes', array( $this, 'ensure_email_classes_loaded' ), 999 );
		add_action( 'init', array( $this, 'check_email_registration' ), 999 );

		// Initialize blocks compatibility.
		if ( class_exists( 'AICOSO_Pickup_Delivery_Blocks_Compatibility' ) ) {
			AICOSO_Pickup_Delivery_Blocks_Compatibility::init();
		}
	}

	/**
	 * Load the required dependencies for this plugin.
	 *
	 * Include the following files that make up the plugin:
	 *
	 * - AICOSO_Pickup_Delivery_Loader. Orchestrates the hooks of the plugin.
	 * - AICOSO_Pickup_Delivery_I18n. Defines internationalization functionality.
	 * - AICOSO_Pickup_Delivery_Admin. Defines all hooks for the admin area.
	 * - AICOSO_Pickup_Delivery_Frontend. Defines all hooks for the public side of the site.
	 *
	 * Create an instance of the loader which will be used to register the hooks
	 * with WordPress.
	 *
	 * @since    1.0.0
	 * @access   private
	 */
	private function load_dependencies() {
		/**
		 * The class responsible for orchestrating the actions and filters of the
		 * core plugin.
		 */
		require_once AICOSO_PICKUP_DELIVERY_PLUGIN_PATH . 'includes/class-aicoso-pickup-delivery-loader.php';

		/**
		 * The class responsible for defining internationalization functionality
		 * of the plugin.
		 */
		require_once AICOSO_PICKUP_DELIVERY_PLUGIN_PATH . 'includes/class-aicoso-pickup-delivery-i18n.php';

		/**
		 * The class responsible for data handling.
		 */
		require_once AICOSO_PICKUP_DELIVERY_PLUGIN_PATH . 'includes/class-aicoso-pickup-delivery-data.php';
		require_once AICOSO_PICKUP_DELIVERY_PLUGIN_PATH . 'includes/data/class-aicoso-pickup-delivery-date-manager.php';
		require_once AICOSO_PICKUP_DELIVERY_PLUGIN_PATH . 'includes/data/class-aicoso-pickup-delivery-timeslot-manager.php';
		require_once AICOSO_PICKUP_DELIVERY_PLUGIN_PATH . 'includes/data/class-aicoso-pickup-delivery-location-manager.php';

		/**
		 * The class responsible for defining all actions that occur in the admin area.
		 */
		require_once AICOSO_PICKUP_DELIVERY_PLUGIN_PATH . 'includes/admin/class-aicoso-pickup-delivery-admin.php';
		require_once AICOSO_PICKUP_DELIVERY_PLUGIN_PATH . 'includes/admin/class-aicoso-pickup-delivery-settings.php';
		require_once AICOSO_PICKUP_DELIVERY_PLUGIN_PATH . 'includes/admin/class-aicoso-pickup-delivery-order-management.php';
		require_once AICOSO_PICKUP_DELIVERY_PLUGIN_PATH . 'includes/admin/class-aicoso-pickup-delivery-reports.php';

		/**
		 * The class responsible for defining all actions that occur in the public-facing
		 * side of the site.
		 */
		require_once AICOSO_PICKUP_DELIVERY_PLUGIN_PATH . 'includes/frontend/class-aicoso-pickup-delivery-frontend.php';
		require_once AICOSO_PICKUP_DELIVERY_PLUGIN_PATH . 'includes/frontend/class-aicoso-pickup-delivery-checkout.php';
		require_once AICOSO_PICKUP_DELIVERY_PLUGIN_PATH . 'includes/frontend/class-aicoso-pickup-delivery-my-account.php';

		/**
		 * Email manager class - this doesn't directly depend on WC_Email
		 */
		if ( file_exists( AICOSO_PICKUP_DELIVERY_PLUGIN_PATH . 'includes/emails/class-aicoso-pickup-delivery-email-manager.php' ) ) {
			require_once AICOSO_PICKUP_DELIVERY_PLUGIN_PATH . 'includes/emails/class-aicoso-pickup-delivery-email-manager.php';
		}

		/**
		 * WooCommerce Blocks compatibility
		 */
		if ( file_exists( AICOSO_PICKUP_DELIVERY_PLUGIN_PATH . 'includes/class-aicoso-pickup-delivery-blocks-compatibility.php' ) ) {
			require_once AICOSO_PICKUP_DELIVERY_PLUGIN_PATH . 'includes/class-aicoso-pickup-delivery-blocks-compatibility.php';
		}

		/**
		 * WooCommerce Blocks integration
		 */
		if ( class_exists( '\Automattic\WooCommerce\Blocks\Package' ) ) {
			if ( file_exists( AICOSO_PICKUP_DELIVERY_PLUGIN_PATH . 'includes/class-aicoso-pickup-delivery-blocks-integration.php' ) ) {
				require_once AICOSO_PICKUP_DELIVERY_PLUGIN_PATH . 'includes/class-aicoso-pickup-delivery-blocks-integration.php';
			}
		}

		$this->loader = new AICOSO_Pickup_Delivery_Loader();
	}

	/**
	 * Define the locale for this plugin for internationalization.
	 *
	 * Uses the AICOSO_Pickup_Delivery_I18n class in order to set the domain and to register the hook
	 * with WordPress.
	 *
	 * @since    1.0.0
	 * @access   private
	 */
	private function set_locale() {
		$plugin_i18n = new AICOSO_Pickup_Delivery_I18n();
		$this->loader->add_action( 'plugins_loaded', $plugin_i18n, 'load_plugin_textdomain' );
	}

	/**
	 * Register all of the hooks related to the admin area functionality
	 * of the plugin.
	 *
	 * @since    1.0.0
	 * @access   private
	 */
	private function define_admin_hooks() {
		// Admin class to handle general admin functionality.
		$plugin_admin = new AICOSO_Pickup_Delivery_Admin( $this->get_plugin_name(), $this->get_version() );
		$this->loader->add_action( 'admin_enqueue_scripts', $plugin_admin, 'enqueue_styles' );
		$this->loader->add_action( 'admin_enqueue_scripts', $plugin_admin, 'enqueue_scripts' );

		// Settings page.
		$plugin_settings = new AICOSO_Pickup_Delivery_Settings( $this->get_plugin_name(), $this->get_version() );
		$this->loader->add_action( 'admin_menu', $plugin_settings, 'add_settings_page' );
		$this->loader->add_action( 'admin_init', $plugin_settings, 'register_settings' );
		$this->loader->add_action( 'admin_enqueue_scripts', $plugin_settings, 'enqueue_settings_scripts' );
		$this->loader->add_filter( 'woocommerce_get_sections_shipping', $plugin_settings, 'add_wc_settings_section' );
		$this->loader->add_filter( 'woocommerce_get_settings_shipping', $plugin_settings, 'add_wc_settings', 10, 2 );

		// Order management.
		$order_management = new AICOSO_Pickup_Delivery_Order_Management( $this->get_plugin_name(), $this->get_version() );
		$this->loader->add_action( 'woocommerce_admin_order_data_after_billing_address', $order_management, 'display_delivery_pickup_info' );
		$this->loader->add_action( 'woocommerce_admin_order_data_after_shipping_address', $order_management, 'display_delivery_pickup_info' );
		$this->loader->add_action( 'woocommerce_process_shop_order_meta', $order_management, 'save_delivery_pickup_info' );

		// Add meta box for better visibility.
		$this->loader->add_action( 'add_meta_boxes', $order_management, 'add_order_meta_box' );
		$this->loader->add_action( 'admin_enqueue_scripts', $order_management, 'enqueue_order_scripts' );

		// Add columns to orders list table.
		$this->loader->add_filter( 'manage_edit-shop_order_columns', $order_management, 'add_order_list_columns', 20 );
		$this->loader->add_action( 'manage_shop_order_posts_custom_column', $order_management, 'add_order_list_column_content', 20, 2 );

		// Add HPOS (High Performance Order Storage) compatibility.
		$this->loader->add_filter( 'manage_woocommerce_page_wc-orders_columns', $order_management, 'add_order_list_columns', 20 );
		$this->loader->add_action( 'manage_woocommerce_page_wc-orders_custom_column', $order_management, 'add_order_list_column_content', 20, 2 );

		// Add date filter to orders list.
		$this->loader->add_action( 'restrict_manage_posts', $order_management, 'add_delivery_pickup_date_filter' );
		$this->loader->add_filter( 'request', $order_management, 'filter_orders_by_date' );

		// Reports.
		$reports = new AICOSO_Pickup_Delivery_Reports( $this->get_plugin_name(), $this->get_version() );
		$this->loader->add_filter( 'woocommerce_admin_reports', $reports, 'add_reports' );
	}

	/**
	 * Register all of the hooks related to the public-facing functionality
	 * of the plugin.
	 *
	 * @since    1.0.0
	 * @access   private
	 */
	private function define_frontend_hooks() {
		// Frontend general.
		$plugin_frontend = new AICOSO_Pickup_Delivery_Frontend( $this->get_plugin_name(), $this->get_version() );
		$this->loader->add_action( 'wp_enqueue_scripts', $plugin_frontend, 'enqueue_styles' );
		$this->loader->add_action( 'wp_enqueue_scripts', $plugin_frontend, 'enqueue_scripts' );

		// Checkout.
		$checkout = new AICOSO_Pickup_Delivery_Checkout( $this->get_plugin_name(), $this->get_version() );
		$this->loader->add_action( 'woocommerce_after_checkout_billing_form', $checkout, 'add_delivery_pickup_selector', 20 );
		$this->loader->add_action( 'woocommerce_checkout_process', $checkout, 'validate_checkout_fields' );
		$this->loader->add_action( 'woocommerce_checkout_create_order', $checkout, 'save_delivery_pickup_details_early', 10, 2 );
		$this->loader->add_action( 'woocommerce_checkout_update_order_meta', $checkout, 'save_delivery_pickup_details' );

		// My Account.
		$my_account = new AICOSO_Pickup_Delivery_My_Account( $this->get_plugin_name(), $this->get_version() );
		$this->loader->add_action( 'woocommerce_order_details_after_order_table', $my_account, 'display_delivery_pickup_details' );
	}

	/**
	 * Register all of the hooks related to email notifications
	 *
	 * @since    1.0.0
	 * @access   public
	 */
	public function define_email_hooks() {
		// Only register if WC_Email is available.
		if ( class_exists( 'WC_Email' ) && class_exists( 'WC_Emails' ) ) {
			$email_manager = new AICOSO_Pickup_Delivery_Email_Manager( $this->get_plugin_name(), $this->get_version() );
			$this->loader->add_filter( 'woocommerce_email_classes', $email_manager, 'register_emails', 10 );
			$this->loader->add_action( 'woocommerce_email_order_meta', $email_manager, 'add_delivery_pickup_details_to_emails', 10, 3 );
		}
	}

	/**
	 * Ensure email classes are loaded - fallback method
	 *
	 * @since    1.0.0
	 * @param    array $email_classes    Existing email classes.
	 * @return   array                   Modified email classes.
	 */
	public function ensure_email_classes_loaded( $email_classes ) {
		// Check if our emails are already registered.
		$pickup_found   = false;
		$delivery_found = false;

		foreach ( $email_classes as $email ) {
			if ( is_object( $email ) ) {
				$class_name = get_class( $email );
				if ( strpos( $class_name, 'AICOSO_Email_Pickup_Confirmation' ) !== false ) {
					$pickup_found = true;
				}
				if ( strpos( $class_name, 'AICOSO_Email_Delivery_Confirmation' ) !== false ) {
					$delivery_found = true;
				}
			}
		}

		// If not found, register them now.
		if ( ! $pickup_found || ! $delivery_found ) {
			if ( class_exists( 'WC_Email' ) && class_exists( 'AICOSO_Pickup_Delivery_Email_Manager' ) ) {
				$email_manager = new AICOSO_Pickup_Delivery_Email_Manager( $this->get_plugin_name(), $this->get_version() );
				$email_classes = $email_manager->register_emails( $email_classes );
			}
		}

		return $email_classes;
	}

	/**
	 * Check if emails are registered and log for debugging
	 *
	 * @since    1.0.0
	 */
	public function check_email_registration() {
		// Only run this check once and only if WooCommerce is active.
		static $checked = false;
		if ( $checked || ! class_exists( 'WC_Emails' ) ) {
			return;
		}
		$checked = true;

		// Get registered email classes.
		$wc_emails = WC_Emails::instance();
		if ( ! $wc_emails ) {
			return;
		}

		$email_classes  = $wc_emails->get_emails();
		$pickup_found   = false;
		$delivery_found = false;

		foreach ( $email_classes as $email ) {
			if ( is_object( $email ) ) {
				$class_name = get_class( $email );
				if ( strpos( $class_name, 'AICOSO_Email_Pickup_Confirmation' ) !== false ) {
					$pickup_found = true;
				}
				if ( strpos( $class_name, 'AICOSO_Email_Delivery_Confirmation' ) !== false ) {
					$delivery_found = true;
				}
			}
		}

		// Email registration check removed for production.
	}

	/**
	 * Run the loader to execute all of the hooks with WordPress.
	 *
	 * @since    1.0.0
	 */
	public function run() {
		$this->loader->run();
	}

	/**
	 * The name of the plugin used to uniquely identify it within the context of
	 * WordPress and to define internationalization functionality.
	 *
	 * @since     1.0.0
	 * @return    string    The name of the plugin.
	 */
	public function get_plugin_name() {
		return $this->plugin_name;
	}

	/**
	 * The reference to the class that orchestrates the hooks with the plugin.
	 *
	 * @since     1.0.0
	 * @return    AICOSO_Pickup_Delivery_Loader    Orchestrates the hooks of the plugin.
	 */
	public function get_loader() {
		return $this->loader;
	}

	/**
	 * Retrieve the version number of the plugin.
	 *
	 * @since     1.0.0
	 * @return    string    The version number of the plugin.
	 */
	public function get_version() {
		return $this->version;
	}
}
