<?php
/**
 * The data handler for the plugin
 *
 * @link       https://example.com
 * @since      1.0.0
 *
 * @package    aicoso_pickup_delivery
 */

if ( ! defined( 'WPINC' ) ) {
	die;
}

/**
 * The data handler class.
 *
 * This class handles general data operations for the plugin.
 *
 * @since      1.0.0
 * @package    aicoso_pickup_delivery
 * @author     Your Name
 */
class AICOSO_Pickup_Delivery_Data {

	/**
	 * Get all plugin settings
	 *
	 * @since    1.0.0
	 * @return   array    All plugin settings
	 */
	public static function get_settings() {
		$default_settings = array(
			// Order settings.
			'enable_order_type_selection' => 'yes',
			'default_order_type'          => 'delivery',
			'max_orders_per_day'          => 50,

			// Delivery settings.
			'enable_delivery_date'        => 'yes',
			'max_delivery_per_day'        => 25,
			'delivery_days'               => array( 1, 2, 3, 4, 5 ), // Monday to Friday.
			'delivery_lead_time'          => 1, // Minimum lead time for delivery in days.

			// Pickup settings.
			'enable_pickup_date'          => 'yes',
			'max_pickup_per_day'          => 25,
			'pickup_days'                 => array( 1, 2, 3, 4, 5 ), // Monday to Friday.
			'pickup_lead_time'            => 1, // Minimum lead time for pickup in days.

			// Time slot settings.
			'enable_delivery_time_slots'  => 'yes',
			'enable_pickup_time_slots'    => 'yes',
		);

		$settings = get_option( 'aicoso_pickup_delivery_settings', $default_settings );

		return $settings;
	}

	/**
	 * Get a specific setting
	 *
	 * @since     1.0.0
	 * @param     string $key          The setting key to retrieve.
	 * @param     mixed  $default      Default value if setting doesn't exist.
	 * @return    mixed                   The setting value or default
	 */
	public static function get_setting( $key, $default = false ) { // phpcs:ignore Universal.NamingConventions.NoReservedKeywordParameterNames.defaultFound -- Standard PHP pattern.
		$settings = self::get_settings();

		if ( isset( $settings[ $key ] ) ) {
			return $settings[ $key ];
		}

		return $default;
	}

	/**
	 * Update a specific setting
	 *
	 * @since     1.0.0
	 * @param     string $key          The setting key to update.
	 * @param     mixed  $value        The value to set.
	 * @return    boolean                 Whether the update was successful
	 */
	public static function update_setting( $key, $value ) {
		$settings         = self::get_settings();
		$settings[ $key ] = $value;

		return update_option( 'aicoso_pickup_delivery_settings', $settings );
	}

	/**
	 * Update multiple settings at once
	 *
	 * @since     1.0.0
	 * @param     array $new_settings   Array of settings to update.
	 * @return    boolean                  Whether the update was successful
	 */
	public static function update_settings( $new_settings ) {
		$settings = self::get_settings();
		$settings = array_merge( $settings, $new_settings );
		return update_option( 'aicoso_pickup_delivery_settings', $settings );
	}

	/**
	 * Get delivery time slots
	 *
	 * @since     1.0.0
	 * @return    array    Array of delivery time slots
	 */
	public static function get_delivery_time_slots() {
		$default_slots = array(
			array(
				'slot_name'  => 'Morning (9:00 AM - 12:00 PM)',
				'start_time' => '09:00',
				'end_time'   => '12:00',
				'max_orders' => 10,
			),
			array(
				'slot_name'  => 'Afternoon (12:00 PM - 3:00 PM)',
				'start_time' => '12:00',
				'end_time'   => '15:00',
				'max_orders' => 10,
			),
			array(
				'slot_name'  => 'Evening (3:00 PM - 6:00 PM)',
				'start_time' => '15:00',
				'end_time'   => '18:00',
				'max_orders' => 10,
			),
		);

		$slots = get_option( 'aicoso_pickup_delivery_delivery_time_slots', $default_slots );

		// Ensure slots are properly formatted and filter out empty entries.
		if ( is_array( $slots ) ) {
			$filtered = array();
			foreach ( $slots as $slot ) {
				if ( isset( $slot['slot_name'] ) && ! empty( $slot['slot_name'] ) ) {
					$filtered[] = $slot;
				}
			}
			if ( ! empty( $filtered ) ) {
				return array_values( $filtered );
			}
		}

		return $default_slots;
	}

	/**
	 * Get pickup time slots
	 *
	 * @since     1.0.0
	 * @return    array    Array of pickup time slots
	 */
	public static function get_pickup_time_slots() {
		$default_slots = array(
			array(
				'slot_name'  => 'Morning (9:00 AM - 12:00 PM)',
				'start_time' => '09:00',
				'end_time'   => '12:00',
				'max_orders' => 10,
			),
			array(
				'slot_name'  => 'Afternoon (12:00 PM - 3:00 PM)',
				'start_time' => '12:00',
				'end_time'   => '15:00',
				'max_orders' => 10,
			),
			array(
				'slot_name'  => 'Evening (3:00 PM - 6:00 PM)',
				'start_time' => '15:00',
				'end_time'   => '18:00',
				'max_orders' => 10,
			),
		);

		$slots = get_option( 'aicoso_pickup_delivery_pickup_time_slots', $default_slots );

		// Ensure slots are properly formatted and filter out empty entries.
		if ( is_array( $slots ) ) {
			$filtered = array();
			foreach ( $slots as $slot ) {
				if ( isset( $slot['slot_name'] ) && ! empty( $slot['slot_name'] ) ) {
					$filtered[] = $slot;
				}
			}
			if ( ! empty( $filtered ) ) {
				return array_values( $filtered );
			}
		}

		return $default_slots;
	}

	/**
	 * Get pickup locations
	 *
	 * @since     1.0.0
	 * @return    array    Array of pickup locations
	 */
	public static function get_pickup_locations() {
		$default_locations = array(
			array(
				'location_name'    => 'Main Store',
				'location_address' => '123 Example Street, City, 12345',
				'location_map_url' => 'https://maps.google.com/?q=123+Example+Street',
				'location_email'   => get_option( 'admin_email' ),
				'max_orders'       => 25,
			),
		);

		return get_option( 'aicoso_pickup_delivery_pickup_locations', $default_locations );
	}

	/**
	 * Get blackout dates
	 *
	 * @since     1.0.0
	 * @param     string $type    'delivery' or 'pickup' (optional, defaults to 'delivery' for backward compatibility).
	 * @return    array    Array of blackout dates
	 */
	public static function get_blackout_dates( $type = 'delivery' ) {
		if ( 'pickup' === $type ) {
			return get_option( 'aicoso_pickup_delivery_pickup_blackout_dates', array() );
		}
		// Default to delivery for backward compatibility.
		return get_option( 'aicoso_pickup_delivery_delivery_blackout_dates', array() );
	}

	/**
	 * Get order type from an order
	 *
	 * @since     1.0.0
	 * @param     WC_Order|int $order    WC_Order object or order ID.
	 * @return    string                    'delivery' or 'pickup'
	 */
	public static function get_order_type( $order ) {
		if ( is_numeric( $order ) ) {
			$order = wc_get_order( $order );
		}

		if ( ! $order instanceof WC_Order ) {
			return self::get_setting( 'default_order_type', 'delivery' );
		}

		// Using get_meta() for HPOS compatibility.
		$order_type = $order->get_meta( '_aicoso_pickup_delivery_type', true );

		if ( empty( $order_type ) ) {
			return self::get_setting( 'default_order_type', 'delivery' );
		}

		return $order_type;
	}

	/**
	 * Get date from an order
	 *
	 * @since     1.0.0
	 * @param     WC_Order|int $order    WC_Order object or order ID.
	 * @return    string                    The selected date or empty string
	 */
	public static function get_order_date( $order ) {
		if ( is_numeric( $order ) ) {
			$order = wc_get_order( $order );
		}

		if ( ! $order instanceof WC_Order ) {
			return '';
		}

		$order_type = self::get_order_type( $order );
		$date_key   = '_aicoso_pickup_delivery_' . $order_type . '_date';

		return $order->get_meta( $date_key, true );
	}

	/**
	 * Get time slot from an order
	 *
	 * @since     1.0.0
	 * @param     WC_Order|int $order    WC_Order object or order ID.
	 * @return    string                    The selected time slot or empty string
	 */
	public static function get_order_time_slot( $order ) {
		if ( is_numeric( $order ) ) {
			$order = wc_get_order( $order );
		}

		if ( ! $order instanceof WC_Order ) {
			return '';
		}

		$order_type    = self::get_order_type( $order );
		$time_slot_key = '_aicoso_pickup_delivery_' . $order_type . '_time_slot';

		return $order->get_meta( $time_slot_key, true );
	}

	/**
	 * Get pickup location from an order
	 *
	 * @since     1.0.0
	 * @param     WC_Order|int $order    WC_Order object or order ID.
	 * @return    string                    The selected pickup location or empty string
	 */
	public static function get_order_pickup_location( $order ) {
		if ( is_numeric( $order ) ) {
			$order = wc_get_order( $order );
		}

		if ( ! $order instanceof WC_Order ) {
			return '';
		}

		$order_type = self::get_order_type( $order );

		if ( 'pickup' !== $order_type ) {
			return '';
		}

		return $order->get_meta( '_aicoso_pickup_delivery_pickup_location', true );
	}

	/**
	 * Count orders for a specific date and type
	 *
	 * @since     1.0.0
	 * @param     string $date         The date to check (Y-m-d format).
	 * @param     string $type         'delivery' or 'pickup'.
	 * @return    int                     Number of orders
	 */
	public static function count_orders_for_date( $date, $type = 'delivery' ) {
		// Using WC Data Store for HPOS compatibility.
		$meta_key = '_aicoso_pickup_delivery_' . $type . '_date';

		$query_args = array(
			'type'       => 'shop_order',
			// Include pending orders so that in-progress/pending orders are counted
			// and reserved when validating availability at checkout.
			'status'     => array( 'wc-pending', 'wc-processing', 'wc-on-hold', 'wc-completed' ),
			'limit'      => -1,
			'return'     => 'ids',
			// phpcs:ignore WordPress.DB.SlowDBQuery.slow_db_query_meta_query

// phpcs:ignore WordPress.DB.SlowDBQuery.slow_db_query_meta_query
			'meta_query' => array(
				array(
					'key'     => $meta_key,
					'value'   => $date,
					'compare' => '=',
				),
			),
		);

		$orders = wc_get_orders( $query_args );

		return count( $orders );
	}

	/**
	 * Count orders for a specific date, type and time slot
	 *
	 * @since     1.0.0
	 * @param     string $date         The date to check (Y-m-d format).
	 * @param     string $type         'delivery' or 'pickup'.
	 * @param     string $time_slot    The time slot name.
	 * @return    int                     Number of orders
	 */
	public static function count_orders_for_time_slot( $date, $type, $time_slot ) {
		// Using WC Data Store for HPOS compatibility.
		$date_meta_key = '_aicoso_pickup_delivery_' . $type . '_date';
		$slot_meta_key = '_aicoso_pickup_delivery_' . $type . '_time_slot';

		$query_args = array(
			'type'       => 'shop_order',
			// Include pending orders to ensure time slot reservations are respected.
			'status'     => array( 'wc-pending', 'wc-processing', 'wc-on-hold', 'wc-completed' ),
			'limit'      => -1,
			'return'     => 'ids',
			// phpcs:ignore WordPress.DB.SlowDBQuery.slow_db_query_meta_query

// phpcs:ignore WordPress.DB.SlowDBQuery.slow_db_query_meta_query
			'meta_query' => array(
				'relation' => 'AND',
				array(
					'key'     => $date_meta_key,
					'value'   => $date,
					'compare' => '=',
				),
				array(
					'key'     => $slot_meta_key,
					'value'   => $time_slot,
					'compare' => '=',
				),
			),
		);

		$orders = wc_get_orders( $query_args );

		return count( $orders );
	}

	/**
	 * Count orders for a specific date, pickup location
	 *
	 * @since     1.0.0
	 * @param     string $date             The date to check (Y-m-d format).
	 * @param     string $location_name    The pickup location name.
	 * @return    int                         Number of orders
	 */
	public static function count_orders_for_pickup_location( $date, $location_name ) {
		// Using WC Data Store for HPOS compatibility.
		$query_args = array(
			'type'       => 'shop_order',
			// Include pending orders to respect pickup location reservations.
			'status'     => array( 'wc-pending', 'wc-processing', 'wc-on-hold', 'wc-completed' ),
			'limit'      => -1,
			'return'     => 'ids',
			// phpcs:ignore WordPress.DB.SlowDBQuery.slow_db_query_meta_query

// phpcs:ignore WordPress.DB.SlowDBQuery.slow_db_query_meta_query
			'meta_query' => array(
				'relation' => 'AND',
				array(
					'key'     => '_aicoso_pickup_delivery_type',
					'value'   => 'pickup',
					'compare' => '=',
				),
				array(
					'key'     => '_aicoso_pickup_delivery_pickup_date',
					'value'   => $date,
					'compare' => '=',
				),
				array(
					'key'     => '_aicoso_pickup_delivery_pickup_location',
					'value'   => $location_name,
					'compare' => '=',
				),
			),
		);

		$orders = wc_get_orders( $query_args );

		return count( $orders );
	}

	/**
	 * Format a time for display
	 *
	 * @since     1.0.0
	 * @param     string $time         Time in 24hr format (HH:MM).
	 * @return    string                  Formatted time based on WP settings
	 */
	public static function format_time( $time ) {
		// Convert HH:MM to a timestamp for today.
		$timestamp = strtotime( 'today ' . $time );

		if ( false === $timestamp ) {
			return $time;
		}

		// Format according to WordPress time format setting.
		$time_format = get_option( 'time_format', 'g:i a' );

		return date_i18n( $time_format, $timestamp );
	}

	/**
	 * Format a date for display
	 *
	 * @since     1.0.0
	 * @param     string $date         Date in Y-m-d format.
	 * @return    string                  Formatted date based on WP settings
	 */
	public static function format_date( $date ) {
		// Convert Y-m-d to a timestamp.
		$timestamp = strtotime( $date );

		if ( false === $timestamp ) {
			return $date;
		}

		// Format according to WordPress date format setting.
		$date_format = get_option( 'date_format', 'F j, Y' );

		return date_i18n( $date_format, $timestamp );
	}
}
