<?php
/**
 * Adds checkout fields support for WooCommerce Blocks
 *
 * This class provides a simpler approach to add pickup/delivery fields
 * to the WooCommerce Blocks checkout without full React integration.
 *
 * @package Aicoso_Pickup_Delivery_For_Woocommerce
 */

if ( ! defined( 'WPINC' ) ) {
	die;
}

// phpcs:disable Generic.Files.OneObjectStructurePerFile -- allow fallback exception and plugin class in same file for early bootstrap

/**
 * Ensure a minimal WP_REST_Exception is available in very early load contexts.
 * Prefer the core implementation if present, otherwise provide a tiny
 * fallback so plugin code can throw exceptions without causing a fatal.
 */
if ( ! class_exists( 'WP_REST_Exception' ) ) {
	$rest_exc_file = ABSPATH . WPINC . '/rest-api/class-wp-rest-exception.php';
	if ( file_exists( $rest_exc_file ) ) {
		require_once $rest_exc_file;
	} else {
		require_once __DIR__ . '/class-wp-rest-exception.php';
	}
}

/**
 * Class AICOSO_Pickup_Delivery_Checkout_Blocks_Support
 *
 * Provides limited integration support for WooCommerce Blocks checkout.
 * Handles registration and enqueuing of scripts used to add pickup/delivery
 * fields into the Blocks-based checkout experience.
 *
 * @package Aicoso_Pickup_Delivery_For_Woocommerce
 */
class AICOSO_Pickup_Delivery_Checkout_Blocks_Support {

	/**
	 * Initialize the blocks support.
	 */
	public static function init() {
		// Add custom fields to checkout blocks.
		add_action( 'woocommerce_blocks_checkout_block_registration', array( __CLASS__, 'register_checkout_fields' ) );

		// Add inline script to handle the fields.
		add_action( 'wp_enqueue_scripts', array( __CLASS__, 'enqueue_checkout_script' ) );

		// Process the fields when order is created.
		add_action( 'woocommerce_store_api_checkout_update_order_from_request', array( __CLASS__, 'save_checkout_fields' ), 10, 2 );

		// Alternative hook for order processing.
		add_action( 'woocommerce_checkout_order_processed', array( __CLASS__, 'save_checkout_fields_alternative' ), 10, 3 );
		add_action( 'woocommerce_checkout_update_order_meta', array( __CLASS__, 'save_checkout_fields_on_update' ), 10, 2 );

		// Add fields via checkout block hooks.
		add_action( 'woocommerce_blocks_enqueue_checkout_block_scripts_after', array( __CLASS__, 'add_checkout_fields_script' ) );

		// Add AJAX handler for fallback saving.
		add_action( 'wp_ajax_aicoso_pickup_delivery_save_order_data', array( __CLASS__, 'ajax_save_order_data' ) );
		add_action( 'wp_ajax_nopriv_aicoso_pickup_delivery_save_order_data', array( __CLASS__, 'ajax_save_order_data' ) );

		// Add AJAX handler for session storage.
		add_action( 'wp_ajax_aicoso_pickup_delivery_save_to_session', array( __CLASS__, 'ajax_save_to_session' ) );
		add_action( 'wp_ajax_nopriv_aicoso_pickup_delivery_save_to_session', array( __CLASS__, 'ajax_save_to_session' ) );

		// Hook into order creation to retrieve session data.
		add_action( 'woocommerce_new_order', array( __CLASS__, 'save_from_session' ), 10, 2 );
	}

	/**
	 * Register checkout fields for blocks.
	 */
	public static function register_checkout_fields() {
		// This is a placeholder for future full blocks integration.
		// For now, we inject via JavaScript.
	}

	/**
	 * Enqueue scripts for checkout.
	 */
	public static function enqueue_checkout_script() {
		if ( ! is_checkout() ) {
			return;
		}

		// Check if blocks checkout is being used.
		if ( ! has_block( 'woocommerce/checkout' ) ) {
			return;
		}

		$settings = AICOSO_Pickup_Delivery_Data::get_settings();

		// Enqueue the new refactored checkout fields script.
		wp_enqueue_script(
			'aicoso-pickup-delivery-blocks-checkout-fields',
			AICOSO_PICKUP_DELIVERY_PLUGIN_URL . 'assets/js/blocks-checkout-fields.js',
			array(),
			AICOSO_PICKUP_DELIVERY_VERSION,
			true
		);

		// Localize the script with settings and data.
		wp_localize_script(
			'aicoso-pickup-delivery-blocks-checkout-fields',
			'wcPickupDeliveryBlocks',
			self::get_localized_script_data( $settings )
		);

		// Enqueue the CSS file for blocks checkout.
		wp_enqueue_style(
			'aicoso-pickup-delivery-blocks-checkout',
			AICOSO_PICKUP_DELIVERY_PLUGIN_URL . 'assets/css/blocks-checkout.css',
			array( 'wc-blocks-style' ),
			AICOSO_PICKUP_DELIVERY_VERSION
		);

		// Enqueue our blocks integration script.
		$integration_path = AICOSO_PICKUP_DELIVERY_PLUGIN_PATH . 'assets/js/blocks-integration.js';
		if ( file_exists( $integration_path ) ) {
			wp_enqueue_script(
				'aicoso-pickup-delivery-blocks-integration',
				AICOSO_PICKUP_DELIVERY_PLUGIN_URL . 'assets/js/blocks-integration.js',
				array( 'wc-blocks-checkout', 'wp-element', 'wp-data' ),
				AICOSO_PICKUP_DELIVERY_VERSION,
				true
			);
		}

		$ajax_fallback_path = AICOSO_PICKUP_DELIVERY_PLUGIN_PATH . 'assets/js/blocks-ajax-fallback.js';
		if ( file_exists( $ajax_fallback_path ) ) {
			// Enqueue AJAX fallback script.
			wp_enqueue_script(
				'aicoso-pickup-delivery-blocks-ajax',
				AICOSO_PICKUP_DELIVERY_PLUGIN_URL . 'assets/js/blocks-ajax-fallback.js',
				array( 'jquery' ),
				AICOSO_PICKUP_DELIVERY_VERSION,
				true
			);

			// Localize script for AJAX.
			wp_localize_script(
				'aicoso-pickup-delivery-blocks-ajax',
				'wcPickupDeliveryAjax',
				array(
					'ajaxUrl' => admin_url( 'admin-ajax.php' ),
					'nonce'   => wp_create_nonce( 'aicoso_pickup_delivery_ajax' ),
				)
			);
		}

		$session_path = AICOSO_PICKUP_DELIVERY_PLUGIN_PATH . 'assets/js/blocks-session-storage.js';
		if ( file_exists( $session_path ) ) {
			wp_enqueue_script(
				'aicoso-pickup-delivery-blocks-session',
				AICOSO_PICKUP_DELIVERY_PLUGIN_URL . 'assets/js/blocks-session-storage.js',
				array( 'jquery' ),
				AICOSO_PICKUP_DELIVERY_VERSION,
				true
			);

			// Localize session script.
			wp_localize_script(
				'aicoso-pickup-delivery-blocks-session',
				'wcPickupDeliverySession',
				array(
					'ajaxUrl' => admin_url( 'admin-ajax.php' ),
					'nonce'   => wp_create_nonce( 'aicoso_pickup_delivery_session' ),
				)
			);
		}
	}

	/**
	 * Get localized script data.
	 *
	 * @param array $settings The settings array.
	 * @return array The localized script data.
	 */
	private static function get_localized_script_data( $settings ) {
		$enable_selection = isset( $settings['enable_order_type_selection'] ) ? $settings['enable_order_type_selection'] : 'yes';
		$default_type     = isset( $settings['default_order_type'] ) ? $settings['default_order_type'] : 'delivery';

		// Check if any fields are enabled.
		$has_delivery_fields = ( isset( $settings['enable_delivery_date'] ) && 'yes' === $settings['enable_delivery_date'] ) ||
								( isset( $settings['enable_delivery_time_slots'] ) && 'yes' === $settings['enable_delivery_time_slots'] );
		$has_pickup_fields   = ( isset( $settings['enable_pickup_date'] ) && 'yes' === $settings['enable_pickup_date'] ) ||
								( isset( $settings['enable_pickup_time_slots'] ) && 'yes' === $settings['enable_pickup_time_slots'] );

		// Prepare date-related settings for Blocks checkout.
		$delivery_blackout_dates = AICOSO_Pickup_Delivery_Data::get_blackout_dates( 'delivery' );
		$pickup_blackout_dates   = AICOSO_Pickup_Delivery_Data::get_blackout_dates( 'pickup' );

		$tz = wp_timezone_string();
		if ( empty( $tz ) ) {
			$tz = 'UTC';
		}

		$now = new \DateTime( 'now', new \DateTimeZone( $tz ) );

		$delivery_lead = (int) ( isset( $settings['delivery_lead_time'] ) ? $settings['delivery_lead_time'] : 1 );
		$pickup_lead   = (int) ( isset( $settings['pickup_lead_time'] ) ? $settings['pickup_lead_time'] : 1 );

		$delivery_min_date = $now->modify( " +{$delivery_lead} days" )->format( 'Y-m-d' );

		// Reset now for pickup calculation.
		$now             = new \DateTime( 'now', new \DateTimeZone( $tz ) );
		$pickup_min_date = $now->modify( " +{$pickup_lead} days" )->format( 'Y-m-d' );

		$delivery_days = isset( $settings['delivery_days'] ) ? (array) $settings['delivery_days'] : array( 1, 2, 3, 4, 5 );
		$pickup_days   = isset( $settings['pickup_days'] ) ? (array) $settings['pickup_days'] : array( 1, 2, 3, 4, 5 );

		return array(
			'dateSettings'    => array(
				'delivery' => array(
					'blackoutDates' => array_values( (array) $delivery_blackout_dates ),
					'minDate'       => $delivery_min_date,
					'availableDays' => array_values( array_map( 'intval', $delivery_days ) ),
				),
				'pickup'   => array(
					'blackoutDates' => array_values( (array) $pickup_blackout_dates ),
					'minDate'       => $pickup_min_date,
					'availableDays' => array_values( array_map( 'intval', $pickup_days ) ),
				),
			),

			// Localized settings and strings.
			'settings'        => array(
				'enableOrderTypeSelection' => $enable_selection,
				'defaultOrderType'         => $default_type,
				'enableDeliveryDate'       => isset( $settings['enable_delivery_date'] ) ? $settings['enable_delivery_date'] : 'yes',
				'enableDeliveryTimeSlots'  => isset( $settings['enable_delivery_time_slots'] ) ? $settings['enable_delivery_time_slots'] : 'yes',
				'enablePickupDate'         => isset( $settings['enable_pickup_date'] ) ? $settings['enable_pickup_date'] : 'yes',
				'enablePickupTimeSlots'    => isset( $settings['enable_pickup_time_slots'] ) ? $settings['enable_pickup_time_slots'] : 'yes',
				'hasDeliveryFields'        => $has_delivery_fields,
				'hasPickupFields'          => $has_pickup_fields,
			),
			'strings'         => array(
				'deliveryPickupOptions'  => __( 'Delivery/Pickup Options', 'aicoso-pickup-and-delivery-for-woocommerce' ),
				'deliveryOptions'        => __( 'Delivery Options', 'aicoso-pickup-and-delivery-for-woocommerce' ),
				'pickupOptions'          => __( 'Pickup Options', 'aicoso-pickup-and-delivery-for-woocommerce' ),
				'orderType'              => __( 'Order Type', 'aicoso-pickup-and-delivery-for-woocommerce' ),
				'delivery'               => __( 'Delivery', 'aicoso-pickup-and-delivery-for-woocommerce' ),
				'pickup'                 => __( 'Pickup', 'aicoso-pickup-and-delivery-for-woocommerce' ),
				'deliveryDate'           => __( 'Delivery Date', 'aicoso-pickup-and-delivery-for-woocommerce' ),
				'deliveryTime'           => __( 'Delivery Time', 'aicoso-pickup-and-delivery-for-woocommerce' ),
				'pickupDate'             => __( 'Pickup Date', 'aicoso-pickup-and-delivery-for-woocommerce' ),
				'pickupTime'             => __( 'Pickup Time', 'aicoso-pickup-and-delivery-for-woocommerce' ),
				'pickupLocation'         => __( 'Pickup Location', 'aicoso-pickup-and-delivery-for-woocommerce' ),
				'selectTime'             => __( 'Select a time', 'aicoso-pickup-and-delivery-for-woocommerce' ),
				'selectLocation'         => __( 'Select a location', 'aicoso-pickup-and-delivery-for-woocommerce' ),
				// Validation / UI messages.
				'dateBlackout'           => __( 'This date is a blackout date and is not available.', 'aicoso-pickup-and-delivery-for-woocommerce' ),
				'dateWeekdayUnavailable' => __( 'Selected day of week is not available for this service.', 'aicoso-pickup-and-delivery-for-woocommerce' ),
				'dateBeforeMin'          => __( 'Selected date is earlier than the required lead time.', 'aicoso-pickup-and-delivery-for-woocommerce' ),
			),
			'timeSlots'       => self::get_time_slots(),
			'pickupLocations' => self::get_pickup_locations(),
			'ajaxUrl'         => admin_url( 'admin-ajax.php' ),
			'nonce'           => wp_create_nonce( 'aicoso_pickup_delivery_checkout' ),
		);
	}

	/**
	 * Get time slots.
	 */
	private static function get_time_slots() {
		// TODO: Get from settings/database.
		return array(
			array(
				'value' => 'morning',
				'label' => __( 'Morning (9AM - 12PM)', 'aicoso-pickup-and-delivery-for-woocommerce' ),
			),
			array(
				'value' => 'afternoon',
				'label' => __( 'Afternoon (12PM - 5PM)', 'aicoso-pickup-and-delivery-for-woocommerce' ),
			),
			array(
				'value' => 'evening',
				'label' => __( 'Evening (5PM - 8PM)', 'aicoso-pickup-and-delivery-for-woocommerce' ),
			),
		);
	}

	/**
	 * Get pickup locations.
	 */
	private static function get_pickup_locations() {
		// TODO: Get from settings/database.
		return array(
			array(
				'value' => 'main-store',
				'label' => __( 'Main Store', 'aicoso-pickup-and-delivery-for-woocommerce' ),
			),
			array(
				'value' => 'warehouse',
				'label' => __( 'Warehouse', 'aicoso-pickup-and-delivery-for-woocommerce' ),
			),
			array(
				'value' => 'downtown',
				'label' => __( 'Downtown Location', 'aicoso-pickup-and-delivery-for-woocommerce' ),
			),
		);
	}


	/**
	 * Add checkout fields script for blocks.
	 */
	public static function add_checkout_fields_script() {
		// This is another hook point for adding scripts specifically for checkout blocks.
		wp_enqueue_script(
			'aicoso-pickup-delivery-checkout-blocks',
			AICOSO_PICKUP_DELIVERY_PLUGIN_URL . 'assets/js/checkout-blocks.js',
			array( 'wc-blocks-checkout', 'wp-element', 'wp-data' ),
			AICOSO_PICKUP_DELIVERY_VERSION,
			true
		);

		// Localize the script.
		wp_localize_script(
			'aicoso-pickup-delivery-checkout-blocks',
			'wcPickupDeliveryData',
			array(
				'ajaxUrl' => admin_url( 'admin-ajax.php' ),
				'nonce'   => wp_create_nonce( 'aicoso_pickup_delivery_blocks' ),
			)
		);
	}

	/**
	 * Save checkout fields from blocks checkout.
	 *
	 * @param WC_Order        $order The order object.
	 * @param WP_REST_Request $request The request object.
	 * @throws \WP_REST_Exception When a supplied date/time is unavailable or an internal error occurs.
	 */
	public static function save_checkout_fields( $order, $request ) {
		// REST exception support is handled at file load; proceed with logic.
		try {
			if ( ! $order instanceof \WC_Order ) {
				return;
			}

			// Try different methods to get the data.
			$data = array();

			// 1. Check extensions data (Blocks way)
			$params = $request->get_params();
			if ( isset( $params['extensions']['aicoso-pickup-and-delivery-for-woocommerce'] ) ) {
				$data = $params['extensions']['aicoso-pickup-and-delivery-for-woocommerce'];
			}

			// 2. Check other_fields (some versions use this).
			if ( empty( $data ) && isset( $params['other_fields'] ) ) {

				foreach ( $params['other_fields'] as $key => $value ) {
					if ( strpos( $key, 'aicoso_pickup_delivery' ) !== false || strpos( $key, 'aicoso-pickup-and-delivery-for-woocommerce' ) !== false ) {
						// Extract field name.
						$field_name          = str_replace( array( 'aicoso_pickup_delivery_', 'aicoso-pickup-delivery/' ), '', $key );
						$data[ $field_name ] = sanitize_text_field( $value );
					}
				}
				// Data collection completed, will be processed below.
			}

			// 3. Check billing/shipping sections.
			if ( empty( $data ) ) {
				$billing    = $request->get_param( 'billing_address' );
				$shipping   = $request->get_param( 'shipping_address' );
				$additional = $request->get_param( 'additional_fields' );

				// Check billing.
				if ( is_array( $billing ) ) {
					foreach ( $billing as $key => $value ) {
						if ( strpos( $key, 'aicoso_pickup_delivery' ) !== false ) {
							$field_name          = str_replace( 'aicoso_pickup_delivery_', '', $key );
							$data[ $field_name ] = sanitize_text_field( $value );
						}
						// Non-pickup-delivery fields are automatically skipped.
					}
				}

				// Check additional fields.
				if ( is_array( $additional ) ) {
					foreach ( $additional as $key => $value ) {
						if ( strpos( $key, 'aicoso_pickup_delivery' ) !== false || strpos( $key, 'aicoso-pickup-and-delivery-for-woocommerce' ) !== false ) {
							$field_name          = str_replace( array( 'aicoso_pickup_delivery_', 'aicoso-pickup-delivery/' ), '', $key );
							$data[ $field_name ] = sanitize_text_field( $value );
						}
					}
				}
			}

			// 4. Also check POST data as fallback.
			// phpcs:disable WordPress.Security.NonceVerification.Missing -- Nonce verification handled by WooCommerce checkout process
			if ( empty( $data ) && ! empty( $_POST ) ) {

				// Check for order type - both field names.
				if ( isset( $_POST['aicoso_pickup_delivery_type'] ) ) {
					$data['type'] = sanitize_text_field( wp_unslash( $_POST['aicoso_pickup_delivery_type'] ) );
				} elseif ( isset( $_POST['order_type'] ) ) {
					$data['type'] = sanitize_text_field( wp_unslash( $_POST['order_type'] ) );
				}

				// Check for delivery fields.
				if ( isset( $_POST['aicoso_pickup_delivery_delivery_date'] ) ) {
					$data['delivery_date'] = sanitize_text_field( wp_unslash( $_POST['aicoso_pickup_delivery_delivery_date'] ) );
				} elseif ( isset( $_POST['delivery_date'] ) ) {
					$data['delivery_date'] = sanitize_text_field( wp_unslash( $_POST['delivery_date'] ) );
				}

				if ( isset( $_POST['aicoso_pickup_delivery_delivery_time_slot'] ) ) {
					$data['delivery_time_slot'] = sanitize_text_field( wp_unslash( $_POST['aicoso_pickup_delivery_delivery_time_slot'] ) );
				} elseif ( isset( $_POST['delivery_time'] ) ) {
					$data['delivery_time_slot'] = sanitize_text_field( wp_unslash( $_POST['delivery_time'] ) );
				}

				// Check for pickup fields.
				if ( isset( $_POST['aicoso_pickup_delivery_pickup_location'] ) ) {
					$data['pickup_location'] = sanitize_text_field( wp_unslash( $_POST['aicoso_pickup_delivery_pickup_location'] ) );
				} elseif ( isset( $_POST['pickup_location'] ) ) {
					$data['pickup_location'] = sanitize_text_field( wp_unslash( $_POST['pickup_location'] ) );
				}
				if ( isset( $_POST['aicoso_pickup_delivery_pickup_date'] ) ) {
					$data['pickup_date'] = sanitize_text_field( wp_unslash( $_POST['aicoso_pickup_delivery_pickup_date'] ) );
				} elseif ( isset( $_POST['pickup_date'] ) ) {
					$data['pickup_date'] = sanitize_text_field( wp_unslash( $_POST['pickup_date'] ) );
				}
				if ( isset( $_POST['aicoso_pickup_delivery_pickup_time_slot'] ) ) {
					$data['pickup_time_slot'] = sanitize_text_field( wp_unslash( $_POST['aicoso_pickup_delivery_pickup_time_slot'] ) );
				} elseif ( isset( $_POST['pickup_time'] ) ) {
					$data['pickup_time_slot'] = sanitize_text_field( wp_unslash( $_POST['pickup_time'] ) );
				}
			}
		// phpcs:enable WordPress.Security.NonceVerification.Missing

			// Save the data if we found any.
			if ( ! empty( $data ) ) {

				// Validate availability for store API requests before saving.
				// If date or timeslot is not available, return a REST error so the checkout can fail.
				try {
					if ( isset( $data['delivery_date'] ) && ! empty( $data['delivery_date'] ) ) {
						if ( ! AICOSO_Pickup_Delivery_Date_Manager::is_date_available( $data['delivery_date'], 'delivery' ) ) {
							self::maybe_send_rest_error( 'aicoso_date_unavailable', __( 'The selected delivery date is not available. Please choose another date.', 'aicoso-pickup-and-delivery-for-woocommerce' ), 400, $request );
							return;
						}
						// If time slot provided, validate it as well.
						if ( isset( $data['delivery_time_slot'] ) && ! empty( $data['delivery_time_slot'] ) ) {
							if ( ! AICOSO_Pickup_Delivery_Timeslot_Manager::is_time_slot_available( $data['delivery_date'], $data['delivery_time_slot'], 'delivery' ) ) {
								self::maybe_send_rest_error( 'aicoso_timeslot_unavailable', __( 'The selected delivery time is not available. Please choose another time.', 'aicoso-pickup-and-delivery-for-woocommerce' ), 400, $request );
								return;
							}
						}
					}

					if ( isset( $data['pickup_date'] ) && ! empty( $data['pickup_date'] ) ) {
						if ( ! AICOSO_Pickup_Delivery_Date_Manager::is_date_available( $data['pickup_date'], 'pickup' ) ) {
							self::maybe_send_rest_error( 'aicoso_date_unavailable', __( 'The selected pickup date is not available. Please choose another date.', 'aicoso-pickup-and-delivery-for-woocommerce' ), 400, $request );
							return;
						}
						if ( isset( $data['pickup_time_slot'] ) && ! empty( $data['pickup_time_slot'] ) ) {
							if ( ! AICOSO_Pickup_Delivery_Timeslot_Manager::is_time_slot_available( $data['pickup_date'], $data['pickup_time_slot'], 'pickup' ) ) {
								self::maybe_send_rest_error( 'aicoso_timeslot_unavailable', __( 'The selected pickup time is not available. Please choose another time.', 'aicoso-pickup-and-delivery-for-woocommerce' ), 400, $request );
								return;
							}
						}
					}
				} catch ( \WP_REST_Exception $e ) {
					// Re-throw to allow REST API to return an error response.
					throw $e;
				}

				// Save order type - handle all possible field names.
				if ( isset( $data['type'] ) || isset( $data['order_type'] ) ) {
					$order_type = isset( $data['type'] ) ? $data['type'] : $data['order_type'];
					$order->update_meta_data( '_aicoso_pickup_delivery_type', $order_type );
				}

				// Save delivery details.
				if ( isset( $data['delivery_date'] ) ) {
					$order->update_meta_data( '_aicoso_pickup_delivery_delivery_date', $data['delivery_date'] );
				}

				if ( isset( $data['delivery_time_slot'] ) || isset( $data['delivery_time'] ) ) {
					$time_slot = isset( $data['delivery_time_slot'] ) ? $data['delivery_time_slot'] : $data['delivery_time'];
					$order->update_meta_data( '_aicoso_pickup_delivery_delivery_time_slot', $time_slot );
				}

				// Save pickup details.
				if ( isset( $data['pickup_location'] ) ) {
					$order->update_meta_data( '_aicoso_pickup_delivery_pickup_location', $data['pickup_location'] );
				}

				if ( isset( $data['pickup_date'] ) ) {
					$order->update_meta_data( '_aicoso_pickup_delivery_pickup_date', $data['pickup_date'] );
				}

				if ( isset( $data['pickup_time_slot'] ) || isset( $data['pickup_time'] ) ) {
					$time_slot = isset( $data['pickup_time_slot'] ) ? $data['pickup_time_slot'] : $data['pickup_time'];
					$order->update_meta_data( '_aicoso_pickup_delivery_pickup_time_slot', $time_slot );
				}

				$order->save();

			}
		} catch ( \WP_REST_Exception $e ) {
			// Re-throw REST exceptions so the Store API can return proper errors.
			throw $e;
		} catch ( \Throwable $e ) {
			// Log unexpected errors and return a generic REST error to avoid a raw 500.
			// phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log -- debug fallback logging
			error_log( 'AICOSO Pickup/Delivery error in save_checkout_fields: ' . $e->getMessage() );
			self::maybe_send_rest_error( 'aicoso_internal_error', __( 'An internal error occurred while processing pickup/delivery data. Please try again or contact support.', 'aicoso-pickup-and-delivery-for-woocommerce' ), 500, isset( $request ) ? $request : null );
			return;
		}
	}

	/**
	 * Helper: send a REST-friendly error response or throw if not a REST request.
	 *
	 * @param string $code Error code.
	 * @param string $message Error message.
	 * @param int    $status HTTP status code.
	 * @param mixed  $request Optional request object.
	 * @throws \WP_REST_Exception When not in a REST context and exception needs to be thrown.
	 */
	private static function maybe_send_rest_error( $code, $message, $status = 400, $request = null ) {
		$in_rest = ( defined( 'REST_REQUEST' ) && REST_REQUEST ) || ( is_object( $request ) && is_a( $request, 'WP_REST_Request' ) );

		if ( $in_rest ) {
			if ( function_exists( 'wp_send_json' ) ) {
				wp_send_json(
					array(
						'code'    => $code,
						'message' => $message,
					),
					$status
				);
			}

			if ( ! headers_sent() ) {
				header( 'Content-Type: application/json', true, $status );
			}
			// phpcs:ignore WordPress.Security.EscapeOutput.ExceptionNotEscaped -- output is JSON encoded via wp_json_encode
			echo wp_json_encode(
				array(
					'code'    => $code,
					'message' => $message,
				)
			);
			exit;
		}

		if ( class_exists( 'WP_REST_Exception' ) ) {
			// phpcs:ignore WordPress.Security.EscapeOutput.ExceptionNotEscaped -- values are passed to WP_REST_Exception for REST response handling.
			throw new \WP_REST_Exception( $code, $message, array( 'status' => $status ) );
		}

		// phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log -- debug fallback logging
		error_log( 'AICOSO Pickup/Delivery: ' . $message );
		if ( function_exists( 'wc_add_notice' ) ) {
			wc_add_notice( $message, 'error' );
		}
	}

	/**
	 * Alternative save method for checkout fields
	 *
	 * @param int      $order_id The order ID.
	 * @param array    $posted_data Posted data.
	 * @param WC_Order $order The order object.
	 */
	public static function save_checkout_fields_alternative( $order_id, $posted_data, $order ) {
		// Try to get data from posted data.
		$order_type = isset( $posted_data['aicoso_pickup_delivery_type'] ) ? sanitize_text_field( $posted_data['aicoso_pickup_delivery_type'] ) : '';

		if ( empty( $order_type ) ) {
			$order_type = isset( $posted_data['order_type'] ) ? sanitize_text_field( $posted_data['order_type'] ) : '';
		}

		if ( ! empty( $order_type ) ) {
			$order->update_meta_data( '_aicoso_pickup_delivery_type', $order_type );

			if ( 'delivery' === $order_type ) {
				if ( isset( $posted_data['aicoso_pickup_delivery_delivery_date'] ) ) {
					$order->update_meta_data( '_aicoso_pickup_delivery_delivery_date', sanitize_text_field( $posted_data['aicoso_pickup_delivery_delivery_date'] ) );
				} elseif ( isset( $posted_data['delivery_date'] ) ) {
					$order->update_meta_data( '_aicoso_pickup_delivery_delivery_date', sanitize_text_field( $posted_data['delivery_date'] ) );
				}

				if ( isset( $posted_data['aicoso_pickup_delivery_delivery_time_slot'] ) ) {
					$order->update_meta_data( '_aicoso_pickup_delivery_delivery_time_slot', sanitize_text_field( $posted_data['aicoso_pickup_delivery_delivery_time_slot'] ) );
				} elseif ( isset( $posted_data['delivery_time'] ) ) {
					$order->update_meta_data( '_aicoso_pickup_delivery_delivery_time_slot', sanitize_text_field( $posted_data['delivery_time'] ) );
				}
			} elseif ( 'pickup' === $order_type ) {
				if ( isset( $posted_data['aicoso_pickup_delivery_pickup_location'] ) ) {
					$order->update_meta_data( '_aicoso_pickup_delivery_pickup_location', sanitize_text_field( $posted_data['aicoso_pickup_delivery_pickup_location'] ) );
				} elseif ( isset( $posted_data['pickup_location'] ) ) {
					$order->update_meta_data( '_aicoso_pickup_delivery_pickup_location', sanitize_text_field( $posted_data['pickup_location'] ) );
				}

				if ( isset( $posted_data['aicoso_pickup_delivery_pickup_date'] ) ) {
					$order->update_meta_data( '_aicoso_pickup_delivery_pickup_date', sanitize_text_field( $posted_data['aicoso_pickup_delivery_pickup_date'] ) );
				} elseif ( isset( $posted_data['pickup_date'] ) ) {
					$order->update_meta_data( '_aicoso_pickup_delivery_pickup_date', sanitize_text_field( $posted_data['pickup_date'] ) );
				}

				if ( isset( $posted_data['aicoso_pickup_delivery_pickup_time_slot'] ) ) {
					$order->update_meta_data( '_aicoso_pickup_delivery_pickup_time_slot', sanitize_text_field( $posted_data['aicoso_pickup_delivery_pickup_time_slot'] ) );
				} elseif ( isset( $posted_data['pickup_time'] ) ) {
					$order->update_meta_data( '_aicoso_pickup_delivery_pickup_time_slot', sanitize_text_field( $posted_data['pickup_time'] ) );
				}
			}

			$order->save();
		}
	}

	/**
	 * Save fields on order update
	 *
	 * @param int   $order_id The order ID.
	 * @param array $data The data array.
	 */
	public static function save_checkout_fields_on_update( $order_id, $data ) {
		$order = wc_get_order( $order_id );
		if ( ! $order ) {
			return;
		}

		// Check for our fields in the data.
		if ( isset( $data['aicoso_pickup_delivery_type'] ) ) {
			$order->update_meta_data( '_aicoso_pickup_delivery_type', sanitize_text_field( $data['aicoso_pickup_delivery_type'] ) );
		}

		// Save delivery fields.
		if ( isset( $data['aicoso_pickup_delivery_delivery_date'] ) ) {
			$order->update_meta_data( '_aicoso_pickup_delivery_delivery_date', sanitize_text_field( $data['aicoso_pickup_delivery_delivery_date'] ) );
		}

		if ( isset( $data['aicoso_pickup_delivery_delivery_time_slot'] ) ) {
			$order->update_meta_data( '_aicoso_pickup_delivery_delivery_time_slot', sanitize_text_field( $data['aicoso_pickup_delivery_delivery_time_slot'] ) );
		}

		// Save pickup fields.
		if ( isset( $data['aicoso_pickup_delivery_pickup_location'] ) ) {
			$order->update_meta_data( '_aicoso_pickup_delivery_pickup_location', sanitize_text_field( $data['aicoso_pickup_delivery_pickup_location'] ) );
		}

		if ( isset( $data['aicoso_pickup_delivery_pickup_date'] ) ) {
			$order->update_meta_data( '_aicoso_pickup_delivery_pickup_date', sanitize_text_field( $data['aicoso_pickup_delivery_pickup_date'] ) );
		}

		if ( isset( $data['aicoso_pickup_delivery_pickup_time_slot'] ) ) {
			$order->update_meta_data( '_aicoso_pickup_delivery_pickup_time_slot', sanitize_text_field( $data['aicoso_pickup_delivery_pickup_time_slot'] ) );
		}

		$order->save();
	}

	/**
	 * AJAX handler to save order data
	 */
	public static function ajax_save_order_data() {
		// Verify nonce.
		if ( ! isset( $_POST['nonce'] ) || ! wp_verify_nonce( sanitize_text_field( wp_unslash( $_POST['nonce'] ) ), 'aicoso_pickup_delivery_blocks_nonce' ) ) {
			wp_send_json_error( 'Invalid nonce' );
		}

		// Get order ID.
		$order_id = isset( $_POST['order_id'] ) ? absint( $_POST['order_id'] ) : 0;
		if ( ! $order_id ) {
			wp_send_json_error( 'Invalid order ID' );
		}

		$order = wc_get_order( $order_id );
		if ( ! $order ) {
			wp_send_json_error( 'Order not found' );
		}

		// Save order type.
		if ( isset( $_POST['order_type'] ) ) {
			$order->update_meta_data( '_aicoso_pickup_delivery_type', sanitize_text_field( wp_unslash( $_POST['order_type'] ) ) );
		}

		// Save delivery fields.
		if ( isset( $_POST['delivery_date'] ) ) {
			$order->update_meta_data( '_aicoso_pickup_delivery_delivery_date', sanitize_text_field( wp_unslash( $_POST['delivery_date'] ) ) );
		}

		if ( isset( $_POST['delivery_time'] ) ) {
			$order->update_meta_data( '_aicoso_pickup_delivery_delivery_time_slot', sanitize_text_field( wp_unslash( $_POST['delivery_time'] ) ) );
		}

		// Save pickup fields.
		if ( isset( $_POST['pickup_location'] ) ) {
			$order->update_meta_data( '_aicoso_pickup_delivery_pickup_location', sanitize_text_field( wp_unslash( $_POST['pickup_location'] ) ) );
		}

		if ( isset( $_POST['pickup_date'] ) ) {
			$order->update_meta_data( '_aicoso_pickup_delivery_pickup_date', sanitize_text_field( wp_unslash( $_POST['pickup_date'] ) ) );
		}

		if ( isset( $_POST['pickup_time'] ) ) {
			$order->update_meta_data( '_aicoso_pickup_delivery_pickup_time_slot', sanitize_text_field( wp_unslash( $_POST['pickup_time'] ) ) );
		}

		$order->save();

		wp_send_json_success( 'Data saved successfully' );
	}

	/**
	 * AJAX handler to save data to session
	 */
	public static function ajax_save_to_session() {
		// Verify nonce.
		if ( ! isset( $_POST['nonce'] ) || ! wp_verify_nonce( sanitize_text_field( wp_unslash( $_POST['nonce'] ) ), 'aicoso_pickup_delivery_session_nonce' ) ) {
			wp_send_json_error( 'Invalid nonce' );
		}

		// Initialize session if not started.
		if ( ! WC()->session ) {
			wp_send_json_error( 'Session not available' );
		}

		// Save data to session.
		if ( isset( $_POST['order_type'] ) ) {
			WC()->session->set( 'aicoso_pickup_delivery_type', sanitize_text_field( wp_unslash( $_POST['order_type'] ) ) );
		}

		// Save delivery fields.
		if ( isset( $_POST['delivery_date'] ) ) {
			WC()->session->set( 'aicoso_pickup_delivery_delivery_date', sanitize_text_field( wp_unslash( $_POST['delivery_date'] ) ) );
		}

		if ( isset( $_POST['delivery_time'] ) ) {
			WC()->session->set( 'aicoso_pickup_delivery_delivery_time_slot', sanitize_text_field( wp_unslash( $_POST['delivery_time'] ) ) );
		}

		// Save pickup fields.
		if ( isset( $_POST['pickup_location'] ) ) {
			WC()->session->set( 'aicoso_pickup_delivery_pickup_location', sanitize_text_field( wp_unslash( $_POST['pickup_location'] ) ) );
		}

		if ( isset( $_POST['pickup_date'] ) ) {
			WC()->session->set( 'aicoso_pickup_delivery_pickup_date', sanitize_text_field( wp_unslash( $_POST['pickup_date'] ) ) );
		}

		if ( isset( $_POST['pickup_time'] ) ) {
			WC()->session->set( 'aicoso_pickup_delivery_pickup_time_slot', sanitize_text_field( wp_unslash( $_POST['pickup_time'] ) ) );
		}

		wp_send_json_success( 'Data saved to session' );
	}

	/**
	 * Save data from session when order is created
	 *
	 * @param int      $order_id The order ID.
	 * @param WC_Order $order The order object.
	 */
	public static function save_from_session( $order_id, $order = null ) {
		if ( ! $order ) {
			$order = wc_get_order( $order_id );
		}

		if ( ! $order || ! WC()->session ) {
			return;
		}

		// Get data from session.
		$order_type = WC()->session->get( 'aicoso_pickup_delivery_type' );

		if ( $order_type ) {
			$order->update_meta_data( '_aicoso_pickup_delivery_type', $order_type );

			if ( 'delivery' === $order_type ) {
				$delivery_date = WC()->session->get( 'aicoso_pickup_delivery_delivery_date' );
				$delivery_time = WC()->session->get( 'aicoso_pickup_delivery_delivery_time_slot' );

				if ( $delivery_date ) {
					$order->update_meta_data( '_aicoso_pickup_delivery_delivery_date', $delivery_date );
				}

				if ( $delivery_time ) {
					$order->update_meta_data( '_aicoso_pickup_delivery_delivery_time_slot', $delivery_time );
				}
			} elseif ( 'pickup' === $order_type ) {
				$pickup_location = WC()->session->get( 'aicoso_pickup_delivery_pickup_location' );
				$pickup_date     = WC()->session->get( 'aicoso_pickup_delivery_pickup_date' );
				$pickup_time     = WC()->session->get( 'aicoso_pickup_delivery_pickup_time_slot' );

				if ( $pickup_location ) {
					$order->update_meta_data( '_aicoso_pickup_delivery_pickup_location', $pickup_location );
				}

				if ( $pickup_date ) {
					$order->update_meta_data( '_aicoso_pickup_delivery_pickup_date', $pickup_date );
				}

				if ( $pickup_time ) {
					$order->update_meta_data( '_aicoso_pickup_delivery_pickup_time_slot', $pickup_time );
				}
			}

			$order->save();

			// Clear session data.
			WC()->session->set( 'aicoso_pickup_delivery_type', null );
			WC()->session->set( 'aicoso_pickup_delivery_delivery_date', null );
			WC()->session->set( 'aicoso_pickup_delivery_delivery_time_slot', null );
			WC()->session->set( 'aicoso_pickup_delivery_pickup_location', null );
			WC()->session->set( 'aicoso_pickup_delivery_pickup_date', null );
			WC()->session->set( 'aicoso_pickup_delivery_pickup_time_slot', null );
		}
	}
}
