<?php
/**
 * WooCommerce Blocks Integration
 *
 * @package Aicoso_Pickup_Delivery_For_Woocommerce
 */

use Automattic\WooCommerce\Blocks\Integrations\IntegrationInterface;

/**
 * Class for integrating with WooCommerce Blocks.
 */
class AICOSO_Pickup_Delivery_Blocks_Integration implements IntegrationInterface {

	/**
	 * The name of the integration.
	 *
	 * @return string
	 */
	public function get_name() {
		return 'aicoso-pickup-and-delivery-for-woocommerce';
	}

	/**
	 * When called invokes any initialization/setup for the integration.
	 */
	public function initialize() {
		$this->register_block_frontend_scripts();
		$this->register_block_editor_scripts();
		$this->register_main_integration();
		add_action( 'woocommerce_blocks_checkout_block_registration', array( $this, 'register_checkout_blocks' ) );
		add_filter( 'woocommerce_blocks_checkout_update_order_from_request', array( $this, 'update_order_from_request' ), 10, 2 );
	}

	/**
	 * Register scripts for the frontend.
	 */
	public function register_block_frontend_scripts() {
		$script_url  = AICOSO_PICKUP_DELIVERY_PLUGIN_URL . 'assets/js/blocks/frontend.js';
		$script_path = AICOSO_PICKUP_DELIVERY_PLUGIN_PATH . 'assets/js/blocks/frontend.js';

		$script_asset_path = AICOSO_PICKUP_DELIVERY_PLUGIN_PATH . 'assets/js/blocks/frontend.asset.php';
		$script_asset      = file_exists( $script_asset_path )
			? require $script_asset_path
			: array(
				'dependencies' => array(),
				'version'      => $this->get_file_version( $script_path ),
			);

		wp_register_script(
			'aicoso-pickup-delivery-blocks-integration',
			$script_url,
			$script_asset['dependencies'],
			$script_asset['version'],
			true
		);

		wp_set_script_translations(
			'aicoso-pickup-delivery-blocks-integration',
			'aicoso-pickup-and-delivery-for-woocommerce',
			AICOSO_PICKUP_DELIVERY_PLUGIN_PATH . 'languages/'
		);

		// Pass data to frontend.
		wp_localize_script(
			'aicoso-pickup-delivery-blocks-integration',
			'wcPickupDeliveryData',
			array(
				'ajaxUrl'  => admin_url( 'admin-ajax.php' ),
				'nonce'    => wp_create_nonce( 'aicoso_pickup_delivery_nonce' ),
				'settings' => AICOSO_Pickup_Delivery_Data::get_settings(),
				'strings'  => array(
					'selectDate'     => __( 'Select a date', 'aicoso-pickup-and-delivery-for-woocommerce' ),
					'selectTime'     => __( 'Select a time', 'aicoso-pickup-and-delivery-for-woocommerce' ),
					'selectLocation' => __( 'Select a location', 'aicoso-pickup-and-delivery-for-woocommerce' ),
					'delivery'       => __( 'Delivery', 'aicoso-pickup-and-delivery-for-woocommerce' ),
					'pickup'         => __( 'Pickup', 'aicoso-pickup-and-delivery-for-woocommerce' ),
					'deliveryDate'   => __( 'Delivery Date', 'aicoso-pickup-and-delivery-for-woocommerce' ),
					'deliveryTime'   => __( 'Delivery Time', 'aicoso-pickup-and-delivery-for-woocommerce' ),
					'pickupDate'     => __( 'Pickup Date', 'aicoso-pickup-and-delivery-for-woocommerce' ),
					'pickupTime'     => __( 'Pickup Time', 'aicoso-pickup-and-delivery-for-woocommerce' ),
					'pickupLocation' => __( 'Pickup Location', 'aicoso-pickup-and-delivery-for-woocommerce' ),
					'orderType'      => __( 'Order Type', 'aicoso-pickup-and-delivery-for-woocommerce' ),
				),
			)
		);
	}

	/**
	 * Register scripts for the editor.
	 */
	public function register_block_editor_scripts() {
		$script_url  = AICOSO_PICKUP_DELIVERY_PLUGIN_URL . 'assets/js/blocks/editor.js';
		$script_path = AICOSO_PICKUP_DELIVERY_PLUGIN_PATH . 'assets/js/blocks/editor.js';

		$script_asset_path = AICOSO_PICKUP_DELIVERY_PLUGIN_PATH . 'assets/js/blocks/editor.asset.php';
		$script_asset      = file_exists( $script_asset_path )
			? require $script_asset_path
			: array(
				'dependencies' => array(),
				'version'      => $this->get_file_version( $script_path ),
			);

		wp_register_script(
			'aicoso-pickup-delivery-blocks-editor',
			$script_url,
			$script_asset['dependencies'],
			$script_asset['version'],
			true
		);

		wp_set_script_translations(
			'aicoso-pickup-delivery-blocks-editor',
			'aicoso-pickup-and-delivery-for-woocommerce',
			AICOSO_PICKUP_DELIVERY_PLUGIN_PATH . 'languages/'
		);
	}

	/**
	 * Register main integration script.
	 */
	public function register_main_integration() {
		$script_path       = '/assets/js/blocks/index.js';
		$script_url        = AICOSO_PICKUP_DELIVERY_PLUGIN_URL . $script_path;
		$script_asset_path = AICOSO_PICKUP_DELIVERY_PLUGIN_PATH . '/assets/js/blocks/index.asset.php';

		$script_asset = file_exists( $script_asset_path )
			? require $script_asset_path
			: array(
				'dependencies' => array(),
				'version'      => $this->get_file_version( AICOSO_PICKUP_DELIVERY_PLUGIN_PATH . $script_path ),
			);

		wp_register_script(
			'aicoso-pickup-delivery-blocks',
			$script_url,
			$script_asset['dependencies'],
			$script_asset['version'],
			true
		);

		wp_set_script_translations(
			'aicoso-pickup-delivery-blocks',
			'aicoso-pickup-and-delivery-for-woocommerce',
			AICOSO_PICKUP_DELIVERY_PLUGIN_PATH . 'languages/'
		);
	}

	/**
	 * Register checkout blocks.
	 *
	 * @param object $integration_registry The integration registry.
	 */
	public function register_checkout_blocks( $integration_registry ) {
		$integration_registry->register( $this );
	}

	/**
	 * Returns an array of script handles to enqueue in the frontend context.
	 *
	 * @return string[]
	 */
	public function get_script_handles() {
		return array( 'aicoso-pickup-delivery-blocks-integration', 'aicoso-pickup-delivery-blocks' );
	}

	/**
	 * Returns an array of script handles to enqueue in the editor context.
	 *
	 * @return string[]
	 */
	public function get_editor_script_handles() {
		return array( 'aicoso-pickup-delivery-blocks-editor', 'aicoso-pickup-delivery-blocks' );
	}

	/**
	 * An array of key, value pairs of data made available to the block on the client side.
	 *
	 * @return array
	 */
	public function get_script_data() {
		$settings = AICOSO_Pickup_Delivery_Data::get_settings();

		return array(
			'enableOrderTypeSelection' => isset( $settings['enable_order_type_selection'] ) ? $settings['enable_order_type_selection'] : 'yes',
			'defaultOrderType'         => isset( $settings['default_order_type'] ) ? $settings['default_order_type'] : 'delivery',
			'enableDeliveryDate'       => isset( $settings['enable_delivery_date'] ) ? $settings['enable_delivery_date'] : 'yes',
			'enableDeliveryTimeSlots'  => isset( $settings['enable_delivery_time_slots'] ) ? $settings['enable_delivery_time_slots'] : 'yes',
			'enablePickupDate'         => isset( $settings['enable_pickup_date'] ) ? $settings['enable_pickup_date'] : 'yes',
			'enablePickupTimeSlots'    => isset( $settings['enable_pickup_time_slots'] ) ? $settings['enable_pickup_time_slots'] : 'yes',
		);
	}

	/**
	 * Update order from the checkout block request.
	 *
	 * @param \WC_Order        $order The order object.
	 * @param \WP_REST_Request $request The request object.
	 */
	public function update_order_from_request( $order, $request ) {
		$data = isset( $request['extensions']['aicoso-pickup-and-delivery-for-woocommerce'] ) ? $request['extensions']['aicoso-pickup-and-delivery-for-woocommerce'] : array();

		if ( ! empty( $data['orderType'] ) ) {
			$order->update_meta_data( '_aicoso_pickup_delivery_type', sanitize_text_field( $data['orderType'] ) );
		}

		if ( ! empty( $data['deliveryDate'] ) ) {
			$order->update_meta_data( '_aicoso_pickup_delivery_delivery_date', sanitize_text_field( $data['deliveryDate'] ) );
		}

		if ( ! empty( $data['deliveryTimeSlot'] ) ) {
			$order->update_meta_data( '_aicoso_pickup_delivery_delivery_time_slot', sanitize_text_field( $data['deliveryTimeSlot'] ) );
		}

		if ( ! empty( $data['pickupDate'] ) ) {
			$order->update_meta_data( '_aicoso_pickup_delivery_pickup_date', sanitize_text_field( $data['pickupDate'] ) );
		}

		if ( ! empty( $data['pickupTimeSlot'] ) ) {
			$order->update_meta_data( '_aicoso_pickup_delivery_pickup_time_slot', sanitize_text_field( $data['pickupTimeSlot'] ) );
		}

		if ( ! empty( $data['pickupLocation'] ) ) {
			$order->update_meta_data( '_aicoso_pickup_delivery_pickup_location', sanitize_text_field( $data['pickupLocation'] ) );
		}
	}

	/**
	 * Get the file modified time as a cache buster if we're in dev mode.
	 *
	 * @param string $file Local path to the file.
	 * @return string The cache buster value to use for the given file.
	 */
	protected function get_file_version( $file ) {
		if ( defined( 'SCRIPT_DEBUG' ) && SCRIPT_DEBUG && file_exists( $file ) ) {
			return filemtime( $file );
		}

		return AICOSO_PICKUP_DELIVERY_VERSION;
	}
}
