<?php
/**
 * WooCommerce Blocks Compatibility
 *
 * @package Aicoso_Pickup_Delivery_For_Woocommerce
 */

if ( ! defined( 'WPINC' ) ) {
	die;
}

/**
 * Class for ensuring compatibility with WooCommerce Blocks.
 */
class AICOSO_Pickup_Delivery_Blocks_Compatibility {

	/**
	 * Initialize the compatibility layer.
	 */
	public static function init() {
		// Check if we should force classic checkout.
		$force_classic = apply_filters( 'aicoso_pickup_delivery_force_classic_checkout', false );

		if ( $force_classic ) {
			// Option 1: Force classic checkout (simpler solution).
			add_filter( 'woocommerce_blocks_force_classic_checkout', '__return_true' );
		} else {
			// Option 2: Add support for blocks checkout.
			// Load the blocks support class.
			require_once AICOSO_PICKUP_DELIVERY_PLUGIN_PATH . 'includes/class-aicoso-pickup-delivery-checkout-blocks-support.php';
			AICOSO_Pickup_Delivery_Checkout_Blocks_Support::init();
		}

		// Add additional fields support to blocks checkout.
		add_action( 'woocommerce_blocks_loaded', array( __CLASS__, 'add_blocks_support' ) );

		// Add notice to admin if blocks are being used.
		add_action( 'admin_notices', array( __CLASS__, 'admin_notice' ) );

		// Enqueue admin scripts for blocks compatibility.
		add_action( 'admin_enqueue_scripts', array( __CLASS__, 'enqueue_admin_scripts' ) );

		// Add AJAX handler for dismissing notice.
		add_action( 'wp_ajax_aicoso_pickup_delivery_dismiss_blocks_notice', array( __CLASS__, 'dismiss_blocks_notice' ) );

		// Add fields to block checkout via additional fields API.
		add_action( 'woocommerce_init', array( __CLASS__, 'register_additional_checkout_fields' ) );
	}

	/**
	 * Add support for blocks.
	 */
	public static function add_blocks_support() {
		// Add fields via the Store API.
		add_filter( '__experimental_woocommerce_blocks_add_data_attributes_to_namespace', array( __CLASS__, 'add_data_attributes' ), 10, 2 );
		add_filter( '__experimental_woocommerce_blocks_checkout_update_order_from_request', array( __CLASS__, 'update_order_from_blocks_request' ), 10, 2 );
	}

	/**
	 * Enqueue admin scripts for blocks compatibility.
	 */
	public static function enqueue_admin_scripts() {
		// Only enqueue on admin pages where our notice might appear.
		$screen = get_current_screen();
		if ( ! $screen ) {
			return;
		}

		// Enqueue the blocks compatibility admin script.
		wp_enqueue_script(
			'aicoso-pickup-delivery-blocks-admin',
			AICOSO_PICKUP_DELIVERY_PLUGIN_URL . 'assets/js/blocks-compatibility-admin.js',
			array( 'jquery' ),
			AICOSO_PICKUP_DELIVERY_VERSION,
			true
		);

		// Localize script with nonce.
		wp_localize_script(
			'aicoso-pickup-delivery-blocks-admin',
			'wcPickupDeliveryBlocksAdmin',
			array(
				'nonce' => wp_create_nonce( 'aicoso_pickup_delivery_dismiss_notice' ),
			)
		);
	}

	/**
	 * Register additional checkout fields for blocks.
	 */
	public static function register_additional_checkout_fields() {
		// Temporarily disabled - fields are being handled through the template.
		// to avoid duplication. This needs proper blocks integration.
	}

	/**
	 * Register additional checkout fields helper function.
	 *
	 * This function is temporarily disabled until proper blocks integration is implemented.
	 * The code below is preserved for future use.
	 *
	 * @return void
	 */
	private function register_fields_helper() {
		$settings         = AICOSO_Pickup_Delivery_Data::get_settings();
		$enable_selection = isset( $settings['enable_order_type_selection'] ) ? $settings['enable_order_type_selection'] : 'yes';
		$default_type     = isset( $settings['default_order_type'] ) ? $settings['default_order_type'] : 'delivery';

		// Register order type field.
		if ( 'yes' === $enable_selection ) {
			woocommerce_register_additional_checkout_field(
				array(
					'id'       => 'aicoso-pickup-delivery/type',
					'label'    => __( 'Order Type', 'aicoso-pickup-and-delivery-for-woocommerce' ),
					'location' => 'order',
					'type'     => 'select',
					'options'  => array(
						array(
							'value' => 'delivery',
							'label' => __( 'Delivery', 'aicoso-pickup-and-delivery-for-woocommerce' ),
						),
						array(
							'value' => 'pickup',
							'label' => __( 'Pickup', 'aicoso-pickup-and-delivery-for-woocommerce' ),
						),
					),
					'required' => true,
					'default'  => $default_type,
				)
			);
		}

		// Register delivery date field.
		if ( isset( $settings['enable_delivery_date'] ) && 'yes' === $settings['enable_delivery_date'] ) {
			woocommerce_register_additional_checkout_field(
				array(
					'id'       => 'aicoso-pickup-delivery/delivery_date',
					'label'    => __( 'Delivery Date', 'aicoso-pickup-and-delivery-for-woocommerce' ),
					'location' => 'order',
					'type'     => 'text', // Date picker will be added via JavaScript.
					'required' => false,
				)
			);
		}

		// Register pickup date field.
		if ( isset( $settings['enable_pickup_date'] ) && 'yes' === $settings['enable_pickup_date'] ) {
			woocommerce_register_additional_checkout_field(
				array(
					'id'       => 'aicoso-pickup-delivery/pickup_date',
					'label'    => __( 'Pickup Date', 'aicoso-pickup-and-delivery-for-woocommerce' ),
					'location' => 'order',
					'type'     => 'text', // Date picker will be added via JavaScript.
					'required' => false,
				)
			);
		}
	}

	/**
	 * Add data attributes for blocks.
	 *
	 * @param array  $allowed_namespaces Allowed namespaces.
	 * @param string $namespace          Current namespace.
	 * @return array
	 */
	public static function add_data_attributes( $allowed_namespaces, $namespace ) { // phpcs:ignore Universal.NamingConventions.NoReservedKeywordParameterNames.namespaceFound
		if ( 'aicoso-pickup-and-delivery-for-woocommerce' === $namespace ) {
			$allowed_namespaces[] = $namespace;
		}
		return $allowed_namespaces;
	}

	/**
	 * Update order from blocks checkout request.
	 *
	 * @param \WC_Order $order   Order object.
	 * @param array     $request Request data.
	 */
	public static function update_order_from_blocks_request( $order, $request ) {
		if ( isset( $request['extensions']['aicoso-pickup-and-delivery-for-woocommerce'] ) ) {
			$data = $request['extensions']['aicoso-pickup-and-delivery-for-woocommerce'];

			if ( isset( $data['order_type'] ) ) {
				$order->update_meta_data( '_aicoso_pickup_delivery_type', sanitize_text_field( $data['order_type'] ) );
			}

			if ( isset( $data['delivery_date'] ) ) {
				$order->update_meta_data( '_aicoso_pickup_delivery_delivery_date', sanitize_text_field( $data['delivery_date'] ) );
			}

			if ( isset( $data['delivery_time'] ) ) {
				$order->update_meta_data( '_aicoso_pickup_delivery_delivery_time_slot', sanitize_text_field( $data['delivery_time'] ) );
			}

			if ( isset( $data['pickup_date'] ) ) {
				$order->update_meta_data( '_aicoso_pickup_delivery_pickup_date', sanitize_text_field( $data['pickup_date'] ) );
			}

			if ( isset( $data['pickup_time'] ) ) {
				$order->update_meta_data( '_aicoso_pickup_delivery_pickup_time_slot', sanitize_text_field( $data['pickup_time'] ) );
			}

			if ( isset( $data['pickup_location'] ) ) {
				$order->update_meta_data( '_aicoso_pickup_delivery_pickup_location', sanitize_text_field( $data['pickup_location'] ) );
			}

			$order->save();
		}
	}

	/**
	 * Handle AJAX request to dismiss blocks notice.
	 */
	public static function dismiss_blocks_notice() {
		check_ajax_referer( 'aicoso_pickup_delivery_dismiss_notice', 'nonce' );

		if ( current_user_can( 'manage_options' ) ) {
			set_transient( 'aicoso_pickup_delivery_blocks_notice_dismissed', true, 7 * DAY_IN_SECONDS );
			wp_send_json_success();
		} else {
			wp_send_json_error( 'Insufficient permissions' );
		}
	}

	/**
	 * Show admin notice about blocks compatibility.
	 */
	public static function admin_notice() {
		if ( ! current_user_can( 'manage_options' ) ) {
			return;
		}

		// Check if we're on a relevant admin page.
		$screen = get_current_screen();
		if ( ! $screen || ( 'woocommerce_page_wc-settings' !== $screen->id && 'plugins' !== $screen->id ) ) {
			return;
		}

		// Check if notice was dismissed.
		if ( get_transient( 'aicoso_pickup_delivery_blocks_notice_dismissed' ) ) {
			return;
		}

		// Check if blocks checkout is being used.
		$checkout_page_id = wc_get_page_id( 'checkout' );
		if ( $checkout_page_id > 0 ) {
			$checkout_page = get_post( $checkout_page_id );
			if ( $checkout_page && has_blocks( $checkout_page->post_content ) ) {
				?>
				<div class="notice notice-warning is-dismissible" id="aicoso-pickup-delivery-blocks-notice">
					<h3><?php esc_html_e( 'WooCommerce Pickup & Delivery - Action Required', 'aicoso-pickup-and-delivery-for-woocommerce' ); ?></h3>
					<p>
						<?php esc_html_e( 'Your checkout page is using the new WooCommerce Blocks checkout. The Pickup & Delivery fields have been added with compatibility mode.', 'aicoso-pickup-and-delivery-for-woocommerce' ); ?>
					</p>
					<p><strong><?php esc_html_e( 'Options to fix this:', 'aicoso-pickup-and-delivery-for-woocommerce' ); ?></strong></p>
					<ol>
						<li><?php esc_html_e( 'Switch to Classic Checkout (Recommended): Replace the checkout block with [woocommerce_checkout] shortcode', 'aicoso-pickup-and-delivery-for-woocommerce' ); ?></li>
						<li><?php esc_html_e( 'Use Compatibility Mode: The fields should appear automatically with our compatibility layer', 'aicoso-pickup-and-delivery-for-woocommerce' ); ?></li>
					</ol>
					<p>
						<a href="<?php echo esc_url( admin_url( 'post.php?post=' . $checkout_page_id . '&action=edit' ) ); ?>" class="button button-primary">
							<?php esc_html_e( 'Edit Checkout Page', 'aicoso-pickup-and-delivery-for-woocommerce' ); ?>
						</a>
						<button type="button" class="button button-secondary" onclick="wcPickupDeliveryDismissNotice()">
							<?php esc_html_e( 'Dismiss for 7 days', 'aicoso-pickup-and-delivery-for-woocommerce' ); ?>
						</button>
					</p>
				</div>
				<?php
			}
		}
	}
}