<?php
/**
 * The admin settings functionality of the plugin.
 *
 * @link       https://example.com
 * @since      1.0.0
 *
 * @package    aicoso_pickup_delivery
 */

if ( ! defined( 'WPINC' ) ) {
	die;
}

/**
 * The admin settings class.
 *
 * Handles the plugin settings page and integration with WooCommerce settings.
 *
 * @since      1.0.0
 * @package    aicoso_pickup_delivery
 * @author     Your Name
 */
class AICOSO_Pickup_Delivery_Settings {

	/**
	 * The ID of this plugin.
	 *
	 * @since    1.0.0
	 * @access   private
	 * @var      string    $plugin_name    The ID of this plugin.
	 */
	private $plugin_name;

	/**
	 * The version of this plugin.
	 *
	 * @since    1.0.0
	 * @access   private
	 * @var      string    $version    The current version of this plugin.
	 */
	private $version;

	/**
	 * Initialize the class and set its properties.
	 *
	 * @since    1.0.0
	 * @param    string $plugin_name       The name of this plugin.
	 * @param    string $version           The version of this plugin.
	 */
	public function __construct( $plugin_name, $version ) {
		$this->plugin_name = $plugin_name;
		$this->version     = $version;
	}

	/**
	 * Enqueue admin scripts for settings pages
	 *
	 * @since    1.0.0
	 * @param    string $hook    The current admin page hook.
	 */
	public function enqueue_settings_scripts( $hook ) {
		// Only load on our settings page.
		if ( 'toplevel_page_aicoso-pickup-delivery-settings' !== $hook ) {
			return;
		}

		// Get current tab.
		// phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Admin page tab navigation
		$current_tab = isset( $_GET['tab'] ) ? sanitize_text_field( wp_unslash( $_GET['tab'] ) ) : 'general';

		// Enqueue locations tab script.
		if ( 'locations' === $current_tab ) {
			wp_enqueue_script(
				'aicoso-pickup-delivery-admin-locations',
				AICOSO_PICKUP_DELIVERY_PLUGIN_URL . 'assets/js/admin-settings-locations.js',
				array( 'jquery' ),
				AICOSO_PICKUP_DELIVERY_VERSION,
				true
			);

			// Localize script with translatable strings.
			wp_localize_script(
				'aicoso-pickup-delivery-admin-locations',
				'wcPickupDeliveryLocations',
				array(
					'editTitle'    => esc_html__( 'Edit Pickup Location', 'aicoso-pickup-and-delivery-for-woocommerce' ),
					'addTitle'     => esc_html__( 'Add New Pickup Location', 'aicoso-pickup-and-delivery-for-woocommerce' ),
					'updateButton' => esc_html__( 'Update Location', 'aicoso-pickup-and-delivery-for-woocommerce' ),
					'addButton'    => esc_html__( 'Add Location', 'aicoso-pickup-and-delivery-for-woocommerce' ),
					'adminEmail'   => get_option( 'admin_email' ),
				)
			);
		}

		// Enqueue timeslots tab script.
		if ( 'timeslots' === $current_tab ) {
			wp_enqueue_script(
				'aicoso-pickup-delivery-admin-timeslots',
				AICOSO_PICKUP_DELIVERY_PLUGIN_URL . 'assets/js/admin-settings-timeslots.js',
				array( 'jquery' ),
				AICOSO_PICKUP_DELIVERY_VERSION,
				true
			);

			// Localize script with translatable strings.
			wp_localize_script(
				'aicoso-pickup-delivery-admin-timeslots',
				'wcPickupDeliveryTimeslots',
				array(
					'editDeliveryTitle' => esc_html__( 'Edit Delivery Time Slot', 'aicoso-pickup-and-delivery-for-woocommerce' ),
					'addDeliveryTitle'  => esc_html__( 'Add New Delivery Time Slot', 'aicoso-pickup-and-delivery-for-woocommerce' ),
					'editPickupTitle'   => esc_html__( 'Edit Pickup Time Slot', 'aicoso-pickup-and-delivery-for-woocommerce' ),
					'addPickupTitle'    => esc_html__( 'Add New Pickup Time Slot', 'aicoso-pickup-and-delivery-for-woocommerce' ),
				)
			);
		}
	}

	/**
	 * Add the settings page to the admin menu
	 *
	 * @since    1.0.0
	 */
	public function add_settings_page() {
		add_menu_page(
			__( 'Pickup & Delivery Settings', 'aicoso-pickup-and-delivery-for-woocommerce' ),
			__( 'Pickup & Delivery', 'aicoso-pickup-and-delivery-for-woocommerce' ),
			'manage_woocommerce', // phpcs:ignore WordPress.WP.Capabilities.Unknown -- WooCommerce custom capability.
			'aicoso-pickup-delivery-settings',
			array( $this, 'display_settings_page' ),
			'dashicons-calendar-alt',
			58
		);
	}

	/**
	 * Register plugin settings
	 *
	 * @since    1.0.0
	 */
	public function register_settings() {
		register_setting(
			'aicoso_pickup_delivery_settings',
			'aicoso_pickup_delivery_settings',
			array( $this, 'validate_settings' )
		);
	}

	/**
	 * Add a new section to WooCommerce settings
	 *
	 * @since    1.0.0
	 * @param    array $sections    WooCommerce settings sections.
	 * @return   array                 Modified settings sections
	 */
	public function add_wc_settings_section( $sections ) {
		$sections['aicoso_pickup_delivery'] = __( 'Pickup & Delivery', 'aicoso-pickup-and-delivery-for-woocommerce' );
		return $sections;
	}

	/**
	 * Add settings to the WooCommerce settings page
	 *
	 * @since    1.0.0
	 * @param    array  $settings           Settings.
	 * @param    string $current_section    Current section.
	 * @return   array                         Modified settings
	 */
	public function add_wc_settings( $settings, $current_section ) {
		if ( 'aicoso_pickup_delivery' !== $current_section ) {
			return $settings;
		}

		$wc_settings = array(
			array(
				'title' => esc_html__( 'Pickup & Delivery Settings', 'aicoso-pickup-and-delivery-for-woocommerce' ),
				'type'  => 'title',
				'desc'  => esc_html__( 'Settings for WooCommerce Pickup & Delivery plugin. For more detailed settings, please use the dedicated settings page.', 'aicoso-pickup-and-delivery-for-woocommerce' ),
				'id'    => 'aicoso_pickup_delivery_settings',
			),
			array(
				'title'   => esc_html__( 'Enable Order Type Selection', 'aicoso-pickup-and-delivery-for-woocommerce' ),
				'desc'    => esc_html__( 'Allow customers to choose between delivery and pickup', 'aicoso-pickup-and-delivery-for-woocommerce' ),
				'id'      => 'aicoso_pickup_delivery_enable_order_type_selection',
				'default' => 'yes',
				'type'    => 'checkbox',
			),
			array(
				'title'   => esc_html__( 'Default Order Type', 'aicoso-pickup-and-delivery-for-woocommerce' ),
				'id'      => 'aicoso_pickup_delivery_default_order_type',
				'default' => 'delivery',
				'type'    => 'select',
				'options' => array(
					'delivery' => esc_html__( 'Delivery', 'aicoso-pickup-and-delivery-for-woocommerce' ),
					'pickup'   => esc_html__( 'Pickup', 'aicoso-pickup-and-delivery-for-woocommerce' ),
				),
			),
			array(
				'title' => esc_html__( 'Configure Additional Settings', 'aicoso-pickup-and-delivery-for-woocommerce' ),
				'desc'  => sprintf(
					/* translators: %s: Settings page URL */
					'<a href="%s" class="button">%s</a>',
					esc_url( admin_url( 'admin.php?page=aicoso-pickup-delivery-settings' ) ),
					esc_html__( 'Go to Pickup & Delivery Settings', 'aicoso-pickup-and-delivery-for-woocommerce' )
				),
				'id'    => 'aicoso_pickup_delivery_go_to_settings',
				'type'  => 'title',
			),
			array(
				'type' => 'sectionend',
				'id'   => 'aicoso_pickup_delivery_settings',
			),
		);

		return $wc_settings;
	}

	/**
	 * Display the settings page
	 *
	 * @since    1.0.0
	 */
	public function display_settings_page() {
		// Check user capabilities.
		// phpcs:ignore WordPress.WP.Capabilities.Unknown -- WooCommerce custom capability.
		if ( ! current_user_can( 'manage_woocommerce' ) ) {
			wp_die( esc_html__( 'You do not have sufficient permissions to access this page.', 'aicoso-pickup-and-delivery-for-woocommerce' ) );
		}

		// Get current tab.
		$current_tab = isset( $_GET['tab'] ) ? sanitize_text_field( wp_unslash( $_GET['tab'] ) ) : 'general';

		// Handle test email sending.
		if ( isset( $_POST['test_delivery_email'] ) && check_admin_referer( 'aicoso_pickup_delivery_save_settings' ) ) {
			$result = $this->send_test_email( 'delivery' );
			if ( $result ) {
				echo '<div class="notice notice-success is-dismissible"><p>' . esc_html__( 'Test delivery email sent successfully! Check your admin email inbox.', 'aicoso-pickup-and-delivery-for-woocommerce' ) . '</p></div>';
			} else {
				echo '<div class="notice notice-error is-dismissible"><p>' . esc_html__( 'Failed to send test delivery email. Please check your email settings.', 'aicoso-pickup-and-delivery-for-woocommerce' ) . '</p></div>';
			}
		} elseif ( isset( $_POST['test_pickup_email'] ) && check_admin_referer( 'aicoso_pickup_delivery_save_settings' ) ) {
			$result = $this->send_test_email( 'pickup' );
			if ( $result ) {
				echo '<div class="notice notice-success is-dismissible"><p>' . esc_html__( 'Test pickup email sent successfully! Check your admin email inbox.', 'aicoso-pickup-and-delivery-for-woocommerce' ) . '</p></div>';
			} else {
				echo '<div class="notice notice-error is-dismissible"><p>' . esc_html__( 'Failed to send test pickup email. Please check your email settings.', 'aicoso-pickup-and-delivery-for-woocommerce' ) . '</p></div>';
			}
		} elseif ( ( isset( $_POST['save_settings'] ) || isset( $_POST['submit'] ) ) && check_admin_referer( 'aicoso_pickup_delivery_save_settings' ) ) {
			// Only save settings if not sending test emails.
			$this->save_settings();

			// Check if we're on the locations tab and performing location operations.
			if ( isset( $_GET['tab'] ) && 'locations' === $_GET['tab'] ) {
				if ( isset( $_POST['add_pickup_location'] ) && isset( $_POST['location_name'] ) && ! empty( $_POST['location_name'] ) ) {
					echo '<div class="notice notice-success is-dismissible"><p>' . esc_html__( 'Pickup location added successfully!', 'aicoso-pickup-and-delivery-for-woocommerce' ) . '</p></div>';
				} elseif ( isset( $_POST['update_pickup_location'] ) && isset( $_POST['location_name'] ) && ! empty( $_POST['location_name'] ) ) {
					echo '<div class="notice notice-success is-dismissible"><p>' . esc_html__( 'Pickup location updated successfully!', 'aicoso-pickup-and-delivery-for-woocommerce' ) . '</p></div>';
				} elseif ( isset( $_POST['delete_pickup_location'] ) && ! empty( $_POST['delete_pickup_location'] ) ) {
					echo '<div class="notice notice-success is-dismissible"><p>' . esc_html__( 'Pickup location deleted successfully!', 'aicoso-pickup-and-delivery-for-woocommerce' ) . '</p></div>';
				} else {
					echo '<div class="notice notice-success is-dismissible"><p>' . esc_html__( 'Settings saved successfully!', 'aicoso-pickup-and-delivery-for-woocommerce' ) . '</p></div>';
				}
			} else {
				echo '<div class="notice notice-success is-dismissible"><p>' . esc_html__( 'Settings saved successfully!', 'aicoso-pickup-and-delivery-for-woocommerce' ) . '</p></div>';
			}
		}

		// Define tabs.
		$tabs = array(
			'general'   => esc_html__( 'General', 'aicoso-pickup-and-delivery-for-woocommerce' ),
			'delivery'  => esc_html__( 'Delivery', 'aicoso-pickup-and-delivery-for-woocommerce' ),
			'pickup'    => esc_html__( 'Pickup', 'aicoso-pickup-and-delivery-for-woocommerce' ),
			'timeslots' => esc_html__( 'Time Slots', 'aicoso-pickup-and-delivery-for-woocommerce' ),
			'locations' => esc_html__( 'Pickup Locations', 'aicoso-pickup-and-delivery-for-woocommerce' ),
			'emails'    => esc_html__( 'Email Notifications', 'aicoso-pickup-and-delivery-for-woocommerce' ),
		);

		// Get settings.
		$settings = AICOSO_Pickup_Delivery_Data::get_settings();

		// Include the appropriate settings page template.
		$template_path = plugin_dir_path( __DIR__ ) . 'admin/partials/admin-settings.php';
		if ( file_exists( $template_path ) ) {
			include_once $template_path;
		} else {
			// If template doesn't exist, show inline settings page.
			$this->display_inline_settings_page( $tabs, $current_tab, $settings );
		}
	}

	/**
	 * Display inline settings page if template is missing
	 *
	 * @since    1.0.0
	 * @param array  $tabs        Available tabs.
	 * @param string $current_tab Current active tab.
	 * @param array  $settings    Plugin settings.
	 */
	private function display_inline_settings_page( $tabs, $current_tab, $settings ) {
		?>
		
		<div class="wrap">
			<h1><?php echo esc_html( get_admin_page_title() ); ?></h1>
			
			<nav class="nav-tab-wrapper">
				<?php foreach ( $tabs as $tab_key => $tab_name ) : ?>
					<a href="<?php echo esc_url( add_query_arg( 'tab', $tab_key ) ); ?>"
						class="nav-tab <?php echo $tab_key === $current_tab ? 'nav-tab-active' : ''; ?>">
						<?php echo esc_html( $tab_name ); ?>
					</a>
				<?php endforeach; ?>
			</nav>
			
			<form method="post" action="">
				<?php wp_nonce_field( 'aicoso_pickup_delivery_save_settings' ); ?>
				
				<table class="form-table">
					<?php if ( 'general' === $current_tab ) : ?>
						<tr>
							<th scope="row">
								<label for="enable_order_type_selection">
									<?php esc_html_e( 'Enable Order Type Selection', 'aicoso-pickup-and-delivery-for-woocommerce' ); ?>
								</label>
							</th>
							<td>
								<div class="toggle-control">
									<label class="toggle-switch">
										<input type="checkbox" id="enable_order_type_selection" name="enable_order_type_selection" 
												value="yes" <?php checked( $settings['enable_order_type_selection'], 'yes' ); ?> />
										<span class="toggle-slider"></span>
									</label>
									<label for="enable_order_type_selection" class="toggle-label">
										<?php esc_html_e( 'Allow customers to choose between delivery and pickup at checkout', 'aicoso-pickup-and-delivery-for-woocommerce' ); ?>
									</label>
								</div>
							</td>
						</tr>
						<tr>
							<th scope="row">
								<label for="default_order_type">
									<?php esc_html_e( 'Default Order Type', 'aicoso-pickup-and-delivery-for-woocommerce' ); ?>
								</label>
							</th>
							<td>
								<select id="default_order_type" name="default_order_type">
									<option value="delivery" <?php selected( $settings['default_order_type'], 'delivery' ); ?>>
										<?php esc_html_e( 'Delivery', 'aicoso-pickup-and-delivery-for-woocommerce' ); ?>
									</option>
									<option value="pickup" <?php selected( $settings['default_order_type'], 'pickup' ); ?>>
										<?php esc_html_e( 'Pickup', 'aicoso-pickup-and-delivery-for-woocommerce' ); ?>
									</option>
								</select>
							</td>
						</tr>
					<?php elseif ( 'delivery' === $current_tab ) : ?>
						<tr>
							<th scope="row">
								<label for="enable_delivery_date">
									<?php esc_html_e( 'Enable Delivery Date', 'aicoso-pickup-and-delivery-for-woocommerce' ); ?>
								</label>
							</th>
							<td>
								<div class="toggle-control">
									<label class="toggle-switch">
										<input type="checkbox" id="enable_delivery_date" name="enable_delivery_date" 
												value="yes" <?php checked( $settings['enable_delivery_date'], 'yes' ); ?> />
										<span class="toggle-slider"></span>
									</label>
									<label for="enable_delivery_date" class="toggle-label">
										<?php esc_html_e( 'Allow customers to select a delivery date', 'aicoso-pickup-and-delivery-for-woocommerce' ); ?>
									</label>
								</div>
							</td>
						</tr>
					<?php elseif ( 'pickup' === $current_tab ) : ?>
						<tr>
							<th scope="row">
								<label for="enable_pickup_date">
									<?php esc_html_e( 'Enable Pickup Date', 'aicoso-pickup-and-delivery-for-woocommerce' ); ?>
								</label>
							</th>
							<td>
								<div class="toggle-control">
									<label class="toggle-switch">
										<input type="checkbox" id="enable_pickup_date" name="enable_pickup_date" 
												value="yes" <?php checked( $settings['enable_pickup_date'], 'yes' ); ?> />
										<span class="toggle-slider"></span>
									</label>
									<label for="enable_pickup_date" class="toggle-label">
										<?php esc_html_e( 'Allow customers to select a pickup date', 'aicoso-pickup-and-delivery-for-woocommerce' ); ?>
									</label>
								</div>
							</td>
						</tr>
					<?php elseif ( 'timeslots' === $current_tab ) : ?>
						<tr>
							<th scope="row">
								<label for="enable_delivery_time_slots">
									<?php esc_html_e( 'Enable Delivery Time Slots', 'aicoso-pickup-and-delivery-for-woocommerce' ); ?>
								</label>
							</th>
							<td>
								<div class="toggle-control">
									<label class="toggle-switch">
										<input type="checkbox" id="enable_delivery_time_slots" name="enable_delivery_time_slots" 
												value="yes" <?php checked( isset( $settings['enable_delivery_time_slots'] ) ? $settings['enable_delivery_time_slots'] : 'no', 'yes' ); ?> />
										<span class="toggle-slider"></span>
									</label>
									<label for="enable_delivery_time_slots" class="toggle-label">
										<?php esc_html_e( 'Allow customers to select specific delivery time slots', 'aicoso-pickup-and-delivery-for-woocommerce' ); ?>
									</label>
								</div>
							</td>
						</tr>
						<tr>
							<th scope="row">
								<label for="enable_pickup_time_slots">
									<?php esc_html_e( 'Enable Pickup Time Slots', 'aicoso-pickup-and-delivery-for-woocommerce' ); ?>
								</label>
							</th>
							<td>
								<div class="toggle-control">
									<label class="toggle-switch">
										<input type="checkbox" id="enable_pickup_time_slots" name="enable_pickup_time_slots" 
												value="yes" <?php checked( isset( $settings['enable_pickup_time_slots'] ) ? $settings['enable_pickup_time_slots'] : 'no', 'yes' ); ?> />
										<span class="toggle-slider"></span>
									</label>
									<label for="enable_pickup_time_slots" class="toggle-label">
										<?php esc_html_e( 'Allow customers to select specific pickup time slots', 'aicoso-pickup-and-delivery-for-woocommerce' ); ?>
									</label>
								</div>
							</td>
						</tr>
					<?php elseif ( 'emails' === $current_tab ) : ?>
						<tr>
							<th scope="row">
								<label for="enable_delivery_confirmation_email">
									<?php esc_html_e( 'Delivery Confirmation Email', 'aicoso-pickup-and-delivery-for-woocommerce' ); ?>
								</label>
							</th>
							<td>
								<div class="toggle-control">
									<label class="toggle-switch">
										<input type="checkbox" id="enable_delivery_confirmation_email" name="enable_delivery_confirmation_email" 
												value="yes" <?php checked( isset( $settings['enable_delivery_confirmation_email'] ) ? $settings['enable_delivery_confirmation_email'] : 'no', 'yes' ); ?> />
										<span class="toggle-slider"></span>
									</label>
									<label for="enable_delivery_confirmation_email" class="toggle-label">
										<?php esc_html_e( 'Send confirmation email for delivery orders', 'aicoso-pickup-and-delivery-for-woocommerce' ); ?>
									</label>
								</div>
							</td>
						</tr>
						<tr>
							<th scope="row">
								<label for="enable_pickup_confirmation_email">
									<?php esc_html_e( 'Pickup Confirmation Email', 'aicoso-pickup-and-delivery-for-woocommerce' ); ?>
								</label>
							</th>
							<td>
								<div class="toggle-control">
									<label class="toggle-switch">
										<input type="checkbox" id="enable_pickup_confirmation_email" name="enable_pickup_confirmation_email" 
												value="yes" <?php checked( isset( $settings['enable_pickup_confirmation_email'] ) ? $settings['enable_pickup_confirmation_email'] : 'no', 'yes' ); ?> />
										<span class="toggle-slider"></span>
									</label>
									<label for="enable_pickup_confirmation_email" class="toggle-label">
										<?php esc_html_e( 'Send confirmation email for pickup orders', 'aicoso-pickup-and-delivery-for-woocommerce' ); ?>
									</label>
								</div>
							</td>
						</tr>
						<tr>
							<th scope="row">
								<label for="enable_admin_notification_email">
									<?php esc_html_e( 'Admin Notification Email', 'aicoso-pickup-and-delivery-for-woocommerce' ); ?>
								</label>
							</th>
							<td>
								<div class="toggle-control">
									<label class="toggle-switch">
										<input type="checkbox" id="enable_admin_notification_email" name="enable_admin_notification_email" 
												value="yes" <?php checked( isset( $settings['enable_admin_notification_email'] ) ? $settings['enable_admin_notification_email'] : 'no', 'yes' ); ?> />
										<span class="toggle-slider"></span>
									</label>
									<label for="enable_admin_notification_email" class="toggle-label">
										<?php esc_html_e( 'Send notification email to admin for new orders', 'aicoso-pickup-and-delivery-for-woocommerce' ); ?>
									</label>
								</div>
							</td>
						</tr>
					<?php endif; ?>
				</table>
				
				<p class="submit">
					<input type="submit" name="save_settings" class="button-primary" 
							value="<?php esc_attr_e( 'Save Settings', 'aicoso-pickup-and-delivery-for-woocommerce' ); ?>" />
				</p>
			</form>
		</div>
		<?php
	}

	/**
	 * Save settings from the settings page
	 *
	 * @since    1.0.0
	 * @return   boolean   Whether settings were saved successfully
	 */
	private function save_settings() {
		// phpcs:disable WordPress.Security.NonceVerification.Missing,WordPress.Security.NonceVerification.Recommended
		// Nonce is already verified in display_settings_page() before calling this function.

		// Initialize settings array.
		$new_settings = array();

		// General settings.
		if ( ( isset( $_GET['tab'] ) && 'general' === $_GET['tab'] ) || ! isset( $_GET['tab'] ) ) {
			$new_settings['enable_order_type_selection'] = isset( $_POST['enable_order_type_selection'] ) ? 'yes' : 'no';
			$new_settings['default_order_type']          = isset( $_POST['default_order_type'] ) ? sanitize_text_field( wp_unslash( $_POST['default_order_type'] ) ) : 'delivery';
			$new_settings['max_orders_per_day']          = isset( $_POST['max_orders_per_day'] ) ? absint( $_POST['max_orders_per_day'] ) : 50;

			// Update delete data on uninstall option.
			update_option( 'aicoso_pickup_delivery_delete_data_on_uninstall', isset( $_POST['delete_data_on_uninstall'] ) ? 'yes' : 'no' );
		}

		// Delivery settings.
		if ( isset( $_GET['tab'] ) && 'delivery' === $_GET['tab'] ) {
			$new_settings['enable_delivery_date'] = isset( $_POST['enable_delivery_date'] ) ? 'yes' : 'no';
			$new_settings['max_delivery_per_day'] = isset( $_POST['max_delivery_per_day'] ) ? absint( $_POST['max_delivery_per_day'] ) : 25;
			$new_settings['delivery_lead_time']   = isset( $_POST['delivery_lead_time'] ) ? absint( $_POST['delivery_lead_time'] ) : 1;

			// Delivery days.
			$delivery_days = array();
			foreach ( range( 1, 7 ) as $day_num ) {
				if ( isset( $_POST[ 'delivery_day_' . $day_num ] ) ) {
					$delivery_days[] = $day_num;
				}
			}
			$new_settings['delivery_days'] = ! empty( $delivery_days ) ? $delivery_days : array( 1, 2, 3, 4, 5 );

			// Add delivery blackout date.
			if ( isset( $_POST['add_delivery_blackout_date'] ) && isset( $_POST['delivery_blackout_date'] ) && ! empty( $_POST['delivery_blackout_date'] ) ) {
				$blackout_date = sanitize_text_field( wp_unslash( $_POST['delivery_blackout_date'] ) );
				if ( AICOSO_Pickup_Delivery_Date_Manager::is_valid_date_format( $blackout_date ) ) {
					AICOSO_Pickup_Delivery_Date_Manager::add_blackout_date( $blackout_date, 'delivery' );
				}
			}

			// Remove delivery blackout date.
			if ( isset( $_POST['remove_blackout_date'] ) && ! empty( $_POST['remove_blackout_date'] ) ) {
				$remove_date = sanitize_text_field( wp_unslash( $_POST['remove_blackout_date'] ) );
				if ( AICOSO_Pickup_Delivery_Date_Manager::is_valid_date_format( $remove_date ) ) {
					AICOSO_Pickup_Delivery_Date_Manager::remove_blackout_date( $remove_date, 'delivery' );
				}
			}
		}

		// Pickup settings.
		if ( isset( $_GET['tab'] ) && 'pickup' === $_GET['tab'] ) {
			$new_settings['enable_pickup_date'] = isset( $_POST['enable_pickup_date'] ) ? 'yes' : 'no';
			$new_settings['max_pickup_per_day'] = isset( $_POST['max_pickup_per_day'] ) ? absint( $_POST['max_pickup_per_day'] ) : 25;
			$new_settings['pickup_lead_time']   = isset( $_POST['pickup_lead_time'] ) ? absint( $_POST['pickup_lead_time'] ) : 1;

			// Pickup days.
			$pickup_days = array();
			foreach ( range( 1, 7 ) as $day_num ) {
				if ( isset( $_POST[ 'pickup_day_' . $day_num ] ) ) {
					$pickup_days[] = $day_num;
				}
			}
			$new_settings['pickup_days'] = ! empty( $pickup_days ) ? $pickup_days : array( 1, 2, 3, 4, 5 );

			// Add pickup blackout date.
			if ( isset( $_POST['add_pickup_blackout_date'] ) && isset( $_POST['pickup_blackout_date'] ) && ! empty( $_POST['pickup_blackout_date'] ) ) {
				$blackout_date = sanitize_text_field( wp_unslash( $_POST['pickup_blackout_date'] ) );
				if ( AICOSO_Pickup_Delivery_Date_Manager::is_valid_date_format( $blackout_date ) ) {
					AICOSO_Pickup_Delivery_Date_Manager::add_blackout_date( $blackout_date, 'pickup' );
				}
			}

			// Remove pickup blackout date.
			if ( isset( $_POST['remove_blackout_date'] ) && ! empty( $_POST['remove_blackout_date'] ) ) {
				$remove_date = sanitize_text_field( wp_unslash( $_POST['remove_blackout_date'] ) );
				if ( AICOSO_Pickup_Delivery_Date_Manager::is_valid_date_format( $remove_date ) ) {
					AICOSO_Pickup_Delivery_Date_Manager::remove_blackout_date( $remove_date, 'pickup' );
				}
			}
		}

		// Time slot settings.
		if ( isset( $_GET['tab'] ) && 'timeslots' === $_GET['tab'] ) {
			$new_settings['enable_delivery_time_slots'] = isset( $_POST['enable_delivery_time_slots'] ) ? 'yes' : 'no';
			$new_settings['enable_pickup_time_slots']   = isset( $_POST['enable_pickup_time_slots'] ) ? 'yes' : 'no';

			// Add new delivery time slot.
			if ( isset( $_POST['add_delivery_time_slot'] ) && isset( $_POST['delivery_slot_name'] ) && ! empty( $_POST['delivery_slot_name'] ) ) {
				$slot_data = array(
					'slot_name'  => sanitize_text_field( wp_unslash( $_POST['delivery_slot_name'] ) ),
					'start_time' => isset( $_POST['delivery_start_time'] ) ? sanitize_text_field( wp_unslash( $_POST['delivery_start_time'] ) ) : '',
					'end_time'   => isset( $_POST['delivery_end_time'] ) ? sanitize_text_field( wp_unslash( $_POST['delivery_end_time'] ) ) : '',
					'max_orders' => isset( $_POST['delivery_max_orders'] ) ? absint( $_POST['delivery_max_orders'] ) : 10,
				);

				AICOSO_Pickup_Delivery_Timeslot_Manager::add_time_slot( $slot_data, 'delivery' );
			}

			// Update delivery time slot.
   // phpcs:ignore WordPress.Security.NonceVerification.Missing -- Nonce verified in display_settings_page
			if ( isset( $_POST['update_delivery_time_slot'] ) && isset( $_POST['delivery_slot_name'] ) && ! empty( $_POST['delivery_slot_name'] ) ) {
	// phpcs:ignore WordPress.Security.NonceVerification.Missing -- Nonce verified in display_settings_page
				$original_name = isset( $_POST['edit_delivery_slot_original'] ) ? sanitize_text_field( wp_unslash( $_POST['edit_delivery_slot_original'] ) ) : '';

				if ( ! empty( $original_name ) ) {
					// Delete the old slot.
					AICOSO_Pickup_Delivery_Timeslot_Manager::delete_time_slot( $original_name, 'delivery' );

					// Add the updated slot.
					$slot_data = array(
						'slot_name'  => sanitize_text_field( wp_unslash( $_POST['delivery_slot_name'] ) ),
						'start_time' => isset( $_POST['delivery_start_time'] ) ? sanitize_text_field( wp_unslash( $_POST['delivery_start_time'] ) ) : '',
						'end_time'   => isset( $_POST['delivery_end_time'] ) ? sanitize_text_field( wp_unslash( $_POST['delivery_end_time'] ) ) : '',
						'max_orders' => isset( $_POST['delivery_max_orders'] ) ? absint( $_POST['delivery_max_orders'] ) : 10,
					);

					AICOSO_Pickup_Delivery_Timeslot_Manager::add_time_slot( $slot_data, 'delivery' );
				}
			}

			// Add new pickup time slot.
   // phpcs:ignore WordPress.Security.NonceVerification.Missing -- Nonce verified in display_settings_page
			if ( isset( $_POST['add_pickup_time_slot'] ) && isset( $_POST['pickup_slot_name'] ) && ! empty( $_POST['pickup_slot_name'] ) ) {
				$slot_data = array(
					'slot_name'  => sanitize_text_field( wp_unslash( $_POST['pickup_slot_name'] ) ),
					'start_time' => isset( $_POST['pickup_start_time'] ) ? sanitize_text_field( wp_unslash( $_POST['pickup_start_time'] ) ) : '',
					'end_time'   => isset( $_POST['pickup_end_time'] ) ? sanitize_text_field( wp_unslash( $_POST['pickup_end_time'] ) ) : '',
					'max_orders' => isset( $_POST['pickup_max_orders'] ) ? absint( $_POST['pickup_max_orders'] ) : 10,
				);

				AICOSO_Pickup_Delivery_Timeslot_Manager::add_time_slot( $slot_data, 'pickup' );
			}

			// Update pickup time slot.
   // phpcs:ignore WordPress.Security.NonceVerification.Missing -- Nonce verified in display_settings_page
			if ( isset( $_POST['update_pickup_time_slot'] ) && isset( $_POST['pickup_slot_name'] ) && ! empty( $_POST['pickup_slot_name'] ) ) {
	// phpcs:ignore WordPress.Security.NonceVerification.Missing -- Nonce verified in display_settings_page
				$original_name = isset( $_POST['edit_pickup_slot_original'] ) ? sanitize_text_field( wp_unslash( $_POST['edit_pickup_slot_original'] ) ) : '';

				if ( ! empty( $original_name ) ) {
					// Delete the old slot.
					AICOSO_Pickup_Delivery_Timeslot_Manager::delete_time_slot( $original_name, 'pickup' );

					// Add the updated slot.
					$slot_data = array(
						'slot_name'  => sanitize_text_field( wp_unslash( $_POST['pickup_slot_name'] ) ),
						'start_time' => isset( $_POST['pickup_start_time'] ) ? sanitize_text_field( wp_unslash( $_POST['pickup_start_time'] ) ) : '',
						'end_time'   => isset( $_POST['pickup_end_time'] ) ? sanitize_text_field( wp_unslash( $_POST['pickup_end_time'] ) ) : '',
						'max_orders' => isset( $_POST['pickup_max_orders'] ) ? absint( $_POST['pickup_max_orders'] ) : 10,
					);

					AICOSO_Pickup_Delivery_Timeslot_Manager::add_time_slot( $slot_data, 'pickup' );
				}
			}

			// Delete delivery time slot.
   // phpcs:ignore WordPress.Security.NonceVerification.Missing -- Nonce verified in display_settings_page
			if ( isset( $_POST['delete_delivery_time_slot'] ) && ! empty( $_POST['delete_delivery_time_slot'] ) ) {
				$slot_name = sanitize_text_field( wp_unslash( $_POST['delete_delivery_time_slot'] ) );
				AICOSO_Pickup_Delivery_Timeslot_Manager::delete_time_slot( $slot_name, 'delivery' );
			}

			// Delete pickup time slot.
   // phpcs:ignore WordPress.Security.NonceVerification.Missing -- Nonce verified in display_settings_page
			if ( isset( $_POST['delete_pickup_time_slot'] ) && ! empty( $_POST['delete_pickup_time_slot'] ) ) {
				$slot_name = sanitize_text_field( wp_unslash( $_POST['delete_pickup_time_slot'] ) );
				AICOSO_Pickup_Delivery_Timeslot_Manager::delete_time_slot( $slot_name, 'pickup' );
			}
		}

		// Pickup location settings.
		if ( isset( $_GET['tab'] ) && 'locations' === $_GET['tab'] ) {
			// Add new pickup location.
   // phpcs:ignore WordPress.Security.NonceVerification.Missing -- Nonce verified in display_settings_page
			if ( isset( $_POST['add_pickup_location'] ) && isset( $_POST['location_name'] ) && ! empty( $_POST['location_name'] ) ) {
				$location_data = array(
					'location_name'    => sanitize_text_field( wp_unslash( $_POST['location_name'] ) ),
					'location_address' => isset( $_POST['location_address'] ) ? sanitize_text_field( wp_unslash( $_POST['location_address'] ) ) : '',
					'location_map_url' => isset( $_POST['location_map_url'] ) ? esc_url_raw( wp_unslash( $_POST['location_map_url'] ) ) : '',
					'location_email'   => isset( $_POST['location_email'] ) ? sanitize_email( wp_unslash( $_POST['location_email'] ) ) : '',
					'max_orders'       => isset( $_POST['location_max_orders'] ) ? absint( $_POST['location_max_orders'] ) : 25,
				);

				AICOSO_Pickup_Delivery_Location_Manager::add_location( $location_data );
			}

			// Update pickup location.
   // phpcs:ignore WordPress.Security.NonceVerification.Missing -- Nonce verified in display_settings_page
			if ( isset( $_POST['update_pickup_location'] ) && isset( $_POST['location_name'] ) && ! empty( $_POST['location_name'] ) ) {
				$original_name = isset( $_POST['original_location_name'] ) ? sanitize_text_field( wp_unslash( $_POST['original_location_name'] ) ) : '';

				$location_data = array(
					'location_name'    => sanitize_text_field( wp_unslash( $_POST['location_name'] ) ),
					'location_address' => isset( $_POST['location_address'] ) ? sanitize_text_field( wp_unslash( $_POST['location_address'] ) ) : '',
					'location_map_url' => isset( $_POST['location_map_url'] ) ? esc_url_raw( wp_unslash( $_POST['location_map_url'] ) ) : '',
					'location_email'   => isset( $_POST['location_email'] ) ? sanitize_email( wp_unslash( $_POST['location_email'] ) ) : '',
					'max_orders'       => isset( $_POST['location_max_orders'] ) ? absint( $_POST['location_max_orders'] ) : 25,
				);

				if ( ! empty( $original_name ) ) {
					AICOSO_Pickup_Delivery_Location_Manager::update_location( $original_name, $location_data );
				}
			}

			// Delete pickup location.
   // phpcs:ignore WordPress.Security.NonceVerification.Missing -- Nonce verified in display_settings_page
			if ( isset( $_POST['delete_pickup_location'] ) && ! empty( $_POST['delete_pickup_location'] ) ) {
				$location_name = sanitize_text_field( wp_unslash( $_POST['delete_pickup_location'] ) );
				AICOSO_Pickup_Delivery_Location_Manager::delete_location( $location_name );
			}
		}

		// Email settings.
		if ( isset( $_GET['tab'] ) && 'emails' === $_GET['tab'] ) {
   // phpcs:ignore WordPress.Security.NonceVerification.Missing -- Nonce verified in display_settings_page
			$new_settings['enable_delivery_confirmation_email'] = isset( $_POST['enable_delivery_confirmation_email'] ) ? 'yes' : 'no';
   // phpcs:ignore WordPress.Security.NonceVerification.Missing -- Nonce verified in display_settings_page
			$new_settings['enable_pickup_confirmation_email'] = isset( $_POST['enable_pickup_confirmation_email'] ) ? 'yes' : 'no';
			$new_settings['enable_admin_notification_email']  = isset( $_POST['enable_admin_notification_email'] ) ? 'yes' : 'no';
		}

		// Update settings.
		if ( ! empty( $new_settings ) ) {
			$temp = AICOSO_Pickup_Delivery_Data::update_settings( $new_settings );
			return $temp;
		}

		return false;
		// phpcs:enable WordPress.Security.NonceVerification.Missing,WordPress.Security.NonceVerification.Recommended
	}

	/**
	 * Validate settings before saving
	 *
	 * @since    1.0.0
	 * @param    array $input    Settings input.
	 * @return   array              Validated settings
	 */
	public function validate_settings( $input ) {
		// Validate settings here if needed.
		return $input;
	}

	/**
	 * Send test email for delivery or pickup
	 *
	 * @since    1.0.0
	 * @param    string $type    The type of test email to send ('delivery' or 'pickup').
	 * @return   boolean           Whether the email was sent successfully
	 */
	private function send_test_email( $type = 'delivery' ) {
		// Get admin email.
		$admin_email = get_option( 'admin_email' );

		if ( empty( $admin_email ) ) {
			return false;
		}

		// Get site name.
		$site_name = get_bloginfo( 'name' );

		// Set up email content based on type.
		if ( 'delivery' === $type ) {
			/* translators: %1$s: string placeholder */
			$subject = sprintf( __( '[%s] Test Delivery Confirmation Email', 'aicoso-pickup-and-delivery-for-woocommerce' ), $site_name );
			$heading = __( 'Test Delivery Confirmation', 'aicoso-pickup-and-delivery-for-woocommerce' );
			$message = __( 'This is a test delivery confirmation email from WooCommerce Pickup & Delivery plugin.', 'aicoso-pickup-and-delivery-for-woocommerce' );
			$details = array(
				__( 'Delivery Date', 'aicoso-pickup-and-delivery-for-woocommerce' ) => date_i18n( get_option( 'date_format' ), strtotime( '+2 days' ) ),
				__( 'Delivery Time', 'aicoso-pickup-and-delivery-for-woocommerce' ) => '10:00 AM - 12:00 PM',
				__( 'Delivery Address', 'aicoso-pickup-and-delivery-for-woocommerce' ) => __( 'Customer shipping address', 'aicoso-pickup-and-delivery-for-woocommerce' ),
			);
		} else {
			/* translators: %1$s: string placeholder */
			$subject = sprintf( __( '[%s] Test Pickup Confirmation Email', 'aicoso-pickup-and-delivery-for-woocommerce' ), $site_name );
			$heading = __( 'Test Pickup Confirmation', 'aicoso-pickup-and-delivery-for-woocommerce' );
			$message = __( 'This is a test pickup confirmation email from WooCommerce Pickup & Delivery plugin.', 'aicoso-pickup-and-delivery-for-woocommerce' );
			$details = array(
				__( 'Pickup Date', 'aicoso-pickup-and-delivery-for-woocommerce' ) => date_i18n( get_option( 'date_format' ), strtotime( '+1 day' ) ),
				__( 'Pickup Time', 'aicoso-pickup-and-delivery-for-woocommerce' ) => '2:00 PM - 4:00 PM',
				__( 'Pickup Location', 'aicoso-pickup-and-delivery-for-woocommerce' ) => __( 'Main Store - 123 Main St', 'aicoso-pickup-and-delivery-for-woocommerce' ),
			);
		}

		// Build email HTML.
		$email_content = '<html><head>';
		// NOTE: Inline styles in email templates are acceptable and necessary for email client compatibility.
		// Email clients do not support external stylesheets, so inline CSS is the WordPress standard for emails.
		$email_content .= '<style>
            body { font-family: Arial, sans-serif; color: #333; }
            .container { max-width: 600px; margin: 0 auto; padding: 20px; }
            .header { background-color: #7f54b3; color: white; padding: 20px; text-align: center; }
            .content { padding: 20px; background-color: #f9f9f9; }
            .details-table { width: 100%; border-collapse: collapse; margin: 20px 0; }
            .details-table th { text-align: left; padding: 10px; background-color: #f1f1f1; border-bottom: 1px solid #ddd; }
            .details-table td { padding: 10px; border-bottom: 1px solid #ddd; }
            .footer { padding: 20px; text-align: center; font-size: 12px; color: #666; }
        </style>';
		$email_content .= '</head><body>';
		$email_content .= '<div class="container">';
		$email_content .= '<div class="header">';
		$email_content .= '<h1>' . esc_html( $heading ) . '</h1>';
		$email_content .= '</div>';
		$email_content .= '<div class="content">';
		$email_content .= '<p>' . esc_html( $message ) . '</p>';
		$email_content .= '<table class="details-table">';

		foreach ( $details as $label => $value ) {
			$email_content .= '<tr>';
			$email_content .= '<th>' . esc_html( $label ) . '</th>';
			$email_content .= '<td>' . esc_html( $value ) . '</td>';
			$email_content .= '</tr>';
		}

		$email_content .= '</table>';
		$email_content .= '<p><strong>' . __( 'Note:', 'aicoso-pickup-and-delivery-for-woocommerce' ) . '</strong> ' . __( 'This is a test email. No actual order has been placed.', 'aicoso-pickup-and-delivery-for-woocommerce' ) . '</p>';
		$email_content .= '</div>';
		$email_content .= '<div class="footer">';
		/* translators: %1$s: string placeholder */
		$email_content .= '<p>' . sprintf( __( 'This test email was sent from %s', 'aicoso-pickup-and-delivery-for-woocommerce' ), esc_html( $site_name ) ) . '</p>';
		$email_content .= '</div>';
		$email_content .= '</div>';
		$email_content .= '</body></html>';

		// Set up email headers.
		$headers = array(
			'Content-Type: text/html; charset=UTF-8',
			'From: ' . $site_name . ' <' . $admin_email . '>',
		);

		// Send the email.
		$sent = wp_mail( $admin_email, $subject, $email_content, $headers );

		return $sent;
	}
}