<?php
/**
 * The reporting functionality of the plugin.
 *
 * @link       https://example.com
 * @since      1.0.0
 *
 * @package    aicoso_pickup_delivery
 */

if ( ! defined( 'WPINC' ) ) {
	die;
}

/**
 * The reports class.
 *
 * Handles the integration with WooCommerce reports to provide delivery/pickup
 * reporting functionality.
 *
 * @since      1.0.0
 * @package    aicoso_pickup_delivery
 * @author     Your Name
 */
class AICOSO_Pickup_Delivery_Reports {

	/**
	 * The ID of this plugin.
	 *
	 * @since    1.0.0
	 * @access   private
	 * @var      string    $plugin_name    The ID of this plugin.
	 */
	private $plugin_name;

	/**
	 * The version of this plugin.
	 *
	 * @since    1.0.0
	 * @access   private
	 * @var      string    $version    The current version of this plugin.
	 */
	private $version;

	/**
	 * Initialize the class and set its properties.
	 *
	 * @since    1.0.0
	 * @param    string $plugin_name       The name of this plugin.
	 * @param    string $version           The version of this plugin.
	 */
	public function __construct( $plugin_name, $version ) {
		$this->plugin_name = $plugin_name;
		$this->version     = $version;
	}

	/**
	 * Add delivery/pickup reports to WooCommerce reports
	 *
	 * @since    1.0.0
	 * @param    array $reports    Current reports.
	 * @return   array                Modified reports
	 */
	public function add_reports( $reports ) {
		$reports['orders']['reports']['pickup_delivery'] = array(
			'title'       => __( 'Delivery & Pickup', 'aicoso-pickup-and-delivery-for-woocommerce' ),
			'description' => __( 'View reports for delivery and pickup orders.', 'aicoso-pickup-and-delivery-for-woocommerce' ),
			'callback'    => array( $this, 'get_delivery_pickup_report' ),
		);

		return $reports;
	}

	/**
	 * Get delivery/pickup report content
	 *
	 * @since    1.0.0
	 */
	public function get_delivery_pickup_report() {
		// phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Admin report page filtering.
		$current_range = ! empty( $_GET['range'] ) ? sanitize_text_field( wp_unslash( $_GET['range'] ) ) : '7day';

		// Get date ranges.
		$ranges = array(
			'7day'       => __( 'Last 7 Days', 'aicoso-pickup-and-delivery-for-woocommerce' ),
			'month'      => __( 'This Month', 'aicoso-pickup-and-delivery-for-woocommerce' ),
			'last_month' => __( 'Last Month', 'aicoso-pickup-and-delivery-for-woocommerce' ),
			'year'       => __( 'Year', 'aicoso-pickup-and-delivery-for-woocommerce' ),
		);

		$start_date = '';
		$end_date   = gmdate( 'Y-m-d' );

		switch ( $current_range ) {
			case '7day':
				$start_date = gmdate( 'Y-m-d', strtotime( '-6 days' ) );
				break;
			case 'month':
				$start_date = gmdate( 'Y-m-01' );
				break;
			case 'last_month':
				$start_date = gmdate( 'Y-m-01', strtotime( '-1 month' ) );
				$end_date   = gmdate( 'Y-m-t', strtotime( '-1 month' ) );
				break;
			case 'year':
				$start_date = gmdate( 'Y-01-01' );
				break;
			default:
				$start_date = gmdate( 'Y-m-d', strtotime( '-6 days' ) );
				break;
		}

		// Get reports data.
		$data = $this->get_report_data( $start_date, $end_date );

		include plugin_dir_path( dirname( __DIR__, 1 ) ) . 'admin/partials/reports/delivery-pickup-report.php';
	}

	/**
	 * Get report data
	 *
	 * @since    1.0.0
	 * @param    string $start_date    Start date in Y-m-d format.
	 * @param    string $end_date      End date in Y-m-d format.
	 * @return   array                    Report data
	 */
	private function get_report_data( $start_date, $end_date ) {
		// Get order counts by type.
		$delivery_count = $this->get_order_count_by_type( 'delivery', $start_date, $end_date );
		$pickup_count   = $this->get_order_count_by_type( 'pickup', $start_date, $end_date );
		$total_count    = $delivery_count + $pickup_count;

		// Get order totals by type.
		$delivery_total = $this->get_order_total_by_type( 'delivery', $start_date, $end_date );
		$pickup_total   = $this->get_order_total_by_type( 'pickup', $start_date, $end_date );
		$total_amount   = $delivery_total + $pickup_total;

		// Get daily data for chart.
		$daily_data = $this->get_daily_order_data( $start_date, $end_date );

		// Get popular time slots.
		$popular_delivery_slots = $this->get_popular_time_slots( 'delivery', $start_date, $end_date );
		$popular_pickup_slots   = $this->get_popular_time_slots( 'pickup', $start_date, $end_date );

		// Get popular pickup locations.
		$popular_locations = $this->get_popular_pickup_locations( $start_date, $end_date );

		return array(
			'delivery_count'   => $delivery_count,
			'pickup_count'     => $pickup_count,
			'total_count'      => $total_count,
			'delivery_total'   => $delivery_total,
			'pickup_total'     => $pickup_total,
			'total_amount'     => $total_amount,
			'daily_data'       => $daily_data,
			'delivery_slots'   => $popular_delivery_slots,
			'pickup_slots'     => $popular_pickup_slots,
			'pickup_locations' => $popular_locations,
		);
	}

	/**
	 * Get order count by type
	 *
	 * @since    1.0.0
	 * @param    string $type          'delivery' or 'pickup'.
	 * @param    string $start_date    Start date in Y-m-d format.
	 * @param    string $end_date      End date in Y-m-d format.
	 * @return   int                      Order count
	 */
	private function get_order_count_by_type( $type, $start_date, $end_date ) {
		// Using WC Data Store for HPOS compatibility.
		$query_args = array(
			'type'         => 'shop_order',
			'status'       => array( 'wc-processing', 'wc-completed' ),
			'date_created' => $start_date . '...' . $end_date,
			'return'       => 'ids',
			'limit'        => -1,
			// phpcs:ignore WordPress.DB.SlowDBQuery.slow_db_query_meta_query

		// phpcs:ignore WordPress.DB.SlowDBQuery.slow_db_query_meta_query
			'meta_query'   => array(
				array(
					'key'     => '_aicoso_pickup_delivery_type',
					'value'   => $type,
					'compare' => '=',
				),
			),
		);

		$orders = wc_get_orders( $query_args );

		return count( $orders );
	}

	/**
	 * Get order total by type
	 *
	 * @since    1.0.0
	 * @param    string $type          'delivery' or 'pickup'.
	 * @param    string $start_date    Start date in Y-m-d format.
	 * @param    string $end_date      End date in Y-m-d format.
	 * @return   float                    Order total
	 */
	private function get_order_total_by_type( $type, $start_date, $end_date ) {
		// Using WC Data Store for HPOS compatibility.
		$query_args = array(
			'type'         => 'shop_order',
			'status'       => array( 'wc-processing', 'wc-completed' ),
			'date_created' => $start_date . '...' . $end_date,
			'limit'        => -1,
			// phpcs:ignore WordPress.DB.SlowDBQuery.slow_db_query_meta_query

		// phpcs:ignore WordPress.DB.SlowDBQuery.slow_db_query_meta_query
			'meta_query'   => array(
				array(
					'key'     => '_aicoso_pickup_delivery_type',
					'value'   => $type,
					'compare' => '=',
				),
			),
		);

		$orders = wc_get_orders( $query_args );

		$total = 0;
		foreach ( $orders as $order_id ) {
			$order  = wc_get_order( $order_id );
			$total += $order->get_total();
		}

		return $total;
	}

	/**
	 * Get daily order data for chart
		// phpcs:ignore WordPress.DB.SlowDBQuery.slow_db_query_meta_query
	 *
	 * @since    1.0.0
	 * @param    string $start_date    Start date in Y-m-d format.
	 * @param    string $end_date      End date in Y-m-d format.
	 * @return   array                    Daily order data
	 */
	private function get_daily_order_data( $start_date, $end_date ) {
		$data = array();

		// Get dates between start and end.
		$current_date = new DateTime( $start_date );
		$end_date_obj = new DateTime( $end_date );

		while ( $current_date <= $end_date_obj ) {
			$date_string = $current_date->format( 'Y-m-d' );

			// Get orders for this date.
			$delivery_count = $this->get_order_count_for_date_and_type( $date_string, 'delivery' );
			$pickup_count   = $this->get_order_count_for_date_and_type( $date_string, 'pickup' );

			$data[] = array(
				'date'           => $date_string,
				'formatted_date' => $current_date->format( get_option( 'date_format' ) ),
				'delivery_count' => $delivery_count,
				'pickup_count'   => $pickup_count,
				'total_count'    => $delivery_count + $pickup_count,
			);

			$current_date->modify( '+1 day' );
		}

		return $data;
	}

	/**
	 * Get order count for a specific date and type
	 *
	 * @since    1.0.0
	 * @param    string $date          Date in Y-m-d format.
	 * @param    string $type          'delivery' or 'pickup'.
	 * @return   int                      Order count
	 */
	private function get_order_count_for_date_and_type( $date, $type ) {
		// Using WC Data Store for HPOS compatibility.
		$query_args = array(
			'type'         => 'shop_order',
			'status'       => array( 'wc-processing', 'wc-completed' ),
			'date_created' => $date . '...' . $date . ' 23:59:59',
			'return'       => 'ids',
			'limit'        => -1,
			// phpcs:ignore WordPress.DB.SlowDBQuery.slow_db_query_meta_query

		// phpcs:ignore WordPress.DB.SlowDBQuery.slow_db_query_meta_query
			'meta_query'   => array(
				array(
					'key'     => '_aicoso_pickup_delivery_type',
					'value'   => $type,
					'compare' => '=',
				),
			),
		);

		$orders = wc_get_orders( $query_args );

		return count( $orders );
	}

	/**
	 * Get popular time slots
	 *
	 * @since    1.0.0
	 * @param    string $type          'delivery' or 'pickup'.
	 * @param    string $start_date    Start date in Y-m-d format.
	 * @param    string $end_date      End date in Y-m-d format.
	 * @return   array                    Popular time slots
	 */
	private function get_popular_time_slots( $type, $start_date, $end_date ) {
		$meta_key = '_aicoso_pickup_delivery_' . $type . '_time_slot';

		// Using WC Data Store for HPOS compatibility.
		$query_args = array(
			'type'         => 'shop_order',
			'status'       => array( 'wc-processing', 'wc-completed' ),
			'date_created' => $start_date . '...' . $end_date,
			'limit'        => -1,
			// phpcs:ignore WordPress.DB.SlowDBQuery.slow_db_query_meta_query

		// phpcs:ignore WordPress.DB.SlowDBQuery.slow_db_query_meta_query
			'meta_query'   => array(
				'relation' => 'AND',
				array(
					'key'     => '_aicoso_pickup_delivery_type',
					'value'   => $type,
					'compare' => '=',
				),
				array(
					'key'     => $meta_key,
					'compare' => 'EXISTS',
				),
			),
		);

		$orders = wc_get_orders( $query_args );

		// Count occurrences of each time slot.
		$time_slots = array();
		foreach ( $orders as $order_id ) {
			$order     = wc_get_order( $order_id );
			$time_slot = $order->get_meta( $meta_key, true );

			if ( ! empty( $time_slot ) ) {
				if ( isset( $time_slots[ $time_slot ] ) ) {
					++$time_slots[ $time_slot ];
				} else {
					$time_slots[ $time_slot ] = 1;
				}
			}
		}

		// Sort time slots by popularity.
		arsort( $time_slots );

		// Format results.
		$results = array();
		foreach ( $time_slots as $slot => $count ) {
			$results[] = (object) array(
				'time_slot'   => $slot,
				'order_count' => $count,
			);

			// Limit to top 5.
			if ( count( $results ) >= 5 ) {
				break;
			}
		}

		return $results;
	}

	/**
	 * Get popular pickup locations
	 *
	 * @since    1.0.0
	 * @param    string $start_date    Start date in Y-m-d format.
	 * @param    string $end_date      End date in Y-m-d format.
	 * @return   array                    Popular pickup locations
	 */
	private function get_popular_pickup_locations( $start_date, $end_date ) {
		// Using WC Data Store for HPOS compatibility.
		$query_args = array(
			'type'         => 'shop_order',
			'status'       => array( 'wc-processing', 'wc-completed' ),
			'date_created' => $start_date . '...' . $end_date,
			'limit'        => -1,
			// phpcs:ignore WordPress.DB.SlowDBQuery.slow_db_query_meta_query

		// phpcs:ignore WordPress.DB.SlowDBQuery.slow_db_query_meta_query
			'meta_query'   => array(
				'relation' => 'AND',
				array(
					'key'     => '_aicoso_pickup_delivery_type',
					'value'   => 'pickup',
					'compare' => '=',
				),
				array(
					'key'     => '_aicoso_pickup_delivery_pickup_location',
					'compare' => 'EXISTS',
				),
			),
		);

		$orders = wc_get_orders( $query_args );

		// Count occurrences of each location.
		$locations = array();
		foreach ( $orders as $order_id ) {
			$order    = wc_get_order( $order_id );
			$location = $order->get_meta( '_aicoso_pickup_delivery_pickup_location', true );

			if ( ! empty( $location ) ) {
				if ( isset( $locations[ $location ] ) ) {
					++$locations[ $location ];
				} else {
					$locations[ $location ] = 1;
				}
			}
		}

		// Sort locations by popularity.
		arsort( $locations );

		// Format results.
		$results = array();
		foreach ( $locations as $location => $count ) {
			$results[] = (object) array(
				'location'    => $location,
				'order_count' => $count,
			);
		}

		return $results;
	}
}
