<?php
/**
 * The order management functionality of the plugin.
 *
 * @link       https://example.com
 * @since      1.0.0
 *
 * @package    aicoso_pickup_delivery
 */

if ( ! defined( 'WPINC' ) ) {
	die;
}

/**
 * The order management class.
 *
 * Handles the integration with WooCommerce orders to display and
 * manage delivery/pickup details.
 *
 * @since      1.0.0
 * @package    aicoso_pickup_delivery
 * @author     Your Name
 */
class AICOSO_Pickup_Delivery_Order_Management {

	/**
	 * The ID of this plugin.
	 *
	 * @since  1.0.0
	 * @access private
	 * @var    string $plugin_name The ID of this plugin.
	 */
	private $plugin_name;

	/**
	 * The version of this plugin.
	 *
	 * @since  1.0.0
	 * @access private
	 * @var    string $version The current version of this plugin.
	 */
	private $version;

	/**
	 * Initialize the class and set its properties.
	 *
	 * @since 1.0.0
	 * @param string $plugin_name The name of this plugin.
	 * @param string $version     The version of this plugin.
	 */
	public function __construct( $plugin_name, $version ) {
		$this->plugin_name = $plugin_name;
		$this->version     = $version;
	}

	/**
	 * Enqueue admin scripts for order meta box
	 *
	 * @since    1.0.0
	 * @param    string $hook    The current admin page hook.
	 */
	public function enqueue_order_scripts( $hook ) {
		// Only load on order edit pages.
		if ( 'post.php' !== $hook && 'woocommerce_page_wc-orders' !== $hook ) {
			return;
		}

		$screen = get_current_screen();
		if ( ! $screen || ( 'shop_order' !== $screen->id && 'woocommerce_page_wc-orders' !== $screen->id ) ) {
			return;
		}

		// Enqueue the order meta box script.
		wp_enqueue_script(
			'aicoso-pickup-delivery-order-meta-box',
			AICOSO_PICKUP_DELIVERY_PLUGIN_URL . 'assets/js/admin-order-meta-box.js',
			array( 'jquery' ),
			AICOSO_PICKUP_DELIVERY_VERSION,
			true
		);
	}

	/**
	 * Add meta box to order edit page.
	 *
	 * @since 1.0.0
	 */
	public function add_order_meta_box() {
		$screen = get_current_screen();

		// Add for both classic and HPOS order screens.
		if ( $screen && ( 'shop_order' === $screen->id || 'woocommerce_page_wc-orders' === $screen->id ) ) {
			add_meta_box(
				'aicoso_pickup_delivery_order_details',
				__( '📦 Pickup & Delivery Details', 'aicoso-pickup-and-delivery-for-woocommerce' ),
				array( $this, 'render_order_meta_box' ),
				'shop_order' === $screen->id ? 'shop_order' : 'woocommerce_page_wc-orders',
				'side',
				'high'
			);
		}
	}

	/**
	 * Render the order meta box content.
	 *
	 * @since 1.0.0
	 * @param mixed $post_or_order The post object or order object.
	 */
	public function render_order_meta_box( $post_or_order ) {
		// Get the order object.
		if ( is_a( $post_or_order, 'WC_Order' ) ) {
			$order = $post_or_order;
		} else {
			$order = wc_get_order( $post_or_order->ID );
		}

		if ( ! $order ) {
			return;
		}

		$order_type      = AICOSO_Pickup_Delivery_Data::get_order_type( $order );
		$delivery_date   = $order->get_meta( '_aicoso_pickup_delivery_delivery_date', true );
		$delivery_time   = $order->get_meta( '_aicoso_pickup_delivery_delivery_time_slot', true );
		$pickup_date     = $order->get_meta( '_aicoso_pickup_delivery_pickup_date', true );
		$pickup_time     = $order->get_meta( '_aicoso_pickup_delivery_pickup_time_slot', true );
		$pickup_location = $order->get_meta( '_aicoso_pickup_delivery_pickup_location', true );

		?>
		<div class="aicoso-pickup-delivery-meta-box">
			<p><strong><?php esc_html_e( 'Order Type:', 'aicoso-pickup-and-delivery-for-woocommerce' ); ?></strong>
				<?php if ( 'delivery' === $order_type ) : ?>
					<span style="background: #e8f5e9; color: #2e7d32; padding: 3px 8px; border-radius: 3px; font-weight: bold;">🚚 <?php esc_html_e( 'DELIVERY', 'aicoso-pickup-and-delivery-for-woocommerce' ); ?></span>
				<?php elseif ( 'pickup' === $order_type ) : ?>
					<span style="background: #e3f2fd; color: #1565c0; padding: 3px 8px; border-radius: 3px; font-weight: bold;">📦 <?php esc_html_e( 'PICKUP', 'aicoso-pickup-and-delivery-for-woocommerce' ); ?></span>
				<?php else : ?>
					<span style="color: #999;"><?php esc_html_e( 'Not set', 'aicoso-pickup-and-delivery-for-woocommerce' ); ?></span>
				<?php endif; ?>
			</p>

			<?php if ( 'delivery' === $order_type ) : ?>
				<?php if ( $delivery_date ) : ?>
					<p><strong>📅 <?php esc_html_e( 'Delivery Date:', 'aicoso-pickup-and-delivery-for-woocommerce' ); ?></strong><br>
					<?php echo esc_html( AICOSO_Pickup_Delivery_Data::format_date( $delivery_date ) ); ?></p>
				<?php endif; ?>
				
				<?php if ( $delivery_time ) : ?>
					<p><strong>🕐 <?php esc_html_e( 'Delivery Time:', 'aicoso-pickup-and-delivery-for-woocommerce' ); ?></strong><br>
					<?php echo esc_html( $delivery_time ); ?></p>
				<?php endif; ?>
				
				<?php if ( ! $delivery_date && ! $delivery_time ) : ?>
					<p style="color: #999; font-style: italic;"><?php esc_html_e( 'No delivery details set', 'aicoso-pickup-and-delivery-for-woocommerce' ); ?></p>
				<?php endif; ?>

			<?php elseif ( 'pickup' === $order_type ) : ?>
				<?php if ( $pickup_date ) : ?>
					<p><strong>📅 <?php esc_html_e( 'Pickup Date:', 'aicoso-pickup-and-delivery-for-woocommerce' ); ?></strong><br>
					<?php echo esc_html( AICOSO_Pickup_Delivery_Data::format_date( $pickup_date ) ); ?></p>
				<?php endif; ?>
				
				<?php if ( $pickup_time ) : ?>
					<p><strong>🕐 <?php esc_html_e( 'Pickup Time:', 'aicoso-pickup-and-delivery-for-woocommerce' ); ?></strong><br>
					<?php echo esc_html( $pickup_time ); ?></p>
				<?php endif; ?>
				
				<?php if ( $pickup_location ) : ?>
					<p><strong>📍 <?php esc_html_e( 'Pickup Location:', 'aicoso-pickup-and-delivery-for-woocommerce' ); ?></strong><br>
					<?php echo esc_html( $pickup_location ); ?></p>
				<?php endif; ?>
				
				<?php if ( ! $pickup_date && ! $pickup_time && ! $pickup_location ) : ?>
					<p style="color: #999; font-style: italic;"><?php esc_html_e( 'No pickup details set', 'aicoso-pickup-and-delivery-for-woocommerce' ); ?></p>
				<?php endif; ?>
			<?php endif; ?>
			
			<hr style="margin: 15px 0; border: none; border-top: 1px solid #e0e0e0;">
			
			<p style="margin-bottom: 5px;">
				<a href="#" class="button button-small" onclick="jQuery('#aicoso-pickup-delivery-edit-form').toggle(); return false;">
					<?php esc_html_e( '✏️ Edit Details', 'aicoso-pickup-and-delivery-for-woocommerce' ); ?>
				</a>
			</p>
			
			<div id="aicoso-pickup-delivery-edit-form" style="display: none; margin-top: 10px; padding: 10px; background: #f9f9f9; border-radius: 3px;">
				<?php
				// Include the edit form from the existing meta box partial.
				$order_id = $order->get_id();
				include plugin_dir_path( __FILE__ ) . 'partials/admin-order-meta-box.php';
				?>
			</div>
		</div>
		<?php
	}

	/**
	 * Display delivery/pickup information in the order meta box
	 *
	 * @since    1.0.0
	 * @param    WC_Order $order    The order object.
	 */
	public function display_delivery_pickup_info( $order ) {
		if ( ! $order instanceof WC_Order ) {
			return;
		}

		$order_id   = $order->get_id();
		$order_type = AICOSO_Pickup_Delivery_Data::get_order_type( $order );

		// Get template file.
		include plugin_dir_path( __FILE__ ) . 'partials/admin-order-meta-box.php';
	}

	/**
	 * Save delivery/pickup information when editing an order
	 *
	 * @since    1.0.0
	 * @param    int $order_id    The order ID.
	 */
	public function save_delivery_pickup_info( $order_id ) {
		if ( ! $order_id ) {
			return;
		}

		// Check if this is an update action.
		// phpcs:ignore WordPress.Security.NonceVerification.Missing -- Nonce is verified in the next step
		if ( ! isset( $_POST['aicoso_pickup_delivery_order_update'] ) || ! isset( $_POST['aicoso_pickup_delivery_nonce'] ) ) {
			return;
		}

		// Verify nonce.
		if ( ! wp_verify_nonce( sanitize_text_field( wp_unslash( $_POST['aicoso_pickup_delivery_nonce'] ) ), 'aicoso_pickup_delivery_save_order_meta' ) ) {
			return;
		}

		$order = wc_get_order( $order_id );
		if ( ! $order ) {
			return;
		}

		// Get old order type.
		$old_order_type = AICOSO_Pickup_Delivery_Data::get_order_type( $order );

		// Get new order type.
		$new_order_type = isset( $_POST['aicoso_pickup_delivery_type'] ) ? sanitize_text_field( wp_unslash( $_POST['aicoso_pickup_delivery_type'] ) ) : $old_order_type;

		// Save order type - using update_meta for HPOS compatibility.
		$order->update_meta_data( '_aicoso_pickup_delivery_type', $new_order_type );

		// Save delivery details.
		if ( 'delivery' === $new_order_type ) {
			if ( isset( $_POST['aicoso_pickup_delivery_delivery_date'] ) ) {
				$order->update_meta_data( '_aicoso_pickup_delivery_delivery_date', sanitize_text_field( wp_unslash( $_POST['aicoso_pickup_delivery_delivery_date'] ) ) );
			}

			if ( isset( $_POST['aicoso_pickup_delivery_delivery_time_slot'] ) ) {
				$order->update_meta_data( '_aicoso_pickup_delivery_delivery_time_slot', sanitize_text_field( wp_unslash( $_POST['aicoso_pickup_delivery_delivery_time_slot'] ) ) );
			}
		}

		// Save pickup details.
		if ( 'pickup' === $new_order_type ) {
			if ( isset( $_POST['aicoso_pickup_delivery_pickup_date'] ) ) {
				$order->update_meta_data( '_aicoso_pickup_delivery_pickup_date', sanitize_text_field( wp_unslash( $_POST['aicoso_pickup_delivery_pickup_date'] ) ) );
			}

			if ( isset( $_POST['aicoso_pickup_delivery_pickup_time_slot'] ) ) {
				$order->update_meta_data( '_aicoso_pickup_delivery_pickup_time_slot', sanitize_text_field( wp_unslash( $_POST['aicoso_pickup_delivery_pickup_time_slot'] ) ) );
			}

			if ( isset( $_POST['aicoso_pickup_delivery_pickup_location'] ) ) {
				$order->update_meta_data( '_aicoso_pickup_delivery_pickup_location', sanitize_text_field( wp_unslash( $_POST['aicoso_pickup_delivery_pickup_location'] ) ) );
			}
		}

		// Save the order - needed for HPOS.
		$order->save();

		// Send notification if order type changed.
		if ( $old_order_type !== $new_order_type ) {
			do_action( 'aicoso_pickup_delivery_order_type_updated', $order_id, $old_order_type, $new_order_type );

			// Add order note.
			$order->add_order_note(
				sprintf(
					/* translators: 1: old order type, 2: new order type */
					__( 'Order type changed from %1$s to %2$s.', 'aicoso-pickup-and-delivery-for-woocommerce' ),
					'delivery' === $old_order_type ? __( 'Delivery', 'aicoso-pickup-and-delivery-for-woocommerce' ) : __( 'Pickup', 'aicoso-pickup-and-delivery-for-woocommerce' ),
					'delivery' === $new_order_type ? __( 'Delivery', 'aicoso-pickup-and-delivery-for-woocommerce' ) : __( 'Pickup', 'aicoso-pickup-and-delivery-for-woocommerce' )
				)
			);

			// Send notification email.
			$email_manager = new AICOSO_Pickup_Delivery_Email_Manager( $this->plugin_name, $this->version );
			$email_manager->send_update_notification( $order_id, $old_order_type, $new_order_type );
		}
	}

	/**
	 * Add delivery/pickup date column to orders list
	 *
	 * @since    1.0.0
	 * @param    array $columns    Current columns.
	 * @return   array                Modified columns
	 */
	public function add_order_list_columns( $columns ) {
		$new_columns = array();

		// Insert our column after order date.
		foreach ( $columns as $key => $value ) {
			$new_columns[ $key ] = $value;

			if ( 'order_date' === $key ) {
				$new_columns['pickup_delivery'] = __( 'Delivery/Pickup', 'aicoso-pickup-and-delivery-for-woocommerce' );
			}
		}

		return $new_columns;
	}

	/**
	 * Add content to delivery/pickup column in orders list
	 *
	 * @since    1.0.0
	 * @param    string $column     Current column.
	 * @param    int    $order_id   Order ID.
	 */
	public function add_order_list_column_content( $column, $order_id ) {
		if ( 'pickup_delivery' !== $column ) {
			return;
		}

		$order = wc_get_order( $order_id );
		if ( ! $order ) {
			return;
		}

		$order_type = AICOSO_Pickup_Delivery_Data::get_order_type( $order );

		if ( 'delivery' === $order_type ) {
			$delivery_date = $order->get_meta( '_aicoso_pickup_delivery_delivery_date', true );
			$delivery_time = $order->get_meta( '_aicoso_pickup_delivery_delivery_time_slot', true );

			echo '<div style="line-height: 1.4;">';
			echo '<span style="display: inline-block; background: #e8f5e9; color: #2e7d32; padding: 2px 6px; border-radius: 3px; font-size: 11px; font-weight: bold; margin-bottom: 4px;">🚚 ' . esc_html__( 'DELIVERY', 'aicoso-pickup-and-delivery-for-woocommerce' ) . '</span><br>';

			if ( $delivery_date ) {
				echo '<small>📅 ' . esc_html( AICOSO_Pickup_Delivery_Data::format_date( $delivery_date ) ) . '</small>';
			}

			if ( $delivery_time ) {
				echo '<br><small>🕐 ' . esc_html( $delivery_time ) . '</small>';
			}

			if ( ! $delivery_date && ! $delivery_time ) {
				echo '<small style="color: #999;">' . esc_html__( 'No date/time set', 'aicoso-pickup-and-delivery-for-woocommerce' ) . '</small>';
			}
			echo '</div>';
		} elseif ( 'pickup' === $order_type ) {
			$pickup_date     = $order->get_meta( '_aicoso_pickup_delivery_pickup_date', true );
			$pickup_time     = $order->get_meta( '_aicoso_pickup_delivery_pickup_time_slot', true );
			$pickup_location = $order->get_meta( '_aicoso_pickup_delivery_pickup_location', true );

			echo '<div style="line-height: 1.4;">';
			echo '<span style="display: inline-block; background: #e3f2fd; color: #1565c0; padding: 2px 6px; border-radius: 3px; font-size: 11px; font-weight: bold; margin-bottom: 4px;">📦 ' . esc_html__( 'PICKUP', 'aicoso-pickup-and-delivery-for-woocommerce' ) . '</span><br>';

			if ( $pickup_date ) {
				echo '<small>📅 ' . esc_html( AICOSO_Pickup_Delivery_Data::format_date( $pickup_date ) ) . '</small>';
			}

			if ( $pickup_time ) {
				echo '<br><small>🕐 ' . esc_html( $pickup_time ) . '</small>';
			}

			if ( $pickup_location ) {
				echo '<br><small>📍 ' . esc_html( $pickup_location ) . '</small>';
			}

			if ( ! $pickup_date && ! $pickup_time && ! $pickup_location ) {
				echo '<small style="color: #999;">' . esc_html__( 'No details set', 'aicoso-pickup-and-delivery-for-woocommerce' ) . '</small>';
			}
			echo '</div>';
		} else {
			echo '<span style="color: #999;">—</span>';
		}
	}

	/**
	 * Add order filter for delivery/pickup date
	 *
	 * @since    1.0.0
	 */
	public function add_delivery_pickup_date_filter() {
		global $pagenow, $typenow;

		if ( 'shop_order' === $typenow && 'edit.php' === $pagenow ) {
// phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Admin page filtering
			$current_date = isset( $_GET['pickup_delivery_date'] ) ? sanitize_text_field( wp_unslash( $_GET['pickup_delivery_date'] ) ) : '';
			?>
			<input type="text" name="pickup_delivery_date" placeholder="<?php esc_attr_e( 'Filter by date', 'aicoso-pickup-and-delivery-for-woocommerce' ); ?>" value="<?php echo esc_attr( $current_date ); ?>" class="date-picker" />
			<!-- JavaScript functionality moved to assets/js/aicoso-pickup-delivery-admin-orders.js for WordPress standards compliance -->
			<?php
		}
	}

	/**
	 * Filter orders by delivery/pickup date
	 *
	 * @since    1.0.0
	 * @param    array $query_vars    Current query variables.
	 * @return   array                   Modified query variables
	 */
	public function filter_orders_by_date( $query_vars ) {
		global $pagenow, $typenow;

// phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Admin page filtering
		if ( 'shop_order' === $typenow && 'edit.php' === $pagenow && isset( $_GET['pickup_delivery_date'] ) && ! empty( $_GET['pickup_delivery_date'] ) ) {
			// phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Admin page filtering
			$date = sanitize_text_field( wp_unslash( $_GET['pickup_delivery_date'] ) );

			// Using meta_query for HPOS compatibility.
			// phpcs:ignore WordPress.DB.SlowDBQuery.slow_db_query_meta_query
			if ( ! isset( $query_vars['meta_query'] ) ) {
			// phpcs:ignore WordPress.DB.SlowDBQuery.slow_db_query_meta_query
				$query_vars['meta_query'] = array();
			}

			// phpcs:ignore WordPress.DB.SlowDBQuery.slow_db_query_meta_query
			$query_vars['meta_query'][] = array(
				'relation' => 'OR',
				array(
					'key'     => '_aicoso_pickup_delivery_delivery_date',
					'value'   => $date,
					'compare' => '=',
				),
				array(
					'key'     => '_aicoso_pickup_delivery_pickup_date',
					'value'   => $date,
					'compare' => '=',
				),
			);
		}

		return $query_vars;
	}
}