#!/usr/bin/env php
<?php
/**
 * Extract translatable strings script.
 *
 * Generates a POT file for the plugin by scanning PHP files
 * for translation function calls using the plugin text domain.
 *
 * NOTE: This script expects to be run within a WordPress environment
 * where `ABSPATH` and translation helper functions are available.
 *
 * @package Aicoso_Pickup_Delivery_For_Woocommerce
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}
/**
 * Simple POT file generator for the aicoso-pickup-delivery plugin
 * Run this script from within WordPress environment
 */

// Plugin details.
$plugin_name       = 'AICOSO Pickup and Delivery for WooCommerce';
$text_domain       = 'aicoso-pickup-and-delivery-for-woocommerce';
$plugin_version    = '1.0.0';
$plugin_author     = 'AICOSO';
$plugin_author_uri = 'https://aicoso.com';
$bugs_url          = 'https://aicoso.com/support';

// Output POT file header.
$pot_header  = "# Copyright (C) 2024 $plugin_author\n";
$pot_header .= "# This file is distributed under the GPL-2.0+.\n";
$pot_header .= "msgid \"\"\n";
$pot_header .= "msgstr \"\"\n";
$pot_header .= "\"Project-Id-Version: $plugin_name $plugin_version\\n\"\n";
$pot_header .= "\"Report-Msgid-Bugs-To: $bugs_url\\n\"\n";
$pot_header .= "\"POT-Creation-Date: 2024-09-18 08:00+0000\\n\"\n";
$pot_header .= "\"MIME-Version: 1.0\\n\"\n";
$pot_header .= "\"Content-Type: text/plain; charset=UTF-8\\n\"\n";
$pot_header .= "\"Content-Transfer-Encoding: 8bit\\n\"\n";
$pot_header .= "\"PO-Revision-Date: YEAR-MO-DA HO:MI+ZONE\\n\"\n";
$pot_header .= "\"Last-Translator: FULL NAME <EMAIL@ADDRESS>\\n\"\n";
$pot_header .= "\"Language-Team: LANGUAGE <LL@li.org>\\n\"\n";
$pot_header .= "\"Language: \\n\"\n";
$pot_header .= "\"Plural-Forms: nplurals=INTEGER; plural=EXPRESSION;\\n\"\n";
$pot_header .= "\"X-Generator: WC Pickup Delivery POT Generator\\n\"\n";
$pot_header .= "\"X-Domain: $text_domain\\n\"\n";
$pot_header .= "\n";

// Function to extract strings from PHP files.
/**
 * Extract translatable strings from PHP files.
 *
 * @param string $directory  The directory to scan.
 * @param string $text_domain The text domain to look for.
 * @return array Array of translatable strings with their locations.
 */
function aicoso_pickup_delivery_extract_translatable_strings( $directory, $text_domain ) {
	$strings = array();
	$files   = new RecursiveIteratorIterator(
		new RecursiveDirectoryIterator( $directory, RecursiveDirectoryIterator::SKIP_DOTS ),
		RecursiveIteratorIterator::LEAVES_ONLY
	);

	foreach ( $files as $file ) {
		if ( $file->getExtension() !== 'php' ) {
			continue;
		}
		if ( strpos( $file->getPathname(), '/node_modules/' ) !== false ) {
			continue;
		}
		if ( strpos( $file->getPathname(), '/vendor/' ) !== false ) {
			continue;
		}

        // phpcs:ignore WordPress.WP.AlternativeFunctions.file_get_contents_file_get_contents
		$content       = file_get_contents( $file->getPathname() );
		$relative_path = str_replace( $directory . '/', '', $file->getPathname() );

		// Match various translation functions.
		$patterns = array(
			// Pattern for double underscore, esc_html__, and esc_attr__ functions.
			'/(?:__|esc_html__|esc_attr__)\s*\(\s*[\'"]([^\'"]+)[\'"]\s*,\s*[\'"]' . preg_quote( $text_domain, '/' ) . '[\'"]\s*\)/i',
			// Pattern for underscore e function.
			'/_e\s*\(\s*[\'"]([^\'"]+)[\'"]\s*,\s*[\'"]' . preg_quote( $text_domain, '/' ) . '[\'"]\s*\)/i',
			// Pattern for escaped HTML echo function.
			'/esc_html_e\s*\(\s*[\'"]([^\'"]+)[\'"]\s*,\s*[\'"]' . preg_quote( $text_domain, '/' ) . '[\'"]\s*\)/i',
			// Pattern for underscore x function with context parameter.
			'/_x\s*\(\s*[\'"]([^\'"]+)[\'"]\s*,\s*[\'"]([^\'"]+)[\'"]\s*,\s*[\'"]' . preg_quote( $text_domain, '/' ) . '[\'"]\s*\)/i',
		);

		foreach ( $patterns as $pattern ) {
			if ( preg_match_all( $pattern, $content, $matches, PREG_OFFSET_CAPTURE ) ) {
				foreach ( $matches[1] as $match ) {
					$string   = $match[0];
					$position = $match[1];

					// Get line number.
					$line_number = substr_count( substr( $content, 0, $position ), "\n" ) + 1;

					if ( ! isset( $strings[ $string ] ) ) {
						$strings[ $string ] = array();
					}
					$strings[ $string ][] = "$relative_path:$line_number";
				}
			}
		}
	}

	return $strings;
}

// Get plugin directory.
$plugin_dir = __DIR__;

// Extract strings.
$translatable_strings = aicoso_pickup_delivery_extract_translatable_strings( $plugin_dir, $text_domain );

// Generate POT content.
$pot_content = $pot_header;

foreach ( $translatable_strings as $string => $locations ) {
	$pot_content .= "\n";
	foreach ( $locations as $location ) {
		$pot_content .= "#: $location\n";
	}

	// Escape quotes in the string.
	$escaped_string = str_replace( '"', '\\"', $string );

	$pot_content .= "msgid \"$escaped_string\"\n";
	$pot_content .= "msgstr \"\"\n";
}

// Write POT file.
$pot_file = $plugin_dir . '/languages/aicoso-pickup-delivery.pot';
if ( ! is_dir( $plugin_dir . '/languages' ) ) {
	// Use wp_mkdir_p() for WordPress compliance.
	if ( function_exists( 'wp_mkdir_p' ) ) {
		wp_mkdir_p( $plugin_dir . '/languages' );
	} else {
		// Fallback for command line usage.
        // phpcs:ignore WordPress.WP.AlternativeFunctions.file_system_operations_mkdir
		mkdir( $plugin_dir . '/languages', 0755, true );
	}
}

// phpcs:ignore WordPress.WP.AlternativeFunctions.file_system_operations_file_put_contents
file_put_contents( $pot_file, $pot_content );

echo esc_html( "POT file generated successfully at: $pot_file\n" );
echo esc_html( 'Total translatable strings found: ' . count( $translatable_strings ) . "\n" );

// Show summary.
echo esc_html( "\nStrings extracted:\n" );
foreach ( $translatable_strings as $string => $locations ) {
	echo esc_html( "- \"$string\" (found in " . count( $locations ) . " location(s))\n" );
}
