/**
 * WooCommerce Blocks Checkout Integration
 * 
 * This file adds pickup/delivery fields to the WooCommerce Blocks checkout
 */

(function() {
    'use strict';

    // Wait for WooCommerce Blocks to be available
    if (!window.wc || !window.wc.blocksCheckout) {
        console.warn('WooCommerce Blocks not available');
        return;
    }

    const { registerPlugin } = wp.plugins;
    const { ExperimentalOrderMeta } = window.wc.blocksCheckout;
    const { useState, useEffect } = wp.element;
    const { RadioControl, SelectControl, Notice } = wp.components;
    const { getSetting } = window.wc.wcSettings;

    // Get plugin settings
    const pluginSettings = getSetting('aicoso_pickup_delivery_data', {});
    const isEnabled = pluginSettings.enable_order_type_selection === 'yes';

    if (!isEnabled) {
        console.log('Pickup/Delivery selection is disabled in settings');
        return;
    }

    const PickupDeliveryFields = () => {
        const [orderType, setOrderType] = useState('delivery');
        const [deliveryDate, setDeliveryDate] = useState('');
        const [deliveryTime, setDeliveryTime] = useState('');
        const [pickupDate, setPickupDate] = useState('');
        const [pickupTime, setPickupTime] = useState('');
        const [pickupLocation, setPickupLocation] = useState('');

        // Save data to checkout
        useEffect(() => {
            if (window.wp && window.wp.data) {
                const { dispatch } = window.wp.data;
                const checkoutStore = 'wc/store/checkout';
                
                // Update extension data
                dispatch(checkoutStore).__internalSetExtensionData('aicoso-pickup-delivery', {
                    order_type: orderType,
                    delivery_date: deliveryDate,
                    delivery_time: deliveryTime,
                    pickup_date: pickupDate,
                    pickup_time: pickupTime,
                    pickup_location: pickupLocation
                });
            }
        }, [orderType, deliveryDate, deliveryTime, pickupDate, pickupTime, pickupLocation]);

        // Get time slots from settings
        const getTimeSlots = (type) => {
            const slots = type === 'delivery' 
                ? pluginSettings.delivery_time_slots 
                : pluginSettings.pickup_time_slots;
            
            if (!slots || !Array.isArray(slots)) {
                return [];
            }
            
            return slots.map(slot => ({
                label: slot,
                value: slot
            }));
        };

        // Get pickup locations from settings
        const getPickupLocations = () => {
            const locations = pluginSettings.pickup_locations;
            
            if (!locations || !Array.isArray(locations)) {
                return [];
            }
            
            return locations.map(location => ({
                label: location.location_name || location,
                value: location.location_name || location
            }));
        };

        // Get available dates (simplified - in production, use proper date calculation)
        const getAvailableDates = () => {
            const dates = [];
            const today = new Date();
            
            for (let i = 1; i <= 30; i++) {
                const date = new Date(today);
                date.setDate(today.getDate() + i);
                const dateStr = date.toISOString().split('T')[0];
                dates.push({
                    label: date.toLocaleDateString(),
                    value: dateStr
                });
            }
            
            return dates;
        };

        return (
            <div className="aicoso-pickup-delivery-fields">
                <h3>Delivery or Pickup</h3>
                
                <RadioControl
                    label="How would you like to receive your order?"
                    selected={orderType}
                    options={[
                        { label: 'Delivery', value: 'delivery' },
                        { label: 'Pickup', value: 'pickup' }
                    ]}
                    onChange={setOrderType}
                />

                {orderType === 'delivery' && (
                    <div className="delivery-fields">
                        <SelectControl
                            label="Delivery Date"
                            value={deliveryDate}
                            options={[
                                { label: 'Select a date', value: '' },
                                ...getAvailableDates()
                            ]}
                            onChange={setDeliveryDate}
                            required
                        />
                        
                        {deliveryDate && (
                            <SelectControl
                                label="Delivery Time"
                                value={deliveryTime}
                                options={[
                                    { label: 'Select a time', value: '' },
                                    ...getTimeSlots('delivery')
                                ]}
                                onChange={setDeliveryTime}
                                required
                            />
                        )}
                    </div>
                )}

                {orderType === 'pickup' && (
                    <div className="pickup-fields">
                        <SelectControl
                            label="Pickup Location"
                            value={pickupLocation}
                            options={[
                                { label: 'Select a location', value: '' },
                                ...getPickupLocations()
                            ]}
                            onChange={setPickupLocation}
                            required
                        />
                        
                        {pickupLocation && (
                            <>
                                <SelectControl
                                    label="Pickup Date"
                                    value={pickupDate}
                                    options={[
                                        { label: 'Select a date', value: '' },
                                        ...getAvailableDates()
                                    ]}
                                    onChange={setPickupDate}
                                    required
                                />
                                
                                {pickupDate && (
                                    <SelectControl
                                        label="Pickup Time"
                                        value={pickupTime}
                                        options={[
                                            { label: 'Select a time', value: '' },
                                            ...getTimeSlots('pickup')
                                        ]}
                                        onChange={setPickupTime}
                                        required
                                    />
                                )}
                            </>
                        )}
                    </div>
                )}
            </div>
        );
    };

    // Register the component
    const render = () => {
        return (
            <ExperimentalOrderMeta>
                <PickupDeliveryFields />
            </ExperimentalOrderMeta>
        );
    };

    registerPlugin('aicoso-pickup-delivery', {
        render,
        scope: 'woocommerce-checkout'
    });

})();