/**
 * WooCommerce Pickup & Delivery - Blocks Checkout Fields
 * 
 * Handles the dynamic insertion and management of pickup/delivery fields
 * in the WooCommerce Blocks checkout page.
 */

(function(window, document) {
    'use strict';

    // Configuration object passed from PHP
    const wcPickupDeliveryBlocks = window.wcPickupDeliveryBlocks || {};

    /**
     * Main checkout fields handler
     */
    const WCPickupDeliveryCheckoutFields = {
        
        // Cache DOM elements
        elements: {
            checkoutForm: null,
            fieldsContainer: null,
            orderTypeSelect: null,
            deliveryFields: null,
            pickupFields: null
        },

        // Settings from PHP
        settings: {
            enableOrderTypeSelection: 'yes',
            defaultOrderType: 'delivery',
            enableDeliveryDate: 'yes',
            enableDeliveryTimeSlots: 'yes',
            enablePickupDate: 'yes',
            enablePickupTimeSlots: 'yes',
            hasDeliveryFields: true,
            hasPickupFields: true,
            // Override with passed settings
            ...wcPickupDeliveryBlocks.settings
        },

        // Localized strings
        strings: {
            deliveryPickupOptions: 'Delivery/Pickup Options',
            deliveryOptions: 'Delivery Options',
            pickupOptions: 'Pickup Options',
            orderType: 'Order Type',
            delivery: 'Delivery',
            pickup: 'Pickup',
            deliveryDate: 'Delivery Date',
            deliveryTime: 'Delivery Time',
            pickupDate: 'Pickup Date',
            pickupTime: 'Pickup Time',
            pickupLocation: 'Pickup Location',
            selectTime: 'Select a time',
            selectLocation: 'Select a location',
            // Override with passed strings
            ...wcPickupDeliveryBlocks.strings
        },

        // Time slots configuration
        timeSlots: wcPickupDeliveryBlocks.timeSlots || [
            { value: 'morning', label: 'Morning (9AM - 12PM)' },
            { value: 'afternoon', label: 'Afternoon (12PM - 5PM)' },
            { value: 'evening', label: 'Evening (5PM - 8PM)' }
        ],

        // Pickup locations configuration
        pickupLocations: wcPickupDeliveryBlocks.pickupLocations || [
            { value: 'main-store', label: 'Main Store' },
            { value: 'warehouse', label: 'Warehouse' },
            { value: 'downtown', label: 'Downtown Location' }
        ],

        /**
         * Initialize the checkout fields
         */
        init: function() {
            // Don't initialize if all fields are disabled
            if (!this.shouldShowFields()) {
                return;
            }

            // Wait for DOM to be ready
            if (document.readyState === 'loading') {
                document.addEventListener('DOMContentLoaded', () => this.onDOMReady());
            } else {
                this.onDOMReady();
            }
        },

        /**
         * Check if we should show any fields
         */
        shouldShowFields: function() {
            const { enableOrderTypeSelection, hasDeliveryFields, hasPickupFields, defaultOrderType } = this.settings;
            
            // If order type selection is disabled, check if we have any relevant fields to show
            if (enableOrderTypeSelection === 'no') {
                // Only show if we have fields for the default order type
                if (defaultOrderType === 'delivery' && !hasDeliveryFields) {
                    return false;
                }
                if (defaultOrderType === 'pickup' && !hasPickupFields) {
                    return false;
                }
                // If no fields at all, don't show
                if (!hasDeliveryFields && !hasPickupFields) {
                    return false;
                }
            }
            
            // If order type selection is enabled, always show the container
            return true;
        },

        /**
         * DOM ready handler
         */
        onDOMReady: function() {
            // Try to add fields with retry logic
            this.addFieldsWithRetry();
            // Start observing for generic Blocks notices so we can suppress them
            this.observeAndRemoveGenericNotices();
        },

        /**
         * Add fields with retry logic for dynamic checkout loading
         */
        addFieldsWithRetry: function(attempts = 0) {
            const maxAttempts = 20;
            const retryDelay = 500;

            if (attempts >= maxAttempts) {
                console.warn('WC Pickup/Delivery: Could not find checkout form after maximum attempts');
                return;
            }

            // Try to add fields
            if (!this.addPickupDeliveryFields()) {
                // Retry after delay
                setTimeout(() => this.addFieldsWithRetry(attempts + 1), retryDelay);
            }
        },

        /**
         * Add pickup/delivery fields to checkout
         */
        addPickupDeliveryFields: function() {
            // Find checkout form
            this.elements.checkoutForm = document.querySelector('.wc-block-checkout__form');
            if (!this.elements.checkoutForm) {
                return false; // Retry
            }

            // Check if fields already exist
            if (document.getElementById('aicoso-pickup-delivery-fields')) {
                return true; // Already added
            }

            // Find insertion point
            const insertPoint = this.findInsertionPoint();
            if (!insertPoint) {
                return false; // Retry
            }

            // Create fields container
            this.elements.fieldsContainer = this.createFieldsContainer();
            
            // If container is null (no fields to show), we're done
            if (!this.elements.fieldsContainer) {
                return true; // Success - nothing to show
            }

            // Insert the container
            insertPoint.parentNode.insertBefore(this.elements.fieldsContainer, insertPoint.nextSibling);

            // Setup event listeners
            this.setupEventListeners();

            // Hook into checkout submission
            this.setupCheckoutSubmission();

            return true; // Success
        },

        /**
         * Find where to insert the fields
         */
        findInsertionPoint: function() {
            const contactSection = this.elements.checkoutForm.querySelector('.wc-block-checkout__contact-fields');
            const shippingSection = this.elements.checkoutForm.querySelector('.wc-block-checkout__shipping-fields');
            return contactSection || shippingSection;
        },

        /**
         * Create the fields container with all fields
         */
        createFieldsContainer: function() {
            // Build the HTML structure
            const fieldsHTML = this.getFieldsHTML();
            
            // If no HTML to show (order type disabled and no fields), return null
            if (!fieldsHTML) {
                return null;
            }

            const container = document.createElement('div');
            container.id = 'aicoso-pickup-delivery-fields';
            container.className = 'wc-block-components-checkout-step aicoso-pickup-delivery-section';
            container.innerHTML = fieldsHTML;

            // Store references to important elements
            this.elements.orderTypeSelect = container.querySelector('#aicoso_pickup_delivery_type');
            this.elements.deliveryFields = container.querySelector('#delivery-fields');
            this.elements.pickupFields = container.querySelector('#pickup-fields');

            return container;
        },

        /**
         * Generate the HTML for all fields
         */
        getFieldsHTML: function() {
            const { 
                enableOrderTypeSelection, 
                defaultOrderType,
                hasDeliveryFields,
                hasPickupFields
            } = this.settings;

            // Don't generate any HTML if order type selection is disabled and no fields to show
            if (enableOrderTypeSelection === 'no') {
                // Check if we should show anything based on default type
                if (defaultOrderType === 'delivery' && !hasDeliveryFields) {
                    return ''; // Return empty string - no container
                }
                if (defaultOrderType === 'pickup' && !hasPickupFields) {
                    return ''; // Return empty string - no container
                }
            }

            let html = '<div class="wc-block-components-checkout-step__container">';

            // Add title
            html += this.getTitleHTML();

            html += '<div class="wc-block-components-checkout-step__content">';

            // Add order type selector if enabled
            if (enableOrderTypeSelection === 'yes') {
                html += this.getOrderTypeSelectorHTML();
            }

            // Add delivery fields
            if (enableOrderTypeSelection === 'yes' || defaultOrderType === 'delivery') {
                const hideDelivery = enableOrderTypeSelection === 'yes' && defaultOrderType !== 'delivery';
                html += this.getDeliveryFieldsHTML(hideDelivery);
            }

            // Add pickup fields
            if (enableOrderTypeSelection === 'yes' || defaultOrderType === 'pickup') {
                const hidePickup = enableOrderTypeSelection === 'yes' && defaultOrderType !== 'pickup';
                html += this.getPickupFieldsHTML(hidePickup);
            }

            html += '</div></div>';

            return html;
        },

        /**
         * Get the appropriate title HTML
         */
        getTitleHTML: function() {
            const { enableOrderTypeSelection, defaultOrderType, hasDeliveryFields, hasPickupFields } = this.settings;
            let title = '';

            if (enableOrderTypeSelection === 'yes') {
                title = this.strings.deliveryPickupOptions;
            } else if (defaultOrderType === 'delivery' && hasDeliveryFields) {
                title = this.strings.deliveryOptions;
            } else if (defaultOrderType === 'pickup' && hasPickupFields) {
                title = this.strings.pickupOptions;
            }

            if (!title) return '';

            return `<h2 class="wc-block-components-title wc-block-components-checkout-step__title">${title}</h2>`;
        },

        /**
         * Get order type selector HTML
         */
        getOrderTypeSelectorHTML: function() {
            const { defaultOrderType } = this.settings;
            
            return `
                <div class="wc-block-components-form aicoso-pickup-delivery-type-selector">
                    <label for="aicoso_pickup_delivery_type">${this.strings.orderType}</label>
                    <select id="aicoso_pickup_delivery_type" name="aicoso_pickup_delivery_type" class="wc-block-components-select">
                        <option value="delivery" ${defaultOrderType === 'delivery' ? 'selected' : ''}>
                            ${this.strings.delivery}
                        </option>
                        <option value="pickup" ${defaultOrderType === 'pickup' ? 'selected' : ''}>
                            ${this.strings.pickup}
                        </option>
                    </select>
                </div>
            `;
        },

        /**
         * Get delivery fields HTML
         */
        getDeliveryFieldsHTML: function(hidden = false) {
            const { enableDeliveryDate, enableDeliveryTimeSlots } = this.settings;
            // If jQuery UI datepicker is available we prefer a text input (datepicker attaches),
            // otherwise keep native date input to leverage browser picker.
            const prefersTextDate = (window.jQuery && jQuery.fn && jQuery.fn.datepicker);
            const dateInputType = prefersTextDate ? 'text' : 'date';

            let html = `<div id="delivery-fields" class="aicoso-pickup-delivery-fields" ${hidden ? 'style="display:none;"' : ''}>`;

            if (enableDeliveryDate === 'yes') {
                html += `
                    <div class="wc-block-components-form">
                        <label for="aicoso_pickup_delivery_delivery_date">${this.strings.deliveryDate}</label>
                        <div class="aicoso-date-input-wrap">
                            <input type="${dateInputType}" id="aicoso_pickup_delivery_delivery_date" 
                                   name="aicoso_pickup_delivery_delivery_date" 
                                   class="wc-block-components-text-input" required />
                            <span class="aicoso-date-input-icon" aria-hidden="true">📅</span>
                        </div>
                    </div>
                `;
            }

            if (enableDeliveryTimeSlots === 'yes') {
                html += `
                    <div class="wc-block-components-form">
                        <label for="aicoso_pickup_delivery_delivery_time_slot">${this.strings.deliveryTime}</label>
                        <select id="aicoso_pickup_delivery_delivery_time_slot" 
                                name="aicoso_pickup_delivery_delivery_time_slot" 
                                class="wc-block-components-select">
                            <option value="">${this.strings.selectTime}</option>
                            ${this.timeSlots.map(slot => 
                                `<option value="${slot.value}">${slot.label}</option>`
                            ).join('')}
                        </select>
                    </div>
                `;
            }

            html += '</div>';
            return html;
        },

        /**
         * Get pickup fields HTML
         */
        getPickupFieldsHTML: function(hidden = false) {
            const { enablePickupDate, enablePickupTimeSlots } = this.settings;
            const prefersTextDate = (window.jQuery && jQuery.fn && jQuery.fn.datepicker);
            const dateInputType = prefersTextDate ? 'text' : 'date';

            let html = `<div id="pickup-fields" class="aicoso-pickup-delivery-fields" ${hidden ? 'style="display:none;"' : ''}>`;

            if (enablePickupDate === 'yes') {
                html += `
                    <div class="wc-block-components-form">
                        <label for="aicoso_pickup_delivery_pickup_date">${this.strings.pickupDate}</label>
                        <div class="aicoso-date-input-wrap">
                            <input type="${dateInputType}" id="aicoso_pickup_delivery_pickup_date" 
                                   name="aicoso_pickup_delivery_pickup_date" 
                                   class="wc-block-components-text-input" required />
                            <span class="aicoso-date-input-icon" aria-hidden="true">📅</span>
                        </div>
                    </div>
                `;
            }

            if (enablePickupTimeSlots === 'yes') {
                html += `
                    <div class="wc-block-components-form">
                        <label for="aicoso_pickup_delivery_pickup_time_slot">${this.strings.pickupTime}</label>
                        <select id="aicoso_pickup_delivery_pickup_time_slot" 
                                name="aicoso_pickup_delivery_pickup_time_slot" 
                                class="wc-block-components-select">
                            <option value="">${this.strings.selectTime}</option>
                            ${this.timeSlots.map(slot => 
                                `<option value="${slot.value}">${slot.label}</option>`
                            ).join('')}
                        </select>
                    </div>
                `;
            }

            // Always show pickup location field
            html += `
                <div class="wc-block-components-form">
                    <label for="aicoso_pickup_delivery_pickup_location">${this.strings.pickupLocation}</label>
                    <select id="aicoso_pickup_delivery_pickup_location" 
                            name="aicoso_pickup_delivery_pickup_location" 
                            class="wc-block-components-select">
                        <option value="">${this.strings.selectLocation}</option>
                        ${this.pickupLocations.map(location => 
                            `<option value="${location.value}">${location.label}</option>`
                        ).join('')}
                    </select>
                </div>
            `;

            html += '</div>';
            return html;
        },

        /**
         * Setup event listeners
         */
        setupEventListeners: function() {
            // Order type change listener
            if (this.elements.orderTypeSelect) {
                this.elements.orderTypeSelect.addEventListener('change', (e) => this.handleOrderTypeChange(e));
            }

            // Date field validation
            this.setupDateValidation();
        },

        /**
         * Validate date fields against localized rules.
         * Returns true if valid, false otherwise.
         */
        validateDateFields: function() {
            const dateSettings = wcPickupDeliveryBlocks.dateSettings || {};
            let valid = true;

            const deliveryCfg = dateSettings.delivery || {};
            const pickupCfg = dateSettings.pickup || {};

            const deliveryDateField = document.getElementById('aicoso_pickup_delivery_delivery_date');
            const pickupDateField = document.getElementById('aicoso_pickup_delivery_pickup_date');

            // Helper: check a single field
            const checkField = function(field, cfg) {
                if (!field || !cfg) return true;
                const val = field.value;
                if (!val) return true;

                // Reset previous validity
                field.setCustomValidity('');

                // Min date check
                if (cfg.minDate && val < cfg.minDate) {
                    var msg = (wcPickupDeliveryBlocks.strings && wcPickupDeliveryBlocks.strings.dateBeforeMin) ? wcPickupDeliveryBlocks.strings.dateBeforeMin : 'Selected date is earlier than allowed.';
                    field.setCustomValidity(msg);
                    field.reportValidity();
                    return false;
                }

                // Blackout dates
                if (Array.isArray(cfg.blackoutDates) && cfg.blackoutDates.indexOf(val) !== -1) {
                    var msg = (wcPickupDeliveryBlocks.strings && wcPickupDeliveryBlocks.strings.dateBlackout) ? wcPickupDeliveryBlocks.strings.dateBlackout : 'Selected date is not available.';
                    field.setCustomValidity(msg);
                    field.reportValidity();
                    return false;
                }

                // Weekday availability: PHP uses 1 = Monday ... 7 = Sunday. Map to JS getDay (0 = Sunday ... 6 = Saturday).
                if (Array.isArray(cfg.availableDays) && cfg.availableDays.length) {
                    const jsDay = new Date(val + 'T00:00:00').getDay();
                    // Map PHP day to JS day: php 7 -> js 0, otherwise php n -> js n
                    const allowedJsDays = cfg.availableDays.map(function(d) { return (parseInt(d, 10) === 7) ? 0 : parseInt(d, 10); });
                    if (allowedJsDays.indexOf(jsDay) === -1) {
                        var msg = (wcPickupDeliveryBlocks.strings && wcPickupDeliveryBlocks.strings.dateWeekdayUnavailable) ? wcPickupDeliveryBlocks.strings.dateWeekdayUnavailable : 'Selected day of week is not available.';
                        field.setCustomValidity(msg);
                        field.reportValidity();
                        return false;
                    }
                }

                return true;
            };

            if (deliveryDateField && !checkField(deliveryDateField, deliveryCfg)) {
                valid = false;
            }

            if (pickupDateField && !checkField(pickupDateField, pickupCfg)) {
                valid = false;
            }

            return valid;
        },

        /**
         * Handle order type change
         */
        handleOrderTypeChange: function(event) {
            const orderType = event.target.value;
            
            if (this.elements.deliveryFields && this.elements.pickupFields) {
                if (orderType === 'delivery') {
                    this.elements.deliveryFields.style.display = 'block';
                    this.elements.pickupFields.style.display = 'none';
                } else {
                    this.elements.deliveryFields.style.display = 'none';
                    this.elements.pickupFields.style.display = 'block';
                }
            }

            // Trigger custom event
            this.triggerEvent('orderTypeChanged', { orderType });
        },

        /**
         * Setup date field validation
         */
        setupDateValidation: function() {
            // Use localized dateSettings when available.
            const dateSettings = wcPickupDeliveryBlocks.dateSettings || {};

            const deliveryCfg = dateSettings.delivery || {};
            const pickupCfg = dateSettings.pickup || {};

            const deliveryDateField = document.getElementById('aicoso_pickup_delivery_delivery_date');
            const pickupDateField = document.getElementById('aicoso_pickup_delivery_pickup_date');

            if (deliveryDateField && deliveryCfg.minDate) {
                deliveryDateField.setAttribute('min', deliveryCfg.minDate);
            }

            if (pickupDateField && pickupCfg.minDate) {
                pickupDateField.setAttribute('min', pickupCfg.minDate);
            }

            // If jQuery UI datepicker is available, initialize it so we can disable dates/weekdays.
            if (window.jQuery && jQuery.fn && jQuery.fn.datepicker) {
                try {
                    if (deliveryDateField) {
                        jQuery(deliveryDateField).datepicker('destroy');
                        jQuery(deliveryDateField).datepicker({
                            dateFormat: 'yy-mm-dd',
                            minDate: deliveryCfg.minDate || 0,
                            beforeShowDay: function(date) {
                                var dateString = jQuery.datepicker.formatDate('yy-mm-dd', date);
                                var day = date.getDay(); // 0 is Sunday
                                var isoDay = day === 0 ? 7 : day; // 1 = Monday ... 7 = Sunday

                                if (Array.isArray(deliveryCfg.blackoutDates) && deliveryCfg.blackoutDates.indexOf(dateString) > -1) {
                                    return [false, 'date-blackout', 'Not available'];
                                }

                                if (Array.isArray(deliveryCfg.availableDays) && deliveryCfg.availableDays.indexOf(isoDay) === -1) {
                                    return [false, '', 'Not available'];
                                }

                                return [true, 'date-available', 'Available'];
                            }
                        });
                    }

                    if (pickupDateField) {
                        jQuery(pickupDateField).datepicker('destroy');
                        jQuery(pickupDateField).datepicker({
                            dateFormat: 'yy-mm-dd',
                            minDate: pickupCfg.minDate || 0,
                            beforeShowDay: function(date) {
                                var dateString = jQuery.datepicker.formatDate('yy-mm-dd', date);
                                var day = date.getDay();
                                var isoDay = day === 0 ? 7 : day;

                                if (Array.isArray(pickupCfg.blackoutDates) && pickupCfg.blackoutDates.indexOf(dateString) > -1) {
                                    return [false, 'date-blackout', 'Not available'];
                                }

                                if (Array.isArray(pickupCfg.availableDays) && pickupCfg.availableDays.indexOf(isoDay) === -1) {
                                    return [false, '', 'Not available'];
                                }

                                return [true, 'date-available', 'Available'];
                            }
                        });
                    }
                } catch (e) {
                    // If datepicker initialization fails, fall back to native validation.
                    console.warn('WC Pickup/Delivery: datepicker init failed, falling back to native inputs', e);
                }
            }

            // Add input listeners to validate on change and fetch time slots.
            if (deliveryDateField) {
                deliveryDateField.addEventListener('change', () => {
                    this.validateDateFields();
                    this.fetchTimeSlotsFor('delivery', deliveryDateField.value);
                });
            }
            if (pickupDateField) {
                pickupDateField.addEventListener('change', () => {
                    this.validateDateFields();
                    this.fetchTimeSlotsFor('pickup', pickupDateField.value);
                });
            }

            // If dates already filled (e.g. browser autofill), fetch time slots on init.
            if (deliveryDateField && deliveryDateField.value) {
                this.fetchTimeSlotsFor('delivery', deliveryDateField.value);
            }
            if (pickupDateField && pickupDateField.value) {
                this.fetchTimeSlotsFor('pickup', pickupDateField.value);
            }
        },

        /**
         * Helper to get ajax URL and a valid nonce from any localized object.
         * Tries multiple global objects to maximize compatibility with different localizations.
         */
        getAjaxConfig: function() {
            // Common places where ajaxUrl and nonce may be localized
            const sources = [
                window.wcPickupDeliveryBlocks || {},
                window.wcPickupDeliveryAjax || {},
                window.wcPickupDeliveryData || {},
                window.wcPickupDeliverySession || {},
                window.aicoso_pickup_delivery_params || {}
            ];

            let ajaxUrl = '';
            let nonce = '';

            sources.forEach(src => {
                if (!ajaxUrl) {
                    ajaxUrl = src.ajaxUrl || src.ajax_url || '';
                }
                if (!nonce) {
                    nonce = src.nonce || '';
                }
            });

            // Fallback to admin-ajax.php if nothing localized (unlikely)
            if (!ajaxUrl && typeof window.ajaxurl !== 'undefined') {
                ajaxUrl = window.ajaxurl;
            }

            return { ajaxUrl: ajaxUrl, nonce: nonce };
        },

        /**
         * Fetch time slots for the given type and date from server (uses admin-ajax).
         * Populates the corresponding select element.
         */
        fetchTimeSlotsFor: function( type, date ) {
            if ( ! date ) {
                this.populateTimeSlots(type, []);
                return;
            }

            const cfg = this.getAjaxConfig();
            const ajaxUrl = cfg.ajaxUrl || (window.location.origin + '/wp-admin/admin-ajax.php');
            const nonce = cfg.nonce || '';

            const form = new FormData();
            form.append('action', 'aicoso_pickup_delivery_get_time_slots');
            form.append('type', type);
            form.append('date', date);
            if (nonce) {
                form.append('nonce', nonce);
            }

            fetch(ajaxUrl, {
                method: 'POST',
                credentials: 'same-origin',
                body: form
            }).then(response => response.json()).then(data => {
                if (!data) {
                    this.populateTimeSlots(type, []);
                    return;
                }

                if (data.success && data.data && Array.isArray(data.data.time_slots)) {
                    this.populateTimeSlots(type, data.data.time_slots);
                } else if (data.success && data.data && Array.isArray(data.data) ) {
                    // Some endpoints may return the array directly
                    this.populateTimeSlots(type, data.data);
                } else {
                    this.populateTimeSlots(type, []);
                }
            }).catch(err => {
                console.warn('WC Pickup/Delivery: failed to fetch time slots', err);
                this.populateTimeSlots(type, []);
            });
        },

        /**
         * Populate the time slot select for delivery or pickup
         */
        populateTimeSlots: function(type, slots) {
            const isDelivery = type === 'delivery';
            const selectId = isDelivery ? 'aicoso_pickup_delivery_delivery_time_slot' : 'aicoso_pickup_delivery_pickup_time_slot';
            const select = document.getElementById(selectId);
            const notAvailableText = (wcPickupDeliveryBlocks.strings && wcPickupDeliveryBlocks.strings.notAvailable) ? wcPickupDeliveryBlocks.strings.notAvailable : 'No times available.';

            if (!select) return;

            // Clear existing options
            while (select.firstChild) select.removeChild(select.firstChild);

            // Add default option
            const defaultOpt = document.createElement('option');
            defaultOpt.value = '';
            defaultOpt.textContent = this.strings.selectTime || 'Select a time';
            select.appendChild(defaultOpt);

            if (!Array.isArray(slots) || slots.length === 0) {
                const opt = document.createElement('option');
                opt.value = '';
                opt.textContent = notAvailableText;
                select.appendChild(opt);
                select.disabled = true;
                return;
            }

            // Populate with returned slots. Expected format from server:
            // [ { name: 'morning', formatted_time: 'Morning (9AM - 12PM)', remaining_capacity: 5 }, ... ]
            slots.forEach(slot => {
                const opt = document.createElement('option');
                // Support different key names gracefully
                opt.value = slot.name || slot.slot_name || slot.value || '';
                opt.textContent = slot.formatted_time || slot.label || slot.name || String(opt.value);
                select.appendChild(opt);
            });

            select.disabled = false;
        },

        /**
         * Setup checkout submission handling
         */
        setupCheckoutSubmission: function() {
            // Find checkout submit button
            const checkoutSubmit = document.querySelector('.wc-block-components-checkout-place-order-button');
            
            if (checkoutSubmit) {
                checkoutSubmit.addEventListener('click', (e) => {
                    // Validate date fields and prevent submission if invalid.
                    if (!this.validateDateFields()) {
                        e.preventDefault();
                        return;
                    }

                    this.handleCheckoutSubmission();
                });
            }

            // Also intercept fetch requests for checkout
            this.interceptCheckoutFetch();
        },

        /**
         * Handle checkout submission
         */
        handleCheckoutSubmission: function() {
            const fieldsData = this.collectFieldData();
            
            // Store in session storage for retrieval
            sessionStorage.setItem('aicoso_pickup_delivery_data', JSON.stringify(fieldsData));
            
            // Trigger custom event
            this.triggerEvent('checkoutSubmission', fieldsData);
        },

        /**
         * Collect field data
         */
        collectFieldData: function() {
            const { defaultOrderType } = this.settings;
            
            return {
                order_type: document.getElementById('aicoso_pickup_delivery_type')?.value || defaultOrderType,
                delivery_date: document.getElementById('aicoso_pickup_delivery_delivery_date')?.value || '',
                delivery_time: document.getElementById('aicoso_pickup_delivery_delivery_time_slot')?.value || '',
                pickup_date: document.getElementById('aicoso_pickup_delivery_pickup_date')?.value || '',
                pickup_time: document.getElementById('aicoso_pickup_delivery_pickup_time_slot')?.value || '',
                pickup_location: document.getElementById('aicoso_pickup_delivery_pickup_location')?.value || ''
            };
        },

        /**
         * Intercept checkout fetch requests to add our data
         */
        interceptCheckoutFetch: function() {
            const originalFetch = window.fetch;
            const self = this;

            window.fetch = function(...args) {
                var resource = args[0];
                var config = args[1] || {};

                // Normalize resource URL for Request objects
                var resourceUrl = '';
                try {
                    if (resource && resource.url) {
                        resourceUrl = resource.url;
                    } else if (typeof resource === 'string') {
                        resourceUrl = resource;
                    }
                } catch (e) {
                    resourceUrl = '';
                }

                // Check if this is a checkout request
                if (resourceUrl && resourceUrl.indexOf('/wc/store/v1/checkout') !== -1) {
                    // Abort if invalid fields are present.
                    if (!self.validateDateFields()) {
                        return Promise.reject(new Error('Invalid pickup/delivery date or selection.'));
                    }

                    // Modify the request to include our fields
                    if (config && config.body) {
                        try {
                            let bodyData = JSON.parse(config.body);
                            const fieldsData = self.collectFieldData();
                            
                            // Add to extensions
                            if (!bodyData.extensions) {
                                bodyData.extensions = {};
                            }
                            bodyData.extensions['aicoso-pickup-delivery'] = {
                                orderType: fieldsData.order_type,
                                deliveryDate: fieldsData.delivery_date,
                                deliveryTimeSlot: fieldsData.delivery_time,
                                pickupDate: fieldsData.pickup_date,
                                pickupTimeSlot: fieldsData.pickup_time,
                                pickupLocation: fieldsData.pickup_location
                            };

                            // Also add to other_fields for compatibility
                            if (!bodyData.other_fields) {
                                bodyData.other_fields = {};
                            }
                            Object.keys(fieldsData).forEach(key => {
                                bodyData.other_fields['aicoso_pickup_delivery_' + key] = fieldsData[key];
                            });

                            config.body = JSON.stringify(bodyData);
                        } catch (e) {
                            console.error('WC Pickup/Delivery: Failed to inject data into checkout request', e);
                        }
                    }

                    // Call original fetch and inspect response for REST errors so we can show a friendly notice.
                    return originalFetch.apply(this, args).then(function(response) {
                        if (response && !response.ok) {
                            // Try to parse JSON error response and show message
                            try {
                                return response.clone().json().then(function(json) {
                                    var message = '';

                                    if (json) {
                                        if (json.message) {
                                            message = json.message;
                                        } else if (json.data && json.data.message) {
                                            message = json.data.message;
                                        } else if (json.errors && Array.isArray(json.errors) && json.errors.length) {
                                            // WP_Error style
                                            message = json.errors[0].message || JSON.stringify(json.errors[0]);
                                        } else {
                                            message = JSON.stringify(json);
                                        }
                                    }

                                    if (!message) {
                                        message = 'The selected delivery date is not available. Please choose another date.';
                                    }

                                    self.showCheckoutNotice(message, 'error');
                                    self.removeGenericBlocksNotices();

                                    return Promise.reject(new Error(message));
                                }).catch(function() {
                                    // Parsing failed - show canonical delivery-unavailable message
                                    var fallbackMessage = 'The selected delivery date is not available. Please choose another date.';
                                    self.showCheckoutNotice(fallbackMessage, 'error');
                                    // Remove generic Blocks error notices if present
                                    self.removeGenericBlocksNotices();
                                    return Promise.reject(new Error(fallbackMessage));
                                });
                            } catch (e) {
                                var fallbackMessage = 'The selected delivery date is not available. Please choose another date.';
                                self.showCheckoutNotice(fallbackMessage, 'error');
                                self.removeGenericBlocksNotices();
                                return Promise.reject(new Error(fallbackMessage));
                            }
                        }

                        return response;
                    }).catch(function(err) {
                        // If fetch was rejected earlier (e.g. validation), rethrow so calling code can handle as well.
                        return Promise.reject(err);
                    });
                }

                return originalFetch.apply(this, args);
            };
        },

        /**
         * Show a transient checkout notice inside the blocks checkout near our fields.
         * This is intentionally minimal to avoid depending on WC internals.
         */
        showCheckoutNotice: function(message, type) {
            try {
                var container = document.getElementById('aicoso-pickup-delivery-fields');
                if (!container) {
                    // Fallback to top of checkout form
                    container = document.querySelector('.wc-block-checkout__form') || document.body;
                }

                // Remove existing notice if present
                var existing = container.querySelector('.aicoso-blocks-notice');
                if (existing) {
                    existing.parentNode.removeChild(existing);
                }

                var notice = document.createElement('div');
                notice.className = 'aicoso-blocks-notice aicoso-blocks-notice--' + (type || 'error');
                notice.setAttribute('role', 'alert');
                notice.style.margin = '0 0 12px 0';
                notice.style.padding = '12px';
                notice.style.border = '1px solid rgba(0,0,0,0.1)';
                notice.style.background = '#fff6f6';
                notice.style.color = '#800';
                notice.textContent = message;

                container.insertBefore(notice, container.firstChild);

                // Auto-remove after 8 seconds
                setTimeout(function() {
                    if (notice && notice.parentNode) {
                        notice.parentNode.removeChild(notice);
                    }
                }, 8000);
            } catch (e) {
                // Ignore UI failures
                console.warn('WC Pickup/Delivery: showCheckoutNotice failed', e);
            }
        },

        /**
         * Remove known generic Blocks/WC error notices from the checkout UI.
         * This attempts to reduce duplicate or confusing messages when we
         * already surface a specific availability notice.
         */
        removeGenericBlocksNotices: function() {
            try {
                // Common selector for Blocks notices
                var selectors = [
                    '.wc-block-components-notice',
                    '.wc-block-components-checkout__notice',
                    '.woocommerce-error',
                    '.woocommerce-notice'
                ];

                selectors.forEach(function(sel) {
                    var nodes = document.querySelectorAll(sel);
                    nodes.forEach(function(n) {
                        var txt = (n.textContent || '').trim();
                        if (!txt) return;
                        // Remove generic messages we want to suppress (broad matching)
                        var lower = txt.toLowerCase();
                        if (lower.indexOf('something went wrong') !== -1 ||
                            lower.indexOf('check your account') !== -1 ||
                            lower.indexOf('an error occurred while processing pickup/delivery') !== -1 ||
                            lower.indexOf('please try again') !== -1 ||
                            lower.indexOf('please try again or contact support') !== -1) {
                            if (n.parentNode) {
                                n.parentNode.removeChild(n);
                            }
                        }
                    });
                });
            } catch (e) {
                // Non-fatal
            }
        },

        /**
         * Observe the checkout form for generic Blocks notices and remove them as they appear.
         * This helps suppress the default "Something went wrong" messages that may be
         * inserted by the Blocks checkout after our fetch interception.
         */
        observeAndRemoveGenericNotices: function() {
            try {
                var form = document.querySelector('.wc-block-checkout__form');
                if (!form || !window.MutationObserver) return;

                var observer = new MutationObserver(function(mutations) {
                    mutations.forEach(function(m) {
                        if (m.addedNodes && m.addedNodes.length) {
                            m.addedNodes.forEach(function(node) {
                                if (!node || !node.textContent) return;
                                var txt = (node.textContent || '').trim().toLowerCase();
                                if (!txt) return;
                                if (txt.indexOf('something went wrong') !== -1 || txt.indexOf('check your account') !== -1) {
                                    try { if (node.parentNode) node.parentNode.removeChild(node); } catch (e) {}
                                }
                            });
                        }
                    });
                });

                observer.observe(form, { childList: true, subtree: true });
                // Keep reference to disconnect later if needed
                this._noticeObserver = observer;
            } catch (e) {
                // ignore
            }
        },

        /**
         * Trigger custom event
         */
        triggerEvent: function(eventName, detail = {}) {
            const event = new CustomEvent('wcPickupDelivery:' + eventName, {
                detail: detail,
                bubbles: true,
                cancelable: true
            });
            document.dispatchEvent(event);
        }
    };

    // Initialize when ready
    WCPickupDeliveryCheckoutFields.init();

    // Export for external use
    window.WCPickupDeliveryCheckoutFields = WCPickupDeliveryCheckoutFields;

})(window, document);