/**
 * Frontend JavaScript for WooCommerce Pickup & Delivery
 *
 * @package    aicoso_pickup_delivery
 * @version    1.0.0
 */

jQuery(document).ready(function($) {
    'use strict';
    
    // Variables
    var params = typeof aicoso_pickup_delivery_params !== 'undefined' ? aicoso_pickup_delivery_params : {};
    
    // Initialize delivery date picker
    function initDeliveryDatepicker() {
        var deliveryDatepicker = $('#aicoso-pickup-delivery-delivery-date');
        
        if (deliveryDatepicker.length) {
            // Prepare datepicker options
            var datepickerDefaults = {
                dateFormat: 'yy-mm-dd',
                minDate: 0,
                beforeShowDay: function(date) {
                    // Default to not selectable
                    return [false, '', 'Not available'];
                },
                onSelect: function(dateText) {
                    // When date is selected, fetch available time slots
                    if (dateText) {
                        fetchDeliveryTimeSlots(dateText);
                    }
                }
            };
            
            // Initialize datepicker with default options
            deliveryDatepicker.datepicker(datepickerDefaults);
            
            // Fetch available dates for delivery and update datepicker
            fetchDeliveryDates();
        }
    }
    
    // Initialize pickup date picker
    function initPickupDatepicker() {
        var pickupDatepicker = $('#aicoso-pickup-delivery-pickup-date');
        
        if (pickupDatepicker.length) {
            // Prepare datepicker options
            var datepickerDefaults = {
                dateFormat: 'yy-mm-dd',
                minDate: 0,
                beforeShowDay: function(date) {
                    // Default to not selectable
                    return [false, '', 'Not available'];
                },
                onSelect: function(dateText) {
                    // When date is selected, fetch available locations and time slots
                    if (dateText) {
                        fetchPickupLocations(dateText);
                        fetchPickupTimeSlots(dateText);
                    }
                }
            };
            
            // Initialize datepicker with default options
            pickupDatepicker.datepicker(datepickerDefaults);
            
            // Fetch available dates for pickup and update datepicker
            fetchPickupDates();
        }
    }
    
    // Initialize order type selector
    function initOrderTypeSelector() {
        // Handle toggle between delivery and pickup options
        $('input[name="aicoso_pickup_delivery_type"]').on('change', function() {
            var selectedType = $(this).val();
            
            if (selectedType === 'delivery') {
                $('.aicoso-pickup-delivery-delivery-options').show();
                $('.aicoso-pickup-delivery-pickup-options').hide();
                
                // Hide pickup fields in the checkout form
                $('.pickup-field').closest('.form-row').hide();
                // Show delivery fields in the checkout form
                $('.delivery-field').closest('.form-row').show();
                
                // Update hidden field in checkout form
                $('#aicoso_pickup_delivery_type').val('delivery');
            } else {
                $('.aicoso-pickup-delivery-delivery-options').hide();
                $('.aicoso-pickup-delivery-pickup-options').show();
                
                // Hide delivery fields in the checkout form
                $('.delivery-field').closest('.form-row').hide();
                // Show pickup fields in the checkout form
                $('.pickup-field').closest('.form-row').show();
                
                // Update hidden field in checkout form
                $('#aicoso_pickup_delivery_type').val('pickup');
            }
            
            // Save selection to session via AJAX
            saveOrderTypeToSession(selectedType);
        });
        
        // Trigger change event on page load to ensure fields are properly shown/hidden
        $('input[name="aicoso_pickup_delivery_type"]:checked').trigger('change');
    }
    
    // Save order type to session via AJAX
    function saveOrderTypeToSession(type) {
        if (params.ajax_url && params.nonce) {
            $.ajax({
                type: 'POST',
                url: params.ajax_url,
                data: {
                    action: 'aicoso_pickup_delivery_save_type',
                    type: type,
                    nonce: params.nonce
                }
            });
        }
    }
    
    // Fetch delivery dates via AJAX
    function fetchDeliveryDates() {
        if (params.ajax_url && params.nonce) {
            $.ajax({
                type: 'POST',
                url: params.ajax_url,
                data: {
                    action: 'aicoso_pickup_delivery_get_dates',
                    type: 'delivery',
                    nonce: params.nonce
                },
                success: function(response) {
                    if (response.success && response.data.datepicker_data) {
                        updateDeliveryDatepicker(response.data.datepicker_data);
                    }
                }
            });
        }
    }
    
    // Fetch pickup dates via AJAX
    function fetchPickupDates() {
        if (params.ajax_url && params.nonce) {
            $.ajax({
                type: 'POST',
                url: params.ajax_url,
                data: {
                    action: 'aicoso_pickup_delivery_get_dates',
                    type: 'pickup',
                    nonce: params.nonce
                },
                success: function(response) {
                    if (response.success && response.data.datepicker_data) {
                        updatePickupDatepicker(response.data.datepicker_data);
                    }
                }
            });
        }
    }
    
    // Update delivery datepicker with available dates
    function updateDeliveryDatepicker(data) {
        var deliveryDatepicker = $('#aicoso-pickup-delivery-delivery-date');
        
        if (deliveryDatepicker.length) {
            // Update beforeShowDay function to filter available dates
            deliveryDatepicker.datepicker('option', 'beforeShowDay', function(date) {
                var dateString = $.datepicker.formatDate('yy-mm-dd', date);
                var day = date.getDay(); // 0 is Sunday, 1 is Monday, etc.
                
                // Convert from JavaScript (0 = Sunday) to ISO-8601 (1 = Monday)
                var isoDay = day === 0 ? 7 : day;
                
                // Check if date is in blackout dates
                if ($.inArray(dateString, data.blackout_dates) > -1) {
                    return [false, 'date-blackout', 'Not available'];
                }
                
                // Check if day is in available days
                if ($.inArray(isoDay, data.available_days) === -1) {
                    return [false, '', 'Not available'];
                }
                
                // Check if date is in available dates
                if ($.inArray(dateString, data.available_dates) > -1) {
                    return [true, 'date-available', 'Available'];
                }
                
                return [false, '', 'Not available'];
            });
            
            // Set minimum date
            if (data.min_date) {
                deliveryDatepicker.datepicker('option', 'minDate', data.min_date);
            }
        }
    }
    
    // Update pickup datepicker with available dates
    function updatePickupDatepicker(data) {
        var pickupDatepicker = $('#aicoso-pickup-delivery-pickup-date');
        
        if (pickupDatepicker.length) {
            // Update beforeShowDay function to filter available dates
            pickupDatepicker.datepicker('option', 'beforeShowDay', function(date) {
                var dateString = $.datepicker.formatDate('yy-mm-dd', date);
                var day = date.getDay(); // 0 is Sunday, 1 is Monday, etc.
                
                // Convert from JavaScript (0 = Sunday) to ISO-8601 (1 = Monday)
                var isoDay = day === 0 ? 7 : day;
                
                // Check if date is in blackout dates
                if ($.inArray(dateString, data.blackout_dates) > -1) {
                    return [false, 'date-blackout', 'Not available'];
                }
                
                // Check if day is in available days
                if ($.inArray(isoDay, data.available_days) === -1) {
                    return [false, '', 'Not available'];
                }
                
                // Check if date is in available dates
                if ($.inArray(dateString, data.available_dates) > -1) {
                    return [true, 'date-available', 'Available'];
                }
                
                return [false, '', 'Not available'];
            });
            
            // Set minimum date
            if (data.min_date) {
                pickupDatepicker.datepicker('option', 'minDate', data.min_date);
            }
        }
    }
    
    // Fetch delivery time slots via AJAX
    function fetchDeliveryTimeSlots(date) {
        var timeSlotSelect = $('#aicoso-pickup-delivery-delivery-time-slot');
        
        // Only proceed if the element exists
        if (timeSlotSelect.length === 0) {
            return;
        }
        
        // Reset and disable time slot select while loading
        timeSlotSelect.empty().append($('<option></option>').val('').text(params.i18n.loading_time_slots)).prop('disabled', true);
        
        if (params.ajax_url && params.nonce) {
            $.ajax({
                type: 'POST',
                url: params.ajax_url,
                data: {
                    action: 'aicoso_pickup_delivery_get_time_slots',
                    type: 'delivery',
                    date: date,
                    nonce: params.nonce
                },
                success: function(response) {
                    timeSlotSelect.empty();
                    
                    if (response.success && response.data.time_slots && response.data.time_slots.length > 0) {
                        // Add default option
                        timeSlotSelect.append($('<option></option>').val('').text(params.i18n.select_time_slot));
                        
                        // Add time slots
                        $.each(response.data.time_slots, function(index, slot) {
                            timeSlotSelect.append(
                                $('<option></option>').val(slot.name).text(slot.formatted_time)
                            );
                        });
                        
                        // Enable select
                        timeSlotSelect.prop('disabled', false);
                    } else {
                        // No time slots available
                        timeSlotSelect.append($('<option></option>').val('').text(params.i18n.no_time_slots));
                    }
                },
                error: function() {
                    timeSlotSelect.empty().append($('<option></option>').val('').text('Error loading time slots'));
                }
            });
        }
    }
    
    // Fetch pickup time slots via AJAX
    function fetchPickupTimeSlots(date) {
        var timeSlotSelect = $('#aicoso-pickup-delivery-pickup-time-slot');
        
        // Only proceed if the element exists
        if (timeSlotSelect.length === 0) {
            return;
        }
        
        // Reset and disable time slot select while loading
        timeSlotSelect.empty().append($('<option></option>').val('').text(params.i18n.loading_time_slots)).prop('disabled', true);
        
        if (params.ajax_url && params.nonce) {
            $.ajax({
                type: 'POST',
                url: params.ajax_url,
                data: {
                    action: 'aicoso_pickup_delivery_get_time_slots',
                    type: 'pickup',
                    date: date,
                    nonce: params.nonce
                },
                success: function(response) {
                    timeSlotSelect.empty();
                    
                    if (response.success && response.data.time_slots && response.data.time_slots.length > 0) {
                        // Add default option
                        timeSlotSelect.append($('<option></option>').val('').text(params.i18n.select_time_slot));
                        
                        // Add time slots
                        $.each(response.data.time_slots, function(index, slot) {
                            timeSlotSelect.append(
                                $('<option></option>').val(slot.name).text(slot.formatted_time)
                            );
                        });
                        
                        // Enable select
                        timeSlotSelect.prop('disabled', false);
                    } else {
                        // No time slots available
                        timeSlotSelect.append($('<option></option>').val('').text(params.i18n.no_time_slots));
                    }
                },
                error: function() {
                    timeSlotSelect.empty().append($('<option></option>').val('').text('Error loading time slots'));
                }
            });
        }
    }
    
    // Fetch pickup locations via AJAX
    function fetchPickupLocations(date) {
        var locationSelect = $('#aicoso-pickup-delivery-pickup-location');
        
        // Reset and disable location select while loading
        locationSelect.empty().append($('<option></option>').val('').text(params.i18n.loading_locations)).prop('disabled', true);
        
        // Hide location details
        $('.aicoso-pickup-delivery-location-details').hide();
        
        if (params.ajax_url && params.nonce) {
            $.ajax({
                type: 'POST',
                url: params.ajax_url,
                data: {
                    action: 'aicoso_pickup_delivery_get_locations',
                    date: date,
                    nonce: params.nonce
                },
                success: function(response) {
                    locationSelect.empty();
                    
                    if (response.success && response.data.locations && response.data.locations.length > 0) {
                        // Add default option
                        locationSelect.append($('<option></option>').val('').text(params.i18n.select_location));
                        
                        // Store location data for reference
                        locationSelect.data('locations', {});
                        
                        // Add locations
                        $.each(response.data.locations, function(index, location) {
                            locationSelect.append(
                                $('<option></option>').val(location.name).text(location.name)
                            );
                            
                            // Store location data
                            var locations = locationSelect.data('locations');
                            locations[location.name] = location;
                            locationSelect.data('locations', locations);
                        });
                        
                        // Enable select
                        locationSelect.prop('disabled', false);
                    } else {
                        // No locations available
                        locationSelect.append($('<option></option>').val('').text(params.i18n.no_locations));
                    }
                },
                error: function() {
                    locationSelect.empty().append($('<option></option>').val('').text('Error loading locations'));
                }
            });
        }
    }
    
    // Show location details when a location is selected
    function initLocationDetails() {
        $('#aicoso-pickup-delivery-pickup-location').on('change', function() {
            var locationName = $(this).val();
            
            if (locationName) {
                var locations = $(this).data('locations');
                var location = locations[locationName];
                
                if (location) {
                    // Update location details
                    var detailsContainer = $('.aicoso-pickup-delivery-location-details');
                    
                    if (location.address) {
                        detailsContainer.find('.pickup-address').html('<strong>Address:</strong> ' + location.address).show();
                    } else {
                        detailsContainer.find('.pickup-address').hide();
                    }
                    
                    if (location.map_url) {
                        detailsContainer.find('.pickup-map-link').html('<a href="' + location.map_url + '" target="_blank">View on Google Maps</a>').show();
                    } else {
                        detailsContainer.find('.pickup-map-link').hide();
                    }
                    
                    detailsContainer.show();
                }
            } else {
                $('.aicoso-pickup-delivery-location-details').hide();
            }
        });
    }
    
    // Initialize all functions
    function init() {
        initOrderTypeSelector();
        initDeliveryDatepicker();
        initPickupDatepicker();
        initLocationDetails();
    }
    
    // Run initialization
    init();
});