/**
 * Frontend JavaScript for WooCommerce Pickup & Delivery
 * 
 * This file contains all JavaScript functionality for the checkout templates.
 * Moved from inline scripts to comply with WordPress standards.
 * 
 * @since 1.0.0
 */

(function($){


/* ================================
   DELIVERY-PICKUP SELECTOR FUNCTIONALITY
   (Moved from delivery-pickup-selector.php inline script)
   ================================ */

jQuery(document).ready(function($) {
    // Function to initialize the selector.
    function initializePickupDeliverySelector() {
        // Prevent duplicate initialization.
        if ($('#aicoso-pickup-delivery-main-selector').data('initialized')) {
            return;
        }
        
        // Remove any duplicate selectors (keep only the first one).
        var $selectors = $('.aicoso-pickup-delivery-selector-wrapper');
        if ($selectors.length > 1) {
            $selectors.not('#aicoso-pickup-delivery-main-selector').remove();
        }
        
        // Hide any blocks-generated duplicate fields.
        $('.wc-block-components-checkout-step__content .wc-block-components-combobox').each(function() {
            var $label = $(this).find('label');
            if ($label.text().indexOf('Order Type') !== -1 || 
                $label.text().indexOf('Delivery Date') !== -1 || 
                $label.text().indexOf('Pickup Date') !== -1) {
                $(this).closest('.wc-block-components-checkout-step__content').hide();
            }
        });
        
        // Mark as initialized.
        $('#aicoso-pickup-delivery-main-selector').data('initialized', true);
    }
    
    // Initialize immediately.
    initializePickupDeliverySelector();
    
    // Re-initialize on WooCommerce blocks updates (for React checkout).
    $(document.body).on('updated_checkout', function() {
        initializePickupDeliverySelector();
    });
    
    // Handle toggle between delivery and pickup options.
    $('#aicoso-pickup-delivery-type-select, select[name="aicoso_pickup_delivery_type_selector"]').on('change', function() {
        var selectedType = $(this).val();
        
        if (selectedType === 'delivery') {
            $('.aicoso-pickup-delivery-delivery-options').fadeIn(200);
            $('.aicoso-pickup-delivery-pickup-options').hide();
            
            // Hide pickup fields in the checkout form.
            $('.pickup-field').closest('.form-row').hide();
            // Show delivery fields in the checkout form.
            $('.delivery-field').closest('.form-row').show();
            
            // Update hidden field in checkout form.
            $('input[name="aicoso_pickup_delivery_type"]').val('delivery');
        } else {
            $('.aicoso-pickup-delivery-delivery-options').hide();
            $('.aicoso-pickup-delivery-pickup-options').fadeIn(200);
            
            // Hide delivery fields in the checkout form.
            $('.delivery-field').closest('.form-row').hide();
            // Show pickup fields in the checkout form.
            $('.pickup-field').closest('.form-row').show();
            
            // Update hidden field in checkout form.
            $('input[name="aicoso_pickup_delivery_type"]').val('pickup');
        }
        
        // Save selection to session via AJAX.
        if (typeof aicoso_pickup_delivery_params !== 'undefined') {
            $.ajax({
                type: 'POST',
                url: aicoso_pickup_delivery_params.ajax_url,
                data: {
                    action: 'aicoso_pickup_delivery_save_type',
                    type: selectedType,
                    nonce: aicoso_pickup_delivery_params.nonce
                }
            });
        }
    });
    
    // Trigger change event on page load to ensure fields are properly shown/hidden.
    $('#aicoso-pickup-delivery-type-select').trigger('change');
});

/* ================================
   CALENDAR ICON FUNCTIONALITY
   (Moved from pickup-options.php and delivery-options.php)
   ================================ */

jQuery(document).ready(function($) {
    // Make calendar icon clickable - applies to both pickup and delivery options
    function initializeCalendarIcons() {
        $('.aicoso-pickup-delivery-calendar-icon').css('cursor', 'pointer').off('click.pickupdelivery').on('click.pickupdelivery', function() {
            $(this).siblings('input').focus().trigger('click');
        });
    }

    // Initialize calendar icons on page load
    initializeCalendarIcons();

    // Re-initialize after checkout updates
    $(document.body).on('updated_checkout checkout_error', function() {
        setTimeout(initializeCalendarIcons, 100);
    });
});

/* ================================
   PICKUP DATE PICKER FUNCTIONALITY
   (Moved from pickup-options.php inline script)
   ================================ */

jQuery(document).ready(function($) {
    // Initialize pickup date picker functionality
    function initializePickupDatePicker() {
        var pickupDatepicker = $('#aicoso-pickup-delivery-pickup-date');
        
        if (pickupDatepicker.length && !pickupDatepicker.hasClass('hasDatepicker')) {
            // Prepare datepicker options
            var datepickerDefaults = {
                dateFormat: 'yy-mm-dd',
                minDate: 0,
                beforeShowDay: function(date) {
                    // Default to not selectable
                    var notAvailableText = (typeof aicoso_pickup_delivery_params !== 'undefined' && aicoso_pickup_delivery_params.i18n && aicoso_pickup_delivery_params.i18n.not_available) 
                        ? aicoso_pickup_delivery_params.i18n.not_available 
                        : 'Not available';
                    return [false, '', notAvailableText];
                },
                onSelect: function(dateText) {
                    // When date is selected, fetch available locations and time slots
                    if (dateText) {
                        fetchPickupLocations(dateText);
                        fetchPickupTimeSlots(dateText);
                    }
                }
            };
            
            // Initialize datepicker with default options
            pickupDatepicker.datepicker(datepickerDefaults);
            
            // Initialize by fetching available dates
            fetchPickupDates(pickupDatepicker);
        }
    }

    // Fetch available dates for pickup and update datepicker
    function fetchPickupDates(pickupDatepicker) {
        if (typeof aicoso_pickup_delivery_params === 'undefined') {
            return;
        }

        $.ajax({
            type: 'POST',
            url: aicoso_pickup_delivery_params.ajax_url,
            data: {
                action: 'aicoso_pickup_delivery_get_dates',
                type: 'pickup',
                nonce: aicoso_pickup_delivery_params.nonce
            },
            success: function(response) {
                if (response.success && response.data.datepicker_data) {
                    // Update datepicker with available dates
                    var data = response.data.datepicker_data;
                    
                    // Update beforeShowDay function to filter available dates
                    pickupDatepicker.datepicker('option', 'beforeShowDay', function(date) {
                        var dateString = $.datepicker.formatDate('yy-mm-dd', date);
                        var day = date.getDay(); // 0 is Sunday, 1 is Monday, etc.
                        
                        // Get localized strings
                        var notAvailableText = (typeof aicoso_pickup_delivery_params !== 'undefined' && aicoso_pickup_delivery_params.i18n && aicoso_pickup_delivery_params.i18n.not_available) 
                            ? aicoso_pickup_delivery_params.i18n.not_available 
                            : 'Not available';
                        var availableText = (typeof aicoso_pickup_delivery_params !== 'undefined' && aicoso_pickup_delivery_params.i18n && aicoso_pickup_delivery_params.i18n.available) 
                            ? aicoso_pickup_delivery_params.i18n.available 
                            : 'Available';
                        
                        // Check if date is in blackout dates
                        if ($.inArray(dateString, data.blackout_dates) > -1) {
                            return [false, 'date-blackout', notAvailableText];
                        }
                        
                        // Check if day is in available days
                        if ($.inArray(day, data.available_days) === -1) {
                            return [false, '', notAvailableText];
                        }
                        
                        // Check if date is in available dates
                        if ($.inArray(dateString, data.available_dates) > -1) {
                            return [true, 'date-available', availableText];
                        }
                        
                        return [false, '', notAvailableText];
                    });
                    
                    // Set minimum date
                    if (data.min_date) {
                        pickupDatepicker.datepicker('option', 'minDate', data.min_date);
                    }
                }
            }
        });
    }

    // Fetch locations when a date is selected
    function fetchPickupLocations(date) {
        if (typeof aicoso_pickup_delivery_params === 'undefined') {
            return;
        }

        var locationSelect = $('#aicoso-pickup-delivery-pickup-location');
        
        if (!locationSelect.length) {
            return;
        }

        // Get localized strings
        var loadingText = (aicoso_pickup_delivery_params.i18n && aicoso_pickup_delivery_params.i18n.loading_locations) 
            ? aicoso_pickup_delivery_params.i18n.loading_locations 
            : 'Loading locations...';
        var selectLocationText = (aicoso_pickup_delivery_params.i18n && aicoso_pickup_delivery_params.i18n.select_location) 
            ? aicoso_pickup_delivery_params.i18n.select_location 
            : 'Select a pickup location';
        var noLocationsText = (aicoso_pickup_delivery_params.i18n && aicoso_pickup_delivery_params.i18n.no_locations) 
            ? aicoso_pickup_delivery_params.i18n.no_locations 
            : 'No pickup locations available';
        var errorLoadingText = (aicoso_pickup_delivery_params.i18n && aicoso_pickup_delivery_params.i18n.error_loading_locations) 
            ? aicoso_pickup_delivery_params.i18n.error_loading_locations 
            : 'Error loading locations';
        
        // Reset and disable location select while loading
        locationSelect.empty().append($('<option></option>').val('').text(loadingText)).prop('disabled', true);
        
        // Hide location details
        $('.aicoso-pickup-delivery-location-details').hide();
        
        $.ajax({
            type: 'POST',
            url: aicoso_pickup_delivery_params.ajax_url,
            data: {
                action: 'aicoso_pickup_delivery_get_locations',
                date: date,
                nonce: aicoso_pickup_delivery_params.nonce
            },
            success: function(response) {
                locationSelect.empty();
                
                if (response.success && response.data.locations && response.data.locations.length > 0) {
                    // Filter out locations with no remaining capacity (if provided).
                    var availableLocations = $.grep(response.data.locations, function(loc) {
                        return (typeof loc.remaining_capacity === 'undefined') || (parseInt(loc.remaining_capacity, 10) > 0);
                    });

                    if (availableLocations.length > 0) {
                        // Add default option
                        locationSelect.append($('<option></option>').val('').text(selectLocationText));

                        // Store location data for reference
                        var storedLocations = {};

                        // Add locations
                        $.each(availableLocations, function(index, location) {
                            var name = location.name || location.location_name || '';
                            if (!name) {
                                return;
                            }
                            locationSelect.append(
                                $('<option></option>').val(name).text(name)
                            );

                            // Store location data
                            storedLocations[name] = location;
                        });

                        locationSelect.data('locations', storedLocations);

                        // Enable select
                        locationSelect.prop('disabled', false);
                    } else {
                        // No locations available after filtering
                        locationSelect.append($('<option></option>').val('').text(noLocationsText));
                    }
                } else {
                    // No locations available
                    locationSelect.append($('<option></option>').val('').text(noLocationsText));
                }
            },
            error: function() {
                locationSelect.empty().append($('<option></option>').val('').text(errorLoadingText));
            }
        });
    }

    // Fetch time slots when a date is selected
    function fetchPickupTimeSlots(date) {
        if (typeof aicoso_pickup_delivery_params === 'undefined') {
            return;
        }

        var timeSlotSelect = $('#aicoso-pickup-delivery-pickup-time-slot');
        
        // Only proceed if the element exists (time slots are enabled)
        if (timeSlotSelect.length === 0) {
            return;
        }

        // Get localized strings
        var loadingText = (aicoso_pickup_delivery_params.i18n && aicoso_pickup_delivery_params.i18n.loading_time_slots) 
            ? aicoso_pickup_delivery_params.i18n.loading_time_slots 
            : 'Loading time slots...';
        var selectTimeText = (aicoso_pickup_delivery_params.i18n && aicoso_pickup_delivery_params.i18n.select_time_slot) 
            ? aicoso_pickup_delivery_params.i18n.select_time_slot 
            : 'Select a time slot';
        var noTimeSlotsText = (aicoso_pickup_delivery_params.i18n && aicoso_pickup_delivery_params.i18n.no_time_slots) 
            ? aicoso_pickup_delivery_params.i18n.no_time_slots 
            : 'No time slots available';
        var errorLoadingText = (aicoso_pickup_delivery_params.i18n && aicoso_pickup_delivery_params.i18n.error_loading_time_slots) 
            ? aicoso_pickup_delivery_params.i18n.error_loading_time_slots 
            : 'Error loading time slots';
        
        // Reset and disable time slot select while loading
        timeSlotSelect.empty().append($('<option></option>').val('').text(loadingText)).prop('disabled', true);
        
        $.ajax({
            type: 'POST',
            url: aicoso_pickup_delivery_params.ajax_url,
            data: {
                action: 'aicoso_pickup_delivery_get_time_slots',
                type: 'pickup',
                date: date,
                nonce: aicoso_pickup_delivery_params.nonce
            },
            success: function(response) {
                timeSlotSelect.empty();
                
                if (response.success && response.data.time_slots && response.data.time_slots.length > 0) {
                    // Filter out slots with no remaining capacity (if provided).
                    var availableSlots = $.grep(response.data.time_slots, function(slot) {
                        return (typeof slot.remaining_capacity === 'undefined') || (parseInt(slot.remaining_capacity, 10) > 0);
                    });

                    if (availableSlots.length > 0) {
                        // Add default option
                        timeSlotSelect.append($('<option></option>').val('').text(selectTimeText));

                        // Add time slots
                        $.each(availableSlots, function(index, slot) {
                            var value = slot.name || slot.slot_name || slot.value || '';
                            var label = slot.formatted_time || slot.slot_name || value;
                            if (!value) {
                                return;
                            }
                            timeSlotSelect.append(
                                $('<option></option>').val(value).text(label)
                            );
                        });

                        // Enable select
                        timeSlotSelect.prop('disabled', false);
                    } else {
                        // No time slots available after filtering
                        timeSlotSelect.append($('<option></option>').val('').text(noTimeSlotsText));
                    }
                } else {
                    // No time slots available
                    timeSlotSelect.append($('<option></option>').val('').text(noTimeSlotsText));
                }
            },
            error: function() {
                timeSlotSelect.empty().append($('<option></option>').val('').text(errorLoadingText));
            }
        });
    }

    // Show location details when a location is selected
    function initializeLocationDetailsHandler() {
        $('#aicoso-pickup-delivery-pickup-location').off('change.pickupdelivery').on('change.pickupdelivery', function() {
            var locationName = $(this).val();
            
            if (locationName) {
                var locations = $(this).data('locations');
                var location = locations ? locations[locationName] : null;
                
                if (location) {
                    // Update location details
                    var detailsContainer = $('.aicoso-pickup-delivery-location-details');
                    
                    // Get localized strings
                    var addressLabel = (typeof aicoso_pickup_delivery_params !== 'undefined' && aicoso_pickup_delivery_params.i18n && aicoso_pickup_delivery_params.i18n.address_label) 
                        ? aicoso_pickup_delivery_params.i18n.address_label 
                        : 'Address:';
                    var mapLinkText = (typeof aicoso_pickup_delivery_params !== 'undefined' && aicoso_pickup_delivery_params.i18n && aicoso_pickup_delivery_params.i18n.view_on_maps) 
                        ? aicoso_pickup_delivery_params.i18n.view_on_maps 
                        : 'View on Google Maps';
                    
                    if (location.address) {
                        detailsContainer.find('.pickup-address').html('<strong>' + addressLabel + '</strong> ' + location.address).show();
                    } else {
                        detailsContainer.find('.pickup-address').hide();
                    }
                    
                    if (location.map_url) {
                        detailsContainer.find('.pickup-map-link').html('<a href="' + location.map_url + '" target="_blank">' + mapLinkText + '</a>').show();
                    } else {
                        detailsContainer.find('.pickup-map-link').hide();
                    }
                    
                    detailsContainer.show();
                }
            } else {
                $('.aicoso-pickup-delivery-location-details').hide();
            }
        });
    }

    // Initialize pickup datepicker on page load
    initializePickupDatePicker();
    initializeDeliveryDatePicker();
    initializeLocationDetailsHandler();

    // Re-initialize on checkout updates
    $(document.body).on('updated_checkout checkout_error', function() {
        setTimeout(function() {
            initializePickupDatePicker();
            initializeDeliveryDatePicker();
            initializeLocationDetailsHandler();
        }, 100);
    });
});

/* ================================
   DELIVERY DATEPICKER FUNCTIONALITY
   ================================ */

/**
 * Initialize delivery date picker with dynamic data loading
 */
function initializeDeliveryDatePicker() {
    var $deliveryDateField = $('#aicoso-pickup-delivery-delivery-date');
    
    if ($deliveryDateField.length && typeof $.datepicker !== 'undefined') {
        // Get localized strings with fallbacks
        var dateFormat = (typeof aicoso_pickup_delivery_params !== 'undefined' && aicoso_pickup_delivery_params.i18n && aicoso_pickup_delivery_params.i18n.date_format) 
            ? aicoso_pickup_delivery_params.i18n.date_format 
            : 'yy-mm-dd';
        var notAvailableText = (typeof aicoso_pickup_delivery_params !== 'undefined' && aicoso_pickup_delivery_params.i18n && aicoso_pickup_delivery_params.i18n.delivery_not_available) 
            ? aicoso_pickup_delivery_params.i18n.delivery_not_available 
            : 'Not available';
        
        // Prepare datepicker default options
        var datepickerDefaults = {
            dateFormat: dateFormat,
            minDate: 0,
            beforeShowDay: function(date) {
                // Default to not selectable
                return [false, '', notAvailableText];
            },
            onSelect: function(dateText) {
                // When date is selected, fetch available time slots
                if (dateText) {
                    fetchDeliveryTimeSlots(dateText);
                }
            }
        };
        
        // Initialize datepicker with default options
        $deliveryDateField.datepicker(datepickerDefaults);
        
        // Fetch available dates for delivery and update datepicker
        fetchDeliveryDates();
    }
}

/**
 * Fetch available delivery dates via AJAX
 */
function fetchDeliveryDates() {
    var $deliveryDateField = $('#aicoso-pickup-delivery-delivery-date');
    
    if (!$deliveryDateField.length) {
        return;
    }
    
    // Check if required parameters exist
    if (typeof aicoso_pickup_delivery_params === 'undefined' || !aicoso_pickup_delivery_params.ajax_url) {
        return;
    }
    
    $.ajax({
        type: 'POST',
        url: aicoso_pickup_delivery_params.ajax_url,
        data: {
            action: 'aicoso_pickup_delivery_get_dates',
            type: 'delivery',
            nonce: aicoso_pickup_delivery_params.nonce
        },
        success: function(response) {
            if (response.success && response.data.datepicker_data) {
                // Update datepicker with available dates
                var data = response.data.datepicker_data;
                
                // Get localized strings
                var notAvailableText = (typeof aicoso_pickup_delivery_params !== 'undefined' && aicoso_pickup_delivery_params.i18n && aicoso_pickup_delivery_params.i18n.delivery_not_available) 
                    ? aicoso_pickup_delivery_params.i18n.delivery_not_available 
                    : 'Not available';
                var availableText = (typeof aicoso_pickup_delivery_params !== 'undefined' && aicoso_pickup_delivery_params.i18n && aicoso_pickup_delivery_params.i18n.delivery_available) 
                    ? aicoso_pickup_delivery_params.i18n.delivery_available 
                    : 'Available';
                
                // Update beforeShowDay function to filter available dates
                $deliveryDateField.datepicker('option', 'beforeShowDay', function(date) {
                    var dateString = $.datepicker.formatDate('yy-mm-dd', date);
                    var day = date.getDay(); // 0 is Sunday, 1 is Monday, etc.
                    
                    // Check if date is in blackout dates
                    if ($.inArray(dateString, data.blackout_dates) > -1) {
                        return [false, 'date-blackout', notAvailableText];
                    }
                    
                    // Check if day is in available days
                    if ($.inArray(day, data.available_days) === -1) {
                        return [false, '', notAvailableText];
                    }
                    
                    // Check if date is in available dates
                    if ($.inArray(dateString, data.available_dates) > -1) {
                        return [true, 'date-available', availableText];
                    }
                    
                    return [false, '', notAvailableText];
                });
                
                // Set minimum date
                if (data.min_date) {
                    $deliveryDateField.datepicker('option', 'minDate', data.min_date);
                }
            }
        },
        error: function(xhr, status, error) {
            // Silently handle AJAX errors - could add user-friendly error display here if needed
        }
    });
}

/**
 * Fetch time slots when a delivery date is selected
 * @param {string} date - Selected date in YYYY-MM-DD format
 */
function fetchDeliveryTimeSlots(date) {
    var $timeSlotSelect = $('#aicoso-pickup-delivery-delivery-time-slot');
    
    if (!$timeSlotSelect.length) {
        return;
    }
    
    // Get localized strings
    var loadingText = (typeof aicoso_pickup_delivery_params !== 'undefined' && aicoso_pickup_delivery_params.i18n && aicoso_pickup_delivery_params.i18n.delivery_loading_slots) 
        ? aicoso_pickup_delivery_params.i18n.delivery_loading_slots 
        : 'Loading time slots...';
    var selectSlotText = (typeof aicoso_pickup_delivery_params !== 'undefined' && aicoso_pickup_delivery_params.i18n && aicoso_pickup_delivery_params.i18n.delivery_select_slot) 
        ? aicoso_pickup_delivery_params.i18n.delivery_select_slot 
        : 'Select a time slot';
    var noSlotsText = (typeof aicoso_pickup_delivery_params !== 'undefined' && aicoso_pickup_delivery_params.i18n && aicoso_pickup_delivery_params.i18n.delivery_no_slots) 
        ? aicoso_pickup_delivery_params.i18n.delivery_no_slots 
        : 'No time slots available';
    var errorLoadingText = (typeof aicoso_pickup_delivery_params !== 'undefined' && aicoso_pickup_delivery_params.i18n && aicoso_pickup_delivery_params.i18n.delivery_error_loading) 
        ? aicoso_pickup_delivery_params.i18n.delivery_error_loading 
        : 'Error loading time slots';
    
    // Reset and disable time slot select while loading
    $timeSlotSelect.empty()
        .append($('<option></option>').val('').text(loadingText))
        .prop('disabled', true);
    
    // Check if required parameters exist
    if (typeof aicoso_pickup_delivery_params === 'undefined' || !aicoso_pickup_delivery_params.ajax_url) {
        $timeSlotSelect.empty().append($('<option></option>').val('').text(errorLoadingText));
        return;
    }
    
    $.ajax({
        type: 'POST',
        url: aicoso_pickup_delivery_params.ajax_url,
        data: {
            action: 'aicoso_pickup_delivery_get_time_slots',
            type: 'delivery',
            date: date,
            nonce: aicoso_pickup_delivery_params.nonce
        },
        success: function(response) {
            $timeSlotSelect.empty();
            
            if (response.success && response.data.time_slots && response.data.time_slots.length > 0) {
                // Filter out slots with no remaining capacity (if provided).
                var availableSlots = $.grep(response.data.time_slots, function(slot) {
                    return (typeof slot.remaining_capacity === 'undefined') || (parseInt(slot.remaining_capacity, 10) > 0);
                });

                if (availableSlots.length > 0) {
                    // Add default option
                    $timeSlotSelect.append($('<option></option>').val('').text(selectSlotText));

                    // Add time slots
                    $.each(availableSlots, function(index, slot) {
                        var value = slot.name || slot.slot_name || slot.value || '';
                        var label = slot.formatted_time || slot.slot_name || value;
                        if (!value) {
                            return;
                        }
                        $timeSlotSelect.append(
                            $('<option></option>').val(value).text(label)
                        );
                    });

                    // Enable select
                    $timeSlotSelect.prop('disabled', false);
                } else {
                    // No time slots available after filtering
                    $timeSlotSelect.append($('<option></option>').val('').text(noSlotsText));
                }
            } else {
                // No time slots available
                $timeSlotSelect.append($('<option></option>').val('').text(noSlotsText));
            }
        },
        error: function(xhr, status, error) {
            // Handle AJAX errors gracefully with user-friendly message
            $timeSlotSelect.empty().append($('<option></option>').val('').text(errorLoadingText));
        }
    });
}

/* ================================
   UTILITY FUNCTIONS
   ================================ */

// Add any utility functions that might be needed for the templates

})(jQuery);
