<?php
/**
 * AICOSO Pickup & Delivery
 *
 * @package   Aicoso_Pickup_Delivery_For_Woocommerce
 * @author    AICOSO
 * @license   GPL-3.0+
 * @link      https://example.com
 *
 * @wordpress-plugin
 * Plugin Name:       AICOSO Pickup and Delivery for WooCommerce
 * Plugin URI:        https://wordpress.org/plugins/aicoso-pickup-and-delivery-for-woocommerce
 * Description:       Enables customers to choose between home delivery and pickup with scheduling.
 * Version:           1.0.0
 * Author:            Team ePLUGINS
 * Text Domain:       aicoso-pickup-and-delivery-for-woocommerce
 * License:           GPL v3 or later
 * License URI:       https://www.gnu.org/licenses/gpl-3.0.html
 * Domain Path:       /languages
 * Requires Plugins:  woocommerce
 * WC requires at least: 9.0
 * WC tested up to: 10.3
 * Requires PHP: 7.4
 */

// If this file is called directly, abort.
if ( ! defined( 'WPINC' ) ) {
	die;
}
/**
 * Currently plugin version.
 */
define( 'AICOSO_PICKUP_DELIVERY_VERSION', '1.0.0' );
define( 'AICOSO_PICKUP_DELIVERY_PLUGIN_PATH', plugin_dir_path( __FILE__ ) );
define( 'AICOSO_PICKUP_DELIVERY_PLUGIN_URL', plugin_dir_url( __FILE__ ) );

/**
 * Check if WooCommerce is active
 */
function aicoso_pickup_delivery_is_woocommerce_active() {
	$active_plugins = (array) get_option( 'active_plugins', array() );

	if ( is_multisite() ) {
		$active_plugins = array_merge( $active_plugins, get_site_option( 'active_sitewide_plugins', array() ) );
	}

	return in_array( 'woocommerce/woocommerce.php', $active_plugins, true ) || array_key_exists( 'woocommerce/woocommerce.php', $active_plugins );
}

/**
 * The code that runs during plugin activation.
 * This action is documented in includes/class-aicoso-pickup-delivery-activator.php
 */
function aicoso_pickup_delivery_activate() {
	// Check if WooCommerce is active before allowing activation.
	if ( ! aicoso_pickup_delivery_is_woocommerce_active() ) {
		// Deactivate the plugin.
		deactivate_plugins( plugin_basename( __FILE__ ) );

		// Show error message.
		wp_die(
			esc_html__( 'AICOSO Pickup & Delivery requires WooCommerce to be installed and activated. Please install and activate WooCommerce first.', 'aicoso-pickup-and-delivery-for-woocommerce' ),
			esc_html__( 'Plugin Activation Error', 'aicoso-pickup-and-delivery-for-woocommerce' ),
			array(
				'response'  => 200,
				'back_link' => true,
			)
		);
	}

	require_once AICOSO_PICKUP_DELIVERY_PLUGIN_PATH . 'includes/class-aicoso-pickup-delivery-activator.php';
	AICOSO_Pickup_Delivery_Activator::activate();
}

/**
 * The code that runs during plugin deactivation.
 * This action is documented in includes/class-aicoso-pickup-delivery-deactivator.php
 */
function aicoso_pickup_delivery_deactivate() {
	require_once AICOSO_PICKUP_DELIVERY_PLUGIN_PATH . 'includes/class-aicoso-pickup-delivery-deactivator.php';
	AICOSO_Pickup_Delivery_Deactivator::deactivate();
}

register_activation_hook( __FILE__, 'aicoso_pickup_delivery_activate' );
register_deactivation_hook( __FILE__, 'aicoso_pickup_delivery_deactivate' );

/**
 * Auto-deactivate this plugin when WooCommerce is deactivated
 */
add_action( 'deactivated_plugin', 'aicoso_pickup_delivery_check_woocommerce_deactivation', 10, 2 );

/**
 * Check if WooCommerce was deactivated and deactivate this plugin.
 *
 * @param string $plugin               Path to the plugin file.
 * @param bool   $network_deactivating Whether the plugin is deactivated for all sites in the network.
 * @return void
 */
function aicoso_pickup_delivery_check_woocommerce_deactivation( $plugin, $network_deactivating ) { // phpcs:ignore Generic.CodeAnalysis.UnusedFunctionParameter.FoundAfterLastUsed -- Required by WordPress hook signature.
	// Check if WooCommerce was just deactivated.
	if ( 'woocommerce/woocommerce.php' === $plugin ) {
		// Check if our plugin is active.
		if ( is_plugin_active( plugin_basename( __FILE__ ) ) ) {
			// Deactivate our plugin.
			deactivate_plugins( plugin_basename( __FILE__ ) );

			// Store a flag to show notice on next page load.
			set_transient( 'aicoso_pickup_delivery_auto_deactivated', true, 60 );

			// Add a note about why the plugin was deactivated.
			add_option( 'aicoso_pickup_delivery_deactivation_reason', 'woocommerce_deactivated' );

			// Prevent redirect to WooCommerce setup wizard.
			// phpcs:ignore WordPress.Security.NonceVerification.Recommended
			if ( isset( $_GET['activate'] ) ) {
				unset( $_GET['activate'] );
			}
		}
	}
}

/**
 * Show notice when plugin was auto-deactivated
 */
add_action( 'admin_notices', 'aicoso_pickup_delivery_auto_deactivated_notice' );

/**
 * Display admin notice when plugin was auto-deactivated.
 *
 * @return void
 */
function aicoso_pickup_delivery_auto_deactivated_notice() {
	// Show auto-deactivation notice.
	if ( get_transient( 'aicoso_pickup_delivery_auto_deactivated' ) ) {
		delete_transient( 'aicoso_pickup_delivery_auto_deactivated' );
		delete_option( 'aicoso_pickup_delivery_deactivation_reason' );
		?>
		<div class="notice notice-warning is-dismissible">
			<h3><?php echo esc_html__( 'AICOSO Pickup & Delivery - Auto Deactivated', 'aicoso-pickup-and-delivery-for-woocommerce' ); ?></h3>
			<p><?php echo esc_html__( 'The AICOSO Pickup & Delivery plugin has been automatically deactivated because WooCommerce is no longer active.', 'aicoso-pickup-and-delivery-for-woocommerce' ); ?></p>
			<p><?php echo esc_html__( 'To use this plugin again:', 'aicoso-pickup-and-delivery-for-woocommerce' ); ?></p>
			<ol>
				<li><?php echo esc_html__( 'Activate WooCommerce', 'aicoso-pickup-and-delivery-for-woocommerce' ); ?></li>
				<li><?php echo esc_html__( 'Activate AICOSO Pickup & Delivery', 'aicoso-pickup-and-delivery-for-woocommerce' ); ?></li>
			</ol>
			<p>
				<a href="<?php echo esc_url( admin_url( 'plugins.php' ) ); ?>" class="button button-primary">
					<?php echo esc_html__( 'Go to Plugins', 'aicoso-pickup-and-delivery-for-woocommerce' ); ?>
				</a>
			</p>
		</div>
		<?php
	}

	// Also check if trying to run without WooCommerce (but not on plugins page to avoid duplicate notices).
	$screen = get_current_screen();
	if ( $screen && 'plugins' !== $screen->id ) {
		if ( is_plugin_active( plugin_basename( __FILE__ ) ) && ! aicoso_pickup_delivery_is_woocommerce_active() ) {
			?>
			<div class="notice notice-error">
				<p><strong><?php echo esc_html__( 'AICOSO Pickup & Delivery Error', 'aicoso-pickup-and-delivery-for-woocommerce' ); ?></strong></p>
				<p><?php echo esc_html__( 'This plugin requires WooCommerce to be active. Please activate WooCommerce or deactivate this plugin.', 'aicoso-pickup-and-delivery-for-woocommerce' ); ?></p>
			</div>
			<?php
		}
	}
}

/**
 * Display notice if WooCommerce is not active
 */
function aicoso_pickup_delivery_woocommerce_missing_notice() {
	?>
	<div class="error">
		<p><?php echo esc_html__( 'AICOSO Pickup & Delivery requires WooCommerce to be installed and activated!', 'aicoso-pickup-and-delivery-for-woocommerce' ); ?></p>
	</div>
	<?php
}

/**
 * The core plugin class that is used to define internationalization,
 * admin-specific hooks, and public-facing site hooks.
 */
function aicoso_pickup_delivery_run() {
	// If WooCommerce is not active, display notice and return.
	if ( ! aicoso_pickup_delivery_is_woocommerce_active() ) {
		add_action( 'admin_notices', 'aicoso_pickup_delivery_woocommerce_missing_notice' );
		return;
	}

	// Wait until WooCommerce is fully loaded to initialize our plugin.
	if ( did_action( 'woocommerce_loaded' ) ) {
		// WooCommerce is loaded, safe to initialize.
		require_once AICOSO_PICKUP_DELIVERY_PLUGIN_PATH . 'includes/class-aicoso-pickup-delivery.php';
		$plugin = new AICOSO_Pickup_Delivery();
		$plugin->run();
	} else {
		// Wait for WooCommerce to load first.
		add_action( 'woocommerce_loaded', 'aicoso_pickup_delivery_init' );
	}
}

/**
 * Initialize the plugin after WooCommerce is loaded
 */
function aicoso_pickup_delivery_init() {
	require_once AICOSO_PICKUP_DELIVERY_PLUGIN_PATH . 'includes/class-aicoso-pickup-delivery.php';
	$plugin = new AICOSO_Pickup_Delivery();
	$plugin->run();
}

// Initialize the plugin only after WooCommerce is fully loaded.
add_action( 'plugins_loaded', 'aicoso_pickup_delivery_init_check', 5 );

/**
 * Check if WooCommerce is loaded before initializing our plugin
 */
function aicoso_pickup_delivery_init_check() {
	// Wait for WooCommerce to be fully loaded.
	if ( class_exists( 'WooCommerce' ) ) {
		// WooCommerce is loaded, initialize immediately.
		add_action( 'init', 'aicoso_pickup_delivery_run', 1 );
	} else {
		// Wait for WooCommerce to load.
		add_action( 'woocommerce_loaded', 'aicoso_pickup_delivery_run' );
	}
}

/**
 * Add plugin action links (Settings link on plugins page)
 */
add_filter( 'plugin_action_links_' . plugin_basename( __FILE__ ), 'aicoso_pickup_delivery_action_links' );

/**
 * Add action links to the plugin list.
 *
 * @param array $links Existing links.
 * @return array Modified links.
 */
function aicoso_pickup_delivery_action_links( $links ) {
	// Check if WooCommerce is active first.
	if ( ! aicoso_pickup_delivery_is_woocommerce_active() ) {
		return $links;
	}

	// Primary settings URL - Direct to our custom settings page.
	$primary_settings_url = admin_url( 'admin.php?page=aicoso-pickup-delivery-settings' );

	// Add settings link at the beginning.
	$settings_link = '<a href="' . esc_url( $primary_settings_url ) . '" style="color: #2271b1; font-weight: 600;">' . esc_html__( 'Settings', 'aicoso-pickup-and-delivery-for-woocommerce' ) . '</a>';
	array_unshift( $links, $settings_link );

	return $links;
}

/**
 * Add plugin meta links (appears after plugin description)
 */
add_filter( 'plugin_row_meta', 'aicoso_pickup_delivery_plugin_meta_links', 10, 2 );

/**
 * Add meta links to the plugin list.
 *
 * @param array  $links Existing links.
 * @param string $file  Plugin file.
 * @return array Modified links.
 */
function aicoso_pickup_delivery_plugin_meta_links( $links, $file ) {
	if ( plugin_basename( __FILE__ ) === $file ) {
		// Only add links if WooCommerce is active.
		if ( aicoso_pickup_delivery_is_woocommerce_active() ) {
			// Add additional meta links.
			$meta_links = array(
				'<a href="' . esc_url( admin_url( 'admin.php?page=aicoso-pickup-delivery-settings' ) ) . '" style="font-weight: 600;">' . esc_html__( 'Configure Settings', 'aicoso-pickup-and-delivery-for-woocommerce' ) . '</a>',
				'<a href="' . esc_url( admin_url( 'edit.php?post_type=shop_order' ) ) . '">' . esc_html__( 'View Orders', 'aicoso-pickup-and-delivery-for-woocommerce' ) . '</a>',
			);

			$links = array_merge( $links, $meta_links );
		}
	}

	return $links;
}

/**
 * Declare HPOS (High Performance Order Storage) compatibility
 */
add_action(
	'before_woocommerce_init',
	function () {
		if ( class_exists( '\Automattic\WooCommerce\Utilities\FeaturesUtil' ) ) {
			\Automattic\WooCommerce\Utilities\FeaturesUtil::declare_compatibility(
				'custom_order_tables',
				__FILE__,
				true
			);

			// Also declare compatibility with Cart and Checkout blocks.
			\Automattic\WooCommerce\Utilities\FeaturesUtil::declare_compatibility(
				'cart_checkout_blocks',
				__FILE__,
				true
			);
		}
	}
);

add_action( 'plugins_loaded', 'aicoso_pickup_delivery_check_email' );

/**
 * Debug function to check WooCommerce email functionality.
 *
 * @return bool True if WC_Email class exists, false otherwise.
 */
function aicoso_pickup_delivery_check_email() {
	if ( class_exists( 'WC_Email' ) ) {
		// WC_Email is available for use.
		return true;
	} else {
		// WC_Email is not available.
		return false;
	}
}