<?php
/**
 * Asset Loader Utility Class
 *
 * Provides static methods to enqueue scripts and styles with WordPress.
 * Automatically handles .asset.php metadata files generated by build tools like wp-scripts.
 * Falls back to file modification time for versioning if asset metadata is not available.
 *
 * Usage:
 *     AssetLoader::enqueue_script( 'my-handle', 'my-script' );
 *     AssetLoader::enqueue_style( 'my-style', 'my-stylesheet' );
 *
 * @package WordPress\AI
 */

declare( strict_types=1 );

namespace WordPress\AI;

/**
 * Class Asset_Loader
 *
 * A utility class for registering and enqueuing assets (scripts and styles)
 * with support for asset metadata files.
 *
 * @since 0.1.0
 */
class Asset_Loader {

	/**
	 * Enqueue a script using a script path and its asset metadata.
	 *
	 * @since 0.1.0
	 *
	 * @param string                    $handle     The handle for the script.
	 * @param string                    $file_name  The script file name.
	 * @param array<string, mixed>|null $asset_data Optional asset metadata (dependencies and version).
	 */
	public static function enqueue_script( string $handle, string $file_name, ?array $asset_data = null ): void {
		$script_path       = AI_EXPERIMENTS_DIR . 'build/' . $file_name . '.js';
		$script_url        = AI_EXPERIMENTS_PLUGIN_URL . 'build/' . $file_name . '.js';
		$script_asset_path = substr( $script_path, 0, -3 ) . '.asset.php';

		if ( is_array( $asset_data ) ) {
			if ( ! isset( $asset_data['dependencies'] ) ) {
				$asset_data['dependencies'] = array();
			}
			if ( ! isset( $asset_data['version'] ) ) {
				$asset_data['version'] = filemtime( $script_path );
			}
		} elseif ( file_exists( $script_asset_path ) ) {
			$asset_data = require $script_asset_path; // phpcs:ignore WordPressVIPMinimum.Files.IncludingFile.UsingVariable
		} else {
			$asset_data = array(
				'dependencies' => array(),
				'version'      => filemtime( $script_path ),
			);
		}

		wp_enqueue_script(
			'ai_' . $handle,
			$script_url,
			$asset_data['dependencies'],
			$asset_data['version'],
			array( 'strategy' => 'defer' )
		);
	}

	/**
	 * Enqueue a style using a style path and its asset metadata.
	 *
	 * @since 0.1.0
	 *
	 * @param string                    $handle     The handle for the style.
	 * @param string                    $file_name  The script file name.
	 * @param array<string, mixed>|null $asset_data Optional asset metadata (dependencies and version).
	 */
	public static function enqueue_style( string $handle, string $file_name, ?array $asset_data = null ): void {
		$style_path       = AI_EXPERIMENTS_DIR . 'build/' . $file_name . '.css';
		$style_url        = AI_EXPERIMENTS_PLUGIN_URL . 'build/' . $file_name . '.css';
		$style_asset_path = substr( $style_path, 0, -4 ) . '.asset.php';

		if ( is_array( $asset_data ) ) {
			if ( ! isset( $asset_data['dependencies'] ) ) {
				$asset_data['dependencies'] = array();
			}
			if ( ! isset( $asset_data['version'] ) ) {
				$asset_data['version'] = filemtime( $style_path );
			}
		} elseif ( file_exists( $style_asset_path ) ) {
			$asset_data = require $style_asset_path; // phpcs:ignore WordPressVIPMinimum.Files.IncludingFile.UsingVariable
		} else {
			$asset_data = array(
				'dependencies' => array(),
				'version'      => filemtime( $style_path ),
			);
		}

		wp_enqueue_style(
			'ai_' . $handle,
			$style_url,
			array(),
			$asset_data['version']
		);
		wp_style_add_data( 'ai_' . $handle, 'path', $style_path );

		$rtl_style_path = str_replace( '.css', '-rtl.css', $style_path );
		if ( ! file_exists( $rtl_style_path ) ) {
			return;
		}
		wp_style_add_data( 'ai_' . $handle, 'rtl', 'replace' );
		if ( ! is_rtl() ) {
			return;
		}
		wp_style_add_data( 'ai_' . $handle, 'path', $rtl_style_path );
	}

	/**
	 * Localize data for an enqueued script.
	 *
	 * This method allows passing PHP data to JavaScript using `wp_localize_script()`.
	 * It must be called after the script has been enqueued using `enqueue_script()`.
	 *
	 * @since 0.1.0
	 *
	 * @param string $handle The script handle used in `enqueue_script()` (without prefix).
	 * @param string $object_name The name of the JavaScript object to contain the data.
	 * @param array<string, mixed> $data The data to localize.
	 */
	public static function localize_script( string $handle, string $object_name, array $data ): void {
		wp_localize_script(
			'ai_' . $handle,
			'ai' . $object_name,
			$data
		);
	}
}
