<?php

/**
 * Product meta box class
 *
 * @package ReplicateWP_Virtual_TryOn
 */

// Exit if accessed directly
if (! defined('ABSPATH')) {
    exit;
}

/**
 * Product meta box class
 */
class RWPVTO_Product_Meta
{

    /**
     * Single instance
     *
     * @var RWPVTO_Product_Meta
     */
    private static $instance = null;

    /**
     * Get instance
     *
     * @return RWPVTO_Product_Meta
     */
    public static function instance()
    {
        if (is_null(self::$instance)) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    /**
     * Constructor
     */
    private function __construct()
    {
        add_action('add_meta_boxes', array($this, 'add_meta_box'));
    }

    /**
     * Add meta box
     */
    public function add_meta_box()
    {
        add_meta_box(
            'rwpvto_product_meta',
            __('Virtual Try-On Generator', 'ai-virtual-try-on-for-woocommerce'),
            array($this, 'render_meta_box'),
            'product',
            'side',
            'default'
        );
    }

    /**
     * Render meta box
     *
     * @param WP_Post $post Post object
     */
    public function render_meta_box($post)
    {
        // Check if API key is set
        $settings = get_option('rwpvto_settings', array());
        $api_key_set = ! empty($settings['api_key']);

        if (! $api_key_set) {
?>
            <div class="rwpvto-notice">
                <p>
                    <?php
                    echo wp_kses_post(
                        sprintf(
                            /* translators: %s: Settings page link */
                            __('Please configure your Replicate API key in %s first.', 'ai-virtual-try-on-for-woocommerce'),
                            '<a href="' . esc_url(admin_url('admin.php?page=rwpvto-settings')) . '">' . esc_html__('Settings', 'ai-virtual-try-on-for-woocommerce') . '</a>'
                        )
                    );
                    ?>
                </p>
            </div>
        <?php
            return;
        }

        // Get product
        $product = wc_get_product($post->ID);

        if (! $product) {
            return;
        }

        // Get product images
        $featured_image = get_the_post_thumbnail_url($post->ID, 'full');
        $gallery_images = array();

        if ($featured_image) {
            $gallery_images[] = array(
                'id' => get_post_thumbnail_id($post->ID),
                'url' => $featured_image,
            );
        }

        foreach ($product->get_gallery_image_ids() as $image_id) {
            $gallery_images[] = array(
                'id' => $image_id,
                'url' => wp_get_attachment_url($image_id),
            );
        }

        // Get default model images (you can expand this)
        $default_models = $this->get_default_models();

        // Get previously generated images
        $db = RWPVTO_Database::instance();
        $generated_images = $db->get_images_by_product($post->ID, array('limit' => 5));

        ?>
        <div class="rwpvto-product-meta">
            <div class="rwpvto-form-group">
                <label for="rwpvto_garment_image">
                    <strong><?php esc_html_e('Select Garment Image:', 'ai-virtual-try-on-for-woocommerce'); ?></strong>
                </label>
                <select id="rwpvto_garment_image" class="rwpvto-garment-select">
                    <option value=""><?php esc_html_e('Choose an image...', 'ai-virtual-try-on-for-woocommerce'); ?></option>
                    <?php foreach ($gallery_images as $image) : ?>
                        <option value="<?php echo esc_url($image['url']); ?>" data-image-id="<?php echo esc_attr($image['id']); ?>">
                            <?php echo esc_html(basename($image['url'])); ?>
                        </option>
                    <?php endforeach; ?>
                </select>
                <div class="rwpvto-image-preview-container">
                    <img id="rwpvto_garment_preview" src="" alt="" class="rwpvto-thumbnail-preview">
                </div>
            </div>

            <div class="rwpvto-form-group">
                <label>
                    <strong><?php esc_html_e('Select Model:', 'ai-virtual-try-on-for-woocommerce'); ?></strong>
                </label>

                <!-- Hidden select for compatibility -->
                <select id="rwpvto_model_image" style="display:none;">
                    <option value=""><?php esc_html_e('Choose a model...', 'ai-virtual-try-on-for-woocommerce'); ?></option>
                    <?php foreach ($default_models as $model) : ?>
                        <option value="<?php echo esc_url($model['url']); ?>"
                            data-gender="<?php echo esc_attr($model['gender']); ?>">
                            <?php echo esc_html($model['name']); ?>
                        </option>
                    <?php endforeach; ?>
                </select>

                <!-- Visual Model Picker Button -->
                <button type="button" class="button button-secondary rwpvto-open-model-picker" style="padding: 10px; margin-bottom: 10px;">
                    <span class="dashicons dashicons-admin-users" style="margin-top: 3px;"></span>
                    <span id="rwpvto_selected_model_text"><?php esc_html_e('Browse Models', 'ai-virtual-try-on-for-woocommerce'); ?></span>
                </button>

                <!-- Selected Model Preview -->
                <div class="rwpvto-image-preview-container">
                    <img id="rwpvto_model_preview" src="" alt="" class="rwpvto-thumbnail-preview">
                </div>
                <p class="description" style="margin-top: 10px; margin-bottom: 5px;">
                    <?php esc_html_e('Or use your own model image:', 'ai-virtual-try-on-for-woocommerce'); ?>
                </p>
                <div style="display: flex; gap: 10px; margin-bottom: 10px;">
                    <button type="button" class="button rwpvto-choose-model-btn">
                        <span class="dashicons dashicons-format-image" style="margin-top: 3px;"></span>
                        <?php esc_html_e('Choose from Library', 'ai-virtual-try-on-for-woocommerce'); ?>
                    </button>
                    <button type="button" class="button rwpvto-clear-model-btn" style="display:none;">
                        <?php esc_html_e('Clear', 'ai-virtual-try-on-for-woocommerce'); ?>
                    </button>
                </div>
                <input type="url" id="rwpvto_custom_model_url" class="widefat" placeholder="<?php esc_attr_e('Or paste image URL here...', 'ai-virtual-try-on-for-woocommerce'); ?>" readonly style="background: #f5f5f5;">

                <!-- Model Picker Modal -->
                <div id="rwpvto-model-picker-modal" class="rwpvto-picker-modal" style="display:none;">
                    <div class="rwpvto-picker-modal-overlay"></div>
                    <div class="rwpvto-picker-modal-content">
                        <div class="rwpvto-picker-modal-header">
                            <h2><?php esc_html_e('Select a Model', 'ai-virtual-try-on-for-woocommerce'); ?></h2>
                            <button type="button" class="rwpvto-picker-modal-close">
                                <span class="dashicons dashicons-no-alt"></span>
                            </button>
                        </div>
                        <div class="rwpvto-picker-modal-filters">
                            <div class="rwpvto-filter-group">
                                <label style="display: block; margin-bottom: 8px; font-weight: 600;">
                                    <?php esc_html_e('Gender:', 'ai-virtual-try-on-for-woocommerce'); ?>
                                </label>
                                <div class="rwpvto-filter-buttons">
                                    <button type="button" class="rwpvto-filter-btn rwpvto-gender-filter active" data-filter-type="gender" data-filter="all">
                                        <?php esc_html_e('All', 'ai-virtual-try-on-for-woocommerce'); ?>
                                    </button>
                                    <button type="button" class="rwpvto-filter-btn rwpvto-gender-filter" data-filter-type="gender" data-filter="male">
                                        <?php esc_html_e('Male', 'ai-virtual-try-on-for-woocommerce'); ?>
                                    </button>
                                    <button type="button" class="rwpvto-filter-btn rwpvto-gender-filter" data-filter-type="gender" data-filter="female">
                                        <?php esc_html_e('Female', 'ai-virtual-try-on-for-woocommerce'); ?>
                                    </button>
                                </div>
                            </div>
                            <div class="rwpvto-filter-group" style="margin-top: 15px;">
                                <label style="display: block; margin-bottom: 8px; font-weight: 600;">
                                    <?php esc_html_e('Category:', 'ai-virtual-try-on-for-woocommerce'); ?>
                                </label>
                                <div class="rwpvto-filter-buttons">
                                    <button type="button" class="rwpvto-filter-btn rwpvto-category-filter active" data-filter-type="category" data-filter="all">
                                        <?php esc_html_e('All', 'ai-virtual-try-on-for-woocommerce'); ?>
                                    </button>
                                    <button type="button" class="rwpvto-filter-btn rwpvto-category-filter" data-filter-type="category" data-filter="upper_body">
                                        <?php esc_html_e('Upper Body', 'ai-virtual-try-on-for-woocommerce'); ?>
                                    </button>
                                    <button type="button" class="rwpvto-filter-btn rwpvto-category-filter" data-filter-type="category" data-filter="lower_body">
                                        <?php esc_html_e('Lower Body', 'ai-virtual-try-on-for-woocommerce'); ?>
                                    </button>
                                    <button type="button" class="rwpvto-filter-btn rwpvto-category-filter" data-filter-type="category" data-filter="dresses">
                                        <?php esc_html_e('Dresses', 'ai-virtual-try-on-for-woocommerce'); ?>
                                    </button>
                                </div>
                            </div>
                        </div>
                        <div class="rwpvto-picker-modal-body">
                            <div class="rwpvto-model-grid">
                                <?php foreach ($default_models as $index => $model) : ?>
                                    <div class="rwpvto-model-item" data-model-url="<?php echo esc_url($model['url']); ?>"
                                        data-model-name="<?php echo esc_attr($model['name']); ?>"
                                        data-gender="<?php echo esc_attr($model['gender']); ?>"
                                        data-category="<?php echo esc_attr($model['category']); ?>">
                                        <div class="rwpvto-model-thumbnail">
                                            <img src="<?php echo esc_url($model['url']); ?>" alt="<?php echo esc_attr($model['name']); ?>">
                                            <div class="rwpvto-model-overlay">
                                                <span class="dashicons dashicons-yes-alt"></span>
                                            </div>
                                        </div>
                                        <div class="rwpvto-model-label">
                                            <?php echo esc_html($model['name']); ?>
                                        </div>
                                    </div>
                                <?php endforeach; ?>
                            </div>
                        </div>
                    </div>
                </div>
            </div>

            <div class="rwpvto-form-group">
                <label for="rwpvto_category">
                    <strong><?php esc_html_e('Category:', 'ai-virtual-try-on-for-woocommerce'); ?></strong>
                </label>
                <select id="rwpvto_category">
                    <option value="upper_body"><?php esc_html_e('Upper Body', 'ai-virtual-try-on-for-woocommerce'); ?></option>
                    <option value="lower_body"><?php esc_html_e('Lower Body', 'ai-virtual-try-on-for-woocommerce'); ?></option>
                    <option value="dresses"><?php esc_html_e('Dresses', 'ai-virtual-try-on-for-woocommerce'); ?></option>
                </select>
            </div>

            <div class="rwpvto-form-group">
                <label>
                    <input type="checkbox" id="rwpvto_auto_crop" value="1">
                    <strong><?php esc_html_e('Auto-crop model image', 'ai-virtual-try-on-for-woocommerce'); ?></strong>
                </label>
                <p class="description">
                    <?php esc_html_e('Enable if your model image has wrong aspect ratio. May help with stretched images.', 'ai-virtual-try-on-for-woocommerce'); ?>
                </p>
            </div>

            <div class="rwpvto-form-group">
                <label for="rwpvto_garment_description">
                    <strong><?php esc_html_e('Garment Description (Optional):', 'ai-virtual-try-on-for-woocommerce'); ?></strong>
                </label>
                <input type="text" id="rwpvto_garment_description" class="widefat" placeholder="<?php esc_attr_e('e.g., slim-fit and stretchy', 'ai-virtual-try-on-for-woocommerce'); ?>">
                <p class="description">
                    <?php esc_html_e('Brief description to help the AI understand the garment better.', 'ai-virtual-try-on-for-woocommerce'); ?>
                </p>
            </div>

            <div class="rwpvto-form-group">
                <label for="rwpvto_custom_filename">
                    <strong><?php esc_html_e('Custom Filename (Optional):', 'ai-virtual-try-on-for-woocommerce'); ?></strong>
                </label>
                <input type="text" id="rwpvto_custom_filename" class="widefat" placeholder="<?php esc_attr_e('e.g., blue-t-shirt-on-male-model', 'ai-virtual-try-on-for-woocommerce'); ?>">
                <p class="description">
                    <?php esc_html_e('Custom filename for the generated image (SEO-friendly). Leave empty for auto-generated filename.', 'ai-virtual-try-on-for-woocommerce'); ?>
                </p>
            </div>

            <div class="rwpvto-form-actions">
                <button type="button" class="button button-primary button-large rwpvto-generate-btn" data-product-id="<?php echo esc_attr($post->ID); ?>">
                    <span class="dashicons dashicons-camera" style="margin-top: 3px;"></span>
                    <?php esc_html_e('Generate Try-On Image', 'ai-virtual-try-on-for-woocommerce'); ?>
                </button>
            </div>

            <div class="rwpvto-result" style="display:none; margin-top: 15px;">
                <div class="rwpvto-result-message"></div>
                <div class="rwpvto-result-image"></div>
            </div>

            <?php if (! empty($generated_images)) : ?>
                <div class="rwpvto-previous-images">
                    <h4><?php esc_html_e('Recently Generated:', 'ai-virtual-try-on-for-woocommerce'); ?></h4>
                    <div class="rwpvto-thumbnails">
                        <?php foreach ($generated_images as $gen_image) : ?>
                            <?php
                            // Use local WordPress URL if available, otherwise fall back to replicate URL
                            $image_url = ! empty( $gen_image->local_image_url ) ? $gen_image->local_image_url : $gen_image->generated_image_url;
                            ?>
                            <div class="rwpvto-thumbnail">
                                <img src="<?php echo esc_url($image_url); ?>" alt="">
                                <div class="rwpvto-thumbnail-actions">
                                    <a href="<?php echo esc_url($image_url); ?>" target="_blank" class="button button-small">
                                        <?php esc_html_e('View', 'ai-virtual-try-on-for-woocommerce'); ?>
                                    </a>
                                </div>
                            </div>
                        <?php endforeach; ?>
                    </div>
                </div>
            <?php endif; ?>
        </div>
<?php
    }

    /**
     * Get default model images
     *
     * @return array
     */
    private function get_default_models()
    {
        // Local models from plugin assets
        return array(
            // Male models
            array(
                'name' => __('Male (Upper Body, White T-Shirt)', 'ai-virtual-try-on-for-woocommerce'),
                'url' => RWPVTO_PLUGIN_URL . 'assets/models/0VEL3LbUaixNznNk.jpg',
                'gender' => 'male',
                'category' => 'upper_body',
            ),
            array(
                'name' => __('Male (Full Body, White T-Shirt)', 'ai-virtual-try-on-for-woocommerce'),
                'url' => RWPVTO_PLUGIN_URL . 'assets/models/ctAy7FJ2C5V6Dwcv.jpg',
                'gender' => 'male',
                'category' => 'upper_body,lower_body',
            ),
            array(
                'name' => __('Male (Upper Body, Jumper)', 'ai-virtual-try-on-for-woocommerce'),
                'url' => RWPVTO_PLUGIN_URL . 'assets/models/8rTm4Dxpy6wcxYUf.jpg',
                'gender' => 'male',
                'category' => 'upper_body',
            ),
            array(
                'name' => __('Male (Upper Body, Long Sleeve)', 'ai-virtual-try-on-for-woocommerce'),
                'url' => RWPVTO_PLUGIN_URL . 'assets/models/vOYTtZF4JKEdoi0w.jpg',
                'gender' => 'male',
                'category' => 'upper_body',
            ),
            array(
                'name' => __('Male (Upper Body, Green T-Shirt)', 'ai-virtual-try-on-for-woocommerce'),
                'url' => RWPVTO_PLUGIN_URL . 'assets/models/BOHnclom8c6EBNqO.jpg',
                'gender' => 'male',
                'category' => 'upper_body',
            ),
            array(
                'name' => __('Male (Upper Body, Green T-Shirt)', 'ai-virtual-try-on-for-woocommerce'),
                'url' => RWPVTO_PLUGIN_URL . 'assets/models/MmES6yTKmqV7po8V.jpg',
                'gender' => 'male',
                'category' => 'upper_body',
            ),
            array(
                'name' => __('Male (Upper Body, Multicolor Hoodie)', 'ai-virtual-try-on-for-woocommerce'),
                'url' => RWPVTO_PLUGIN_URL . 'assets/models/G42ML2w5kMjgA4CT.jpg',
                'gender' => 'male',
                'category' => 'upper_body',
            ),
            array(
                'name' => __('Male (Upper Body, Multicolor Hoodie)', 'ai-virtual-try-on-for-woocommerce'),
                'url' => RWPVTO_PLUGIN_URL . 'assets/models/rgVfO2fkFXuL1raG.jpg',
                'gender' => 'male',
                'category' => 'upper_body',
            ),
            array(
                'name' => __('Male (Upper Body, Blue Hoodie)', 'ai-virtual-try-on-for-woocommerce'),
                'url' => RWPVTO_PLUGIN_URL . 'assets/models/Ootpb7RGERk9mj5V.jpg',
                'gender' => 'male',
                'category' => 'upper_body',
            ),
            array(
                'name' => __('Male (Upper Body, Green Hoodie)', 'ai-virtual-try-on-for-woocommerce'),
                'url' => RWPVTO_PLUGIN_URL . 'assets/models/a1VSKA2nO6RTJrIf.jpg',
                'gender' => 'male',
                'category' => 'upper_body',
            ),
            array(
                'name' => __('Male (Upper Body, Green Hoodie 2)', 'ai-virtual-try-on-for-woocommerce'),
                'url' => RWPVTO_PLUGIN_URL . 'assets/models/e1aNpkRUbTPRGTbN.jpg',
                'gender' => 'male',
                'category' => 'upper_body',
            ),
            array(
                'name' => __('Male (Upper Body, Green Hoodie 3)', 'ai-virtual-try-on-for-woocommerce'),
                'url' => RWPVTO_PLUGIN_URL . 'assets/models/3m8skxI6bDA19gJF.jpg',
                'gender' => 'male',
                'category' => 'upper_body',
            ),
            array(
                'name' => __('Male (Full Body, White Shirt)', 'ai-virtual-try-on-for-woocommerce'),
                'url' => RWPVTO_PLUGIN_URL . 'assets/models/0u6ghBQ2A1iUAr7b.jpg',
                'gender' => 'male',
                'category' => 'upper_body,lower_body',
            ),
            array(
                'name' => __('Male (Full Body, Blue Shirt)', 'ai-virtual-try-on-for-woocommerce'),
                'url' => RWPVTO_PLUGIN_URL . 'assets/models/34NpEjzaelQLRSi4.jpg',
                'gender' => 'male',
                'category' => 'upper_body,lower_body',
            ),
            array(
                'name' => __('Male (Full Body, Cyan T-Shirt)', 'ai-virtual-try-on-for-woocommerce'),
                'url' => RWPVTO_PLUGIN_URL . 'assets/models/5AoQm4XYXoemhg8q.jpg',
                'gender' => 'male',
                'category' => 'upper_body,lower_body',
            ),
            array(
                'name' => __('Male (Full Body, Cyan T-Shirt)', 'ai-virtual-try-on-for-woocommerce'),
                'url' => RWPVTO_PLUGIN_URL . 'assets/models/68S4ERsMDwb2Gz9G.jpg',
                'gender' => 'male',
                'category' => 'upper_body,lower_body',
            ),
            array(
                'name' => __('Male (Full Body, Red Hoodie)', 'ai-virtual-try-on-for-woocommerce'),
                'url' => RWPVTO_PLUGIN_URL . 'assets/models/9AyNsz0DMtjpdfQW.jpg',
                'gender' => 'male',
                'category' => 'upper_body,lower_body',
            ),
            array(
                'name' => __('Male (Full Body, Silver Hoodie)', 'ai-virtual-try-on-for-woocommerce'),
                'url' => RWPVTO_PLUGIN_URL . 'assets/models/A9beKNJDNEGORgmn.jpg',
                'gender' => 'male',
                'category' => 'upper_body,lower_body',
            ),
            array(
                'name' => __('Male (Full Body, Gray Pants)', 'ai-virtual-try-on-for-woocommerce'),
                'url' => RWPVTO_PLUGIN_URL . 'assets/models/DsyK3y17kUBeTwvk.jpg',
                'gender' => 'male',
                'category' => 'upper_body,lower_body',
            ),
            array(
                'name' => __('Male (Full Body, White Shirt)', 'ai-virtual-try-on-for-woocommerce'),
                'url' => RWPVTO_PLUGIN_URL . 'assets/models/GlfyxdCgg8I18N8m.jpg',
                'gender' => 'male',
                'category' => 'upper_body,lower_body',
            ),
            array(
                'name' => __('Male (Full Body, Black Jacket)', 'ai-virtual-try-on-for-woocommerce'),
                'url' => RWPVTO_PLUGIN_URL . 'assets/models/HjRsXFZ7zZvdDqQN.jpg',
                'gender' => 'male',
                'category' => 'upper_body,lower_body',
            ),
            array(
                'name' => __('Male (Full Body, Blue Jeans)', 'ai-virtual-try-on-for-woocommerce'),
                'url' => RWPVTO_PLUGIN_URL . 'assets/models/iiLNWsndtbTQVSig.jpg',
                'gender' => 'male',
                'category' => 'upper_body,lower_body',
            ),
            array(
                'name' => __('Male (Upper Body, Yellow T-Shirt)', 'ai-virtual-try-on-for-woocommerce'),
                'url' => RWPVTO_PLUGIN_URL . 'assets/models/PGYIk30J845hOkxD.jpg',
                'gender' => 'male',
                'category' => 'upper_body',
            ),
            // Female models
            array(
                'name' => __('Female (Upper Body, T-Shirt)', 'ai-virtual-try-on-for-woocommerce'),
                'url' => RWPVTO_PLUGIN_URL . 'assets/models/sjAGaMtcLkaRJ945.jpg',
                'gender' => 'female',
                'category' => 'upper_body',
            ),
            array(
                'name' => __('Female (Full Body, Navy Dress)', 'ai-virtual-try-on-for-woocommerce'),
                'url' => RWPVTO_PLUGIN_URL . 'assets/models/ouo5PQIK7L8W60RJ.jpg',
                'gender' => 'female',
                'category' => 'dresses',
            ),
            array(
                'name' => __('Female (Full Body, Pink Dress)', 'ai-virtual-try-on-for-woocommerce'),
                'url' => RWPVTO_PLUGIN_URL . 'assets/models/TNZm6epsIZd4HTuR.jpg',
                'gender' => 'female',
                'category' => 'dresses',
            ),
            array(
                'name' => __('Female (Upper Body, Red Dress)', 'ai-virtual-try-on-for-woocommerce'),
                'url' => RWPVTO_PLUGIN_URL . 'assets/models/LJsDVxbUixV1Xr6s.jpg',
                'gender' => 'female',
                'category' => 'dresses',
            ),
            array(
                'name' => __('Female (Full Body, Blue Pants)', 'ai-virtual-try-on-for-woocommerce'),
                'url' => RWPVTO_PLUGIN_URL . 'assets/models/0ORrdEmqBOKanPmR.jpg',
                'gender' => 'female',
                'category' => 'upper_body,lower_body',
            ),
            array(
                'name' => __('Female (Full Body, Sky Blue Pants)', 'ai-virtual-try-on-for-woocommerce'),
                'url' => RWPVTO_PLUGIN_URL . 'assets/models/syq7rmBh1g0kM6vX.jpg',
                'gender' => 'female',
                'category' => 'upper_body,lower_body',
            ),
            array(
                'name' => __('Female (Upper Body, Black Hoodie)', 'ai-virtual-try-on-for-woocommerce'),
                'url' => RWPVTO_PLUGIN_URL . 'assets/models/tAAMXycOo1e11nYx.jpg',
                'gender' => 'female',
                'category' => 'upper_body',
            ),
            array(
                'name' => __('Female (Upper Body, Hoodie with Hat)', 'ai-virtual-try-on-for-woocommerce'),
                'url' => RWPVTO_PLUGIN_URL . 'assets/models/NGvv6N61QXpNPR9r.jpg',
                'gender' => 'female',
                'category' => 'upper_body',
            ),
            array(
                'name' => __('Female (Upper Body, Pink Hoodie)', 'ai-virtual-try-on-for-woocommerce'),
                'url' => RWPVTO_PLUGIN_URL . 'assets/models/lc7ev5QZO3iNZVIe.jpg',
                'gender' => 'female',
                'category' => 'upper_body',
            ),
            array(
                'name' => __('Female (Upper Body, Beige Sweater)', 'ai-virtual-try-on-for-woocommerce'),
                'url' => RWPVTO_PLUGIN_URL . 'assets/models/WNgN1rH03owPB3X0.jpg',
                'gender' => 'female',
                'category' => 'upper_body',
            ),
            array(
                'name' => __('Female (Upper Body, Silver Sweater)', 'ai-virtual-try-on-for-woocommerce'),
                'url' => RWPVTO_PLUGIN_URL . 'assets/models/pPnNZJs15rcDJozC.jpg',
                'gender' => 'female',
                'category' => 'upper_body',
            ),
            array(
                'name' => __('Female (Upper Body, Light Gray Sweater)', 'ai-virtual-try-on-for-woocommerce'),
                'url' => RWPVTO_PLUGIN_URL . 'assets/models/PY0CcwefZ9WH3pbJ.jpg',
                'gender' => 'female',
                'category' => 'upper_body',
            ),
            array(
                'name' => __('Female (Upper Body, Light Gray Sweater)', 'ai-virtual-try-on-for-woocommerce'),
                'url' => RWPVTO_PLUGIN_URL . 'assets/models/iJ2yzYVolnex0oqk.jpg',
                'gender' => 'female',
                'category' => 'upper_body',
            ),
            array(
                'name' => __('Female (Upper Body, Cream Knitwear)', 'ai-virtual-try-on-for-woocommerce'),
                'url' => RWPVTO_PLUGIN_URL . 'assets/models/6FTBzY6ohWmJQjhj.jpg',
                'gender' => 'female',
                'category' => 'upper_body',
            ),
            array(
                'name' => __('Female (Full Body, Flowers Dress)', 'ai-virtual-try-on-for-woocommerce'),
                'url' => RWPVTO_PLUGIN_URL . 'assets/models/91JXDKwwXc479nlI.jpg',
                'gender' => 'female',
                'category' => 'dresses',
            ),
            array(
                'name' => __('Female (Full Body, Pink Hoodie)', 'ai-virtual-try-on-for-woocommerce'),
                'url' => RWPVTO_PLUGIN_URL . 'assets/models/NAWNmd9O4qW0ydWL.jpg',
                'gender' => 'female',
                'category' => 'upper_body,lower_body',
            ),
            array(
                'name' => __('Female (Full Body, Blue Jeans)', 'ai-virtual-try-on-for-woocommerce'),
                'url' => RWPVTO_PLUGIN_URL . 'assets/models/T69CWc4d4wPi6LxZ.jpg',
                'gender' => 'female',
                'category' => 'upper_body,lower_body',
            ),
            array(
                'name' => __('Female (Full Body, Brown Hoodie)', 'ai-virtual-try-on-for-woocommerce'),
                'url' => RWPVTO_PLUGIN_URL . 'assets/models/QIYXhaRneRHsrFAd.jpg',
                'gender' => 'female',
                'category' => 'upper_body,lower_body',
            ),
            array(
                'name' => __('Female (Full Body, White Hoodie)', 'ai-virtual-try-on-for-woocommerce'),
                'url' => RWPVTO_PLUGIN_URL . 'assets/models/5VVFgrY6NjREzOZe.jpg',
                'gender' => 'female',
                'category' => 'upper_body,lower_body',
            ),
            array(
                'name' => __('Female (Full Body, Gray Dress)', 'ai-virtual-try-on-for-woocommerce'),
                'url' => RWPVTO_PLUGIN_URL . 'assets/models/3vLLJAjHQwWXY77l.jpg',
                'gender' => 'female',
                'category' => 'dresses',
            ),
            array(
                'name' => __('Female (Full Body, Silver Trousers)', 'ai-virtual-try-on-for-woocommerce'),
                'url' => RWPVTO_PLUGIN_URL . 'assets/models/J7PAT0Jk5NSPn6xY.jpg',
                'gender' => 'female',
                'category' => 'upper_body,lower_body',
            ),
            array(
                'name' => __('Female (Full Body, Gray Dress)', 'ai-virtual-try-on-for-woocommerce'),
                'url' => RWPVTO_PLUGIN_URL . 'assets/models/46kJZEmZXyydhlQ9.jpg',
                'gender' => 'female',
                'category' => 'dresses',
            ),
            array(
                'name' => __('Female (Full Body, White T-Shirt)', 'ai-virtual-try-on-for-woocommerce'),
                'url' => RWPVTO_PLUGIN_URL . 'assets/models/hEK0vOWp5iufkVRB.jpg',
                'gender' => 'female',
                'category' => 'upper_body,lower_body',
            ),
            array(
                'name' => __('Female (Full Body, Black Mini Dress)', 'ai-virtual-try-on-for-woocommerce'),
                'url' => RWPVTO_PLUGIN_URL . 'assets/models/mUOQAsVR6Ik5MNZZ.jpg',
                'gender' => 'female',
                'category' => 'upper_body,lower_body',
            ),
            array(
                'name' => __('Female (Full Body, White Pants)', 'ai-virtual-try-on-for-woocommerce'),
                'url' => RWPVTO_PLUGIN_URL . 'assets/models/wXBrHNWlp1E4l1RA.jpg',
                'gender' => 'female',
                'category' => 'upper_body,lower_body',
            ),
            array(
                'name' => __('Female (Full Body, Black Dress)', 'ai-virtual-try-on-for-woocommerce'),
                'url' => RWPVTO_PLUGIN_URL . 'assets/models/ftRZq40D9MR3zqsV.jpg',
                'gender' => 'female',
                'category' => 'dresses',
            ),
            array(
                'name' => __('Female (Full Body, Black Pants)', 'ai-virtual-try-on-for-woocommerce'),
                'url' => RWPVTO_PLUGIN_URL . 'assets/models/F7CjgwuEVpGwonYp.jpg',
                'gender' => 'female',
                'category' => 'upper_body,lower_body',
            ),
        );
    }
}
