<?php

/**
 * The public-facing functionality of the plugin.
 *
 * This class handles loading all stylesheets and JavaScript needed for the
 * NimBot chat interface on the front-end of the WordPress site. It also
 * provides a placeholder for loading the public-facing PHP template.
 *
 * @package    ai-tool-center
 * @subpackage Aitoce/public/enqueues
 * @author     sjvision <contact@ai-tool-center.com>
 */
class Aitoce_Public
{
    private $aitoce_plugin_name;
    private $aitoce_version;

    /**
     * Constructor.
     * Stores the plugin name and version for use in asset loading.
     */
    public function __construct($aitoce_plugin_name, $aitoce_version)
    {
        $this->aitoce_plugin_name = "aitoce-{$aitoce_plugin_name}";
        $this->aitoce_version = $aitoce_version;
    }

    /**
     * Register the stylesheets for the public-facing side of the site.
     *
     * Loads all CSS needed for the NimBot interface, Google Fonts, and FontAwesome icons.
     */
    public function aitoce_enqueue_styles(): void
    {
        wp_enqueue_style("{$this->aitoce_plugin_name}-nimbot", plugin_dir_url(__FILE__) . 'css/aitoce-nimbot.css', [], $this->aitoce_version, 'all');

        // Nimbot interface code parser styles
        wp_enqueue_style("{$this->aitoce_plugin_name}-nimbot-highlight", plugin_dir_url(__FILE__) . 'css/aitoce-nimbot-googlecode.css', [], "11.11.1", 'all');

        // Enqueue Google Fonts stylesheet
        wp_enqueue_style("{$this->aitoce_plugin_name}-google-fonts", 'https://fonts.googleapis.com/css?family=Audiowide:400', [], $this->aitoce_version, false);

        // Load FontAwesome icon styles
        wp_enqueue_style("{$this->aitoce_plugin_name}-fontAwesome-main", plugin_dir_url(__FILE__) . 'css/fontAwesome-main.min.css', [], "6.7.2", 'all');
        wp_enqueue_style("{$this->aitoce_plugin_name}-fontAwesome-regular", plugin_dir_url(__FILE__) . 'css/fontAwesome-regular.min.css', [], "6.7.2", 'all');
        wp_enqueue_style("{$this->aitoce_plugin_name}-fontAwesome-solid", plugin_dir_url(__FILE__) . 'css/fontAwesome-solid.min.css', [], "6.7.2", 'all');
    }

    /**
     * Register the JavaScript for the public-facing side of the site.
     *
     * Loads all JS needed for the NimBot interface, including drag-and-drop,
     * markdown parsing, syntax highlighting, and AJAX functionality.
     * Also localizes the AJAX URL and security nonce for use in JS.
     */
    public function aitoce_enqueue_scripts(): void
    {
        // Nimbot interface draggable functionality
        wp_enqueue_script("{$this->aitoce_plugin_name}-nimbot-draggable", plugin_dir_url(__FILE__) . 'js/aitoce-public-nimbot-draggable.js', ['jquery', 'jquery-ui-draggable'], $this->aitoce_version, false);

        // Nimbot interface markdown parser
        wp_enqueue_script("{$this->aitoce_plugin_name}-nimbot-marked", plugin_dir_url(__FILE__) . 'js/aitoce-public-nimbot-marked.js', ['jquery'], '15.0.7', false);

        // Nimbot interface syntax highlighter
        wp_enqueue_script("{$this->aitoce_plugin_name}-nimbot-highlight", plugin_dir_url(__FILE__) . 'js/aitoce-public-nimbot-highlight.js', ['jquery'], '11.11.1', false);

        // Nimbot AJAX handler
        wp_enqueue_script("{$this->aitoce_plugin_name}-ajax", plugin_dir_url(__FILE__) . 'js/aitoce-public-nimbot.js', ['jquery'], $this->aitoce_version, false);

        // Setup local variable for the ajax route, name must match js script name
        wp_localize_script("{$this->aitoce_plugin_name}-ajax", 'aitoce_ai_tool_center_ajax', [
            "name" => "AI Tool Center Ajax",
            "author" => "sjvision",
            'ajax_url' => admin_url('admin-ajax.php'),
            'security' => wp_create_nonce('aitoce_ajax_nonce') // Optional for security
        ]);

    }

    /**
     * Register the PHP for the public-facing side of the site.
     *
     * Placeholder for loading the NimBot popup template on the front-end.
     * (Currently commented out; can be enabled to conditionally load the chat interface.)
     */
    public function aitoce_enqueue_footer_scripts(): void
    {
        $options = new Aitoce_Admin_Options();
        $popup_settings = $options->aitoce_get_option(AITOCE_OPTION_NAME_POPUP);

        if (!isset($popup_settings['popup_activate']) || $popup_settings['popup_activate'] !== 1) {
            return; // Popup not activated
        }

        // At this point, popup is enabled and account level is known
        $popup_agent_id = $popup_settings['popup_agent'];

        $agent = new Aitoce_Agent_Settings();

        $mysql_agent_data = $agent->aitoce_get_agent_data($popup_agent_id);
        if($mysql_agent_data){
                    
            $agent->aitoce_load_mysql_into_form($mysql_agent_data);
        }

        include plugin_dir_path(__FILE__) . 'partials/aitoce-public-nimbot-display.php';
    }
}