<?php

/**
 * The admin options utility class for the plugin.
 *
 * This class provides helper methods for retrieving options and rendering
 * various types of settings fields (text, textarea, checkbox, select, hidden)
 * in the WordPress admin area for the plugin's settings pages.
 *
 * @package    ai-tool-center
 * @subpackage Aitoce/admin/options
 * @author     sjvision <contact@ai-tool-center.com>
 */
class Aitoce_Admin_Options
{
    /**
     * Retrieve an option value or the entire option array.
     *
     * @param string $option_name The name of the option.
     * @param mixed $key Optional. The specific key within the option array to retrieve.
     * @param mixed $default Optional. The default value to return if not found.
     * @return mixed The option value or array.
     */
    public function aitoce_get_option(string $option_name, $key = null, $default = '')
    {
        // Return either the entire option array or a single option value
        $options = get_option($option_name, []);
        if ($key === null)
            return $options ?? $default;
        else
            return $options[$key] ?? $default;
    }

    /**
     * Add a text input field to a settings section.
     *
     * @param string $id Field ID.
     * @param string $label Field label.
     * @param string $option_name Option group name.
     * @param string $option_slug Settings page slug.
     * @param string $option_section Section ID.
     * @param string $description Optional. Field description.
     */
    public function aitoce_add_text_field(string $id, string $label, string $placeholder, string $option_name, string $option_slug, string $option_section, string $description = '')
    {
        add_settings_field(
            $id,
            $label,
            function () use ($id, $option_name, $placeholder, $label, $description) {
                $value = esc_attr($this->aitoce_get_option($option_name, $id));
                printf(
                    '<input type="text" name="%s[%s]" value="%s" class="regular-text" placeholder="%s"  aria-label="%s" /><br /><div class="aitoce-description">%s</div>',
                    esc_attr($option_name),
                    esc_attr($id),
                    esc_attr($value),
                    esc_attr($placeholder),
                    esc_attr($label),
                    wp_kses_post($description)
                );
            },
            esc_attr($option_slug),
            esc_attr($option_section)
        );
    }

    /**
     * Add a textarea field to a settings section.
     *
     * @param string $id Field ID.
     * @param string $label Field label.
     * @param int $rows Number of rows for the textarea.
     * @param string $option_name Option group name.
     * @param string $option_slug Settings page slug.
     * @param string $option_section Section ID.
     * @param string $description Optional. Field description.
     */
    public function aitoce_add_textarea_field(string $id, string $label, string $placeholder, int $rows, string $option_name, string $option_slug, string $option_section, string $description = '')
    {
        add_settings_field(
            $id,
            $label,
            function () use ($id, $option_name, $placeholder, $rows, $label, $description) {
                $value = $this->aitoce_get_option($option_name, $id);
                printf(
                    '<textarea name="%s[%s]" class="regular-text" rows="%d" aria-label="%s" placeholder="%s">%s</textarea><br /><div class="aitoce-description">%s</div>',
                    esc_attr($option_name),
                    esc_attr($id),
                    esc_attr($rows),
                    esc_attr($label),
                    esc_attr($placeholder),
                    esc_attr($value),
                    wp_kses_post($description)
                );
            },
            esc_attr($option_slug),
            esc_attr($option_section)
        );
    }

    /**
     * Add a checkbox field to a settings section.
     *
     * @param string $id Field ID.
     * @param string $label Field label.
     * @param string $option_name Option group name.
     * @param string $option_slug Settings page slug.
     * @param string $option_section Section ID.
     * @param string $description Optional. Field description.
     */
    public function aitoce_add_checkbox_field(string $id, string $label, string $option_name, string $option_slug, string $option_section, string $description = '')
    {
        add_settings_field(
            $id,
            $label,
            function () use ($id, $option_name, $label, $description) {
                $checked = checked(1, (int) $this->aitoce_get_option($option_name, $id), false);
                printf(
                    '<label class="switch"><input type="checkbox" id="%s[%s]" name="%s[%s]" %s /><span class="slider round"></span></label><br /><div class="aitoce-description">%s</div>',
                    esc_attr($option_name),
                    esc_attr($id),
                    esc_attr($option_name),
                    esc_attr($id),
                    esc_attr($checked),
                    wp_kses_post($description)
                );
            },
            esc_attr($option_slug),
            esc_attr($option_section)
        );
    }

    /**
     * Add a select dropdown field to a settings section.
     *
     * @param string $id Field ID.
     * @param string $label Field label.
     * @param array $options Array of value => label pairs for the dropdown.
     * @param string $option_name Option group name.
     * @param string $option_slug Settings page slug.
     * @param string $option_section Section ID.
     * @param string $description Optional. Field description.
     */
    public function aitoce_add_select_field(string $id, string $aria_label, array $options, string $option_name, string $option_slug, string $option_section, string $description = '', string $default = '')
    {
        add_settings_field(
            $id,
            $aria_label,
            function () use ($id, $option_name, $options, $aria_label, $description, $default) {
                $selected = esc_attr($this->aitoce_get_option($option_name, $id));
                if(empty($selected) && !empty($default)){$selected = $default;}
                echo '<div class="aitoce-select-box">';
                printf('<select name="%s[%s]" aria-label="%s" >', esc_attr($option_name), esc_attr($id), esc_attr($aria_label));
                echo '<option>Please Select</option>';
                foreach ($options as $value => $label) {
                    printf(
                        '<option value="%s"%s>%s</option>',
                        esc_attr($value),
                        selected($selected, $value, false),
                        esc_html($label)
                    );
                }
                printf(
                    '</select></div><br /><div class="aitoce-description">%s</div>',
                    wp_kses_post($description)
                );
            },
            esc_attr($option_slug),
            esc_attr($option_section)
        );
    }

    /**
     * Add a hidden field to a settings section.
     *
     * @param string $id Field ID.
     * @param string $value Field value.
     * @param string $option_name Option group name.
     * @param string $option_slug Settings page slug.
     * @param string $option_section Section ID.
     */
    public function aitoce_add_hidden_field(string $id, string $value, string $option_name, string $option_slug, string $option_section)
    {
        add_settings_field(
            $id,
            '',
            function () use ($id, $option_name, $value) {
                printf(
                    '<input type="hidden" name="%s[%s]" value="%s" aria-label="%s" >',
                    esc_attr($option_name),
                    esc_attr($id),
                    esc_attr($value),
                    esc_attr($id)
                );
            },
            esc_attr($option_slug),
            esc_attr($option_section)
        );
    }

    /**
     * Add a text input field to a settings section.
     *
     * @param string $id Field ID.
     * @param string $label Field label.
     * @param string $option_name Option group name.
     * @param string $option_slug Settings page slug.
     * @param string $option_section Section ID.
     * @param string $description Optional. Field description.
     */
    public function aitoce_add_color_picker_field(string $id, string $label, string $default_color, string $option_name, string $option_slug, string $option_section, string $description = '')
    {
        add_settings_field(
            $id,
            $label,
            function () use ($id, $option_name, $label, $default_color, $description) {
                $value = esc_attr($this->aitoce_get_option($option_name, $id));
                if (empty($value)) {
                    $value = $default_color;
                }

                printf(
                    '<input class="my-color-field" type="text" name="%s[%s]" value="%s" aria-label="%s" data-default-color="%s" /><br /><div class="aitoce-description">%s</div>',
                    esc_attr($option_name),
                    esc_attr($id),
                    esc_attr($value),
                    esc_attr($label),
                    esc_attr($default_color),
                    wp_kses_post($description)
                );
            },
            esc_attr($option_slug),
            esc_attr($option_section)
        );
    }

    /**
     * Get the current HTTP host as a referer string.
     *
     * @return string The sanitized HTTP host or a fallback message.
     */
    public function aitoce_get_referer(): string
    {
        return !empty($_SERVER['HTTP_HOST'])
            ? sanitize_text_field(wp_unslash($_SERVER['HTTP_HOST']))
            : esc_html__('Host name not found! Please contact your server support.', 'ai-tool-center');
    }

    /**
     * Show an admin notice message in the WordPress dashboard.
     *
     * @param string $aitoce_message The message to display.
     * @param string $aitoce_type    Notice type: 'success', 'error', 'warning', 'info'.
     */
    public function aitoce_admin_notice(string $aitoce_message, string $aitoce_type = 'info'): void
    {
        echo '<div class="notice notice-' . esc_attr($aitoce_type) . ' is-dismissible"><p>' . wp_kses_post($aitoce_message) . '</p></div>';
    }
}