<?php
/**
 * The API integration functionality of the plugin.
 *
 * This class handles communication with the AI Tool Center backend API,
 * including account and message endpoints, and provides helper methods
 * for making authenticated API requests from the WordPress admin.
 *
 * @package    ai-tool-center
 * @subpackage Aitoce/admin-menu-nimbot-account
 * @author     sjvision <contact@ai-tool-center.com>
 */
class Aitoce_Api
{
    /**
     * Stores an instance of the options handler for retrieving plugin settings.
     * @var Aitoce_Admin_Options
     */
    public $options;

    /**
     * Constructor.
     * Initializes the options handler object.
     */
    public function __construct()
    {
        $this->options = new Aitoce_Admin_Options();
    }

    /**
     * Make an authenticated API call to the specified endpoint.
     *
     * Prepares headers (including API key and referer), sends a POST request,
     * and returns the decoded response or a default error structure.
     *
     * @param string $api_url The API endpoint URL.
     * @param array $args Optional. Additional arguments for the request.
     * @return mixed The decoded API response or error array.
     */
    private function aitoce_api_call($api_url, $args = [])
    {
        $api_key = esc_attr($this->options->aitoce_get_option(AITOCE_OPTION_NAME_ACCOUNT, 'api_key'));
        $referer = $this->options->aitoce_get_referer();

        $args['method'] = 'POST';
        $args['headers']['Content-Type'] = 'application/json';
        $args['headers']['Authorization'] = esc_attr($api_key);
        $args['headers']['Referer'] = esc_attr($referer);

        $response = wp_remote_get($api_url, $args);

        if (is_wp_error($response)) {
            // Return a default error structure if the API call fails
            return [
                'label' => 'none',
                'level' => 0,
                'status' => $api_url
            ];
        } else {
            // Decode and return the API response
            $body = wp_remote_retrieve_body($response);
            $data = json_decode($body, true);
            return $data;
        }
    }

    /**
     * Retrieve account data from the AI Tool Center API.
     *
     * Calls the /account/ endpoint and returns account info such as label, level, and status.
     *
     * @param array $args Optional. Additional request arguments.
     * @return mixed Account data array or error structure.
     */
    public function aitoce_get_account($args = []): mixed
    {
        $api_url = 'https://ai-tool-center.com/wp-content/plugins/api-responder/api/v1/account/';
        $account_data = $this->aitoce_api_call($api_url, $args);
        return $account_data;
    }

    /**
     * Retrieve message data from the AI Tool Center API.
     *
     * Calls the /message/ endpoint to send or receive chatbot messages.
     *
     * @param array $args Optional. Additional request arguments.
     * @return mixed Message data array or error structure.
     */
    public function aitoce_get_message($args = []): mixed
    {
        $api_url = 'https://ai-tool-center.com/wp-content/plugins/api-responder/api/v1/message/';
        $message_data = $this->aitoce_api_call($api_url, $args);
        return $message_data;
    }

    /**
     * Send an error message as a JSON error response.
     *
     * Used for AJAX or API error handling in the plugin.
     *
     * @param string $aitoce_message The error message.
     * @param string $aitoce_code    The error code.
     */
    private function aitoce_send_error_message(string $aitoce_message, string $aitoce_code): void
    {
        wp_send_json_error([
            'message' => sprintf('%s - Code: %s', $aitoce_message, $aitoce_code),
            'assistant' => 'Assistant Offline!',
        ]);
    }
}
