<?php

/**
 * The ajax requests functionality of the plugin.
 *
 * Handles AJAX requests for the Nimbot assistant, including retrieving user inputs,
 * sanitizing data, and sending API requests to the backend.
 *
 * @package    ai-tool-center
 * @subpackage Aitoce/includes
 * @author     sjvision <contact@ai-tool-center.com>
 */
class Aitoce_Ajax
{
    private $api_call;
    private $options;
    private $agent;

    /**
     * Constructor.
     * Initializes API and options handler objects.
     */
    public function __construct()
    {
        $this->api_call = new Aitoce_Api();
        $this->options = new Aitoce_Admin_Options();
        $this->agent = new Aitoce_Agent_Settings();
    }

    /**
     * Handles the main AJAX request for the Nimbot assistant.
     *
     * - Checks the AJAX nonce for security.
     * - Validates and sanitizes required input fields.
     * - Retrieves plugin settings for popup, page, and model.
     * - Builds the message body for the API request.
     * - If using a custom LLM, adds those settings to the request.
     * - Sends the request to the backend API and returns the response as JSON.
     */
    public function aitoce_ajax_request()
    {
        check_ajax_referer('aitoce_ajax_nonce', 'aitoce_security');

        // Define required fields and their error codes
        $required_fields = [
            'aitoce_agent_id' => '0x80001001',
            'aitoce_message_id' => '0x80001002',
            'aitoce_last_prompt' => '0x80001003',
            'aitoce_prompt_format' => '0x80001004',
        ];

        // Validate and sanitize required fields
        $inputs = [];
        foreach ($required_fields as $field => $code) {
            if (empty($_REQUEST[$field])) {
                $this->aitoce_send_error_message("Missing field: $field", $code);
            }
            $inputs[$field] = sanitize_text_field(wp_unslash($_REQUEST[$field]));
        }

        $account_membership = $this->api_call->aitoce_get_account();

        if($account_membership['label'] === AITOCE_ACCOUNT_LEVELS[1]){
            $agent_reply[] = $this->agent->aitoce_get_level_2_agents($inputs['aitoce_agent_id']);
        }elseif($account_membership['label'] === AITOCE_ACCOUNT_LEVELS[2]){
            $agent_reply[] = $this->agent->aitoce_get_level_3_agents($inputs['aitoce_agent_id']);
        }else{
            $agent_reply[] = $this->agent->aitoce_get_level_1_agents();
        }

        if(!isset($agent_reply[0]['error']))
        {
            $this->agent->aitoce_load_mysql_into_form($agent_reply);
        }else{
            $this->aitoce_send_error_message($agent_reply[0]['error'], $agent_reply[0]['code']);
        }

        // Extract correct Model
        if($this->agent->connection['aitoce_agent_provider'] === 'openai'){
            $config = 'input';
            $model = $this->agent->connection['aitoce_agent_openai_model'];

        }elseif($this->agent->connection['aitoce_agent_provider'] === 'claude'){
            $config = 'messages';
            $model = $this->agent->connection['aitoce_agent_openai_model'];

        }else{
            if($this->agent->connection['aitoce_agent_custom_interface'] === 'openai'){
                $config = 'input';

            }elseif($this->agent->connection['aitoce_agent_custom_interface'] === 'claude'){
                $config = 'input';

            }elseif($this->agent->connection['aitoce_agent_custom_interface'] === 'open_webui'){
                $config = 'messages';

            }else{
                $config = 'messages';

            }
                
            $model = $this->agent->connection['aitoce_agent_custom_model'];

        }

        // Build the message body for the API request
        $message_body = [
            'message' => [
                'assistant_role' => [
                    'role'      => 'system',
                    'content'   => esc_attr($this->agent->rag['aitoce_agent_role']),
                ],
                'message' => [
                    'role'      => 'user',
                    'content'   => $inputs['aitoce_last_prompt'],
                ],
                'temperature'       => 0.8,
                'message_id'        => esc_attr($inputs['aitoce_message_id']),
                'history_count'     => esc_attr($this->agent->rag['aitoce_agent_retention']),
                'transcript_count'  => esc_attr($this->agent->rag['aitoce_agent_transcript']),
                'page_inclusion'    => esc_attr($this->agent->rag['aitoce_agent_pages']),
                'post_inclusion'    => esc_attr($this->agent->rag['aitoce_agent_posts']),
            ],
            'llm' => [
                'aitoce_llm'        => 1,
                'aitoce_llm_config' => esc_attr($config),
                'aitoce_api_url'    => esc_url($this->agent->connection['aitoce_agent_url']),
                'aitoce_api_key'    => esc_attr($this->agent->connection['aitoce_agent_api_key']),
                'aitoce_model'      => esc_attr($model),
            ]
        ];

        // Prepare arguments for the API call
        $args = [
            'body' => wp_json_encode($message_body),
            'timeout' => 240,
        ];

        // Send the request to the backend API and get the response
        $aitoce_response_data = $this->api_call->aitoce_get_message($args);

        // Return the API response as a JSON success response
        wp_send_json_success([
            'success' => $aitoce_response_data['success'],
            'message' => $aitoce_response_data['message'],
            'assistant' => $aitoce_response_data['assistant'],
        ]);
    }

    /**
     * Send an error message as a JSON error response.
     *
     * @param string $aitoce_message The error message.
     * @param string $aitoce_code    The error code.
     */
    private function aitoce_send_error_message(string $aitoce_message, string $aitoce_code): void
    {
        wp_send_json_error([
            'message' => sprintf('%s - Code: %s', $aitoce_message, $aitoce_code),
            'assistant' => 'Assistant Offline!',
        ]);
    }

}