<?php

/**
 * The admin-specific functionality for the Settings settings page.
 *
 * This class manages the admin menu, settings registration, and rendering for the Settings configuration page.
 * It allows users to configure and save their own LLM provider settings (such as OpenAI or CustomGPT.ai).
 *
 * @package    ai-tool-center
 * @subpackage Aitoce/admin/settings-page
 * @author     sjvision <contact@ai-tool-center.com>
 */

class Aitoce_Admin_Settings_Page
{
    /**
     * Handles admin notifications.
     * @var Aitoce_Admin_Notifications
     */
    private $notify;

    /**
     * Handles options and field rendering.
     * @var Aitoce_Admin_Options
     */
    private $options;

    /**
     * Constructor.
     * Initializes notification and options handler objects.
     */
    public function __construct()
    {
        $this->notify = new Aitoce_Admin_Notifications();
        $this->options = new Aitoce_Admin_Options();
    }

    /**
     * Adds the Settings submenu page to the WordPress admin menu.
     */
    public function aitoce_add_settings_menu(): void
    {
        add_submenu_page(
            parent_slug: AITOCE_MENU_SLUG,
            page_title: __('Settings', 'ai-tool-center'),
            menu_title: __('Settings', 'ai-tool-center'),
            capability: 'manage_options',
            menu_slug: AITOCE_MENU_SLUG_SETTINGS,
            callback: [$this, 'aitoce_render_settings_page']
        );
    }

    /**
     * Renders the Settings admin page.
     * Displays admin errors and includes the settings page template.
     */
    public function aitoce_render_settings_page(): void
    {
        $this->notify->aitoce_display_admin_errors();
        include plugin_dir_path(__FILE__) . './aitoce-settings-display.php';
    }

    /**
     * Registers settings, sections, and fields for the Settings settings page.
     * Adds fields for using a custom LLM provider, base URL, API key, and settings selection.
     */
    public function aitoce_register_settings_settings(): void
    {
        $agent = new Aitoce_Agent_Settings();

        // Register the PopUp settings group and sanitize callback
        register_setting(
            AITOCE_OPTION_SLUG_POPUP,
            AITOCE_OPTION_NAME_POPUP,
            ['sanitize_callback' => [$this, 'aitoce_sanitize_popup']]
        );

        // Add the main settings section
        add_settings_section(
            AITOCE_OPTION_SECTION_POPUP,
            __('NimBot PopUp', 'ai-tool-center'),
            '__return_false',
            AITOCE_OPTION_SLUG_POPUP
        );

        // Checkbox to activate the popup chatbot
        $this->options->aitoce_add_checkbox_field(
            'popup_activate',
            __('Activate PopUp', 'ai-tool-center'),
            esc_attr(AITOCE_OPTION_NAME_POPUP),
            esc_attr(AITOCE_OPTION_SLUG_POPUP),
            esc_attr(AITOCE_OPTION_SECTION_POPUP),
            'Check this if you want to activate NimBot PopUp Chatbot on the front-end.'
        );

        $mysql_agent_data = $agent->aitoce_get_agent_data('ALL');

        if(count($mysql_agent_data) >= 1){
            foreach ($mysql_agent_data as $key => $value) {

                // Get Model display name
                if($value['agent_provider'] === 'openai'){
                    $model_name = $agent->openai_models[$value['agent_model']]['name'];
                }elseif($value['agent_provider'] === 'claude'){
                    $model_name = $agent->claude_models[$value['agent_model']]['name'];
                }else{
                    $model_name = $value['agent_model'];
                }

                $agents[$value['agent_id']] = 'ID:'.$value['agent_id'].' - ' .  $value['agent_name'].' - ' .  $model_name;
            }

        }else{
            // No Agents
            $agents[] = 'No Agents';
            
        }
        
        // Select the PopUp default PopUP
        $this->options->aitoce_add_select_field(
            'popup_agent',
            __('PopUp Agent', 'ai-tool-center'),
            $agents,
            esc_attr(AITOCE_OPTION_NAME_POPUP),
            esc_attr(AITOCE_OPTION_SLUG_POPUP),
            esc_attr(AITOCE_OPTION_SECTION_POPUP),
            'Select the Default Agent for the NimBot PopUp when activated.<br>If an Agent has been selected when creating a post/page, the post/page selected Agent will <strong class="aitoce-red">override</strong> this Default Agent.',
        );


        // Register the Database settings group and sanitize callback
        register_setting(
            AITOCE_OPTION_SLUG_DATABASE,
            AITOCE_OPTION_NAME_DATABASE,
            ['sanitize_callback' => [$this, 'aitoce_sanitize_database']]
        );

        // Add the main database settings section
        add_settings_section(
            AITOCE_OPTION_SECTION_DATABASE,
            __('Database Settings', 'ai-tool-center'),
            [$this, 'aitoce_display_database_section'],
            AITOCE_OPTION_SLUG_DATABASE
        );

        // Checkbox to enable using a custom LLM provider
        $this->options->aitoce_add_checkbox_field(
            'delete_account_options',
            __('Delete Account Settings', 'ai-tool-center'),
            esc_attr(AITOCE_OPTION_NAME_DATABASE),
            esc_attr(AITOCE_OPTION_SLUG_DATABASE),
            esc_attr(AITOCE_OPTION_SECTION_DATABASE),
        );

        // Checkbox to enable using a custom LLM provider
        $this->options->aitoce_add_checkbox_field(
            'delete_agents_options',
            __('Delete Agents', 'ai-tool-center'),
            esc_attr(AITOCE_OPTION_NAME_DATABASE),
            esc_attr(AITOCE_OPTION_SLUG_DATABASE),
            esc_attr(AITOCE_OPTION_SECTION_DATABASE),
        );

        // Checkbox to enable using a custom LLM provider
        $this->options->aitoce_add_checkbox_field(
            'delete_popup_options',
            __('Delete PopUp Settings', 'ai-tool-center'),
            esc_attr(AITOCE_OPTION_NAME_DATABASE),
            esc_attr(AITOCE_OPTION_SLUG_DATABASE),
            esc_attr(AITOCE_OPTION_SECTION_DATABASE),
        );
    }

    public function aitoce_display_database_section(){
        echo '
            <table class="form-table" role="presentation">
                <tr>
                    <td><p>Check the settings you want to <span class="aitoce-red"><strong>DELETE</strong></span>, this will delete all <span class="aitoce-red"><strong>SETTINGS/OPTIONS</strong></span> and <span class="aitoce-red"><strong>VALUES</strong></span> on plugin <span class="aitoce-red"><strong>DEACTIVATION</strong></span>.</p></td>
                </tr>
            </table>';

    }

    /**
     * Sanitizes the settings input for the Settings options.
     *
     * @param array $input The input array from the settings form.
     * @return array Sanitized settings.
     */
    public function aitoce_sanitize_popup($input): array
    {
        return [
            'popup_activate' => isset($input['popup_activate']) ? 1 : 0,
            'popup_agent' => absint($input['popup_agent'] ?? ''),
        ];
    }
    public function aitoce_sanitize_database($input): array
    {
        return [
            'delete_account_options' => isset($input['delete_account_options']) ? 1 : 0,
            'delete_agents_options' => isset($input['delete_agents_options']) ? 1 : 0,
            'delete_popup_options' => isset($input['delete_popup_options']) ? 1 : 0,
        ];
    }

}
