<?php

/**
 * The admin-specific functionality for the Agents settings page.
 *
 * This class manages the admin menu, settings registration, and rendering for the Agents configuration page.
 * It allows users to configure and save their own LLM provider settings (such as OpenAI or CustomGPT.ai).
 *
 * @package    ai-tool-center
 * @subpackage Aitoce/admin/agents-page
 * @author     sjvision <contact@ai-tool-center.com>
 */

class Aitoce_Agents_Page extends Aitoce_Agent_Settings
{
    /**
     * Handles admin notifications.
     * @var Aitoce_Admin_Notifications
     */
    private $notify;

    /**
     * Handles admin notifications.
     * @var Aitoce_Agent
     */
    private $agent;

    public $account_membership;

    /**
     * Constructor.
     * Initializes notification and options handler objects.
     */
    public function __construct()
    {
        $this->notify = new Aitoce_Admin_Notifications();

    }

    public function aitoce_add_agents_menu(): void
    {
        global $nimbot_agents_page;

        $nimbot_agents_page = add_submenu_page(
            parent_slug: AITOCE_MENU_SLUG,
            page_title: __('Agents', 'ai-tool-center'),
            menu_title: __('Agents', 'ai-tool-center'),
            capability: 'manage_options',
            menu_slug: AITOCE_MENU_SLUG_AGENTS,
            callback: [$this, 'aitoce_render_agent_page']
        );

        add_action("load-$nimbot_agents_page", [$this, "aitoce_agent_list_screen_options"]);

    }

    // add screen options
    public function aitoce_agent_list_screen_options($submitted = '')
    {
        // phpcs:disable WordPress.Security.NonceVerification
        if (!empty($submitted) || (!isset($_GET['action']) || ($_GET['action'] !== 'add_agent' && $_GET['action'] !== 'edit_agent'))) {
            global $nimbot_agents_page;
            global $table;

            $screen = get_current_screen();

            // get out of here if we are not on our settings page
            if (!is_object($screen) || $screen->id != $nimbot_agents_page)
                return;

            $args = [
                'label' => __('Elements per page', 'ai-tool-center'),
                'default' => 5,
                'option' => 'elements_per_page'
            ];
            add_screen_option('per_page', $args);

            $table = new Aitoce_Agent();
        }
    }

    // add screen options
    public function aitoce_render_agent_page()
    {
        // phpcs:disable PluginCheck.CodeAnalysis.ImageFunctions.NonEnqueuedImage
        // phpcs:disable WordPress.DB
        
        global $wpdb;
        
        $api_call = new Aitoce_Api;
        $account_membership = $api_call->aitoce_get_account();

        $this->notify->aitoce_display_admin_errors();
        // Creating an instance
        $args = [
            'plural'   => 'Agents',
			'singular' => 'Agent',
			'ajax'     => false,
			'screen'   => null,
        ];

        $table = new Aitoce_Agent($args);
        if (isset($_GET['action']) && $_GET['action'] === 'add_agent') {

            if(isset($_POST['submit'])){

                // Check and serialize inputs
                $clean_data = $this->aitoce_sanitize_post_data($account_membership);

                // If errors display them and display add form
                if(!$clean_data){

                    $this->aitoce_display_agent_add($account_membership);

                }else{
                    // No errors enter into database
                    // Get correct Model
                    switch ($this->connection['aitoce_agent_provider']) {
                        case 'claude':
                            $agent_interface = $this->connection['aitoce_agent_claude_interface'];
                            $agent_model = $this->connection['aitoce_agent_claude_model'];
                            $agent_url = $this->claude_models[$agent_model]['url'];
                            break;
                        
                        case 'custom':
                            $agent_interface = $this->connection['aitoce_agent_custom_interface'];
                            $agent_model = $this->connection['aitoce_agent_custom_model'];
                            $agent_url = $this->connection['aitoce_agent_url'];
                            break;
                        
                        default:
                            $agent_interface = $this->connection['aitoce_agent_openai_interface'];
                            $agent_model = $this->connection['aitoce_agent_openai_model'];
                            $agent_url = $this->openai_models[$agent_model]['url'];
                            break;
                    }

                    // Table name with WP prefix
                    $table_name = esc_sql($wpdb->prefix . AITOCE_AGENT_TABLE);

                    $inserted = $wpdb->insert(
                        $table_name,
                        [
                            'agent_name' => $this->connection['aitoce_agent_name'],
                            'agent_description'         => $this->connection['aitoce_agent_description'],
                            'agent_provider'            => $this->connection['aitoce_agent_provider'],
                            'agent_custom_interface'    => $agent_interface,
                            'agent_url'                 => $agent_url,
                            'agent_api_key'             => $this->connection['aitoce_agent_api_key'],
                            'agent_model'               => $agent_model,
                            'agent_role'                => $this->rag['aitoce_agent_role'],
                            'agent_retention'           => $this->rag['aitoce_agent_retention'],
                            'agent_transcript'          => $this->rag['aitoce_agent_transcript'],
                            'agent_pages'               => $this->rag['aitoce_agent_pages'],
                            'agent_posts'               => $this->rag['aitoce_agent_posts'],
                            'agent_styling'             => wp_json_encode($this->styling),
                            'created_at'                => current_time('mysql', 1)
                        ],
                        [
                            '%s',  // agent_name
                            '%s',  // agent_description
                            '%s',  // agent_provider
                            '%s',  // agent_custom_interface
                            '%s',  // agent_url
                            '%s',  // agent_api_key
                            '%s',  // agent_model
                            '%s',  // agent_role
                            '%s',  // agent_retention
                            '%s',  // agent_transcript
                            '%s',  // agent_pages
                            '%s',  // agent_posts
                            '%s',  // agent_styling
                            '%s',  // created_at
                        ]
                    );

                    // If successful return to agent list else add agent form
                    if($inserted){
                        $type = 'success';
                        $message = __( 'Added new Agent.', 'ai-tool-center' );

                        add_settings_error(
                            'aitoce-admin-notice',
                            esc_attr( 'settings_updated' ),
                            $message,
                            $type
                        );
                    }else{
                        $type = 'error';
                        $message = __( 'Error inserting data into database!', 'ai-tool-center' );

                        add_settings_error(
                            'aitoce-admin-notice',
                            esc_attr( 'settings_updated' ),
                            $message,
                            $type
                        );
                    }

                    $this->aitoce_display_agent_add($account_membership);
                }

            }else{

                $this->aitoce_display_agent_add($account_membership);

            }

        } elseif (isset($_GET['action']) && $_GET['action'] === 'edit_agent') {
            
            // Check nonce
            if (!isset($_GET['_wpnonce']) || !wp_verify_nonce(sanitize_text_field(wp_unslash($_GET['_wpnonce'])), 'edit_agent_wpnonce')) {
                return;
            }
            $agent_id = isset($_GET['element']) ? absint($_GET['element'] ?? ''):'';

            if(isset($_POST['submit'])){

                // Check and serialize inputs
                $clean_data = $this->aitoce_sanitize_post_data($account_membership);
                
                // If errors display them and display add form
                if(!$clean_data){

                    $this->aitoce_display_agent_edit($account_membership);

                }else{
                    // Get correct Model
                    switch ($this->connection['aitoce_agent_provider']) {
                        case 'claude':
                            $agent_interface = $this->connection['aitoce_agent_claude_interface'];
                            $agent_model = $this->connection['aitoce_agent_claude_model'];
                            $agent_url = $this->claude_models[$agent_model]['url'];
                            break;
                        
                        case 'custom':
                            $agent_interface = $this->connection['aitoce_agent_custom_interface'];
                            $agent_model = $this->connection['aitoce_agent_custom_model'];
                            $agent_url = $this->connection['aitoce_agent_url'];
                            break;
                        
                        default:
                            $agent_interface = $this->connection['aitoce_agent_openai_interface'];
                            $agent_model = $this->connection['aitoce_agent_openai_model'];
                            $agent_url = $this->openai_models[$agent_model]['url'];
                            break;
                    }

                    // Table name with WP prefix
                    $table_name = esc_sql($wpdb->prefix . AITOCE_AGENT_TABLE);
                    
                    // Update agent in database
                    $inserted = $wpdb->update(
                        $table_name,
                        [
                            'agent_name' => $this->connection['aitoce_agent_name'],
                            'agent_description' => $this->connection['aitoce_agent_description'],
                            'agent_provider' => $this->connection['aitoce_agent_provider'],
                            'agent_custom_interface' => $agent_interface,
                            'agent_url' => $agent_url,
                            'agent_api_key' => $this->connection['aitoce_agent_api_key'],
                            'agent_model' => $agent_model,
                            'agent_role' => $this->rag['aitoce_agent_role'],
                            'agent_retention' => $this->rag['aitoce_agent_retention'],
                            'agent_transcript' => $this->rag['aitoce_agent_transcript'],
                            'agent_pages' => $this->rag['aitoce_agent_pages'],
                            'agent_posts' => $this->rag['aitoce_agent_posts'],
                            'agent_styling' => wp_json_encode($this->styling)
                        ],
                        ['agent_id' => $agent_id],
                        [
                            '%s',  // agent_name
                            '%s',  // agent_description
                            '%s',  // agent_provider
                            '%s',  // agent_custom_interface
                            '%s',  // agent_url
                            '%s',  // agent_api_key
                            '%s',  // agent_model
                            '%s',  // agent_role
                            '%s',  // agent_retention
                            '%s',  // agent_transcript
                            '%s',  // agent_pages
                            '%s',  // agent_posts
                            '%s',  // agent_styling
                            '%s',  // created_at
                        ]
                    );

                    $type = 'success';
                    $message = 'Agent ID:<span class="aitoce-red">'.esc_attr($agent_id).'</span> Updated.';

                    add_settings_error(
                        'aitoce-admin-notice',
                        esc_attr( 'settings_updated' ),
                        $message,
                        $type
                    );

                    $this->aitoce_display_agent_edit($account_membership);

                }
            }else{

                // Load Agent data from database
                $mysql_agent_data = $this->aitoce_get_agent_data($agent_id);
                // Output Agent data into form
                if($mysql_agent_data){
                    
                    $this->aitoce_load_mysql_into_form($mysql_agent_data);
                    $this->aitoce_display_agent_edit($account_membership);
                }

            }

        } else {

            $table->aitoce_display_agent_list();
        }

    }

    private function aitoce_redirect($url)
    {
        $string = "<script type='text/javascript'>";
        $string .= "window.location = '{$url}'";
        $string .= "</script>";

        echo wp_kses($string, [
            'script' => [
                'type' => [],
            ]
        ]);
    }
        
    // To show bulk action dropdown
    public function aitoce_display_agent_add($account_membership)
    {
        // phpcs:disable PluginCheck.CodeAnalysis.ImageFunctions.NonEnqueuedImage
        echo '
                <div class="aitoce-header">
                    <h1 class="aitoce-admin-header">
                        <img src="' . esc_url(plugins_url("images/knot.png", __DIR__)) . '"  alt="AI Tool Center Icon" style="vertical-align: middle;" width="35" height="35" /> AI Tool Center <img src="' . esc_url(plugins_url("images/nimbot-head-left.png", __DIR__)) . '" alt="Nimbot Head" style="vertical-align: middle; margin-left: 10px; margin-bottom: 5px;" width="46" height="37" />
                    </h1>';
                    // Display errors
                    settings_errors();
        echo '
                    <div id="aitoce-settings-buttons" class="aitoce-tabs">
                        <ul>
                            <li><a href="/wp-admin/admin.php?page='.esc_attr(AITOCE_MENU_SLUG_AGENTS).'">Back</a></li>
                        </ul>
                    </div>
                </div>
            <form id="aitoce-add-agent-form" class="aitoce-admin-form" method="post">
                <div class="aitoce-wrapper">
                    <h2 class="aitoce-section-header">Add Agent</h2>'; 

        $this->aitoce_get_connection_form();

        $this->aitoce_get_rag_form($account_membership['label']);

        $this->aitoce_get_styling_form();

        echo '</form>';
       
    }
    
    // To show bulk action dropdown
    public function aitoce_display_agent_edit($account_membership)
    {
        // phpcs:disable PluginCheck.CodeAnalysis.ImageFunctions.NonEnqueuedImage
        echo '
                <div class="aitoce-header">
                    <h1 class="aitoce-admin-header">
                        <img src="' . esc_url(plugins_url("images/knot.png", __DIR__)) . '"  alt="AI Tool Center Icon" style="vertical-align: middle;" width="35" height="35" /> AI Tool Center <img src="' . esc_url(plugins_url("images/nimbot-head-left.png", __DIR__)) . '" alt="Nimbot Head" style="vertical-align: middle; margin-left: 10px; margin-bottom: 5px;" width="46" height="37" />
                    </h1>';
        // Display errors
        settings_errors();
        echo '
                    <div id="aitoce-settings-buttons" class="aitoce-tabs">
                        <ul>
                            <li><a href="/wp-admin/admin.php?page='.esc_attr(AITOCE_MENU_SLUG_AGENTS).'">Back</a></li>
                        </ul>
                    </div>
                </div>
                <form id="aitoce-add-agent-form" class="aitoce-admin-form" method="post">
                    <div class="aitoce-wrapper">
                        <h2 class="aitoce-section-header">Edit Agent</h2>'; 
                
                $this->aitoce_get_connection_form();

                $this->aitoce_get_rag_form($account_membership['label']);

                $this->aitoce_get_styling_form();

        echo '
                </form>';
    }
    
    private function aitoce_get_connection_form(){
        echo '
                        <table class="form-table" role="presentation">
                        <tbody>
                            <tr>
                                <th scope="row">Name</th>
                                <td><input type="text" name="aitoce_agent_name" value="' . esc_html($this->connection["aitoce_agent_name"]) . '" aria-label="Agent Name" placeholder="Name..." /> <i class="fa-solid fa-triangle-exclamation aitoce-red" title="Required. &nbsp;Give the Agent a name."></i><br />
                                <div class="aitoce-description">Give the Agent a name.</div></td>
                            </tr>
                            <tr>
                                <th scope="row">Description</th>
                                <td><textarea name="aitoce_agent_description" aria-label="aitoce_agent_description" placeholder="Description..." rows="4" maxlength="1000">' . esc_html($this->connection["aitoce_agent_description"]) . '</textarea><br />
                                <div class="aitoce-description">Give the Agent a description.</div></td>
                            </tr>
                            <tr>
                                <th scope="row">LLM Provider</th>
                                <td>
                                    <ul class="aitoce_provider">
                                        <li>
                                            <input type="radio" name="aitoce_agent_provider" id="aitoce-openai-options" value="openai"'. checked($this->connection['aitoce_agent_provider'], 'openai', false).' />
                                            <label for="aitoce-openai-options">OpenAI</label>
                                        </li>
                                        <li>
                                            <input type="radio" name="aitoce_agent_provider" id="aitoce-claude-options" value="claude"'. checked($this->connection['aitoce_agent_provider'], 'claude', false).' />
                                            <label for="aitoce-claude-options">Claude</label>
                                        </li>
                                        <li>
                                            <input type="radio" name="aitoce_agent_provider" id="aitoce-custom-options" value="custom"'. checked($this->connection['aitoce_agent_provider'], 'custom', false).' />
                                            <label for="aitoce-custom-options">Custom</label>
                                        </li>
                                    </ul>
                                </td>
                            </tr>
                            <tr id="aitoce-custom-interface" style="display: none">
                                <th scope="row">Provider API Interface</th>
                                <td>';

            echo '<div class="aitoce-select-box"><select id="aitoce_agent_custom_interface" name="aitoce_agent_custom_interface" aria-label="aitoce_agent_custom_interface" class="">';
                
            echo '<option>Please Select</option>';

            foreach ($this->interfaces as $key => $value) {
                echo '<option value="'. esc_attr($key) .'" '. selected($this->connection['aitoce_agent_custom_interface'], esc_attr($key), false) . '>'. esc_attr($value) .'</option>';
            }

            echo '  </select></div><br />
                            <div class="aitoce-description">Select the OpenAI Model you wish to use.</div>';

            echo '      </td>
                            </tr>
                            <tr>
                                <th scope="row">API Key</th>
                                <td><input type="text" name="aitoce_agent_api_key" value="' . esc_html($this->connection["aitoce_agent_api_key"]) . '" aria-label="aitoce_agent_api_key" placeholder="API Key..." /> <i class="fa-solid fa-triangle-exclamation aitoce-red" title="Required. &nbsp;Enter your OpenAI API Key."></i><br />
                                <div class="aitoce-description">Enter your OpenAI API Key.</div></td>
                            </tr>
                            <tr id="aitoce-openai-model">
                                <th scope="row">Model</th>
                                <td>';

            echo '<div class="aitoce-select-box"><select id="aitoce_agent_openai_model" name="aitoce_agent_openai_model" aria-label="aitoce_agent_openai_model" class="">';
                
            echo '<option>Please Select</option>';

            foreach ($this->openai_models as $key => $value) {
                echo '<option value="'. esc_attr($key) .'" '. selected($this->connection["aitoce_agent_openai_model"], esc_attr($key), false) . '>'. esc_attr($value['name']) .'</option>';
            }

            echo '  </select></div>
                    <input type="hidden" id="aitoce_agent_openai_interface" name="aitoce_agent_openai_interface" aria-label="aitoce_agent_openai_interface" value="openai"><br />
                            <div class="aitoce-description">Select the OpenAI Model you wish to use.</div>';

            echo '      </td>
                            </tr>



                            <tr id="aitoce-claude-model" style="display: none">
                                <th scope="row">Model</th>
                                <td>';

            echo '<div class="aitoce-select-box"><select id="aitoce_agent_claude_model" name="aitoce_agent_claude_model" aria-label="aitoce_agent_claude_model" class="">';
                
            echo '<option>Please Select</option>';

            foreach ($this->claude_models as $key => $value) {
                echo '<option value="'. esc_attr($key) .'" '. selected($this->connection["aitoce_agent_claude_model"], esc_attr($key), false) . '>'. esc_attr($value['name']) .'</option>';
            }

            echo '  </select></div>
                    <input type="hidden" id="aitoce_agent_claude_interface" name="aitoce_agent_claude_interface" aria-label="aitoce_agent_claude_interface" value="claude"><br />
                            <div class="aitoce-description">Select the Claude Model you wish to use.</div>';
            echo '  </td>
                        </tr>
                            <tr id="aitoce-custom-model" style="display: none">
                                <th scope="row">Model</th>
                                <td><input type="text" name="aitoce_agent_custom_model" value="' . esc_html($this->connection["aitoce_agent_custom_model"]) . '" aria-label="aitoce_agent_custom_model" placeholder="Model..." /> <i class="fa-solid fa-triangle-exclamation aitoce-red" title="Required. &nbsp;Enter the Model you wish to use."></i><br />
                                <div class="aitoce-description">Enter the Model you wish to use.</div></td>
                            </tr>
                            <tr id="aitoce-custom-url" style="display: none">
                                <th scope="row">API URL</th>
                                <td><input type="text" name="aitoce_agent_url" value="' . esc_html($this->connection["aitoce_agent_url"]) . '" aria-label="aitoce_agent_url" placeholder="API URL..." /> <i class="fa-solid fa-triangle-exclamation aitoce-red" title="Required. &nbsp;Enter the API URL you wish to use."></i><br />
                                <div class="aitoce-description">Enter the API URL you wish to use.</div></td>
                            </tr>
                    </div>
                    </tbody>
                </table>';

            $submit_text = (isset($_GET['action']) && $_GET['action'] === 'add_agent')? 'Add New':'Update';
            echo '<p><input type="submit" name="submit" id="submit" class="button button-primary aitoce-button" value="'.esc_attr($submit_text).'"></p>
                </div>';

    }
    
    private function aitoce_get_rag_form($account_membership){
        if(empty($this->rag['aitoce_agent_retention'])){
            $this->rag['aitoce_agent_retention'] = $this->rag_defaults[$account_membership]['aitoce_agent_retention']['default'];
        }

        if(empty($this->rag['aitoce_agent_transcript']) && $this->rag['aitoce_agent_transcript'] !== '0'){
            $this->rag['aitoce_agent_transcript'] = $this->rag_defaults[$account_membership]['aitoce_agent_transcript']['default'];
        }

        echo '
            <div class="aitoce-wrapper">
                <h2 class="aitoce-section-header">RAG</h2>
                    <table class="form-table" role="presentation">
                        <tbody>
                            <tr>
                                <th scope="row">Role</th>
                                <td><textarea id="aitoce_agent_role" name="aitoce_agent_role" aria-label="Role" placeholder="Role..." rows="4" >' . esc_html($this->rag["aitoce_agent_role"]) . '</textarea> <i class="fa-solid fa-triangle-exclamation aitoce-red" title="Required. &nbsp;Enter how you want the Agent to act."></i><br />
                                <div class="aitoce-description">Enter how you want the Agent to act.<br /><br />
                                Example: " You are a helpful assistant. "</div></td>
                            </tr>
                            <tr>
                                <th scope="row">Memory Retention</th>
                                <td>';

                echo '<div class="aitoce-select-box"><select id="aitoce_agent_retention" name="aitoce_agent_retention" aria-label="Configuration" class="">';
                
                echo '<option>Please Select</option>';

                for ($i=1; $i <= 50; $i++) { 
                    echo '<option value="'.esc_attr($i).'" '. selected($this->rag["aitoce_agent_retention"], $i, false) . (($i > $this->rag_defaults[$account_membership]["aitoce_agent_retention"]['max'])? ' disabled':'') . '>'.esc_attr($i).'</option>';
                }

                echo '</select></div><br />
                                <div class="aitoce-description">The higher the retention the higher the token costs when using your own API end-points.
        </div>';

                echo '</td>
                            </tr>
                            <tr>
                                <th scope="row">Transcript Retention</th>
                                <td>';

                echo '<div class="aitoce-select-box"><select id="aitoce_agent_transcript" name="aitoce_agent_transcript" aria-label="Configuration" class="">';
                
                echo '<option>Please Select</option>';

                for ($i=0; $i <= 72; $i+=12) { 
                    echo '<option value="'.esc_attr($i).'"'. selected($this->rag["aitoce_agent_transcript"], $i, false) . (($i > $this->rag_defaults[$account_membership]["aitoce_agent_transcript"]['max'])? ' disabled':'') . '>'.esc_attr($i).'</option>';
                }

                echo '</select></div><br />
                                <div class="aitoce-description">How many hours to save message transcripts for, so they can be reviewed later.
        </div>';

                echo '</td>
                            </tr>
                            <tr>
                                <th scope="row">Inclusion by Page</th>
                                <td><textarea id="aitoce_agent_pages" name="aitoce_agent_pages" aria-label="Inclusion by Page" placeholder="Inclusion by Page..." rows="4">' . esc_html($this->rag["aitoce_agent_pages"]) . '</textarea><br />
                                <div class="aitoce-description">With a Free account only the first page slugs will be included. Add page slugs and separate each page with a comma. <br /><br />
                                Example: page-1, page-2, page/sub-page-3</div></td>
                            </tr>
                            <tr>
                                <th scope="row">Inclusion by Post</th>
                                <td><textarea id="aitoce_agent_posts" name="aitoce_agent_posts" aria-label="Inclusion by Post" placeholder="Inclusion by Post..." rows="4">' . esc_html($this->rag["aitoce_agent_posts"]) . '</textarea><br />
                                <div class="aitoce-description">With a Free account only the first post slugs will be included. Add Post slugs and separate each post with a comma. <br /><br />
                                Example: post-1, post-2, post/sub-post-3</div></td>
                            </tr>
                        </tbody>
                    </table>';

            $submit_text = (isset($_GET['action']) && $_GET['action'] === 'add_agent')? 'Add New':'Update';
            echo '
                <p><input type="submit" name="submit" id="submit" class="button button-primary aitoce-button" value="'.esc_attr($submit_text).'"></p>
            </div>';

    }
    
    private function aitoce_get_styling_form(){
        echo '
                <div class="aitoce-wrapper">
                    <h2 class="aitoce-section-header">Styling</h2>
                        <table class="form-table" role="presentation">
                            <tbody>
                                <tr>
                                    <th scope="row">Radius</th>
                                    <td>';

        echo '<div class="aitoce-select-box"><select id="aitoce_agent_header_radius" name="aitoce_agent_header_radius" aria-label="Header Radius" class="">';
                
        echo '<option>Please Select</option>';

        for ($i=0; $i <= 20; $i++) { 
            echo '<option value="'.esc_attr($i).'"'. selected($this->styling["aitoce_agent_header_radius"], $i, false) . '>'.esc_attr($i).'</option>';
        }

        echo '</select></div>';

        echo '</td>
                    </tr>
                    <tr>
                        <th scope="row">Text</th>
                        <td><input class="my-color-field" type="text" name="aitoce_agent_header_text" value="' . esc_html($this->styling["aitoce_agent_header_text"]) . '" aria-label="Header Text" data-default-color="#fbb53b" /></td>
                    </tr>
                    <tr>
                        <th scope="row">Background</th>
                        <td><input class="my-color-field" type="text" name="aitoce_agent_header_background" value="' . esc_html($this->styling["aitoce_agent_header_background"]) . '" aria-label="Header Background" data-default-color="#2d2b33" /></td>
                    </tr>
                </tbody>
            </table>
            <h2 class="aitoce-section-header">Background Style</h2>
            <table class="form-table" role="presentation">
                <tbody>
                    <tr>
                        <th scope="row">Height</th>
                        <td>';

        echo '<div class="aitoce-select-box"><select id="aitoce_agent_background_height" name="aitoce_agent_background_height" aria-label="Header Radius" class="">';
                
        echo '<option>Please Select</option>';

        for ($i=200; $i <= 1000; $i+=10) { 
            echo '<option value="'.esc_attr($i).'"'. selected($this->styling["aitoce_agent_background_height"], $i, false) . '>'.esc_attr($i).'</option>';
        }

        echo '</select></div>';

        echo '</td>
                    </tr>
                    <tr>
                        <th scope="row">Background</th>
                        <td><input class="my-color-field" type="text" name="aitoce_agent_background_background" value="' . esc_html($this->styling["aitoce_agent_background_background"]) . '" aria-label="Background" data-default-color="#3e3c47" /></td>
                    </tr>
                </tbody>
            </table>
            <h2 class="aitoce-section-header">Client Message Style</h2>
            <table class="form-table" role="presentation">
                <tbody>
                    <tr>
                        <th scope="row">Radius</th>
                        <td>';

        echo '<div class="aitoce-select-box"><select id="aitoce_agent_client_radius" name="aitoce_agent_client_radius" aria-label="Client Message Radius" class="">';
                
        echo '<option>Please Select</option>';

        for ($i=0; $i <= 20; $i++) { 
            echo '<option value="'.esc_attr($i).'"'. selected($this->styling["aitoce_agent_client_radius"], $i, false) . '>'.esc_attr($i).'</option>';
        }

        echo '</select></div>';

        echo '</td>
                    </tr>
                    <tr>
                        <th scope="row">Text</th>
                        <td><input class="my-color-field" type="text" name="aitoce_agent_client_text" value="' . esc_html($this->styling["aitoce_agent_client_text"]) . '" aria-label="Header Text" data-default-color="#2d2b33" /></td>
                    </tr>
                    <tr>
                        <th scope="row">Background</th>
                        <td><input class="my-color-field" type="text" name="aitoce_agent_client_background" value="' . esc_html($this->styling["aitoce_agent_client_background"]) . '" aria-label="Header Background" data-default-color="#e4e4e4" /></td>
                    </tr>
                </tbody>
            </table>
            <h2 class="aitoce-section-header">NimBot Reply Style</h2>
            <table class="form-table" role="presentation">
                <tbody>
                    <tr>
                        <th scope="row">Initial Greeting</th>
                        <td><textarea id="aitoce_agent_nimbot_greeting" name="aitoce_agent_nimbot_greeting" aria-label="Initial Greeting" placeholder="Hi, How can I help?..." rows="4">' . esc_html($this->styling["aitoce_agent_nimbot_greeting"]) . '</textarea></td>
                    </tr>
                    <tr>
                        <th scope="row">Radius</th>
                        <td>';

        echo '<div class="aitoce-select-box"><select id="aitoce_agent_nimbot_radius" name="aitoce_agent_nimbot_radius" aria-label="NimBot Reply Radius" class="">';
                
        echo '<option>Please Select</option>';

        for ($i=0; $i <= 20; $i++) { 
            echo '<option value="'.esc_attr($i).'"'. selected($this->styling["aitoce_agent_nimbot_radius"], $i, false) . '>'.esc_attr($i).'</option>';
        }

        echo '</select></div>';

        echo '</td>
                    </tr>
                    <tr>
                        <th scope="row">Text</th>
                        <td><input class="my-color-field" type="text" name="aitoce_agent_nimbot_text" value="' . esc_html($this->styling["aitoce_agent_nimbot_text"]) . '" aria-label="Header Text" data-default-color="#e4e4e4" /></td>
                    </tr>
                    <tr>
                        <th scope="row">Background</th>
                        <td><input class="my-color-field" type="text" name="aitoce_agent_nimbot_background" value="' . esc_html($this->styling["aitoce_agent_nimbot_background"]) . '" aria-label="Header Background" data-default-color="#2d2b33" /></td>
                    </tr>
                </tbody>
            </table>
            <h2 class="aitoce-section-header">Client Input Style</h2>
            <table class="form-table" role="presentation">
                <tbody>
                    <tr>
                        <th scope="row">Radius</th>
                        <td>';

        echo '<div class="aitoce-select-box"><select id="aitoce_agent_input_radius" name="aitoce_agent_input_radius" aria-label="Client Input Radius" class="">';
                
        echo '<option>Please Select</option>';

        for ($i=0; $i <= 20; $i++) { 
            echo '<option value="'.esc_attr($i).'"'. selected($this->styling["aitoce_agent_input_radius"], $i, false) . '>'.esc_attr($i).'</option>';
        }

        echo '</select><div>';

        echo '</td>
                    </tr>
                    <tr>
                        <th scope="row">Text</th>
                        <td><input class="my-color-field" type="text" name="aitoce_agent_input_text" value="' . esc_html($this->styling["aitoce_agent_input_text"]) . '" aria-label="Client Input Text" data-default-color="#3e3c47" /></td>
                    </tr>
                    <tr>
                        <th scope="row">Background</th>
                        <td><input class="my-color-field" type="text" name="aitoce_agent_input_background" value="' . esc_html($this->styling["aitoce_agent_input_background"]) . '" aria-label="Client Input Background" data-default-color="#e4e4e4" /></td>
                    </tr>
                </tbody>
            </table>
            <h2 class="aitoce-section-header">Button Style</h2>
            <table class="form-table" role="presentation">
                <tbody>
                    <tr>
                        <th scope="row">Radius</th>
                        <td>';

        echo '<div class="aitoce-select-box"><select id="aitoce_agent_button_radius" name="aitoce_agent_button_radius" aria-label="Button Radius" class="">';
                
        echo '<option>Please Select</option>';

        for ($i=0; $i <= 20; $i++) { 
            echo '<option value="'.esc_attr($i).'"'. selected($this->styling["aitoce_agent_button_radius"], $i, false) . '>'.esc_attr($i).'</option>';
        }

        echo '</select></div>';

        echo '</td>
                    </tr>
                    <tr>
                        <th scope="row">Text</th>
                        <td><input class="my-color-field" type="text" name="aitoce_agent_button_text" value="' . esc_html($this->styling["aitoce_agent_button_text"]) . '" aria-label="Button Text" data-default-color="#ffffff" /></td>
                    </tr>
                    <tr>
                        <th scope="row">Background</th>
                        <td><input class="my-color-field" type="text" name="aitoce_agent_button_background" value="' . esc_html($this->styling["aitoce_agent_button_background"]) . '" aria-label="Button Background" data-default-color="#fbb53b" /></td>
                    </tr>
                    <tr>
                        <th scope="row">Text</th>
                        <td><input class="my-color-field" type="text" name="aitoce_agent_button_hover_text" value="' . esc_html($this->styling["aitoce_agent_button_hover_text"]) . '" aria-label="Button Hover Text" data-default-color="#ffffff" /></td>
                    </tr>
                    <tr>
                        <th scope="row">Background</th>
                        <td><input class="my-color-field" type="text" name="aitoce_agent_button_hover_background" value="' . esc_html($this->styling["aitoce_agent_button_hover_background"]) . '" aria-label="Button Hover Background" data-default-color="#cb9231" /></td>
                    </tr>
                </tbody>
            </table>';

        $submit_text = (isset($_GET['action']) && $_GET['action'] === 'add_agent')? 'Add New':'Update';
        echo '
                        <p><input type="submit" name="submit" id="submit" class="button button-primary aitoce-button" value="'.esc_attr($submit_text).'"></p>
                        ';
                        wp_nonce_field("add_agent_wpnonce", "_wpnonce");
        echo '
                </div>';

    }

}