<?php

/**
 * The admin-specific functionality for the Account settings page.
 *
 * This class manages the admin menu, settings registration, and rendering for the Account page
 * in the WordPress admin area. It allows users to enter and view their API key and account status.
 *
 * @package    ai-tool-center
 * @subpackage Aitoce/admin/account-page
 * @author     sjvision <contact@ai-tool-center.com>
 */
class Aitoce_Admin_Account_Page
{
    /**
     * Handles admin notifications.
     * @var Aitoce_Admin_Notifications
     */
    private $notify;

    /**
     * Handles options and field rendering.
     * @var Aitoce_Admin_Options
     */
    private $options;

    /**
     * Constructor.
     * Initializes notification and options handler objects.
     */
    public function __construct()
    {
        $this->notify = new Aitoce_Admin_Notifications();
        $this->options = new Aitoce_Admin_Options();
    }

    /**
     * Adds the Account menu and submenu pages to the WordPress admin menu.
     */
    public function aitoce_add_account_menu(): void
    {
        $aitoce_base64_icon = $this->aitoce_get_icon();

        // Add main menu AI Tool Center page.
        add_menu_page(
            page_title: __('AI Tool Center', 'ai-tool-center'),
            menu_title: __('AI Tool Center', 'ai-tool-center'),
            capability: 'manage_options',
            menu_slug: AITOCE_MENU_SLUG,
            callback: [$this, 'aitoce_render_account_page'],   // Function to render the page.
            icon_url: 'data:image/svg+xml;base64,' . $aitoce_base64_icon, //'dashicons-editor-quote',
            position: 76
        );

        // Add submenu page for Account.
        add_submenu_page(
            parent_slug: AITOCE_MENU_SLUG,
            page_title: __('Account', 'ai-tool-center'),
            menu_title: __('Account', 'ai-tool-center'),
            capability: 'manage_options',
            menu_slug: AITOCE_MENU_SLUG_ACCOUNT,
            callback: [$this, 'aitoce_render_account_page']
        );

        if(!category_exists('NimBot Rag')){
            // Add Categories for NimBot Assistant.
            wp_insert_category([
                'cat_name' => 'NimBot Rag',
                'category_description' => '<strong>NimBot RAG</strong> posts are designed for use in the AI Tool Center’s Retrieval-Augmented Generation system. These entries provide structured, high-quality reference content for the <strong>NimBot AI Assistant</strong> to retrieve and respond accurately in real time.',
                'category_nicename' => 'nimbot-rag',
                'category_parent' => '',
            ]);
        }
    }

    /**
     * Renders the Account admin page.
     * Displays admin errors and includes the settings page template.
     */
    public function aitoce_render_account_page(): void
    {
        $this->notify->aitoce_display_admin_errors();
        include plugin_dir_path(__FILE__) . './aitoce-account-display.php';
    }

    /**
     * Registers settings, sections, and fields for the Account settings page.
     * Adds a field for entering the API key.
     */
    public function aitoce_register_account_settings(): void
    {
        // Register the settings group and sanitize callback
        register_setting(
            AITOCE_OPTION_SLUG_ACCOUNT,
            AITOCE_OPTION_NAME_ACCOUNT,
            ['sanitize_callback' => [$this, 'aitoce_sanitize_settings']]
        );

        // Add the main settings section
        add_settings_section(
            AITOCE_OPTION_SECTION_ACCOUNT,
            __('Account', 'ai-tool-center'),
            [$this, 'aitoce_account_section_callback'],
            AITOCE_OPTION_SLUG_ACCOUNT
        );

        // Add a text field for the API key
        $this->options->aitoce_add_text_field(
            'api_key',
            __('Account Key', 'ai-tool-center'),
            __('Account Key', 'ai-tool-center'),
            esc_attr(AITOCE_OPTION_NAME_ACCOUNT),
            esc_attr(AITOCE_OPTION_SLUG_ACCOUNT),
            esc_attr(AITOCE_OPTION_SECTION_ACCOUNT),
            'Your official API key from ai-tool-center.com'
        );
    }

    /**
     * Sanitizes the settings input for the Account options.
     *
     * @param array $input The input array from the settings form.
     * @return array Sanitized settings.
     */
    public function aitoce_sanitize_settings(array $input): array
    {
        return [
            'api_key' => sanitize_text_field($input['api_key'] ?? ''),
        ];
    }

    /**
     * Outputs the Account section, showing account level and domain name.
     * Displays the current account status and domain name for reference.
     */
    public function aitoce_account_section_callback()
    {
        $api_call = new Aitoce_Api;
        $account_membership = $api_call->aitoce_get_account();
        
        $http_referer = $this->options->aitoce_get_referer();

        $status = (in_array($account_membership['label'], AITOCE_ACCOUNT_LEVELS, true)) ?
            '<span class="aitoce-' . esc_html(strtolower($account_membership["label"])) . '"><i class="fa-solid fa-award"> </i> ' . esc_html($account_membership["label"]) . '</span>'
            :
            '<span class="aitoce-no-membership"><i class="fa-solid fa-award"></i> No Membership</span>';
        ?>
        <table class="form-table" role="presentation">
            <tbody>
                <tr>
                    <th scope="row"><?php esc_html_e('Account Membership', 'ai-tool-center'); ?></th>
                    <td class="aitoce-account-status"><?php echo wp_kses_post($status); ?><span class="aitoce-membership-upgrade"><a href="https://ai-tool-center.com/#prices" target="_blank"><i id="aitoce-membership-upgrade" class="fa-solid fa-up-right-from-square aitoce-blue" title="Change Membership"></i></a></span></td>
                </tr>
                <tr>
                    <th scope="row"><?php esc_html_e('Account Domain Name', 'ai-tool-center'); ?></th>
                    <td>
                        <span id="aitoce-domain" class="aitoce-domain"><?php echo esc_html($http_referer); ?></span><span class="aitoce-copy-domain"><i id="aitoce-copy-domain" class="fa-solid fa-copy aitoce-blue" title="Copy to clipboard" onclick="aitoceCopyToClipboard('aitoce-domain', 'aitoce-copy-domain')"></i></span><br /><br />
                        <div class="aitoce-description">
                            <?php esc_html_e('You will need this Domain Name when you create a ', 'ai-tool-center');
                            echo esc_html(implode(', ', array_slice(AITOCE_ACCOUNT_LEVELS, 0, 2)));
                            esc_html_e(' or ', 'ai-tool-center');
                            echo esc_html(AITOCE_ACCOUNT_LEVELS[2]);
                            esc_html_e(' Account at', 'ai-tool-center');
                            ?><br /><strong>https://ai-tool-center.com/membership-account/</strong>  <a href="https://ai-tool-center.com/membership-account/" target="_blank"><i class="fa-solid fa-up-right-from-square aitoce-blue" title="https://ai-tool-center.com/membership-account/"></i></a>
                        </div>
                    </td>
                </tr>
            </tbody>
        </table>
        <?php
    }

    private function aitoce_get_icon() {
        return "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";
    }

}
