/*!
 * AI Snippet SEO Helper — Admin JS
 * Version: 1.0.0
 * ---------------------------------------------
 * - Adds the “Generate with AI” UX in the editor
 * - Syncs Rank Math fields & live SERP preview
 * - Shows a real-time score badge (0–100)
 * - Performs a masked hard reload after generate
 * - Drives bulk jobs on list tables
 */

jQuery(function ($) {

	/* helpers */
	const minT=30,maxT=60,minD=120,maxD=160,maxS=75,
	      fold=s=>(toStr(s)).normalize('NFD').replace(/[\u0300-\u036f]/g,''),
	      norm=s=>fold(s).toLocaleLowerCase().replace(/[^a-z0-9]+/gi,' '),
	      hasKW=(s,k)=>!!toStr(k) && norm(s).includes(norm(k).trim());
	      
	function getBadgeColor(score){
		if (score >= 100) return '#2980b9';   // blue
		if (score >= 66)  return '#f39c12';   // orange
		return '#e74c3c';                     // red
	}

	/* null/undefined-safe primitives */
	function toStr(v){ return (typeof v==='string') ? v : (v==null ? '' : String(v)); }
	const charLen = s => [...toStr(s)].length;

	// update existing sidebar score badge
	function setAishScore(score){
		var $badge = $('#aish-score-box .aish-score-badge');
		if (!$badge.length) return;
		$badge.css('background', getBadgeColor(score))
		      .text('SEO Snippet Score: ' + score + ' / 100');
	}

	function calcAishScore(fw, ti, de, slug){
		var kw  = (String(fw||'').split(',')[0]||'').trim().toLowerCase();
		if (!kw) return 0;
		function ntext(s){ 
			return String(s||'').toLowerCase()
				.normalize('NFD').replace(/[\u0300-\u036f]/g,'')
				.replace(/[^a-z0-9]+/g,' ').trim(); 
		}
		function nslug(s){ 
			return String(s||'').toLowerCase()
				.normalize('NFD').replace(/[\u0300-\u036f]/g,'')
				.replace(/[^a-z0-9]+/g,'-').replace(/^-+|-+$/g,''); 
		}
		var hit = 0;
		if ((' '+ntext(ti)+' ').indexOf(' '+ntext(kw)+' ')!==-1) hit++;
		if ((' '+ntext(de)+' ').indexOf(' '+ntext(kw)+' ')!==-1) hit++;
		if (nslug(slug).indexOf(nslug(kw))!==-1) hit++;
		return hit===0 ? 0 : (hit===1 ? 33 : (hit===2 ? 66 : 100));
	}

	/* write into Rank Math fields and refresh its store */
	const setRM=(field,val)=>{
		val = toStr(val);
		let $i=$(`#rank_math_${field}`); if(!$i.length)$i=$(`[name="rank_math_${field}"]`);
		if($i.length)$i.val(val).trigger('input change');
		try{
			window.rankMath?.app?.$store?.commit('UPDATE_FIELD',{field,value:val});
			window.rankMath?.app?.$store?.commit('FORCE_UPDATE');
			window.rankMath?.analysis?.refresh?.();
		}catch(_){}
	};

	/* Post ID helpers & cleanup of legacy "Done" flag */
	function aishCurrentPostId(){
	  return parseInt($('#post_ID').val(), 10)
	      || parseInt(new URLSearchParams(location.search).get('post'), 10)
	      || 0;
	}
	function aishRestoreDone(){
	  const pid = aishCurrentPostId(); if(!pid) return;
	  try{ sessionStorage.removeItem('AISH_DONE_'+pid); }catch(_){}
	}
	aishRestoreDone();

	/* live score badge */
	const $kw = $('#rank_math_focus_keyword'),
	      $ti = $('#rank_math_title'),
	      $de = $('#rank_math_description');

	const getSlug = () => {
	  // 1) try editable spans
	  var t = ($('#editable-post-name-full,#editable-post-name').text()||'').trim();
	  // 2) input#post_name (exists in some themes)
	  if (!t) t = String($('input#post_name').val()||'').trim();
	  // 3) derive from Rank Math SERP preview URL
	  if (!t) {
	    var u = String($('.rank-math-serp-url, .snippet-preview-url').text()||'').trim();
	    if (u) {
	      try {
	        var p = new URL(u);
	        t = p.pathname || '';
	      } catch(_){
	        t = u.replace(/^https?:\/\/[^/]+/i,'');
	      }
	      t = (t||'').replace(/^\/|\/$/g,''); // trim leading/trailing slash
	    }
	  }
	  return t; // path works too (about-rankpilotai/who-we-are...)
	};

	function ensureBadge(){
		let $b = $('#aish-score');
		if(!$b.length){
			const host = $ti.length ? $ti.closest('p') : $('.rank-math-serp-title').first().parent();
			(host.length ? host : $('body')).append(' <span id="aish-score" class="aish-score-badge"></span>');
			$b = $('#aish-score');
		}
		return $b;
	}

	function updScore(){
	  try{
	    var fw = $('#rank_math_focus_keyword').val();
	    var ti = $('#rank_math_title').val();
	    var de = $('#rank_math_description').val();

	    // read slug from UI
	    var slug = (function(){
	      var t = ($('#editable-post-name-full,#editable-post-name').text()||'').trim();
	      if (!t) t = String($('input#post_name').val()||'').trim();
	      return t;
	    })();

	    var sc = calcAishScore(fw, ti, de, slug);

	    // 1) small inline badge
	    var $b = ensureBadge();
	    $b.text(sc+' / 100').css('background', getBadgeColor(sc));

	    // 2) right metabox badge
	    var $mb = $('#aish-score-box .aish-score-badge');
	    if ($mb.length) $mb.css('background', getBadgeColor(sc))
	                       .text('SEO Snippet Score: '+sc+' / 100');

	  }catch(err){ /* ignore */ }
	}

	/* instant SERP preview refresh */
	function instantPreview(d){
		if(d?.title) $('.rank-math-serp-title, .snippet-preview-title').text(toStr(d.title));
		if(d?.desc ) $('.rank-math-serp-description, .snippet-preview-description').text(toStr(d.desc));
		if(d?.slug ){
			const base = (window.AISH_Ajax?.home || (window.location.origin + '/'));
			const url = base + toStr(d.slug).replace(/^\/|\/$/g,'') + '/';
			$('.rank-math-serp-url, .snippet-preview-url').text(url);
		}
		refreshRMpopup();
	}

	/* re-render Rank Math snippet editor modal if open */
	function refreshRMpopup(){
		const $m=$('.rank-math-snippet-editor');
		if(!$m.length||!$m.is(':visible')) return;
		const $c=$m.find('.components-modal__header button');
		if($c.length) $c.trigger('click');
		setTimeout(()=>$('#rank-math-snippet-edit').trigger('click'),200);
	}

	/* neutralize beforeunload hooks to avoid prompts */
	function nukeBeforeUnload(){
		try { window.onbeforeunload = null; } catch(_){}
		try { $(window).off('beforeunload'); } catch(_){}
		try {
			const killer = function(e){
				try{
					e.stopImmediatePropagation?.(); e.stopPropagation?.();
					Object.defineProperty(e,'returnValue',{configurable:true,writable:true,value:undefined});
				}catch(_){}
			};
			window.addEventListener('beforeunload', killer, {capture:true});
			try{
				Object.defineProperty(window,'onbeforeunload',{configurable:true,set(){},get(){return null;}});
			}catch(_){}
			const _add = EventTarget.prototype.addEventListener;
			EventTarget.prototype.addEventListener = function(type, listener, options){
				if(type === 'beforeunload') return;
				return _add.call(this, type, listener, options);
			};
		} catch(_){}
	}

	/* detect Gutenberg vs Classic editor */
	function isGutenberg(){
		try{
			const sel = window.wp?.data?.select?.('core/editor');
			return !!(sel && typeof sel.getCurrentPostId === 'function');
		}catch(_){ return false; }
	}

	/* build a cache-busting URL for masked reload */
	function cacheBustingUrl(){
		try{
			const u = new URL(window.location.href);
			u.searchParams.set('_aish','1');
			u.searchParams.set('_ts', Date.now().toString());
			return u.toString();
		}catch(_){
			const base = window.location.href.split('#')[0];
			return base + (base.includes('?') ? '&' : '?') + '_aish=1&_ts=' + Date.now();
		}
	}

	/* masked hard reload (no white flash, no prompts) */
	function maskedHardReload(url){
		nukeBeforeUnload();
		try{
			const m=document.createElement('div');
			m.id='aish-reload-mask';
			m.style.cssText='position:fixed;inset:0;background:#f0f0f1;z-index:999999;display:flex;align-items:center;justify-content:center;gap:10px;font:600 14px/1.4 -apple-system,BlinkMacSystemFont,Segoe UI,Roboto,Arial,sans-serif';
			m.innerHTML='<span class="spinner is-active" style="visibility:visible"></span><span>Refreshing…</span>';
			document.body.appendChild(m);
		}catch(_){}
		try{ window.location.assign(url); }catch(_){ window.location.href=url; }
	}

	/* soft save (kept for reference; currently unused) */
	function softSave(){
	  try{
	    const wpData = window.wp?.data;
	    const sel = wpData?.select?.('core/editor');
	    if (sel){
	      nukeBeforeUnload();
	      const p = wpData.dispatch('core/editor').savePost?.();
	      if (p && typeof p.finally === 'function'){
	        p.finally(()=> setTimeout(refreshRMpopup, 120));
	      } else {
	        setTimeout(refreshRMpopup, 120);
	      }
	      return;
	    }
	  }catch(_){}
	  setTimeout(refreshRMpopup, 120);
	}

	/* AI button handler */
	$(document).on('click','.aish-generate-btn',function(e){
		e.preventDefault();
		const $btn=$(this), pid=$btn.data('post-id');
		if(!pid){alert('Post ID missing');return;}

		try{ sessionStorage.removeItem('AISH_DONE_'+pid); }catch(_){}
		$btn.prop('disabled',true).text('Generating…');

		$.post(AISH_Ajax.ajaxurl,{action:'aish_generate_single',nonce:AISH_Ajax.nonce,post_id:pid})

		.done(resp=>{
			$btn.prop('disabled',false).text('Done! 🎉')
			  .removeClass('button-primary').addClass('button-secondary');

			if(!resp.success) return onErr(resp);

			const d=resp.data||{};

			// reflect results in the UI immediately
			if(d.keyword) setRM('focus_keyword',d.keyword);
			if(d.title)   setRM('title',        d.title);
			if(d.desc)    setRM('description',  d.desc);
			if(d.slug){
				const slugStr = toStr(d.slug);
				$('#editable-post-name,#editable-post-name-full').text(slugStr);
				$('input#post_name').val(slugStr);
			}
			instantPreview(d);
			updScore();

			setTimeout(updScore, 400);
			setTimeout(updScore, 1500);

			// re-calc when Rank Math preview/keyword changes
			['#editable-post-name-full','#editable-post-name','.rank-math-serp-url','.snippet-preview-url','#rank_math_focus_keyword'].forEach(sel=>{
			  const n = document.querySelector(sel);
			  if (n && window.MutationObserver) {
			    try{ new MutationObserver(()=>updScore()).observe(n,{characterData:true,childList:true,subtree:true}); }catch(_){}
			  }
			});

			// update the sidebar score box
			if (resp && resp.data && resp.data.score !== undefined) {
				setAishScore(parseInt(resp.data.score,10)||0);
			}

			// masked reload for a clean state
			try{ sessionStorage.removeItem('AISH_DONE_'+(aishCurrentPostId()||pid)); }catch(_){}
			const url = cacheBustingUrl();

			if (isGutenberg()){
				// Gutenberg: silent save first, then masked reload
				try{
					const p = window.wp?.data?.dispatch?.('core/editor')?.savePost?.();
					if (p && typeof p.finally === 'function'){
						p.finally(()=> setTimeout(()=> maskedHardReload(url), 120));
					}else{
						setTimeout(()=> maskedHardReload(url), 120);
					}
				}catch(_){
					setTimeout(()=> maskedHardReload(url), 120);
				}
			}else{
				// Classic editor: direct masked reload
				setTimeout(()=> maskedHardReload(url), 120);
			}
		})

		.fail(onErr);

		function onErr(err){
			$btn.prop('disabled',false).text('Retry')
				.removeClass('button-secondary').addClass('button-primary');
			console.error(err);
			alert('AI Error: '+(err?.data?.msg||err?.statusText||'unknown'));
			setTimeout(refreshRMpopup, 120);
		}
	});

	// highlight model select after bulk if insufficient tokens
	(function(){
	  function getParam(name){
	    try{ return new URL(location.href).searchParams.get(name); }catch(_){ return null; }
	  }
	  $(function(){
	    var insuf = getParam('aish_binsufficient');
	    if(insuf === '1'){
	      var $sel = $('#model_choice');
	      if($sel.length){
	        $sel.css({ outline: '2px solid #ffb300', transition: 'outline .2s' });
	        $('html,body').animate({scrollTop: $sel.offset().top - 100}, 300);
	      }
	    }
	  });
	})();
});

/* bulk progress controller (post list screens) */
jQuery(function ($) {
  const qs  = new URLSearchParams(window.location.search);
  const job = qs.get('aish_job') || '';
  const $box = $('#aish-bulk-box');
  if (!job || !$box.length) return;

  // safe fallbacks if AISH_Ajax is missing
  const AISH     = window.AISH_Ajax || {};
  const AJAX_URL = AISH.ajaxurl || window.ajaxurl || (window.wp?.ajax?.settings?.url) || '/wp-admin/admin-ajax.php';
  const NONCE    = AISH.nonce   || $('#_wpnonce').val() || '';

  let ticking = false;
  let cancelled = false;

  const upd = (t) => $('#aish-bulk-text').text(t);

  function tick() {
    if (ticking || cancelled) return;
    ticking = true;

    $.post(AJAX_URL, { action: 'aish_bulk_tick', nonce: NONCE, job })
      .done(function (res) {
        ticking = false;
        if (!res || res.success !== true) { upd('error'); return; }

        const d = res.data || {};
        if (d.done) {
          window.location = d.redirect || window.location.href.replace(/([?&])aish_job=[^&]+&?/, '$1');
          return;
        }
        upd(`processed ${d.i}/${d.total} — success ${d.s}, failed ${d.f}`);
        setTimeout(tick, 700);
      })
      .fail(function () {
        ticking = false;
        setTimeout(tick, 1200);
      });
  }

  $('#aish-bulk-cancel').on('click', function (e) {
    e.preventDefault();
    cancelled = true;
    upd('cancelling…');
    $.post(AJAX_URL, { action: 'aish_bulk_cancel', nonce: NONCE, job })
      .always(function () { upd('cancelled'); });
  });

  upd('starting');
  tick();
});
