<?php
// Exit if accessed directly.
if (!defined('ABSPATH')) {
    exit;
}

/**
 * Cart Analysis class for AI Product Assistant
 */
class AIPA_Cart_Analysis {
    
    /**
     * Initialize the Cart Analysis class
     */
    public function init() {
        // Check if cart analysis is enabled
        $options = get_option('aipa_settings');
        $enable_cart_analysis = isset($options['enable_cart_analysis']) ? $options['enable_cart_analysis'] : true;
        
        if (!$enable_cart_analysis) {
            return;
        }
        
        // Add cart analysis to WooCommerce cart page
        add_action('woocommerce_after_cart_table', array($this, 'add_cart_analysis_form'));
        
        // Add AJAX action for handling cart analysis
        add_action('wp_ajax_aipa_cart_analysis', array($this, 'ajax_cart_analysis'));
        add_action('wp_ajax_nopriv_aipa_cart_analysis', array($this, 'ajax_cart_analysis'));
        
        // Enqueue cart analysis scripts and styles
        add_action('wp_enqueue_scripts', array($this, 'enqueue_scripts'));
    }
    
    /**
     * Enqueue necessary scripts and styles
     */
    public function enqueue_scripts() {
        if (!is_admin() && is_cart()) {
            wp_enqueue_style('aipa-cart-css', AIPA_PLUGIN_URL . 'assets/css/cart-analysis.css', array('aipa-frontend-css'), AIPA_VERSION);
            wp_enqueue_script('aipa-cart-js', AIPA_PLUGIN_URL . 'assets/js/cart-analysis.js', array('jquery'), AIPA_VERSION, true);
            
            // Get options for customization
            $options = get_option('aipa_settings');
            
            // Default text values
            $cart_analysis_title = isset($options['cart_analysis_title']) ? $options['cart_analysis_title'] : __('AI Cart Analysis', 'ai-product-assistant');
            $cart_analysis_description = isset($options['cart_analysis_description']) ? $options['cart_analysis_description'] : __('Ask AI if your cart items will work for your specific needs.', 'ai-product-assistant');
            $cart_analysis_placeholder = isset($options['cart_analysis_placeholder']) ? $options['cart_analysis_placeholder'] : __('e.g., Can I make a basic home theater with these items?', 'ai-product-assistant');
            $cart_analysis_button_text = isset($options['cart_analysis_button_text']) ? $options['cart_analysis_button_text'] : __('Analyze with AI', 'ai-product-assistant');
            
            // Localize script with AJAX URL and other data
            wp_localize_script('aipa-cart-js', 'aipa_cart_data', array(
                'ajax_url' => admin_url('admin-ajax.php'),
                'nonce' => wp_create_nonce('aipa_cart_nonce'),
                'analyzing_text' => __('Analyzing...', 'ai-product-assistant'),
                'analysis_placeholder' => $cart_analysis_placeholder,
                'analysis_button' => $cart_analysis_button_text,
                'cart_analysis_title' => $cart_analysis_title,
                'cart_analysis_description' => $cart_analysis_description,
                'error_message' => __('Something went wrong. Please try again.', 'ai-product-assistant'),
                'empty_query_message' => __('Please enter an analysis question.', 'ai-product-assistant'),
                'empty_cart_message' => __('Your cart is empty. Add some products first.', 'ai-product-assistant'),
                'aipa_service_error_message' => __('Sorry, AI feature is temporarily unavailable. Please try again later.', 'ai-product-assistant'),
                'login_url' => function_exists('wc_get_page_permalink') ? wc_get_page_permalink('myaccount') : wp_login_url(get_permalink()),
                'login_text' => __('Log in here', 'ai-product-assistant'),
            ));
            
            // Add custom CSS for cart analysis if set
            $cart_analysis_css = isset($options['cart_analysis_css']) ? $options['cart_analysis_css'] : '';
            if (!empty($cart_analysis_css)) {
                wp_add_inline_style('aipa-cart-css', '.aipa-cart-analysis, .aipa-shortcode-cart-analysis {' . esc_html($cart_analysis_css) . '}');
            }
        }
    }
    
    /**
     * Add cart analysis form to WooCommerce cart page
     */
    public function add_cart_analysis_form() {
        // Only show if cart is not empty
        if (WC()->cart->is_empty()) {
            return;
        }
        
        // Get options for customization
        $options = get_option('aipa_settings');
        
        // Default text values
        $cart_analysis_title = isset($options['cart_analysis_title']) ? $options['cart_analysis_title'] : __('AI Cart Analysis', 'ai-product-assistant');
        $cart_analysis_description = isset($options['cart_analysis_description']) ? $options['cart_analysis_description'] : __('Ask AI if your cart items will work for your specific needs.', 'ai-product-assistant');
        $cart_analysis_placeholder = isset($options['cart_analysis_placeholder']) ? $options['cart_analysis_placeholder'] : __('e.g., Can I make a basic home theater with these items?', 'ai-product-assistant');
        $cart_analysis_button_text = isset($options['cart_analysis_button_text']) ? $options['cart_analysis_button_text'] : __('Analyze with AI', 'ai-product-assistant');
        
        ?>
        <div id="aipa-cart-analysis" class="aipa-cart-analysis">
            <h3><?php echo esc_html($cart_analysis_title); ?></h3>
            <p><?php echo esc_html($cart_analysis_description); ?></p>
            
            <div class="aipa-cart-analysis-form">
                <textarea id="aipa-cart-analysis-input" placeholder="<?php echo esc_attr($cart_analysis_placeholder); ?>"></textarea>
                <button id="aipa-cart-analysis-btn" class="button aipa-cart-analysis-btn">
                    <span class="aipa-spinner"></span>
                    <span class="aipa-btn-text"><?php echo esc_html($cart_analysis_button_text); ?></span>
                </button>
            </div>
            
            <div id="aipa-cart-analysis-error" class="aipa-cart-analysis-error"></div>
            
            <div id="aipa-cart-analysis-results" class="aipa-cart-analysis-results">
                <div id="aipa-analysis-loading" class="aipa-analysis-loading" style="display: none;">
                    <div class="aipa-loading-spinner"></div>
                    <p><?php esc_html_e('AI is analyzing your cart...', 'ai-product-assistant'); ?></p>
                </div>
                
                <div id="aipa-analysis-container" class="aipa-analysis-container"></div>
            </div>
        </div>
        <?php
    }
    
    /**
     * Handle AJAX cart analysis request
     */
    public function ajax_cart_analysis() {
        check_ajax_referer('aipa_cart_nonce', 'nonce');
        
        // Check if user is logged in
        if (!is_user_logged_in()) {
            wp_send_json_error(array(
                'message' => __('Please log in to use AI cart analysis feature.', 'ai-product-assistant'),
                'code' => 'auth_required'
            ));
        }
        
        $query = isset($_POST['query']) ? sanitize_text_field(wp_unslash($_POST['query'])) : '';
        
        AIPA_Debug::log(AIPA_Debug::LEVEL_INFO, 'Cart_Analysis', 'AJAX cart analysis request received', array('query_length' => strlen($query)));
        
        if (empty($query)) {
            AIPA_Debug::log(AIPA_Debug::LEVEL_WARNING, 'Cart_Analysis', 'Empty query provided');
            wp_send_json_error(array(
                'message' => __('Please enter an analysis question.', 'ai-product-assistant')
            ));
        }
        
        if (!function_exists('WC') || !isset(WC()->cart) || WC()->cart->is_empty()) {
            AIPA_Debug::log(AIPA_Debug::LEVEL_WARNING, 'Cart_Analysis', 'Cart is empty or WooCommerce not available');
            wp_send_json_error(array(
                'message' => __('Your cart is empty. Add some products first.', 'ai-product-assistant')
            ));
        }
        
        // Make an internal request to the REST API
        $request = new WP_REST_Request('POST', '/aipa/v1/cart-analysis');
        $request->set_param('query', $query);
        
        $response = rest_do_request($request);
        
        if (is_wp_error($response)) {
            wp_send_json_error(array(
                'message' => $response->get_error_message(),
                'code' => $response->get_error_code()
            ));
        }
        
        $data = $response->get_data();
        
        // Check if the response contains error information (from wp_send_json_error)
        if (isset($data['code']) && isset($data['message'])) {
            wp_send_json_error(array(
                'message' => $data['message'],
                'code' => $data['code']
            ));
            return;
        }
        
        if (empty($data) || !isset($data['analysis'])) {
            wp_send_json_error(array(
                'message' => __('Unable to analyze your cart. Please try again.', 'ai-product-assistant')
            ));
        }
        
        // Generate the HTML for the analysis results
        $html = $this->get_analysis_html($data);
        
        wp_send_json_success(array(
            'html' => $html
        ));
    }
    
    /**
     * Generate HTML for analysis results
     */
    private function get_analysis_html($data) {
        $output = '';
        
        if (empty($data)) {
            return $output;
        }
        
        // Main analysis
        $output .= '<div class="aipa-analysis-main">';
        $output .= '<h4>' . __('Analysis Result', 'ai-product-assistant') . '</h4>';
        $output .= '<p>' . esc_html($data['analysis']) . '</p>';
        $output .= '</div>';
        
        // Result summary
        $output .= '<div class="aipa-analysis-summary">';
        
        // Compatibility status
        $is_complete = isset($data['is_complete']) ? $data['is_complete'] : false;
        $status_class = $is_complete ? 'complete' : 'incomplete';
        
        $output .= '<div class="aipa-analysis-status aipa-status-' . $status_class . '">';
        $output .= '<span class="aipa-status-icon">' . ($is_complete ? '✅' : '⚠️') . '</span>';
        $output .= '<span class="aipa-status-text">';
        $output .= $is_complete ? __('Your cart contains everything needed!', 'ai-product-assistant') : __('Your cart needs some adjustments.', 'ai-product-assistant');
        $output .= '</span>';
        $output .= '</div>';
        
        // Missing items
        if (!$is_complete && isset($data['missing_items']) && !empty($data['missing_items'])) {
            $output .= '<div class="aipa-missing-items">';
            $output .= '<h5>' . __('Missing Items', 'ai-product-assistant') . '</h5>';
            $output .= '<ul>';
            
            foreach ($data['missing_items'] as $item) {
                $output .= '<li>' . esc_html($item) . '</li>';
            }
            
            $output .= '</ul>';
            $output .= '</div>';
        }
        
        // Compatibility issues
        if (isset($data['compatibility_issues']) && !empty($data['compatibility_issues'])) {
            $output .= '<div class="aipa-compatibility-issues">';
            $output .= '<h5>' . __('Compatibility Issues', 'ai-product-assistant') . '</h5>';
            $output .= '<ul>';
            
            foreach ($data['compatibility_issues'] as $issue) {
                $output .= '<li>' . esc_html($issue) . '</li>';
            }
            
            $output .= '</ul>';
            $output .= '</div>';
        }
        
        // Recommendations
        if (isset($data['recommendations']) && !empty($data['recommendations'])) {
            $output .= '<div class="aipa-recommendations">';
            $output .= '<h5>' . __('Recommendations', 'ai-product-assistant') . '</h5>';
            $output .= '<ul>';
            
            foreach ($data['recommendations'] as $recommendation) {
                $output .= '<li>' . esc_html($recommendation) . '</li>';
            }
            
            $output .= '</ul>';
            $output .= '</div>';
        }
        
        $output .= '</div>'; // End analysis summary
        
        return $output;
    }
}