<?php
/**
 * Review Notice Handler
 *
 * @package AI_Product_Assistant
 * @since 1.0.1
 */

// Exit if accessed directly
if (!defined('ABSPATH')) {
    exit;
}

/**
 * Class AIPA_Review_Notice
 *
 * Handles the display of review notices to encourage users to rate the plugin
 */
class AIPA_Review_Notice {
    
    /**
     * Option name for storing review notice status
     *
     * @var string
     */
    private $option_name = 'aipa_review_notice_status';
    
    /**
     * Option name for storing usage statistics
     *
     * @var string
     */
    private $usage_option = 'aipa_usage_stats';
    
    /**
     * Constructor
     */
    public function __construct() {
        add_action('admin_notices', array($this, 'display_review_notice'));
        add_action('wp_ajax_aipa_handle_review_notice', array($this, 'handle_review_action'));
        add_action('admin_enqueue_scripts', array($this, 'enqueue_scripts'));
    }
    
    /**
     * Check if should display review notice
     *
     * @return bool
     */
    private function should_display_notice() {
        // Only show on AI Product Assistant pages
        $screen = get_current_screen();
        if (!$screen || strpos($screen->id, 'ai-product-assistant') === false) {
            return false;
        }
        
        // Check if user can manage options
        if (!current_user_can('manage_options')) {
            return false;
        }
        
        // Get notice status
        $notice_status = get_option($this->option_name, array());
        
        // If permanently dismissed, don't show
        if (isset($notice_status['dismissed']) && $notice_status['dismissed'] === 'permanent') {
            return false;
        }
        
        // If temporarily dismissed, check if time has passed
        if (isset($notice_status['dismissed_until']) && time() < $notice_status['dismissed_until']) {
            return false;
        }
        
        // Check installation time (7 days)
        $install_time = get_option('aipa_install_time', time());
        if (time() - $install_time < 7 * DAY_IN_SECONDS) {
            return false;
        }
        
        // Check usage (10 searches)
        $usage_stats = get_option($this->usage_option, array('search_count' => 0));
        if ($usage_stats['search_count'] < 10) {
            return false;
        }
        
        return true;
    }
    
    /**
     * Display review notice
     */
    public function display_review_notice() {
        if (!$this->should_display_notice()) {
            return;
        }
        
        $review_url = 'https://wordpress.org/support/plugin/ai-product-assistant/reviews/#new-post';
        ?>
        <div class="notice notice-info is-dismissible aipa-review-notice" style="padding: 15px;">
            <p style="font-size: 14px;">
                <strong><?php esc_html_e('Enjoying AI Product Assistant?', 'ai-product-assistant'); ?></strong>
            </p>
            <p>
                <?php esc_html_e('We\'d love to hear your feedback! If you\'re finding the plugin helpful, please consider leaving us a review. It helps us improve and grow!', 'ai-product-assistant'); ?>
            </p>
            <p>
                <a href="<?php echo esc_url($review_url); ?>" class="button button-primary aipa-review-action" data-action="review" target="_blank">
                    <?php esc_html_e('Sure, I\'ll leave a review', 'ai-product-assistant'); ?>
                </a>
                <a href="#" class="button aipa-review-action" data-action="later">
                    <?php esc_html_e('Maybe later', 'ai-product-assistant'); ?>
                </a>
                <a href="#" class="button aipa-review-action" data-action="dismiss">
                    <?php esc_html_e('Don\'t show this again', 'ai-product-assistant'); ?>
                </a>
            </p>
        </div>
        <?php
    }
    
    /**
     * Handle review notice actions via AJAX
     */
    public function handle_review_action() {
        check_ajax_referer('aipa_review_notice', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_die();
        }
        
        $action = isset($_POST['review_action']) ? sanitize_text_field($_POST['review_action']) : '';
        $notice_status = get_option($this->option_name, array());
        
        switch ($action) {
            case 'review':
                // User clicked review - dismiss permanently
                $notice_status['dismissed'] = 'permanent';
                $notice_status['dismissed_time'] = time();
                break;
                
            case 'later':
                // Remind in 7 days
                $notice_status['dismissed'] = 'temporary';
                $notice_status['dismissed_until'] = time() + (7 * DAY_IN_SECONDS);
                break;
                
            case 'dismiss':
                // Don't show again
                $notice_status['dismissed'] = 'permanent';
                $notice_status['dismissed_time'] = time();
                break;
        }
        
        update_option($this->option_name, $notice_status);
        wp_send_json_success();
    }
    
    /**
     * Enqueue scripts for review notice
     *
     * @param string $hook Current admin page hook
     */
    public function enqueue_scripts($hook) {
        // Only on our admin pages
        if (strpos($hook, 'ai-product-assistant') === false) {
            return;
        }
        
        if (!$this->should_display_notice()) {
            return;
        }
        
        wp_add_inline_script('jquery', '
            jQuery(document).ready(function($) {
                $(".aipa-review-action").on("click", function(e) {
                    if ($(this).data("action") !== "review") {
                        e.preventDefault();
                    }
                    
                    var action = $(this).data("action");
                    
                    $.post(ajaxurl, {
                        action: "aipa_handle_review_notice",
                        review_action: action,
                        nonce: "' . wp_create_nonce('aipa_review_notice') . '"
                    }, function() {
                        $(".aipa-review-notice").fadeOut();
                    });
                });
            });
        ');
    }
}

// Initialize
new AIPA_Review_Notice();