<?php
// Exit if accessed directly.
if (!defined('ABSPATH')) {
    exit;
}

/**
 * Admin class for AI Product Assistant (Main Entry Point)
 */
class AIPA_Admin {
    
    private $ai_provider;
    private $tabs;
    private $debug;
    
    /**
     * Initialize the admin class
     */
    public function init() {
        // Initialize sub-classes
        $this->ai_provider = new AIPA_Admin_AI_Provider();
        $this->tabs = new AIPA_Admin_Tabs();
        $this->debug = new AIPA_Admin_Debug();
        
        // Initialize hooks for sub-classes
        $this->ai_provider->init_hooks();
        $this->tabs->init_hooks();
        $this->debug->init_hooks();
        
        // Add admin menu
        add_action('admin_menu', array($this, 'add_admin_menu'));
        
        // Register settings
        add_action('admin_init', array($this, 'register_settings'));
        
        // Add settings link on plugins page
        add_filter('plugin_action_links_' . AIPA_PLUGIN_BASENAME, array($this, 'add_settings_link'));
        
        // Enqueue admin scripts and styles
        add_action('admin_enqueue_scripts', array($this, 'enqueue_admin_scripts'));

        // Add dashboard widget
        add_action('wp_dashboard_setup', array($this, 'add_dashboard_widget'));
        
        // Create statistics table when needed
        add_action('admin_init', array($this, 'create_statistics_table'));
        
        add_action('admin_init', array($this, 'handle_statistics_export'));
    }

    /**
     * Add settings link on plugins page
     */
    public function add_settings_link($links) {
        $settings_link = '<a href="' . admin_url('admin.php?page=ai-product-assistant') . '">' . __('Settings', 'ai-product-assistant') . '</a>';
        array_unshift($links, $settings_link);
        return $links;
    }
    
    /**
     * Proxy method to call do_settings_sections_for_group from tabs
     * This is called from the settings-page.php template
     */
    public function do_settings_sections_for_group($page, $fields) {
        return $this->tabs->do_settings_sections_for_group($page, $fields);
    }
    
    /**
     * Add admin menu items
     */
    public function add_admin_menu() {
        // Use custom 20x20 pixel PNG icon
        $icon_url = AIPA_PLUGIN_URL . 'assets/images/logo_20x20_px.png';
        
        add_menu_page(
            __('AI Product Assistant', 'ai-product-assistant'),
            __('AI Product Assistant', 'ai-product-assistant'),
            'manage_options',
            'ai-product-assistant',
            array($this, 'display_settings_page'),
            $icon_url,
            56
        );
    }
    
    /**
     * Register plugin settings
     */
    public function register_settings() {
        register_setting('aipa_settings_group', 'aipa_settings', array(
            'type' => 'array',
            'sanitize_callback' => array($this->tabs, 'sanitize_settings'),
        ));
        
        // Register AI Provider settings
        $this->ai_provider->register_ai_provider_settings();
        
        // Register debug settings
        $this->debug->register_debug_settings();
        
        // Register other settings
        $this->tabs->register_other_settings();
    }
    
    /**
     * Add dashboard widget
     */
    public function add_dashboard_widget() {
        wp_add_dashboard_widget(
            'aipa_dashboard_widget',
            __('AI Product Assistant', 'ai-product-assistant'),
            array($this, 'display_dashboard_widget')
        );
    }

    /**
     * Display dashboard widget
     */
    public function display_dashboard_widget() {
        // Load dashboard widget template
        include_once AIPA_PLUGIN_DIR . 'templates/admin/dashboard-widget.php';
    }

    /**
     * Create statistics table - delegate to tabs class
     */
    public function create_statistics_table() {
        $this->tabs->create_statistics_table();
    }

    /**
     * Handle statistics export - delegate to tabs class
     */
    public function handle_statistics_export() {
        $this->tabs->handle_statistics_export();
    }

    /**
     * Display settings page
     */
    public function display_settings_page() {
        // Safely get the active tab parameter
        $active_tab = 'ai-provider';
        // phpcs:ignore WordPress.Security.NonceVerification.Recommended
        if (isset($_GET['tab'])) {
            // phpcs:ignore WordPress.Security.NonceVerification.Recommended
            $tab = sanitize_text_field(wp_unslash($_GET['tab']));
            // Validate against allowed tabs
            $allowed_tabs = array('ai-provider', 'fields', 'search', 'cart', 'product-assistant', 'statistics', 'debug', 'shortcodes');
            if (in_array($tab, $allowed_tabs, true)) {
                $active_tab = $tab;
            }
        }
        
        include_once AIPA_PLUGIN_DIR . 'templates/admin/settings-page.php';
    }

    /**
     * Display the statistics page - delegate to tabs class
     */
    public function display_statistics_page() {
        $this->tabs->display_statistics_page();
    }
    
    /**
     * Enqueue admin scripts and styles
     */
    public function enqueue_admin_scripts($hook) {
        if ('toplevel_page_ai-product-assistant' !== $hook) {
            return;
        }
        
        wp_enqueue_style('aipa-admin-style', AIPA_PLUGIN_URL . 'assets/css/admin.css', array(), AIPA_VERSION);
        wp_enqueue_script('aipa-admin-script', AIPA_PLUGIN_URL . 'assets/js/admin.js', array('jquery'), AIPA_VERSION, true);
        
        wp_enqueue_media();
             
        // Pass data to JavaScript
        wp_localize_script('aipa-admin-script', 'aipa_admin_data', array(
            'nonce' => wp_create_nonce('aipa_admin_nonce'),
            'ajax_url' => admin_url('admin-ajax.php'),
            'select_icon_image_text' => __('Select Icon Image', 'ai-product-assistant'),
            'use_this_image_text' => __('Use This Image', 'ai-product-assistant'),
            'testing_text' => __('Testing...', 'ai-product-assistant'),
            'connecting_text' => __('Connecting...', 'ai-product-assistant'),
            'test_button_text' => __('Test Connection', 'ai-product-assistant'),
            'test_model_button_text' => __('Test Model Connection', 'ai-product-assistant'),
            'connect_button_text' => __('Connect API', 'ai-product-assistant'),
            'success_message' => __('Connection successful! API key is valid.', 'ai-product-assistant'),
            'error_message' => __('Connection failed! Please check your API key.', 'ai-product-assistant'),
            'default_urls' => array(
                'openai' => 'https://api.openai.com/v1/chat/completions',
                'claude' => 'https://api.anthropic.com/v1/messages',
                'gemini' => 'https://generativelanguage.googleapis.com/v1/models/gemini-pro:generateContent',
                'deepseek' => 'https://api.deepseek.com/chat/completions',
                'custom' => ''
            ),
            'default_css' => array(
                'search_button' => "/* Normal state */\nbody .aipa-floating-search-btn {\n    position: fixed !important;\n    bottom: 200px !important;\n    right: 20px !important;\n    background-color: #4CAF50 !important;\n    color: white !important;\n    border: none !important;\n    border-radius: 50% !important;\n    width: 60px !important;\n    height: 60px !important;\n    text-align: center !important;\n    box-shadow: 0 4px 8px rgba(0, 0, 0, 0.2) !important;\n    cursor: pointer !important;\n    z-index: 999999 !important;\n    display: flex !important;\n    align-items: center !important;\n    justify-content: center !important;\n    font-size: 24px !important;\n    transition: all 0.3s ease !important;\n}\n\n/* Hover state */\nbody .aipa-floating-search-btn:hover {\n    background-color: #45a049 !important;\n    transform: scale(1.05) !important;\n    box-shadow: 0 6px 12px rgba(0, 0, 0, 0.3) !important;\n}\n\n/* Mobile responsive */\n@media (max-width: 767px) {\n    body .aipa-floating-search-btn {\n        width: 50px !important;\n        height: 50px !important;\n        font-size: 20px !important;\n    }\n}",
                'cart_analysis' => "margin: 30px 0;\npadding: 20px;\nborder: 1px solid #eee;\nborder-radius: 8px;\nbackground-color: #f9f9f9;",
                 'button_css' => "display: inline-flex !important;\nalign-items: center !important;\njustify-content: center !important;\nbackground-color: #4CAF50 !important;\ncolor: white !important;\nborder: none !important;\nborder-radius: 4px !important;\npadding: 8px 16px !important;\ncursor: pointer !important;\nfont-size: 16px !important;\ntransition: all 0.3s ease !important;"
            ),
            'models' => array(
                'openai' => array(),
                'claude' => array(),
                'gemini' => array(),
                'deepseek' => array(),
                'custom' => array(
                    array('value' => 'custom', 'text' => 'Custom Model')
                )
            ),
            'settings' => get_option('aipa_settings'),

             'analysis_error_message' => __('An error occurred during analysis. Please try again.', 'ai-product-assistant'),
             'long_description_text' => __('Product Description', 'ai-product-assistant'),
             'short_description_text' => __('Short Description', 'ai-product-assistant'),
             /* translators: %s: field name (e.g., Description, Short Description) */
             'analysis_summary_text' => __('%s Analysis Summary', 'ai-product-assistant'),
             'total_products_text' => __('Total Products', 'ai-product-assistant'),
             'empty_fields_text' => __('Empty Fields', 'ai-product-assistant'),
             /* translators: %s: character limit number */
             'under_limit_text' => __('Under Limit (%s chars)', 'ai-product-assistant'),
             /* translators: %s: character limit number */
             'over_limit_text' => __('Over Limit (%s chars)', 'ai-product-assistant'),
             'average_length_text' => __('Average Length', 'ai-product-assistant'),
             'max_length_text' => __('Maximum Length', 'ai-product-assistant'),
             'min_length_text' => __('Minimum Length', 'ai-product-assistant'),
             'characters_text' => __('characters', 'ai-product-assistant'),
             'length_distribution_text' => __('Length Distribution', 'ai-product-assistant'),
             'character_range_text' => __('Character Range', 'ai-product-assistant'),
             'product_count_text' => __('Product Count', 'ai-product-assistant'),
             'percentage_text' => __('Percentage', 'ai-product-assistant'),
             'recommendation_text' => __('Recommended Setting', 'ai-product-assistant'),
             /* translators: %s: current character limit */
             'recommendation_most_under' => __('Your current limit of %s characters is sufficient for most of your products. This setting is optimal for token efficiency.', 'ai-product-assistant'),
             /* translators: %s: current character limit */
             'recommendation_many_under' => __('Your current limit of %s characters works well for most products. Consider keeping this setting for a good balance of context and token efficiency.', 'ai-product-assistant'),
             /* translators: 1: current character limit, 2: recommended character limit */
             'recommendation_most_over' => __('Most of your products have descriptions longer than %1$s characters. Consider increasing your limit to around %2$s characters for better AI matching results.', 'ai-product-assistant'),
             /* translators: 1: current character limit, 2: average character length */
             'recommendation_balanced' => __('Your product descriptions vary in length. The current limit of %1$s characters provides a good balance, but consider adjusting to around %2$s characters which is the average length.', 'ai-product-assistant'),
        ));
    }
}