<?php
// Exit if accessed directly.
if (!defined('ABSPATH')) {
    exit;
}

/**
 * Admin Tabs class for AI Product Assistant
 */
class AIPA_Admin_Tabs extends AIPA_Admin_Core {
    
    /**
     * Initialize tabs hooks
     */
    public function init_hooks() {
        add_action('wp_ajax_aipa_get_prompt_details', array($this, 'ajax_get_prompt_details')); 
        add_action('wp_ajax_aipa_analyze_description_length', array($this, 'ajax_analyze_description_length'));
        add_action('wp_ajax_aipa_analyze_short_description_length', array($this, 'ajax_analyze_short_description_length'));
        add_action('wp_ajax_aipa_get_available_meta_keys', array($this, 'ajax_get_available_meta_keys'));
        add_action('wp_ajax_aipa_save_custom_fields', array($this, 'ajax_save_custom_fields'));
    }

    /**
     * Register other settings
     */
    public function register_other_settings() {
        // Product Fields Section
        add_settings_section(
            'aipa_product_fields_section',
            __('Product Fields Matching', 'ai-product-assistant'),
            array($this, 'product_fields_section_callback'),
            'ai-product-assistant-fields'
        );
        
        add_settings_field(
            'search_fields',
            __('Product Fields', 'ai-product-assistant'),
            array($this, 'search_fields_callback'),
            'ai-product-assistant-fields',
            'aipa_product_fields_section'
        );
        
        // Search Settings Section
        add_settings_section(
            'aipa_search_section',
            __('Search Settings', 'ai-product-assistant'),
            array($this, 'search_section_callback'),
            'ai-product-assistant-search'
        );

        add_settings_field(
            'show_explanation',
            __('Show AI Explanation', 'ai-product-assistant'),
            array($this, 'show_explanation_callback'),
            'ai-product-assistant-search',
            'aipa_search_section'
        );

        add_settings_field(
            'explanation_icon',
            __('AI Explanation Icon', 'ai-product-assistant'),
            array($this, 'explanation_icon_callback'),
            'ai-product-assistant-search',
            'aipa_search_section'
        );
        
        add_settings_field(
            'show_add_to_cart',
            __('Show Add to Cart Button', 'ai-product-assistant'),
            array($this, 'show_add_to_cart_callback'),
            'ai-product-assistant-search',
            'aipa_search_section'
        );
        
        add_settings_field(
            'use_native_template',
            __('Use Native Product Template', 'ai-product-assistant'),
            array($this, 'use_native_template_callback'),
            'ai-product-assistant-search',
            'aipa_search_section'
        );
        
        add_settings_field(
            'show_out_of_stock',
            __('Show Out of Stock Products', 'ai-product-assistant'),
            array($this, 'show_out_of_stock_callback'),
            'ai-product-assistant-search',
            'aipa_search_section'
        );
        
        // Search UI Customization Section
        add_settings_section(
            'aipa_search_ui_section',
            __('Search UI Customization', 'ai-product-assistant'),
            array($this, 'search_ui_section_callback'),
            'ai-product-assistant-search'
        );
        
        add_settings_field(
            'search_modal_title',
            __('Search Modal Title', 'ai-product-assistant'),
            array($this, 'search_modal_title_callback'),
            'ai-product-assistant-search',
            'aipa_search_ui_section'
        );
        
        add_settings_field(
            'search_modal_description',
            __('Search Modal Description', 'ai-product-assistant'),
            array($this, 'search_modal_description_callback'),
            'ai-product-assistant-search',
            'aipa_search_ui_section'
        );
        
        add_settings_field(
            'search_placeholder',
            __('Search Input Placeholder', 'ai-product-assistant'),
            array($this, 'search_placeholder_callback'),
            'ai-product-assistant-search',
            'aipa_search_ui_section'
        );
        
        // Voice search hidden from UI
        /*
        add_settings_field(
            'voice_button_text',
            __('Voice Search Button Text', 'ai-product-assistant'),
            array($this, 'voice_button_text_callback'),
            'ai-product-assistant-search',
            'aipa_search_ui_section'
        );
        */

        add_settings_field(
            'search_button_text',
            __('Search Button Text', 'ai-product-assistant'),
            array($this, 'search_button_text_callback'),
            'ai-product-assistant-search',
            'aipa_search_ui_section'
        );

        add_settings_field(
            'show_floating_button',
            __('Floating Search Button', 'ai-product-assistant'),
            array($this, 'show_floating_button_callback'),
            'ai-product-assistant-search',
            'aipa_search_ui_section'
        );
        
        add_settings_field(
            'search_button_icon',
            __('Floating Search Button Icon', 'ai-product-assistant'),
            array($this, 'search_button_icon_callback'),
            'ai-product-assistant-search',
            'aipa_search_ui_section'
        );
        
        add_settings_field(
            'search_button_css',
            __('Floating Search Button CSS', 'ai-product-assistant'),
            array($this, 'search_button_css_callback'),
            'ai-product-assistant-search',
            'aipa_search_ui_section'
        );

        add_settings_field(
            'button_icon',
            __('Search Button Icon', 'ai-product-assistant'),
            array($this, 'button_icon_callback'),
            'ai-product-assistant-search',
            'aipa_search_ui_section'
        );
                
        add_settings_field(
            'button_css',
            __('Search Button CSS', 'ai-product-assistant'),
            array($this, 'button_css_callback'),
            'ai-product-assistant-search',
            'aipa_search_ui_section'
        );
        
        // Cart Analysis Section
        add_settings_section(
            'aipa_cart_section',
            __('Cart Analysis Settings', 'ai-product-assistant'),
            array($this, 'cart_section_callback'),
            'ai-product-assistant-cart'
        );

        add_settings_field(
            'enable_cart_analysis',
            __('Enable Cart Analysis', 'ai-product-assistant'),
            array($this, 'enable_cart_analysis_callback'),
            'ai-product-assistant-cart',
            'aipa_cart_section'
        );
        
        // Cart Analysis UI Customization Section
        add_settings_section(
            'aipa_cart_ui_section',
            __('Cart Analysis UI Customization', 'ai-product-assistant'),
            array($this, 'cart_ui_section_callback'),
            'ai-product-assistant-cart'
        );
        
        add_settings_field(
            'cart_analysis_title',
            __('Cart Analysis Title', 'ai-product-assistant'),
            array($this, 'cart_analysis_title_callback'),
            'ai-product-assistant-cart',
            'aipa_cart_ui_section'
        );
        
        add_settings_field(
            'cart_analysis_description',
            __('Cart Analysis Description', 'ai-product-assistant'),
            array($this, 'cart_analysis_description_callback'),
            'ai-product-assistant-cart',
            'aipa_cart_ui_section'
        );
        
        add_settings_field(
            'cart_analysis_placeholder',
            __('Analysis Input Placeholder', 'ai-product-assistant'),
            array($this, 'cart_analysis_placeholder_callback'),
            'ai-product-assistant-cart',
            'aipa_cart_ui_section'
        );
        
        add_settings_field(
            'cart_analysis_button_text',
            __('Analysis Button Text', 'ai-product-assistant'),
            array($this, 'cart_analysis_button_text_callback'),
            'ai-product-assistant-cart',
            'aipa_cart_ui_section'
        );
        
        add_settings_field(
            'cart_analysis_css',
            __('Cart Analysis CSS', 'ai-product-assistant'),
            array($this, 'cart_analysis_css_callback'),
            'ai-product-assistant-cart',
            'aipa_cart_ui_section'
        );

        // Search performance settings section
        add_settings_section(
            'aipa_performance_section',
            __('Search Performance Settings', 'ai-product-assistant'),
            array($this, 'performance_section_callback'),
            'ai-product-assistant-search'
        );

        add_settings_field(
            'description_length_limit',
            __('Product Description Length Limit', 'ai-product-assistant'),
            array($this, 'description_length_limit_callback'),
            'ai-product-assistant-search',
            'aipa_performance_section'
        );

        add_settings_field(
            'short_description_length_limit',
            __('Short Description Length Limit', 'ai-product-assistant'),
            array($this, 'short_description_length_limit_callback'),
            'ai-product-assistant-search',
            'aipa_performance_section'
        );

        add_settings_field(
            'search_batch_size',
            __('Products Per Batch', 'ai-product-assistant'),
            array($this, 'search_batch_size_callback'),
            'ai-product-assistant-search',
            'aipa_performance_section'
        );

        add_settings_field(
            'max_batches',
            __('Maximum Batches for Product Search', 'ai-product-assistant'),
            array($this, 'max_batches_callback'),
            'ai-product-assistant-search',
            'aipa_performance_section'
        );

        add_settings_field(
            'enable_brute_force_search',
            __('Full Database AI Scan (Brute Force)', 'ai-product-assistant'),
            array($this, 'enable_brute_force_search_callback'),
            'ai-product-assistant-search',
            'aipa_performance_section'
        );

        add_settings_field(
            'ai_temperature',
            __('AI Temperature', 'ai-product-assistant'),
            array($this, 'ai_temperature_callback'),
            'ai-product-assistant-search',
            'aipa_performance_section'
        );

        add_settings_field(
            'ai_max_tokens',
            __('AI Max Tokens', 'ai-product-assistant'),
            array($this, 'ai_max_tokens_callback'),
            'ai-product-assistant-search',
            'aipa_performance_section'
        );

        // Statistics Settings
        add_settings_section(
            'aipa_statistics_section',
            __('API Usage Statistics', 'ai-product-assistant'),
            array($this, 'statistics_section_callback'),
            'ai-product-assistant-statistics'
        );

        add_settings_section(
            'aipa_product_assistant_section',
            __('Product Assistant Settings', 'ai-product-assistant'),
            array($this, 'product_assistant_section_callback'),
            'ai-product-assistant-product-assistant'
        );
        
        add_settings_field(
            'show_product_assistant',
            __('Show on Product Pages', 'ai-product-assistant'),
            array($this, 'show_product_assistant_callback'),
            'ai-product-assistant-product-assistant',
            'aipa_product_assistant_section'
        );
        
        add_settings_field(
            'product_assistant_title',
            __('Assistant Title', 'ai-product-assistant'),
            array($this, 'product_assistant_title_callback'),
            'ai-product-assistant-product-assistant',
            'aipa_product_assistant_section'
        );
        
        add_settings_field(
            'product_assistant_description',
            __('Assistant Description', 'ai-product-assistant'),
            array($this, 'product_assistant_description_callback'),
            'ai-product-assistant-product-assistant',
            'aipa_product_assistant_section'
        );
        
        add_settings_field(
            'product_assistant_placeholder',
            __('Assistant Input Placeholder', 'ai-product-assistant'),
            array($this, 'product_assistant_placeholder_callback'),
            'ai-product-assistant-product-assistant',
            'aipa_product_assistant_section'
        );
        
        add_settings_field(
            'product_assistant_button_text',
            __('Assistant Button Text', 'ai-product-assistant'),
            array($this, 'product_assistant_button_text_callback'),
            'ai-product-assistant-product-assistant',
            'aipa_product_assistant_section'
        );
        
        add_settings_field(
            'product_assistant_button_css',
            __('Assistant Button CSS', 'ai-product-assistant'),
            array($this, 'product_assistant_button_css_callback'),
            'ai-product-assistant-product-assistant',
            'aipa_product_assistant_section'
        );
    }

    /**
     * AJAX analyze description length
     */
    public function ajax_analyze_description_length() {
        check_ajax_referer('aipa_admin_nonce', 'nonce');
        
        $options = get_option('aipa_settings');
        $limit = isset($_POST['limit']) ? intval($_POST['limit']) : 500;
        $field_type = isset($_POST['field_type']) ? sanitize_text_field(wp_unslash($_POST['field_type'])) : 'description';
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error(array('message' => esc_html(__('You do not have permission to perform this action.', 'ai-product-assistant'))));
        }
        
        $field_getter = ($field_type === 'short_description') ? 'get_short_description' : 'get_description';
        
        $args = array(
            'limit' => -1,
            'status' => 'publish',
        );
        
        $products = wc_get_products($args);
        
        if (empty($products)) {
            wp_send_json_error(array('message' => esc_html(__('No products found.', 'ai-product-assistant'))));
        }
        
        $total_count = count($products);
        $empty_count = 0;
        $under_limit_count = 0;
        $over_limit_count = 0;
        $length_distribution = array(
            '0' => 0,                  
            '1-100' => 0,              
            '101-250' => 0,             
            '251-500' => 0,            
            '501-1000' => 0,           
            '1001-2000' => 0,           
            '2001+' => 0                
        );
        
        $max_length = 0;
        $min_length = PHP_INT_MAX;
        $total_length = 0;
        
        foreach ($products as $product) {
            $content = $product->$field_getter();
            $length = mb_strlen($content);
            
            $total_length += $length;
            
            if ($length === 0) {
                $empty_count++;
                $length_distribution['0']++;
            } elseif ($length <= 100) {
                $length_distribution['1-100']++;
            } elseif ($length <= 250) {
                $length_distribution['101-250']++;
            } elseif ($length <= 500) {
                $length_distribution['251-500']++;
            } elseif ($length <= 1000) {
                $length_distribution['501-1000']++;
            } elseif ($length <= 2000) {
                $length_distribution['1001-2000']++;
            } else {
                $length_distribution['2001+']++;
            }
            
            if ($length <= $limit) {
                $under_limit_count++;
            } else {
                $over_limit_count++;
            }
            
            if ($length > $max_length) {
                $max_length = $length;
            }
            
            if ($length < $min_length && $length > 0) {
                $min_length = $length;
            }
        }
        
        $non_empty_count = $total_count - $empty_count;
        $avg_length = $non_empty_count > 0 ? round($total_length / $non_empty_count) : 0;
        
        if ($min_length === PHP_INT_MAX) {
            $min_length = 0;
        }
        
        $result = array(
            'total_count' => $total_count,
            'empty_count' => $empty_count,
            'under_limit_count' => $under_limit_count,
            'over_limit_count' => $over_limit_count,
            'empty_percent' => round(($empty_count / $total_count) * 100, 1),
            'under_limit_percent' => round(($under_limit_count / $total_count) * 100, 1),
            'over_limit_percent' => round(($over_limit_count / $total_count) * 100, 1),
            'max_length' => $max_length,
            'min_length' => $min_length,
            'avg_length' => $avg_length,
            'length_distribution' => $length_distribution,
            'current_limit' => $limit,
            'field_type' => $field_type
        );
        
        wp_send_json_success($result);
    }

    /**
     * AJAX analyze short description length
     */
    public function ajax_analyze_short_description_length() {
        $_POST['field_type'] = 'short_description';
        $this->ajax_analyze_description_length();
    }

    /**
     * AJAX get prompt details
     */
    public function ajax_get_prompt_details() {
        check_ajax_referer('aipa_admin_nonce', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error('Insufficient permissions.');
            return;
        }
        
        $stat_id = isset($_POST['id']) ? intval($_POST['id']) : 0;
        
        if (empty($stat_id)) {
            wp_send_json_error(array('message' => esc_html(__('Invalid statistics ID.', 'ai-product-assistant'))));
        }
        
        global $wpdb;
        $table_name = $wpdb->prefix . 'aipa_statistics';
        
        // Get statistics data with prepared statement for security
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching
        $stat = $wpdb->get_row($wpdb->prepare(
            "SELECT request_prompt, response_text FROM {$wpdb->prefix}aipa_statistics WHERE id = %d",
            $stat_id
        ));
        
        if (!$stat) {
            wp_send_json_error(array('message' => esc_html(__('Statistics not found.', 'ai-product-assistant'))));
        }
        
        wp_send_json_success(array(
            'prompt' => $stat->request_prompt,
            'response' => $stat->response_text
        ));
    }

    /**
     * AJAX Get Available Meta Keys
     */
    public function ajax_get_available_meta_keys() {
        check_ajax_referer('aipa_admin_nonce', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error(array('message' => esc_html(__('You do not have permission to perform this action.', 'ai-product-assistant'))));
        }
        
        global $wpdb;
        
        // Get all product meta keys for admin interface (no user input)
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching
        $all_meta_keys = $wpdb->get_col(
            "SELECT DISTINCT pm.meta_key 
             FROM {$wpdb->postmeta} pm
             INNER JOIN {$wpdb->posts} p ON pm.post_id = p.ID
             WHERE p.post_type = 'product' 
             ORDER BY pm.meta_key"
        );
        
        $exclude_keys = array(
            '_edit_lock',
            '_edit_last',
            '_wp_page_template',
            '_wp_attachment_metadata',
            '_wp_attached_file'
        );
        
        $filtered_meta_keys = array_diff($all_meta_keys, $exclude_keys);
        
        wp_send_json_success(array(
            'meta_keys' => array_values($filtered_meta_keys),
            'total_keys' => count($all_meta_keys),
            'filtered_keys_count' => count($filtered_meta_keys)
        ));
    }
    
    /**
     * AJAX Save Custom Fields
     */
    public function ajax_save_custom_fields() {
        check_ajax_referer('aipa_admin_nonce', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error(array('message' => esc_html(__('You do not have permission to perform this action.', 'ai-product-assistant'))));
        }

        $options = get_option('aipa_settings');
        
        // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized -- JSON data is sanitized after decoding
        $custom_fields = isset($_POST['custom_fields']) ? json_decode(stripslashes(wp_unslash($_POST['custom_fields'])), true) : array();
        
        $sanitized_fields = array();
        if (is_array($custom_fields)) {
            foreach ($custom_fields as $field) {
                if (isset($field['meta_key']) && isset($field['description'])) {
                    $sanitized_fields[] = array(
                        // phpcs:ignore WordPress.DB.SlowDBQuery.slow_db_query_meta_key
                        'meta_key' => sanitize_key($field['meta_key']),
                        'description' => sanitize_text_field($field['description'])
                    );
                }
            }
        }
        
        $options['custom_fields'] = $sanitized_fields;
        update_option('aipa_settings', $options);
        
        wp_send_json_success(array(
            'message' => __('Custom fields saved successfully.', 'ai-product-assistant'),
            'custom_fields' => $sanitized_fields
        ));
    }

    /**
     * Display the statistics page
     */
    public function display_statistics_page() {
        if (!current_user_can('manage_options')) {
            return;
        }
        
        // Safely get pagination parameter
        $current_page = 1;
        // Reading pagination parameter for display purposes only (no data modification)
        // phpcs:ignore WordPress.Security.NonceVerification.Recommended
        if (isset($_GET['paged'])) {
            // phpcs:ignore WordPress.Security.NonceVerification.Recommended
            $page = intval(wp_unslash($_GET['paged']));
            if ($page > 0) {
                $current_page = $page;
            }
        }
        $per_page = 20; 
        
        global $wpdb;
        $table_name = $wpdb->prefix . 'aipa_statistics';
        
        // Safely get time range parameter
        $time_range = 'today';
        // Reading filter parameter for display purposes only (no data modification)
        // phpcs:ignore WordPress.Security.NonceVerification.Recommended
        if (isset($_GET['time_range'])) {
            // phpcs:ignore WordPress.Security.NonceVerification.Recommended
            $range = sanitize_text_field(wp_unslash($_GET['time_range']));
            // Validate against allowed time ranges
            $allowed_ranges = array('today', 'yesterday', '7days', '30days', 'all');
            if (in_array($range, $allowed_ranges, true)) {
                $time_range = $range;
            }
        }
        
        $where_clause = '';
        switch ($time_range) {
            case 'today':
                $where_clause = "WHERE DATE(timestamp) = CURDATE()";
                break;
            case 'yesterday':
                $where_clause = "WHERE DATE(timestamp) = DATE_SUB(CURDATE(), INTERVAL 1 DAY)";
                break;
            case '7days':
                $where_clause = "WHERE timestamp >= DATE_SUB(NOW(), INTERVAL 7 DAY)";
                break;
            case '30days':
                $where_clause = "WHERE timestamp >= DATE_SUB(NOW(), INTERVAL 30 DAY)";
                break;
            default:
                $where_clause = "";
        }
        
        // Build COUNT query with predefined conditions (no user input)
        $count_sql = "SELECT COUNT(*) FROM {$wpdb->prefix}aipa_statistics " . $where_clause;
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching,WordPress.DB.PreparedSQL.NotPrepared
        $total_items = $wpdb->get_var($count_sql);
        $total_pages = ceil($total_items / $per_page);
        
        $current_page = min($current_page, max(1, $total_pages));
        
        $offset = ($current_page - 1) * $per_page;
        
        // Build main query with predefined where clause and prepare only the LIMIT values
        $main_sql = "SELECT * FROM {$wpdb->prefix}aipa_statistics " . $where_clause . " ORDER BY timestamp DESC LIMIT %d, %d";
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching,WordPress.DB.PreparedSQL.NotPrepared
        $stats = $wpdb->get_results($wpdb->prepare($main_sql, $offset, $per_page));
        
        // Build summary query with predefined conditions (no user input)
        $summary_sql = "SELECT 
            COUNT(*) as total_requests,
            SUM(request_tokens) as total_request_tokens,
            SUM(response_tokens) as total_response_tokens,
            SUM(request_tokens + response_tokens) as total_tokens,
            AVG(execution_time) as avg_execution_time,
            COUNT(DISTINCT user_role) as unique_users
            FROM {$wpdb->prefix}aipa_statistics " . $where_clause;
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching,WordPress.DB.PreparedSQL.NotPrepared
        $summary = $wpdb->get_row($summary_sql);
        
        include_once AIPA_PLUGIN_DIR . 'templates/admin/statistics-page.php';
    }

    /**
     * Product Fields Section callback
     */
    public function product_fields_section_callback() {
        echo '<p>' . esc_html(__('Select which product details the AI reads when deciding which products best match a customer\'s search. More fields give the AI more context, but also use more API credits per search.', 'ai-product-assistant')) . '</p>';
    }
    
    /**
     * Search Section callback
     */
    public function search_section_callback() {
        echo '<p>' . esc_html(__('Configure AI product search settings.', 'ai-product-assistant')) . '</p>';
    }

    /**
     * Search UI Section callback
     */
    public function search_ui_section_callback() {
        echo '<p>' . esc_html(__('Customize the text and appearance of the AI search interface.', 'ai-product-assistant')) . '</p>';
        echo '<div class="aipa-settings-section-separator" style="margin: 20px 0; border-bottom: 1px solid #ccc; padding-bottom: 10px;">';
        echo '<h3 style="margin-top: 0;">' . esc_html(__('Search Form Customization', 'ai-product-assistant')) . '</h3>';
        echo '<p>' . esc_html(__('These settings apply to the search form displayed in the modal and shortcode.', 'ai-product-assistant')) . '</p>';
        echo '</div>';
    }

    /**
     * Cart Section callback
     */
    public function cart_section_callback() {
        echo '<p>' . esc_html(__('Configure cart analysis settings.', 'ai-product-assistant')) . '</p>';
    }
    
    /**
     * Cart UI Section callback
     */
    public function cart_ui_section_callback() {
        echo '<p>' . esc_html(__('Customize the text and appearance of the cart analysis interface.', 'ai-product-assistant')) . '</p>';
    }

    /**
     * Performance Section callback
     */
    public function performance_section_callback() {
        echo '<p>' . esc_html(__('Configure AI search performance and efficiency settings.', 'ai-product-assistant')) . '</p>';
    }

    /**
     * Statistics Section callback
     */
    public function statistics_section_callback() {
        echo '<p>' . esc_html(__('View API usage statistics and export data.', 'ai-product-assistant')) . '</p>';
    }

    /**
     * Product Assistant Section callback
     */
    public function product_assistant_section_callback() {
        echo '<p>' . esc_html(__('Configure AI product assistant features for individual products.', 'ai-product-assistant')) . '</p>';
    }

    /**
     * Search Fields callback
     */
    public function search_fields_callback() {
        $options = get_option('aipa_settings');
        $search_fields = isset($options['search_fields']) ? $options['search_fields'] : array();
        $custom_fields = isset($options['custom_fields']) ? $options['custom_fields'] : array();
        
        $fields = array(
            'title' => __('Product Title', 'ai-product-assistant'),
            'short_description' => __('Short Description', 'ai-product-assistant'),
            'content' => __('Long Description/Content', 'ai-product-assistant'),
            'categories' => __('Categories', 'ai-product-assistant'),
            'tags' => __('Tags', 'ai-product-assistant'),
            'brand' => __('Brand', 'ai-product-assistant'),
            'weight' => __('Weight', 'ai-product-assistant'),
            'dimensions' => __('Dimensions (Length x Width x Height)', 'ai-product-assistant'),
            'variation_data' => __('Variation Data (for Variable Products)', 'ai-product-assistant'),
            'attributes' => __('Attributes/Custom Meta Fields', 'ai-product-assistant'),
            'regular_price' => __('Regular Price', 'ai-product-assistant'),
            'sale_price' => __('Sale Price', 'ai-product-assistant')
        );
        
        echo '<p>' . esc_html(__('Select which product fields to include when matching with AI:', 'ai-product-assistant')) . '</p>';
        echo '<fieldset>';
        
        // Display standard fields
        foreach ($fields as $field_id => $field_label) {
            $checked = isset($search_fields[$field_id]) ? $search_fields[$field_id] : false;
            ?>
            <label for="search_fields_<?php echo esc_attr($field_id); ?>">
                <input type="checkbox" id="search_fields_<?php echo esc_attr($field_id); ?>" name="aipa_settings[search_fields][<?php echo esc_attr($field_id); ?>]" <?php checked($checked); ?> />
                <?php echo esc_html($field_label); ?>
            </label><br>
            <?php
        }
        
        // Display custom fields as checkboxes
        foreach ($custom_fields as $custom_field) {
            $field_key = 'custom_' . $custom_field['meta_key'];
            $checked = isset($search_fields[$field_key]) ? $search_fields[$field_key] : false;
            $label = $custom_field['meta_key'];
            
            // Debug output
            if (WP_DEBUG) {
                // phpcs:ignore WordPress.DB.SlowDBQuery.slow_db_query_meta_key
                AIPA_Debug::log(AIPA_Debug::LEVEL_DEBUG, 'Admin_Tabs', 'Processing custom field', array('meta_key' => $custom_field['meta_key'], 'hex' => bin2hex($custom_field['meta_key'])));
            }
            
            if (!empty($custom_field['description'])) {
                $label .= ' - ' . $custom_field['description'];
            }
            ?>
            <div class="aipa-custom-field-wrapper">
                <label for="search_fields_<?php echo esc_attr($field_key); ?>">
                    <input type="checkbox" id="search_fields_<?php echo esc_attr($field_key); ?>" name="aipa_settings[search_fields][<?php echo esc_attr($field_key); ?>]" value="1" <?php checked($checked); ?> />
                    <?php echo esc_html($label); ?>
                </label>
                <button type="button" class="remove-custom-field-btn button-link-delete" data-meta-key="<?php echo esc_attr($custom_field['meta_key']); ?>"><?php esc_html_e('Remove', 'ai-product-assistant'); ?></button>
            </div>
            <?php
        }
        
        echo '</fieldset>';
        echo '<p class="description">' . esc_html(__('These fields are used for both product search and cart analysis.', 'ai-product-assistant')) . '</p>';
        echo '<p class="description"><strong>' . esc_html(__('Note on Price Fields:', 'ai-product-assistant')) . '</strong> ' . esc_html(__('Customers can still search for "sale items" or sort by "cheapest first" without selecting Regular Price or Sale Price here — that filtering is handled automatically by WooCommerce. Selecting price fields here only allows the AI to mention specific prices in its result explanations.', 'ai-product-assistant')) . '</p>';
        
        // Add custom field interface (simplified)
        echo '<div class="aipa-add-custom-field-section">';
        echo '<h4>' . esc_html(__('Add Custom Meta Field', 'ai-product-assistant')) . '</h4>';
        
        // Add new custom field interface
        echo '<div class="aipa-add-custom-field">';
        echo '<select id="aipa-custom-field-select" class="regular-text">';
        echo '<option value="">' . esc_html(__('Select a meta key', 'ai-product-assistant')) . '</option>';
        echo '</select>';
        echo '<input type="text" id="aipa-custom-field-description" placeholder="' . esc_attr(__('Description/Note (optional)', 'ai-product-assistant')) . '" class="regular-text" />';
        echo '<button type="button" id="aipa-add-custom-field-btn" class="button button-primary">' . esc_html(__('Add Field', 'ai-product-assistant')) . '</button>';
        echo '</div>';
        
        echo '</div>';
        
        // Hidden input to store custom fields data
        echo '<input type="hidden" id="aipa_custom_fields" name="aipa_settings[custom_fields]" value="' . esc_attr(json_encode($custom_fields)) . '" />';
    }
    
    /**
     * Show Out of Stock callback
     */
    public function show_out_of_stock_callback() {
        $options = get_option('aipa_settings');
        $show_out_of_stock = isset($options['show_out_of_stock']) ? $options['show_out_of_stock'] : false;
        $use_native_template = isset($options['use_native_template']) ? $options['use_native_template'] : true;
        ?>
        <div id="show_out_of_stock_container" class="aipa-dependent-field"
    <?php echo $use_native_template ? esc_attr('style="display:none;"') : ''; ?>>
            <fieldset>
                <label for="show_out_of_stock">
                    <input type="checkbox" id="show_out_of_stock" name="aipa_settings[show_out_of_stock]" <?php checked($show_out_of_stock); ?>>
                    <?php esc_html_e('Include out of stock products in search results', 'ai-product-assistant'); ?>
                </label>
                <p class="description">
                    <?php esc_html_e('When using the native WooCommerce template, this option is determined by the global WooCommerce settings.', 'ai-product-assistant'); ?>
                </p>
            </fieldset>
        </div>
        <?php
    }

    /**
     * Show Explanation callback
     */
    public function show_explanation_callback() {
        $options = get_option('aipa_settings');
        $show_explanation = isset($options['show_explanation']) ? $options['show_explanation'] : true;
        ?>
        <fieldset>
            <label for="show_explanation">
                <input type="checkbox" id="show_explanation" name="aipa_settings[show_explanation]" <?php checked($show_explanation); ?>>
                <?php esc_html_e('Show AI explanation for why products match the query', 'ai-product-assistant'); ?>
            </label>
        </fieldset>
        <?php
    }

    /**
     * AI Explanation Icon callback
     */
    public function explanation_icon_callback() {
        $options = get_option('aipa_settings');
        $explanation_icon = isset($options['explanation_icon']) ? $options['explanation_icon'] : 'AIℹ️';
        ?>
        <input type="text" id="explanation_icon" name="aipa_settings[explanation_icon]" value="<?php echo esc_attr($explanation_icon); ?>" class="small-text">
        <p class="description"><?php esc_html_e('Enter an emoji (ℹ️, 🔍, 📌, etc.) or symbol to use as the ai explanation icon. Only visible when "Show AI Explanation" is enabled.', 'ai-product-assistant'); ?></p>
        <p class="description"><?php esc_html_e('Tip: The explanation will appear as a tooltip when hovering over this icon.', 'ai-product-assistant'); ?></p>
        <?php
    }

    /**
     * Show Add to Cart callback
     */
    public function show_add_to_cart_callback() {
        $options = get_option('aipa_settings');
        $show_add_to_cart = isset($options['show_add_to_cart']) ? $options['show_add_to_cart'] : true;
        ?>
        <fieldset>
            <label for="show_add_to_cart">
                <input type="checkbox" id="show_add_to_cart" name="aipa_settings[show_add_to_cart]" <?php checked($show_add_to_cart); ?>>
                <?php esc_html_e('Show "Add to Cart" button in search results', 'ai-product-assistant'); ?>
            </label>
        </fieldset>
        <?php
    }
    
    /**
     * Use Native Template callback
     */
    public function use_native_template_callback() {
       $options = get_option('aipa_settings');
       $use_native_template = isset($options['use_native_template']) ? $options['use_native_template'] : true;
       ?>
       <fieldset>
           <label for="use_native_template">
               <input type="checkbox" id="use_native_template" name="aipa_settings[use_native_template]" <?php checked($use_native_template); ?>>
               <?php esc_html_e('Use native WooCommerce product template for search results', 'ai-product-assistant'); ?>
           </label>
           <p class="description"><?php esc_html_e('This ensures search results maintain the same style as your site\'s existing product listings.', 'ai-product-assistant'); ?></p>
           <p class="description"><?php esc_html_e('When using native WooCommerce template, the visibility of out-of-stock products is controlled by WooCommerce settings.', 'ai-product-assistant'); ?></p>
       </fieldset>
       <?php
   }

   /**
    * Search Modal Title callback
    */
   public function search_modal_title_callback() {
       $options = get_option('aipa_settings');
       $search_modal_title = isset($options['search_modal_title']) ? $options['search_modal_title'] : __('AI Product Search', 'ai-product-assistant');
       ?>
       <input type="text" id="search_modal_title" name="aipa_settings[search_modal_title]" value="<?php echo esc_attr($search_modal_title); ?>" class="regular-text" />
       <p class="description"><?php esc_html_e('The title displayed at the top of the search modal.', 'ai-product-assistant'); ?></p>
       <?php
   }
   
   /**
    * Search Modal Description callback
    */
   public function search_modal_description_callback() {
       $options = get_option('aipa_settings');
       $search_modal_description = isset($options['search_modal_description']) ? $options['search_modal_description'] : __('Describe what you\'re looking for in natural language.', 'ai-product-assistant');
       ?>
       <input type="text" id="search_modal_description" name="aipa_settings[search_modal_description]" value="<?php echo esc_attr($search_modal_description); ?>" class="regular-text" />
       <p class="description"><?php esc_html_e('The description text displayed below the title in the search modal.', 'ai-product-assistant'); ?></p>
       <?php
   }
   
   /**
    * Search Placeholder callback
    */
   public function search_placeholder_callback() {
       $options = get_option('aipa_settings');
       $search_placeholder = isset($options['search_placeholder']) ? $options['search_placeholder'] : __('e.g., I need a waterproof jacket for hiking in rainy weather', 'ai-product-assistant');
       ?>
       <input type="text" id="search_placeholder" name="aipa_settings[search_placeholder]" value="<?php echo esc_attr($search_placeholder); ?>" class="regular-text" />
       <p class="description"><?php esc_html_e('The placeholder text shown in the search input field.', 'ai-product-assistant'); ?></p>
       <?php
   }
   
   /**
    * Voice Button Text callback
    */
   public function voice_button_text_callback() {
       $options = get_option('aipa_settings');
       $voice_button_text = isset($options['voice_button_text']) ? $options['voice_button_text'] : __('Voice Search', 'ai-product-assistant');
       ?>
       <input type="text" id="voice_button_text" name="aipa_settings[voice_button_text]" value="<?php echo esc_attr($voice_button_text); ?>" class="regular-text" />
       <p class="description"><?php esc_html_e('The text displayed on the voice search button.', 'ai-product-assistant'); ?></p>
       <?php
   }
   
   /**
    * Search Button Text callback
    */
   public function search_button_text_callback() {
       $options = get_option('aipa_settings');
       $search_button_text = isset($options['search_button_text']) ? $options['search_button_text'] : __('Search with AI', 'ai-product-assistant');
       ?>
       <input type="text" id="search_button_text" name="aipa_settings[search_button_text]" value="<?php echo esc_attr($search_button_text); ?>" class="regular-text" />
       <p class="description"><?php esc_html_e('The text displayed on the search button.', 'ai-product-assistant'); ?></p>
       <?php
   }
   
   /**
     * Inline Search Button Icon callback
     */
    public function button_icon_callback() {
        $options = get_option('aipa_settings');
        $button_icon = isset($options['button_icon']) ? $options['button_icon'] : 'AI Search';
        $button_icon_type = isset($options['button_icon_type']) ? $options['button_icon_type'] : 'text';
        $button_icon_image = isset($options['button_icon_image']) ? $options['button_icon_image'] : '';
        ?>
        <div class="aipa-settings-group">
            <h3><?php esc_html_e('Inline Search Button', 'ai-product-assistant'); ?></h3>
            <p class="description"><?php esc_html_e('Customize the appearance of the inline search button used with the [aipa_search_button] shortcode.', 'ai-product-assistant'); ?></p>
            
            <fieldset>
                <label>
                    <input type="radio" name="aipa_settings[button_icon_type]" value="text" <?php checked($button_icon_type, 'text'); ?>>
                    <?php esc_html_e('Use Text/Emoji Icon', 'ai-product-assistant'); ?>
                </label><br>
                <input type="text" id="button_icon" name="aipa_settings[button_icon]" value="<?php echo esc_attr($button_icon); ?>" class="small-text" placeholder="AI Search">
                <p class="description"><?php esc_html_e('Enter an emoji (🔍, 🔎, etc.) or special character to use as button icon.', 'ai-product-assistant'); ?></p>
            </fieldset>
            
            <fieldset>
                <label>
                    <input type="radio" name="aipa_settings[button_icon_type]" value="image" <?php checked($button_icon_type, 'image'); ?>>
                    <?php esc_html_e('Use Custom Image', 'ai-product-assistant'); ?>
                </label><br>
                <div class="aipa-image-selector">
                    <input type="hidden" id="button_icon_image" name="aipa_settings[button_icon_image]" value="<?php echo esc_attr($button_icon_image); ?>">
                    <button type="button" class="button button-secondary" id="upload_button_icon_image"><?php esc_html_e('Select Image', 'ai-product-assistant'); ?></button>
                    <button type="button" class="button button-secondary" id="remove_button_icon_image" <?php echo empty($button_icon_image) ? 'style="display:none;"' : ''; ?>><?php esc_html_e('Remove Image', 'ai-product-assistant'); ?></button>
                    <div id="button_icon_image_preview" <?php echo empty($button_icon_image) ? 'style="display:none;"' : ''; ?>>
                        <?php // Admin preview image for user-uploaded icon (not WordPress media) ?>
                        <?php // phpcs:ignore PluginCheck.CodeAnalysis.ImageFunctions.NonEnqueuedImage ?>
                        <img src="<?php echo esc_url($button_icon_image); ?>" alt="<?php esc_html_e('Button icon preview', 'ai-product-assistant'); ?>">
                    </div>
                    <p class="description"><?php esc_html_e('Upload a square image to use as button icon (recommended size: 64x64px).', 'ai-product-assistant'); ?></p>
                </div>
            </fieldset>
        </div>
        <?php
    }

    /**
     * Search Button CSS callback
     */
    public function button_css_callback() {
        $options = get_option('aipa_settings');
        $default_css = "display: inline-flex !important;\nalign-items: center !important;\njustify-content: center !important;\nbackground-color: #4CAF50 !important;\ncolor: white !important;\nborder: none !important;\nborder-radius: 4px !important;\npadding: 8px 16px !important;\ncursor: pointer !important;\nfont-size: 16px !important;\ntransition: all 0.3s ease !important;";
        $button_css = isset($options['button_css']) ? $options['button_css'] : $default_css;
        ?>
        <textarea id="button_css" name="aipa_settings[button_css]" rows="8" class="large-text code"><?php echo esc_textarea($button_css); ?></textarea>
        <p class="description"><?php esc_html_e('CSS for the inline search button. You can customize its appearance.', 'ai-product-assistant'); ?></p>
        <button type="button" id="button_css_reset" class="button button-secondary"><?php esc_html_e('Reset to Default', 'ai-product-assistant'); ?></button>
        
        <div class="aipa-preview-container" style="margin-top: 15px; border: 1px solid #ddd; padding: 15px; background: #f9f9f9;">
            <h4><?php esc_html_e('Button Preview', 'ai-product-assistant'); ?></h4>
            <div id="button_preview"></div>
        </div>
        <?php
    }

    /**
     * Show Floating Search Button callback
     */
    public function show_floating_button_callback() {
        $options = get_option('aipa_settings');
        $show_floating_button = isset($options['show_floating_button']) ? $options['show_floating_button'] : true;
        ?>
        <div class="aipa-button-settings-group">
            <h3><?php esc_html_e('Floating Search Button', 'ai-product-assistant'); ?></h3>
            <p class="description"><?php esc_html_e('This button appears floating on the bottom right of your website.', 'ai-product-assistant'); ?></p>
            
            <div class="aipa-field-with-toggle" style="margin-top: 15px;">
                <label class="aipa-toggle-switch">
                    <input type="checkbox" id="show_floating_button" name="aipa_settings[show_floating_button]" <?php checked($show_floating_button); ?>>
                    <span class="aipa-toggle-slider"></span>
                </label>
                <label class="aipa-field-label" for="show_floating_button"><?php esc_html_e('Show floating search button on website', 'ai-product-assistant'); ?></label>
            </div>
        </div>
        <?php
    }

    /**
     * Search Button CSS callback
     */
    public function search_button_css_callback() {
        $options = get_option('aipa_settings');
        $default_css = "/* Normal state */\nbody .aipa-floating-search-btn {\n    position: fixed !important;\n    bottom: 200px !important;\n    right: 20px !important;\n    background-color: #4CAF50 !important;\n    color: white !important;\n    border: none !important;\n    border-radius: 50% !important;\n    width: 60px !important;\n    height: 60px !important;\n    text-align: center !important;\n    box-shadow: 0 4px 8px rgba(0, 0, 0, 0.2) !important;\n    cursor: pointer !important;\n    z-index: 999999 !important;\n    display: flex !important;\n    align-items: center !important;\n    justify-content: center !important;\n    font-size: 24px !important;\n    transition: all 0.3s ease !important;\n}\n\n/* Hover state */\nbody .aipa-floating-search-btn:hover {\n    background-color: #45a049 !important;\n    transform: scale(1.05) !important;\n    box-shadow: 0 6px 12px rgba(0, 0, 0, 0.3) !important;\n}\n\n/* Mobile responsive */\n@media (max-width: 767px) {\n    body .aipa-floating-search-btn {\n        width: 50px !important;\n        height: 50px !important;\n        font-size: 20px !important;\n    }\n}";
        $search_button_css = isset($options['search_button_css']) ? $options['search_button_css'] : $default_css;
        $show_floating_button = isset($options['show_floating_button']) ? $options['show_floating_button'] : true;
        
        $icon_type = isset($options['search_button_icon_type']) ? $options['search_button_icon_type'] : 'text';
        $icon_text = isset($options['search_button_icon']) ? $options['search_button_icon'] : 'AI';
        $icon_image = isset($options['search_button_icon_image']) ? $options['search_button_icon_image'] : '';
        
        ?>
        <textarea id="search_button_css" name="aipa_settings[search_button_css]" rows="15" class="large-text code"><?php echo esc_textarea($search_button_css); ?></textarea>
        <p class="description"><?php esc_html_e('Customize all button styles including normal state, hover effects, and responsive design. Use "body .aipa-floating-search-btn" for normal state and "body .aipa-floating-search-btn:hover" for hover state.', 'ai-product-assistant'); ?></p>
        <button type="button" id="search_button_css_reset" class="button button-secondary"><?php esc_html_e('Reset to Default', 'ai-product-assistant'); ?></button>
        
        <h4 style="margin-top: 20px;"><?php esc_html_e('Floating Button Preview', 'ai-product-assistant'); ?></h4>
        
        <div id="floating_button_preview_container" style="<?php 
    echo esc_attr('position: relative; height: 100px; margin-top: 15px;' . 
    ($show_floating_button ? '' : ' display:none;')); 
?>">
            
            <div id="preview_floating_button" style="<?php echo esc_attr(str_replace('!important', '', $search_button_css)); ?> position: absolute; bottom: 20px; left: 20px;">
                <?php if ($icon_type === 'image' && !empty($icon_image)): ?>
                    <?php // Admin preview image for floating button (user-defined URL) ?>
                    <?php // phpcs:ignore PluginCheck.CodeAnalysis.ImageFunctions.NonEnqueuedImage ?>
                    <img src="<?php echo esc_url($icon_image); ?>" alt="Search" style="max-width: 24px; max-height: 24px;">
                <?php else: ?>
                    <?php echo esc_html($icon_text); ?>
                <?php endif; ?>
            </div>
        </div>
        <?php
    }

   /**
     * Search Button Icon callback
     */
    public function search_button_icon_callback() {
        $options = get_option('aipa_settings');
        $search_button_icon = isset($options['search_button_icon']) ? $options['search_button_icon'] : 'AI';
        $search_button_icon_type = isset($options['search_button_icon_type']) ? $options['search_button_icon_type'] : 'text';
        $search_button_icon_image = isset($options['search_button_icon_image']) ? $options['search_button_icon_image'] : '';
        ?>
        <div class="aipa-icon-settings">
            <div class="aipa-icon-option">
                <label>
                    <input type="radio" name="aipa_settings[search_button_icon_type]" value="text" <?php checked($search_button_icon_type, 'text'); ?>>
                    <?php esc_html_e('Use Text/Emoji Icon', 'ai-product-assistant'); ?>
                </label>
                <input type="text" id="search_button_icon" name="aipa_settings[search_button_icon]" value="<?php echo esc_attr($search_button_icon); ?>" class="regular-text" placeholder="AI" style="max-width: 60px; margin-left: 10px;">
                <p class="description"><?php esc_html_e('Enter an emoji (🔍, 🔎, etc.) or special character to use as button icon.', 'ai-product-assistant'); ?></p>
            </div>
            
            <div class="aipa-icon-option" style="margin-top: 15px;">
                <label>
                    <input type="radio" name="aipa_settings[search_button_icon_type]" value="image" <?php checked($search_button_icon_type, 'image'); ?>>
                    <?php esc_html_e('Use Custom Image', 'ai-product-assistant'); ?>
                </label>
                <div class="aipa-image-selector" style="margin-top: 10px;">
                    <input type="hidden" id="search_button_icon_image" name="aipa_settings[search_button_icon_image]" value="<?php echo esc_attr($search_button_icon_image); ?>">
                    <button type="button" class="button button-secondary" id="upload_icon_image"><?php esc_html_e('Select Image', 'ai-product-assistant'); ?></button>
                    <button type="button" class="button button-secondary" id="remove_icon_image" style="<?php echo empty($search_button_icon_image) ? 'display:none;' : ''; ?>"><?php esc_html_e('Remove Image', 'ai-product-assistant'); ?></button>
                    <div id="icon_image_preview" style="margin-top: 10px; <?php echo empty($search_button_icon_image) ? 'display:none;' : ''; ?>">
                        <?php // Admin preview image for search button icon (user-uploaded file) ?>
                        <?php // phpcs:ignore PluginCheck.CodeAnalysis.ImageFunctions.NonEnqueuedImage ?>
                        <img src="<?php echo esc_url($search_button_icon_image); ?>" style="max-width: 100px; max-height: 100px;">
                    </div>
                    <p class="description"><?php esc_html_e('Upload a square image to use as button icon (recommended size: 64x64px).', 'ai-product-assistant'); ?></p>
                </div>
            </div>
        </div>
        <?php
    }
   
   /**
    * Enable Cart Analysis callback
    */
   public function enable_cart_analysis_callback() {
       $options = get_option('aipa_settings');
       $enable_cart_analysis = isset($options['enable_cart_analysis']) ? $options['enable_cart_analysis'] : true;
       ?>
       <fieldset>
           <label for="enable_cart_analysis">
               <input type="checkbox" id="enable_cart_analysis" name="aipa_settings[enable_cart_analysis]" <?php checked($enable_cart_analysis); ?>>
               <?php esc_html_e('Enable AI cart analysis feature', 'ai-product-assistant'); ?>
           </label>
       </fieldset>
       <?php
   }
   
   /**
    * Cart Analysis Title callback
    */
   public function cart_analysis_title_callback() {
       $options = get_option('aipa_settings');
       $cart_analysis_title = isset($options['cart_analysis_title']) ? $options['cart_analysis_title'] : __('AI Cart Analysis', 'ai-product-assistant');
       ?>
       <input type="text" id="cart_analysis_title" name="aipa_settings[cart_analysis_title]" value="<?php echo esc_attr($cart_analysis_title); ?>" class="regular-text" />
       <p class="description"><?php esc_html_e('The title displayed at the top of the cart analysis section.', 'ai-product-assistant'); ?></p>
       <?php
   }
   
   /**
    * Cart Analysis Description callback
    */
   public function cart_analysis_description_callback() {
       $options = get_option('aipa_settings');
       $cart_analysis_description = isset($options['cart_analysis_description']) ? $options['cart_analysis_description'] : __('Ask AI if your cart items will work for your specific needs.', 'ai-product-assistant');
       ?>
       <input type="text" id="cart_analysis_description" name="aipa_settings[cart_analysis_description]" value="<?php echo esc_attr($cart_analysis_description); ?>" class="regular-text" />
       <p class="description"><?php esc_html_e('The description text displayed below the title in the cart analysis section.', 'ai-product-assistant'); ?></p>
       <?php
   }
   
   /**
    * Cart Analysis Placeholder callback
    */
   public function cart_analysis_placeholder_callback() {
       $options = get_option('aipa_settings');
       $cart_analysis_placeholder = isset($options['cart_analysis_placeholder']) ? $options['cart_analysis_placeholder'] : __('e.g., Can I make a basic home theater with these items?', 'ai-product-assistant');
       ?>
       <input type="text" id="cart_analysis_placeholder" name="aipa_settings[cart_analysis_placeholder]" value="<?php echo esc_attr($cart_analysis_placeholder); ?>" class="regular-text" />
       <p class="description"><?php esc_html_e('The placeholder text shown in the cart analysis input field.', 'ai-product-assistant'); ?></p>
       <?php
   }
   
   /**
    * Cart Analysis Button Text callback
    */
   public function cart_analysis_button_text_callback() {
       $options = get_option('aipa_settings');
       $cart_analysis_button_text = isset($options['cart_analysis_button_text']) ? $options['cart_analysis_button_text'] : __('Analyze with AI', 'ai-product-assistant');
       ?>
       <input type="text" id="cart_analysis_button_text" name="aipa_settings[cart_analysis_button_text]" value="<?php echo esc_attr($cart_analysis_button_text); ?>" class="regular-text" />
       <p class="description"><?php esc_html_e('The text displayed on the cart analysis button.', 'ai-product-assistant'); ?></p>
       <?php
   }
   
   /**
    * Cart Analysis CSS callback
    */
   public function cart_analysis_css_callback() {
       $options = get_option('aipa_settings');
       $default_css = "margin: 30px 0;\npadding: 20px;\nborder: 1px solid #eee;\nborder-radius: 8px;\nbackground-color: #f9f9f9;";
       $cart_analysis_css = isset($options['cart_analysis_css']) ? $options['cart_analysis_css'] : $default_css;
       ?>
       <textarea id="cart_analysis_css" name="aipa_settings[cart_analysis_css]" rows="10" class="large-text code"><?php echo esc_textarea($cart_analysis_css); ?></textarea>
       <p class="description"><?php esc_html_e('CSS for the cart analysis section. You can customize its appearance.', 'ai-product-assistant'); ?></p>
       <button type="button" id="cart_analysis_css_reset" class="button button-secondary"><?php esc_html_e('Reset to Default', 'ai-product-assistant'); ?></button>
       <?php
   }

    /**
     * Description Length Limit callback
     */
    public function description_length_limit_callback() {
        $options = get_option('aipa_settings');
        $description_length_limit = isset($options['description_length_limit']) ? intval($options['description_length_limit']) : 500;
        ?>
        <input type="number" id="description_length_limit" name="aipa_settings[description_length_limit]" value="<?php echo esc_attr($description_length_limit); ?>" min="1" step="1">
        <button type="button" id="analyze_description_length" class="button button-secondary"><?php esc_html_e('Analyze Products', 'ai-product-assistant'); ?></button>
        
        <div id="description_length_analysis_result" style="margin-top: 15px; display: none;">
            <div class="aipa-loading-spinner" style="display: none;"></div>
            <div class="aipa-analysis-results"></div>
        </div>
        
        <p class="description">
            <?php esc_html_e('Maximum number of characters to include from product descriptions when analyzing with AI.', 'ai-product-assistant'); ?>
        </p>
        <div class="aipa-help-box">
            <h4><?php esc_html_e('How this affects AI analysis', 'ai-product-assistant'); ?></h4>
            <ul style="list-style-type: disc; margin-left: 20px;">
                <li><?php echo wp_kses(__('<strong>Higher values</strong> (e.g., 1000+): More accurate product matching but higher token consumption and API costs.', 'ai-product-assistant'), array('strong' => array())); ?></li>
                <li><?php echo wp_kses(__('<strong>Lower values</strong> (e.g., 200-300): Reduced token usage and costs, but less context for the AI to match products.', 'ai-product-assistant'), array('strong' => array())); ?></li>
                <li><?php echo wp_kses(__('<strong>Recommended value</strong>: 500 characters provides a good balance between accuracy and cost for most stores.', 'ai-product-assistant'), array('strong' => array())); ?></li>
            </ul>
            <p><?php esc_html_e('Click "Analyze Products" to see statistics about your product descriptions and make an informed decision.', 'ai-product-assistant'); ?></p>
        </div>
        <?php
    }

    /**
     * Short Description Length Limit callback
     */
    public function short_description_length_limit_callback() {
        $options = get_option('aipa_settings');
        $short_description_length_limit = isset($options['short_description_length_limit']) ? intval($options['short_description_length_limit']) : 200;
        ?>
        <input type="number" id="short_description_length_limit" name="aipa_settings[short_description_length_limit]" value="<?php echo esc_attr($short_description_length_limit); ?>" min="1" step="1">
        <button type="button" id="analyze_short_description_length" class="button button-secondary"><?php esc_html_e('Analyze Products', 'ai-product-assistant'); ?></button>
        
        <div id="short_description_length_analysis_result" style="margin-top: 15px; display: none;">
            <div class="aipa-loading-spinner" style="display: none;"></div>
            <div class="aipa-analysis-results"></div>
        </div>
        
        <p class="description">
            <?php esc_html_e('Maximum number of characters to include from product short descriptions when analyzing with AI.', 'ai-product-assistant'); ?>
        </p>
        <div class="aipa-help-box">
            <h4><?php esc_html_e('Setting the right limit', 'ai-product-assistant'); ?></h4>
            <p><?php esc_html_e('Short descriptions are typically more concise and contain key product features. Consider:', 'ai-product-assistant'); ?></p>
            <ul style="list-style-type: disc; margin-left: 20px;">
                <li><?php esc_html_e('Most short descriptions are under 200 characters', 'ai-product-assistant'); ?></li>
                <li><?php esc_html_e('Higher limits may provide better context for AI matching', 'ai-product-assistant'); ?></li>
                <li><?php esc_html_e('Lower limits help reduce API costs for stores with many products', 'ai-product-assistant'); ?></li>
            </ul>
        </div>
        <?php
    }
   
   /**
    * Search Batch Size callback
    */
   public function search_batch_size_callback() {
       $options = get_option('aipa_settings');
       $batch_size = isset($options['search_batch_size']) ? intval($options['search_batch_size']) : 100;
       
       $product_count = (int) wp_count_posts('product')->publish;
       ?>
       <input type="number" id="search_batch_size" name="aipa_settings[search_batch_size]" value="<?php echo esc_attr($batch_size); ?>" min="1" />
       <p class="description">
           <?php esc_html_e('Number of products to analyze per search batch. Higher values provide more comprehensive results but use more API tokens.', 'ai-product-assistant'); ?>
       </p>
       <p class="description">
           <?php 
           echo esc_html(sprintf(
               /* translators: %d: number of products in the store */
               __('Your store has approximately %d products.', 'ai-product-assistant'),
               $product_count
           )); 
           ?>
       </p>
       <?php
   }

   /**
    * Max Batches callback
    */
   public function max_batches_callback() {
       $options = get_option('aipa_settings');
       $batch_size = isset($options['search_batch_size']) ? intval($options['search_batch_size']) : 100;
       
       $product_count = (int) wp_count_posts('product')->publish;

       // Dynamically calculate max batches based on product count and batch size
       // If no products, default to 1
       // Otherwise, calculate ceil(product_count / batch_size)
       $calculated_max_batches = $product_count > 0 ? ceil($product_count / $batch_size) : 1;
       
       // Get stored value or use calculated value
       $max_batches = isset($options['max_batches']) ? intval($options['max_batches']) : $calculated_max_batches;
       
       // Calculate what percentage of products these batches can search
       $total_searchable = $max_batches * $batch_size;
       $percentage = min(100, round(($total_searchable / max(1, $product_count)) * 100));
       
       ?>
       <input type="number" id="max_batches" name="aipa_settings[max_batches]" value="<?php echo esc_attr($max_batches); ?>" min="1" />
       <p class="description">
           <?php esc_html_e('Maximum number of batches to process during search. Only applies when Full Database AI Scan is enabled.', 'ai-product-assistant'); ?>
       </p>
       <p class="description">
           <?php 
           echo esc_html(sprintf(
               /* translators: 1: calculated max batches based on product count */
               __('Based on your store having %1$d products and %2$d products per batch, the recommended maximum batches is %3$d to search all products.', 'ai-product-assistant'),
               $product_count,
               $batch_size,
               $calculated_max_batches
           )); 
           ?>
       </p>
       <p class="description">
           <?php 
           echo esc_html(sprintf(
               /* translators: 1: products per batch, 2: maximum batches, 3: percentage of products, 4: total number of products */
               __('With current settings (%1$d products per batch, %2$d maximum batches), the search will analyze approximately %3$d%% of your total %4$d products.', 'ai-product-assistant'),
               $batch_size,
               $max_batches,
               $percentage,
               $product_count
           )); 
           ?>
       </p>
       <?php
   }

   /**
    * Enable Brute Force Search callback
    */
   public function enable_brute_force_search_callback() {
       $options = get_option('aipa_settings');
       $enabled = isset($options['enable_brute_force_search']) ? (bool) $options['enable_brute_force_search'] : false;
       $is_pro = class_exists('AIPA_API_Pro') && function_exists('aipa_pro_fs') && aipa_pro_fs()->can_use_premium_code();
       ?>
       <fieldset>
           <label for="enable_brute_force_search">
               <input type="checkbox" id="enable_brute_force_search"
                   name="aipa_settings[enable_brute_force_search]"
                   <?php checked($enabled); ?>
                   <?php echo !$is_pro ? 'disabled="disabled"' : ''; ?>>
               <?php esc_html_e('Full Database AI Scan (Brute Force)', 'ai-product-assistant'); ?>
           </label>
           <?php if (!$is_pro) : ?>
               <p class="description" style="color: #d63638;">
                   <?php esc_html_e('Pro version uses Smart Search by default. This option is only available with Pro.', 'ai-product-assistant'); ?>
               </p>
           <?php else : ?>
               <p class="description">
                   <?php esc_html_e('When enabled, skips AI semantic pre-filtering and scans ALL products in the database with AI analysis. Only recommended for stores with a small number of products.', 'ai-product-assistant'); ?>
               </p>
           <?php endif; ?>
       </fieldset>
       <?php
   }

    /**
     * AI Temperature callback
     */
    public function ai_temperature_callback() {
        $options = get_option('aipa_settings');
        $temperature = isset($options['ai_temperature']) ? (float) $options['ai_temperature'] : 0.2;
        $is_pro = class_exists('AIPA_API_Pro') && function_exists('aipa_pro_fs') && aipa_pro_fs()->can_use_premium_code();
        ?>
        <input type="number" id="ai_temperature" name="aipa_settings[ai_temperature]"
            value="<?php echo esc_attr($temperature); ?>"
            min="0" max="2" step="0.1"
            <?php echo !$is_pro ? 'disabled="disabled"' : ''; ?>
            style="width: 80px;">
        <?php if (!$is_pro) : ?>
            <p class="description" style="color: #d63638;">
                <?php esc_html_e('Pro version only.', 'ai-product-assistant'); ?>
            </p>
        <?php else : ?>
            <p class="description">
                <?php esc_html_e('Controls AI response creativity. Lower values (e.g. 0.1) give more consistent, focused responses. Higher values (e.g. 0.8) give more varied responses. Range: 0.0–2.0. Default: 0.2.', 'ai-product-assistant'); ?>
            </p>
        <?php endif; ?>
        <?php
    }

    /**
     * AI Max Tokens callback
     */
    public function ai_max_tokens_callback() {
        $options = get_option('aipa_settings');
        $max_tokens = isset($options['ai_max_tokens']) ? (int) $options['ai_max_tokens'] : 4096;
        $is_pro = class_exists('AIPA_API_Pro') && function_exists('aipa_pro_fs') && aipa_pro_fs()->can_use_premium_code();
        ?>
        <input type="number" id="ai_max_tokens" name="aipa_settings[ai_max_tokens]"
            value="<?php echo esc_attr($max_tokens); ?>"
            min="256" max="8192" step="256"
            <?php echo !$is_pro ? 'disabled="disabled"' : ''; ?>
            style="width: 100px;">
        <?php if (!$is_pro) : ?>
            <p class="description" style="color: #d63638;">
                <?php esc_html_e('Pro version only.', 'ai-product-assistant'); ?>
            </p>
        <?php else : ?>
            <p class="description">
                <?php esc_html_e('Maximum number of tokens in the AI response. Higher values allow longer responses but increase API cost. Range: 256–8192. Default: 4096.', 'ai-product-assistant'); ?>
            </p>
        <?php endif; ?>
        <?php
    }

    /**
     * Show Product Assistant callback
     */
    public function show_product_assistant_callback() {
        $options = get_option('aipa_settings');
        $show_product_assistant = isset($options['show_product_assistant']) ? $options['show_product_assistant'] : true;
        ?>
        <fieldset>
            <label for="show_product_assistant">
                <input type="checkbox" id="show_product_assistant" name="aipa_settings[show_product_assistant]" <?php checked($show_product_assistant); ?>>
                <?php esc_html_e('Display AI assistant button on product pages', 'ai-product-assistant'); ?>
            </label>
        </fieldset>
        <?php
    }

    /**
     * Product Assistant Title callback
     */
    public function product_assistant_title_callback() {
        $options = get_option('aipa_settings');
        $product_assistant_title = isset($options['product_assistant_title']) ? $options['product_assistant_title'] : __('AI Product Assistant', 'ai-product-assistant');
        ?>
        <input type="text" id="product_assistant_title" name="aipa_settings[product_assistant_title]" value="<?php echo esc_attr($product_assistant_title); ?>" class="regular-text" />
        <p class="description"><?php esc_html_e('The title displayed at the top of the product assistant modal or form.', 'ai-product-assistant'); ?></p>
        <?php
    }

    /**
     * Product Assistant Description callback
     */
    public function product_assistant_description_callback() {
        $options = get_option('aipa_settings');
        $product_assistant_description = isset($options['product_assistant_description']) ? $options['product_assistant_description'] : __('Ask AI about this product', 'ai-product-assistant');
        ?>
        <input type="text" id="product_assistant_description" name="aipa_settings[product_assistant_description]" value="<?php echo esc_attr($product_assistant_description); ?>" class="regular-text" />
        <p class="description"><?php esc_html_e('The description text displayed below the title in the product assistant.', 'ai-product-assistant'); ?></p>
        <?php
    }

    /**
     * Product Assistant Placeholder callback
     */
    public function product_assistant_placeholder_callback() {
        $options = get_option('aipa_settings');
        $product_assistant_placeholder = isset($options['product_assistant_placeholder']) ? $options['product_assistant_placeholder'] : __('e.g., Is this product suitable for beginners?', 'ai-product-assistant');
        ?>
        <input type="text" id="product_assistant_placeholder" name="aipa_settings[product_assistant_placeholder]" value="<?php echo esc_attr($product_assistant_placeholder); ?>" class="regular-text" />
        <p class="description"><?php esc_html_e('The placeholder text shown in the product assistant input field.', 'ai-product-assistant'); ?></p>
        <?php
    }

    /**
     * Product Assistant Button Text callback
     */
    public function product_assistant_button_text_callback() {
        $options = get_option('aipa_settings');
        $product_assistant_button_text = isset($options['product_assistant_button_text']) ? $options['product_assistant_button_text'] : __('Ask AI', 'ai-product-assistant');
        ?>
        <input type="text" id="product_assistant_button_text" name="aipa_settings[product_assistant_button_text]" value="<?php echo esc_attr($product_assistant_button_text); ?>" class="regular-text" />
        <p class="description"><?php esc_html_e('The text displayed on the product assistant button.', 'ai-product-assistant'); ?></p>
        <?php
    }

    /**
     * Product Assistant Button CSS callback
     */
    public function product_assistant_button_css_callback() {
        $options = get_option('aipa_settings');
        $default_css = "display: inline-flex !important;\nalign-items: center !important;\njustify-content: center !important;\nbackground-color: #2196F3 !important;\ncolor: white !important;\nborder: none !important;\nborder-radius: 4px !important;\npadding: 8px 16px !important;\ncursor: pointer !important;\nfont-size: 16px !important;\ntransition: all 0.3s ease !important;";
        $product_assistant_button_css = isset($options['product_assistant_button_css']) ? $options['product_assistant_button_css'] : $default_css;
        ?>
        <textarea id="product_assistant_button_css" name="aipa_settings[product_assistant_button_css]" rows="8" class="large-text code"><?php echo esc_textarea($product_assistant_button_css); ?></textarea>
        <p class="description"><?php esc_html_e('CSS for the product assistant button. You can customize its appearance.', 'ai-product-assistant'); ?></p>
        <button type="button" id="product_assistant_button_css_reset" class="button button-secondary"><?php esc_html_e('Reset to Default', 'ai-product-assistant'); ?></button>
        
        <div class="aipa-preview-container" style="margin-top: 15px; border: 1px solid #ddd; padding: 15px; background: #f9f9f9;">
            <h4><?php esc_html_e('Button Preview', 'ai-product-assistant'); ?></h4>
            <div id="product_assistant_button_preview"></div>
        </div>
        
        <?php
        // Add button preview functionality via wp_add_inline_script
        wp_add_inline_script('aipa-admin-script', 
            sprintf('
                jQuery(document).ready(function($) {
                    updateProductAssistantButtonPreview();
                    
                    $("#product_assistant_button_css").on("input", updateProductAssistantButtonPreview);
                    
                    $("#product_assistant_button_css_reset").on("click", function(e) {
                        e.preventDefault();
                        $("#product_assistant_button_css").val("%s");
                        updateProductAssistantButtonPreview();
                    });
                    
                    function updateProductAssistantButtonPreview() {
                        var buttonCss = $("#product_assistant_button_css").val();
                        var buttonText = $("#product_assistant_button_text").val() || "%s";
                        
                        var previewHtml = "<div style=\\"margin: 10px 0;\\">";
                        previewHtml += "<button id=\\"preview_product_assistant_button\\" style=\\"" + buttonCss.replace(/!important/g, "") + "\\">" + buttonText + "</button>";
                        previewHtml += "</div>";
                        
                        $("#product_assistant_button_preview").html(previewHtml);
                    }
                });
            ',
            esc_js($default_css),
            esc_js(__("Ask AI", 'ai-product-assistant'))
            )
        );
        ?>
        <?php
    }
}