/**
 * AI Product Assistant Frontend JavaScript
 */

// Create global namespace
var AIPA_Frontend = (function($) {
    'use strict';
    
    // Private variables
    var searchButton, searchInput, voiceButton, searchModal, closeButton, floatingButton;
    var resultsContainer, resultsLoading, searchError;
    var isSearching = false;
    var speechRecognition = null;
    var searchSessionKey = '';  // Smart search session key for candidate caching
    var searchMaxBatches = 0;   // Persists Smart Search max_batches across batch calls
    
    // Initialize the frontend
    function init() {
        // Main elements
        searchModal = $('#aipa-search-modal');
        searchInput = $('#aipa-search-input');
        searchButton = $('#aipa-search-btn');
        voiceButton = $('#aipa-voice-search-btn');
        closeButton = $('.aipa-close');
        floatingButton = $('#aipa-floating-search-btn');
        resultsContainer = $('#aipa-results-container');
        resultsLoading = $('#aipa-results-loading');
        searchError = $('#aipa-search-error');
        
        // Initialize speech recognition if available
        initSpeechRecognition();
        
        // Event listeners
        bindEvents();
    }
    
    // Bind event listeners
    function bindEvents() {
        // Open modal when floating button is clicked
        floatingButton.on('click', function() {
            searchModal.css('display', 'flex').hide().fadeIn(300);
            searchInput.focus();
        });
        
        // Close modal when close button is clicked
        closeButton.on('click', function() {
            searchModal.fadeOut(300);
            resetSearchForm();
        });
        
        // Close modal when clicking outside the modal content
        $(window).on('click', function(event) {
            if (event.target === searchModal[0]) {
                searchModal.fadeOut(300);
                resetSearchForm();
            }
        });
        
        // Prevent form submission and perform search when search button is clicked
        searchButton.on('click', function(e) {
            e.preventDefault();
            performSearch();
        });
        
        // Perform search when Enter key is pressed in the search input
        searchInput.on('keypress', function(e) {
            if (e.which === 13 && !e.shiftKey) {
                e.preventDefault();
                performSearch();
            }
        });
        
        // Start voice recognition when voice button is clicked
        if (voiceButton.length) {
            voiceButton.on('click', function() {
                startVoiceSearch();
            });
        }
    }
    
    // Initialize speech recognition
    function initSpeechRecognition() {
        if ('webkitSpeechRecognition' in window) {
            speechRecognition = new webkitSpeechRecognition();
            speechRecognition.continuous = false;
            speechRecognition.interimResults = false;
            speechRecognition.lang = document.documentElement.lang || 'en-US';
            
            speechRecognition.onresult = function(event) {
                var transcript = event.results[0][0].transcript;
                searchInput.val(transcript);
                voiceButton.removeClass('recording').html('<span class="aipa-voice-icon">🎤</span> ' + aipa_data.voice_search_button);
                setTimeout(performSearch, 500); // Auto-search after voice input
            };
            
            speechRecognition.onerror = function() {
                voiceButton.removeClass('recording').html('<span class="aipa-voice-icon">🎤</span> ' + aipa_data.voice_search_button);
                displayError(aipa_data.voice_search_not_supported);
            };
            
            speechRecognition.onend = function() {
                voiceButton.removeClass('recording').html('<span class="aipa-voice-icon">🎤</span> ' + aipa_data.voice_search_button);
            };
        } else {
            voiceButton.prop('disabled', true).attr('title', aipa_data.voice_search_not_supported);
        }
    }
    
    // Start voice search
    function startVoiceSearch() {
        if (!speechRecognition) {
            displayError(aipa_data.voice_search_not_supported);
            return;
        }
        
        try {
            searchError.hide().empty();
            voiceButton.addClass('recording').html('<span class="aipa-voice-icon">🎤</span> ' + aipa_data.voice_search_prompt);
            speechRecognition.start();
        } catch (e) {
            voiceButton.removeClass('recording').html('<span class="aipa-voice-icon">🎤</span> ' + aipa_data.voice_search_button);
            displayError(aipa_data.voice_search_not_supported);
        }
    }
    
    // Perform search
    function performSearch(deepSearch = true, batchOffset = 0) {
        // Get the search query
        var query = searchInput.val().trim();
        
        // Validate input
        if (query === '') {
            displayError(aipa_data.empty_query_message);
            return;
        }
        
        // Prevent multiple searches at once
        if (isSearching) {
            return;
        }
        
        // Clear previous error messages
        searchError.hide().empty();
        
        // Show loading indicator
        isSearching = true;
        searchButton.addClass('searching');
        searchButton.find('.aipa-btn-text').text(aipa_data.searching_text);
        
        // Only clear results container for initial search
        if (!deepSearch || batchOffset === 0) {
            resultsContainer.empty();
        }
        
        resultsLoading.show();
        
        var options = aipa_data.search_options || {};

        // New search: reset session key and restore PHP max_batches as starting point
        // Subsequent batches reuse the searchMaxBatches updated by the server response
        if (batchOffset === 0) {
            searchSessionKey = '';
            searchMaxBatches = options.max_batches || 1;
        }

        var currentBatch = batchOffset + 1;
        var is_final_batch = (currentBatch >= searchMaxBatches);

        // Perform AJAX search
        $.ajax({
            url: aipa_data.ajax_url,
            type: 'POST',
            data: {
                action: 'aipa_product_search',
                query: query,
                deep_search: deepSearch,
                batch_offset: batchOffset,
                nonce: aipa_data.nonce,
                is_final_batch: is_final_batch,
                search_session_key: searchSessionKey
            },
            dataType: 'json',
            success: function(response) {
                // Hide loading indicators
                searchButton.removeClass('searching');
                searchButton.find('.aipa-btn-text').text(aipa_data.search_button);
                resultsLoading.hide();
                isSearching = false;

                if (response.success) {
                    // Store smart search session key for subsequent batch calls
                    if (response.data.search_session_key) {
                        searchSessionKey = response.data.search_session_key;
                    }
                    // Smart search returns its own max_batches based on candidate count
                    // Store in module-level var so next batch call uses the correct limit
                    if (response.data.meta && response.data.meta.smart_search && response.data.meta.max_batches) {
                        searchMaxBatches = response.data.meta.max_batches;
                    }
                    var htmlContent = response.data.html || '';

                    // Display results
                    if (response.data.append) {
                        // For deep search, append new products to existing results
                        $('.aipa-deep-search-container').remove(); // Remove old deep search UI
                        resultsContainer.append(htmlContent);
                    } else {
                        resultsContainer.html(htmlContent);
                    }
                    
                    // Handle case when no products found but still in deep search mode
                    if (response.data.count === 0 && deepSearch && htmlContent === '') {
                        $('.aipa-deep-search-container').remove();
                        
                        // Always show progress UI for deep search, regardless of batch offset
                        var progressHtml = '<div class="aipa-deep-search-container">' +
                                        '<div class="aipa-search-progress">' +
                                        '<div class="aipa-progress-bar">' +
                                        '<div class="aipa-progress-fill" style="width: ' + (response.data.meta.progress || 100) + '%;"></div>' +
                                        '</div>' +
                                        '<div class="aipa-progress-info">';
                        
                        if (!response.data.meta.has_more_batches) {
                            progressHtml += 'Batch ' + response.data.meta.max_batches + ' of ' + response.data.meta.max_batches + ' completed - ' + aipa_data.search_complete_message;
                        } else {
                            progressHtml += 'Batch ' + response.data.meta.current_batch + ' of ' + response.data.meta.max_batches + ' completed - Continue searching for more products?';
                        }
                        
                        progressHtml += '</div>' +
                                    '</div>';
                        
                        if (response.data.meta.has_more_batches) {
                            progressHtml += '<button class="aipa-deep-search-btn" data-batch="' + response.data.meta.next_batch_offset + '">' +
                                        '<span class="aipa-deep-search-icon">AI+</span>' +
                                        '<span class="aipa-deep-search-text">' + aipa_data.deep_search_button + '</span>' +
                                        '</button>';
                        }
                        
                        progressHtml += '</div>';
                        
                        resultsContainer.append(progressHtml);
                    }
                    
                    // Bind event for deep search button if present
                    var $deepSearchBtn = $('.aipa-deep-search-btn'); 
                    if ($deepSearchBtn.length) { 
                        $deepSearchBtn.off('click').on('click', function() { 
                            var nextBatch = $(this).data('batch'); 
                            
                            $(this).addClass('searching').text(aipa_data.searching_text); 
                            
                            performSearch(true, nextBatch);
                        });
                    }
                    
                    // Check if we're in a modal before scrolling
                    var isInModal = resultsContainer.closest('.aipa-search-modal').length > 0;
                    
                    // Scroll to results if available (only if not in modal)
                    if (!isInModal && response.data.count > 0 && !deepSearch) {
                        $('html, body').animate({
                            scrollTop: resultsContainer.offset().top - 100
                        }, 500);
                    } else if (!isInModal && deepSearch && $('.aipa-deep-search-container').length) {
                        // Scroll to the deep search container (only if not in modal)
                        $('html, body').animate({
                            scrollTop: $('.aipa-deep-search-container').offset().top - 100
                        }, 500);
                    } else if (response.data.count === 0 && !deepSearch) {
                        displayError(aipa_data.no_results_message);
                    }
                } else {
                    // Display unified AI service error message
                    if (response.data.code === 'aipa_service_error') {
                        displayError(aipa_data.aipa_service_error_message);
                    } else {
                        displayError(response.data.message || aipa_data.error_message, response.data.code);
                    }
                }
            },
            error: function() {
                // Hide loading indicators
                searchButton.removeClass('searching');
                searchButton.find('.aipa-btn-text').text(aipa_data.search_button);
                resultsLoading.hide();
                isSearching = false;
                
                // Display error message
                displayError(aipa_data.error_message);
            }
        });
    }
    
    // Display error message
    function displayError(message, errorCode) {
        var errorHtml = '';
        
        // Check if this is an authentication error
        if (errorCode === 'aipa_auth_required' || message.indexOf('log in') !== -1) {
            errorHtml = '<p>' + message + ' <a href="' + (aipa_data.login_url || '/wp-login.php') + '">' + (aipa_data.login_text || 'Log in here') + '</a></p>';
        } else {
            errorHtml = '<p>' + message + '</p>';
        }
        
        searchError.html(errorHtml).show();
    }
    
    // Reset search form
    function resetSearchForm() {
        searchInput.val('');
        searchError.hide().empty();
        resultsContainer.empty();
        resultsLoading.hide();
        searchButton.removeClass('searching');
        searchButton.find('.aipa-btn-text').text(aipa_data.search_button);
        isSearching = false;
        searchSessionKey = '';
    }
    
    // Initialize shortcode search form
    function initShortcodeSearchForm(inputId, buttonId, voiceButtonId, errorId, loadingId, resultsId) {
        var $input = $('#' + inputId);
        var $button = $('#' + buttonId);
        var $voiceButton = $('#' + voiceButtonId);
        var $error = $('#' + errorId);
        var $loading = $('#' + loadingId);
        var $results = $('#' + resultsId);
        var $resultsContainer = $results.find('.aipa-shortcode-results-container');
        var isSearchingShortcode = false;
        var shortcodeSpeechRecognition = null;
        var shortcodeState = { sessionKey: '', maxBatches: 0 }; // Mutable state shared via reference
        
        // Initialize speech recognition for shortcode if available
        if ('webkitSpeechRecognition' in window && $voiceButton.length) {
            shortcodeSpeechRecognition = new webkitSpeechRecognition();
            shortcodeSpeechRecognition.continuous = false;
            shortcodeSpeechRecognition.interimResults = false;
            shortcodeSpeechRecognition.lang = document.documentElement.lang || 'en-US';
            
            shortcodeSpeechRecognition.onresult = function(event) {
                var transcript = event.results[0][0].transcript;
                $input.val(transcript);
                $voiceButton.removeClass('recording').html('<span class="aipa-voice-icon">🎤</span> ' + aipa_data.voice_search_button);
                setTimeout(function() {
                    performShortcodeSearch($input, $button, $error, $loading, $results, $resultsContainer, isSearchingShortcode, true, 0, shortcodeState);
                }, 500);
            };
            
            shortcodeSpeechRecognition.onerror = function() {
                $voiceButton.removeClass('recording').html('<span class="aipa-voice-icon">🎤</span> ' + aipa_data.voice_search_button);
                displayShortcodeError($error, aipa_data.voice_search_not_supported);
            };
            
            shortcodeSpeechRecognition.onend = function() {
                $voiceButton.removeClass('recording').html('<span class="aipa-voice-icon">🎤</span> ' + aipa_data.voice_search_button);
            };
            
            $voiceButton.on('click', function() {
                try {
                    $error.hide().empty();
                    $voiceButton.addClass('recording').html('<span class="aipa-voice-icon">🎤</span> ' + aipa_data.voice_search_prompt);
                    shortcodeSpeechRecognition.start();
                } catch (e) {
                    $voiceButton.removeClass('recording').html('<span class="aipa-voice-icon">🎤</span> ' + aipa_data.voice_search_button);
                    displayShortcodeError($error, aipa_data.voice_search_not_supported);
                }
            });
        } else if ($voiceButton.length) {
            $voiceButton.prop('disabled', true).attr('title', aipa_data.voice_search_not_supported);
        }
        
        // Handle search button click
        $button.on('click', function(e) {
            e.preventDefault();
            performShortcodeSearch($input, $button, $error, $loading, $results, $resultsContainer, isSearchingShortcode, true, 0, shortcodeState);
        });

        // Handle Enter key in search input
        $input.on('keypress', function(e) {
            if (e.which === 13 && !e.shiftKey) {
                e.preventDefault();
                performShortcodeSearch($input, $button, $error, $loading, $results, $resultsContainer, isSearchingShortcode, true, 0, shortcodeState);
            }
        });
    }
    
    // Perform shortcode search
    function performShortcodeSearch($input, $button, $error, $loading, $results, $resultsContainer, isSearchingShortcode, deepSearch, batchOffset, state) {
        deepSearch  = (deepSearch  !== undefined) ? deepSearch  : true;
        batchOffset = batchOffset || 0;
        if (!state) { state = { sessionKey: '', maxBatches: 0 }; }
        // Get the search query
        var query = $input.val().trim();
        
        // Validate input
        if (query === '') {
            displayShortcodeError($error, aipa_data.empty_query_message);
            return;
        }
        
        // Prevent multiple searches at once
        if (isSearchingShortcode) {
            return;
        }
        
        // Clear previous error messages
        $error.hide().empty();
        
        // Show loading indicator
        isSearchingShortcode = true;
        $button.addClass('searching');
        $button.find('.aipa-btn-text').text(aipa_data.searching_text);
        
        // Only clear results container for initial search
        if (!deepSearch || batchOffset === 0) {
            $resultsContainer.empty();
        }
        
        $loading.show();
        
        var options = aipa_data.search_options || {};

        // New search: reset state to PHP max_batches as starting point
        // Subsequent batches reuse state.maxBatches updated by the server response
        if (batchOffset === 0) {
            state.sessionKey = '';
            state.maxBatches = options.max_batches || 1;
        }

        var currentBatch = batchOffset + 1;
        var is_final_batch = (currentBatch >= state.maxBatches);

        // Perform AJAX search
        $.ajax({
            url: aipa_data.ajax_url,
            type: 'POST',
            data: {
                action: 'aipa_product_search',
                query: query,
                deep_search: deepSearch,
                batch_offset: batchOffset,
                nonce: aipa_data.nonce,
                is_final_batch: is_final_batch,
                search_session_key: state.sessionKey
            },
            dataType: 'json',
            success: function(response) {
                // Hide loading indicators
                $button.removeClass('searching');
                $button.find('.aipa-btn-text').text(aipa_data.search_button);
                $loading.hide();
                isSearchingShortcode = false;

                if (response.success) {
                    // Store smart search session key for subsequent batch calls
                    if (response.data.search_session_key) {
                        state.sessionKey = response.data.search_session_key;
                    }
                    // Smart search returns its own max_batches based on candidate count
                    // Update state (passed by reference) so next batch uses correct limit
                    if (response.data.meta && response.data.meta.smart_search && response.data.meta.max_batches) {
                        state.maxBatches = response.data.meta.max_batches;
                    }
                    var htmlContent = response.data.html || '';
                    
                    // Display results
                    if (response.data.append) {
                        // For deep search, append new products to existing results
                        $results.find('.aipa-deep-search-container').remove(); // Remove old deep search UI
                        $resultsContainer.append(htmlContent);
                    } else {
                        $resultsContainer.html(htmlContent);
                    }
                    
                    // Handle case when no products found but still in deep search mode
                    if (response.data.count === 0 && deepSearch && htmlContent === '') {
                        $results.find('.aipa-deep-search-container').remove();
                        
                        // Always show progress UI for deep search, regardless of batch offset
                        var progressHtml = '<div class="aipa-deep-search-container">' +
                                        '<div class="aipa-search-progress">' +
                                        '<div class="aipa-progress-bar">' +
                                        '<div class="aipa-progress-fill" style="width: ' + (response.data.meta.progress || 100) + '%;"></div>' +
                                        '</div>' +
                                        '<div class="aipa-progress-info">';
                        
                        if (!response.data.meta.has_more_batches) {
                            progressHtml += 'Batch ' + response.data.meta.max_batches + ' of ' + response.data.meta.max_batches + ' completed - ' + aipa_data.search_complete_message;
                        } else {
                            progressHtml += 'Batch ' + response.data.meta.current_batch + ' of ' + response.data.meta.max_batches + ' completed - Continue searching for more products?';
                        }
                        
                        progressHtml += '</div>' +
                                    '</div>';
                        
                        if (response.data.meta.has_more_batches) {
                            progressHtml += '<button class="aipa-deep-search-btn" data-batch="' + response.data.meta.next_batch_offset + '">' +
                                        '<span class="aipa-deep-search-icon">AI+</span>' +
                                        '<span class="aipa-deep-search-text">' + aipa_data.deep_search_button + '</span>' +
                                        '</button>';
                        }
                        
                        progressHtml += '</div>';
                        
                        $resultsContainer.append(progressHtml);
                    }
                    
                    // Bind event for deep search button if present
                    $results.find('.aipa-deep-search-btn').off('click').on('click', function() {
                        var nextBatch = $(this).data('batch');
                        $(this).addClass('searching').text(aipa_data.searching_text);
                        performShortcodeSearch($input, $button, $error, $loading, $results, $resultsContainer, isSearchingShortcode, true, nextBatch, state);
                    });
                    
                    // Scroll to results if available
                    if (response.data.count > 0 && !deepSearch) {
                        $('html, body').animate({
                            scrollTop: $results.offset().top - 100
                        }, 500);
                    } else if (deepSearch && $results.find('.aipa-deep-search-container').length) {
                        // Scroll to the deep search container
                        $('html, body').animate({
                            scrollTop: $results.find('.aipa-deep-search-container').offset().top - 100
                        }, 500);
                    } else if (response.data.count === 0 && !deepSearch) {
                        displayShortcodeError($error, aipa_data.no_results_message);
                    }
                } else {
                    // Display unified AI service error message
                    if (response.data.code === 'aipa_service_error') {
                        displayShortcodeError($error, aipa_data.aipa_service_error_message);
                    } else {
                        displayShortcodeError($error, response.data.message || aipa_data.error_message, response.data.code);
                    }
                }
            },
            error: function() {
                // Hide loading indicators
                $button.removeClass('searching');
                $button.find('.aipa-btn-text').text(aipa_data.search_button);
                $loading.hide();
                isSearchingShortcode = false;
                
                // Display error message
                displayShortcodeError($error, aipa_data.error_message);
            }
        });
    }
    
    // Display shortcode error message
    function displayShortcodeError($error, message, errorCode) {
        var errorHtml = '';
        
        // Check if this is an authentication error
        if (errorCode === 'aipa_auth_required' || message.indexOf('log in') !== -1) {
            errorHtml = '<p>' + message + ' <a href="' + (aipa_data.login_url || '/wp-login.php') + '">' + (aipa_data.login_text || 'Log in here') + '</a></p>';
        } else {
            errorHtml = '<p>' + message + '</p>';
        }
        
        $error.html(errorHtml).show();
    }
    
    // Public API
    return {
        init: init,
        initShortcodeSearchForm: initShortcodeSearchForm
    };
    
})(jQuery);

// Initialize on document ready
jQuery(document).ready(function() {
    AIPA_Frontend.init();
});
