/**
 * AI Product Assistant Cart Analysis JavaScript
 */

// Create global namespace
var AIPA_CartAnalysis = (function($) {
    'use strict';
    
    // Private variables
    var analysisButton, analysisInput;
    var resultsContainer, resultsLoading, analysisError;
    var isAnalyzing = false;
    
    // Initialize the cart analysis
    function init() {
        // Main elements
        analysisButton = $('#aipa-cart-analysis-btn');
        analysisInput = $('#aipa-cart-analysis-input');
        resultsContainer = $('#aipa-analysis-container');
        resultsLoading = $('#aipa-analysis-loading');
        analysisError = $('#aipa-cart-analysis-error');
        
        // Event listeners
        bindEvents();
    }
    
    // Bind event listeners
    function bindEvents() {
        // Perform analysis when analysis button is clicked
        analysisButton.on('click', function(e) {
            e.preventDefault();
            performAnalysis();
        });
        
        // Perform analysis when Enter key is pressed in the analysis input
        analysisInput.on('keypress', function(e) {
            if (e.which === 13 && !e.shiftKey) {
                e.preventDefault();
                performAnalysis();
            }
        });
    }
    
    // Perform cart analysis
    function performAnalysis() {
        // Get the analysis query
        var query = analysisInput.val().trim();
        
        // Validate input
        if (query === '') {
            displayError(aipa_cart_data.empty_query_message);
            return;
        }
        
        // Prevent multiple analyses at once
        if (isAnalyzing) {
            return;
        }
        
        // Clear previous error messages
        analysisError.hide().empty();
        
        // Show loading indicator
        isAnalyzing = true;
        analysisButton.addClass('analyzing');
        analysisButton.find('.aipa-btn-text').text(aipa_cart_data.analyzing_text);
        resultsContainer.empty();
        resultsLoading.show();
        
        // Perform AJAX analysis
        $.ajax({
            url: aipa_cart_data.ajax_url,
            type: 'POST',
            data: {
                action: 'aipa_cart_analysis',
                query: query,
                nonce: aipa_cart_data.nonce
            },
            dataType: 'json',
            success: function(response) {
                // Hide loading indicators
                analysisButton.removeClass('analyzing');
                analysisButton.find('.aipa-btn-text').text(aipa_cart_data.analysis_button);
                resultsLoading.hide();
                isAnalyzing = false;
                
                if (response.success) {
                    // Display results
                    resultsContainer.html(response.data.html);
                    
                    // Scroll to results
                    $('html, body').animate({
                        scrollTop: resultsContainer.offset().top - 100
                    }, 500);
                } else {
                    // Display unified AI service error message
                    if (response.data.code === 'aipa_service_error') {
                        displayError(aipa_cart_data.aipa_service_error_message);
                    } else {
                        displayError(response.data.message || aipa_cart_data.error_message, response.data.code);
                    }
                }
            },
            error: function() {
                // Hide loading indicators
                analysisButton.removeClass('analyzing');
                analysisButton.find('.aipa-btn-text').text(aipa_cart_data.analysis_button);
                resultsLoading.hide();
                isAnalyzing = false;
                
                // Display error message
                displayError(aipa_cart_data.error_message);
            }
        });
    }
    
    // Display error message
    function displayError(message, errorCode) {
        var errorHtml = '';
        
        // Check if this is an authentication error
        if (errorCode === 'aipa_auth_required' || message.indexOf('log in') !== -1) {
            errorHtml = '<p>' + message + ' <a href="' + (aipa_cart_data.login_url || '/wp-login.php') + '">' + (aipa_cart_data.login_text || 'Log in here') + '</a></p>';
        } else {
            errorHtml = '<p>' + message + '</p>';
        }
        
        analysisError.html(errorHtml).show();
    }
    
    // Initialize shortcode cart analysis form
    function initShortcodeCartAnalysis(inputId, buttonId, errorId, loadingId, resultsId) {
        var $input = $('#' + inputId);
        var $button = $('#' + buttonId);
        var $error = $('#' + errorId);
        var $loading = $('#' + loadingId);
        var $results = $('#' + resultsId);
        var $resultsContainer = $results.find('.aipa-shortcode-cart-analysis-container');
        var isAnalyzingShortcode = false;
        
        // Handle analysis button click
        $button.on('click', function(e) {
            e.preventDefault();
            performShortcodeAnalysis($input, $button, $error, $loading, $results, $resultsContainer, isAnalyzingShortcode);
        });
        
        // Handle Enter key in analysis input
        $input.on('keypress', function(e) {
            if (e.which === 13 && !e.shiftKey) {
                e.preventDefault();
                performShortcodeAnalysis($input, $button, $error, $loading, $results, $resultsContainer, isAnalyzingShortcode);
            }
        });
    }
    
    // Perform shortcode cart analysis
    function performShortcodeAnalysis($input, $button, $error, $loading, $results, $resultsContainer, isAnalyzingShortcode) {
        // Get the analysis query
        var query = $input.val().trim();
        
        // Validate input
        if (query === '') {
            displayShortcodeError($error, aipa_cart_data.empty_query_message);
            return;
        }
        
        // Prevent multiple analyses at once
        if (isAnalyzingShortcode) {
            return;
        }
        
        // Clear previous error messages
        $error.hide().empty();
        
        // Show loading indicator
        isAnalyzingShortcode = true;
        $button.addClass('analyzing');
        $button.find('.aipa-btn-text').text(aipa_cart_data.analyzing_text);
        $resultsContainer.empty();
        $loading.show();
        
        // Perform AJAX analysis
        $.ajax({
            url: aipa_cart_data.ajax_url,
            type: 'POST',
            data: {
                action: 'aipa_cart_analysis',
                query: query,
                nonce: aipa_cart_data.nonce
            },
            dataType: 'json',
            success: function(response) {
                // Hide loading indicators
                $button.removeClass('analyzing');
                $button.find('.aipa-btn-text').text(aipa_cart_data.analysis_button);
                $loading.hide();
                isAnalyzingShortcode = false;
                
                if (response.success) {
                    // Display results
                    $resultsContainer.html(response.data.html);
                    
                    // Scroll to results
                    $('html, body').animate({
                        scrollTop: $results.offset().top - 100
                    }, 500);
                } else {
                    // Display unified AI service error message
                    if (response.data.code === 'aipa_service_error') {
                        displayShortcodeError($error, aipa_cart_data.aipa_service_error_message);
                    } else {
                        displayShortcodeError($error, response.data.message || aipa_cart_data.error_message, response.data.code);
                    }
                }
            },
            error: function() {
                // Hide loading indicators
                $button.removeClass('analyzing');
                $button.find('.aipa-btn-text').text(aipa_cart_data.analysis_button);
                $loading.hide();
                isAnalyzingShortcode = false;
                
                // Display error message
                displayShortcodeError($error, aipa_cart_data.error_message);
            }
        });
    }
    
    // Display shortcode error message
    function displayShortcodeError($error, message, errorCode) {
        var errorHtml = '';
        
        // Check if this is an authentication error
        if (errorCode === 'aipa_auth_required' || message.indexOf('log in') !== -1) {
            errorHtml = '<p>' + message + ' <a href="' + (aipa_cart_data.login_url || '/wp-login.php') + '">' + (aipa_cart_data.login_text || 'Log in here') + '</a></p>';
        } else {
            errorHtml = '<p>' + message + '</p>';
        }
        
        $error.html(errorHtml).show();
    }
    
    // Public API
    return {
        init: init,
        initShortcodeCartAnalysis: initShortcodeCartAnalysis
    };
    
})(jQuery);

// Initialize on document ready
jQuery(document).ready(function() {
    AIPA_CartAnalysis.init();
});
