<?php
/**
 * Plugin Name: AI Product Assistant
 * Plugin URI: https://ki-sum.ai/ai-product-assistant
 * Description: AI-powered product search, instant product AI Q&A, and cart analysis plugin for WooCommerce
 * Version: 1.1.0
 * Author: KI-Sum AI
 * Author URI: https://ki-sum.ai
 * Text Domain: ai-product-assistant
 * Domain Path: /languages
 * License: GPL v2 or later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 * Requires at least: 5.6
 * Requires PHP: 7.4
 * WC requires at least: 5.0
 * WC tested up to: 10.5
 */

// If this file is called directly, abort.
if (!defined('WPINC')) {
    die;
}

// Define plugin constants
define('AIPA_VERSION', '1.1.0');
define('AIPA_PLUGIN_DIR', plugin_dir_path(__FILE__));
define('AIPA_PLUGIN_URL', plugin_dir_url(__FILE__));
define('AIPA_PLUGIN_BASENAME', plugin_basename(__FILE__));
define('AIPA_PRO_URL', 'https://ki-sum.ai/ai-product-assistant');

// Check if WooCommerce is active
function aipa_is_woocommerce_active() {
    $active_plugins = (array) get_option('active_plugins', array());
    
    if (is_multisite()) {
        $active_plugins = array_merge($active_plugins, get_site_option('active_sitewide_plugins', array()));
    }
    
    return in_array('woocommerce/woocommerce.php', $active_plugins) || array_key_exists('woocommerce/woocommerce.php', $active_plugins);
}

// Display admin notice if WooCommerce is not active
function aipa_admin_notice_woocommerce_not_active() {
    ?>
    <div class="notice notice-error">
        <p><?php esc_html_e('AI Product Assistant requires WooCommerce to be installed and activated.', 'ai-product-assistant'); ?></p>
    </div>
    <?php
}

// Add Upgrade to Pro link in plugin actions
function aipa_add_plugin_action_links($links) {
    // Only add if Pro version is not active
    if (!defined('AIPA_PRO_VERSION')) {
        $upgrade_link = '<a href="' . esc_url(AIPA_PRO_URL) . '" target="_blank" style="font-weight:bold; color:#FF5722;">' . __('Upgrade to Pro', 'ai-product-assistant') . '</a>';
        
        // Add the link after the "Settings" link
        $settings_link_index = array_search('<a href="' . admin_url('admin.php?page=ai-product-assistant') . '">Settings</a>', $links);
        if ($settings_link_index !== false) {
            array_splice($links, $settings_link_index + 1, 0, $upgrade_link);
        } else {
            $links[] = $upgrade_link;
        }
    }
    
    return $links;
}
add_filter('plugin_action_links_' . plugin_basename(__FILE__), 'aipa_add_plugin_action_links');

// Add upgrade notification at top of admin pages
function aipa_admin_upgrade_notification() {
    // Only show if Pro version is not active and user can manage options
    if (current_user_can('manage_options') && !defined('AIPA_PRO_VERSION')) {
        // Check if notice was dismissed less than 7 days ago
        $dismissed_time = get_user_meta(get_current_user_id(), 'aipa_upgrade_notice_dismissed', true);
        $show_notice = empty($dismissed_time) || ($dismissed_time < strtotime('-7 days'));
        
        if ($show_notice) {
            ?>
            <div class="notice notice-info is-dismissible aipa-upgrade-notice">
                <p>
                    <strong><?php esc_html_e('Enhance your WooCommerce store with AI Product Assistant Pro!', 'ai-product-assistant'); ?></strong>
                    <?php esc_html_e('Get real AI-powered product search, cart analysis, and more.', 'ai-product-assistant'); ?>
                    <a href="<?php echo esc_url(AIPA_PRO_URL); ?>" target="_blank" class="button button-primary" style="margin-left: 10px; background-color: #FF5722; border-color: #E64A19;">
                        <?php esc_html_e('Upgrade to Pro', 'ai-product-assistant'); ?>
                    </a>
                </p>
            </div>
            <?php
        }
    }
}
add_action('admin_notices', 'aipa_admin_upgrade_notification');

// Add JavaScript to persist the upgrade notice dismissal
function aipa_admin_upgrade_notice_js() {
    // Only add script if Pro version is not active and user can manage options
    if (current_user_can('manage_options') && !defined('AIPA_PRO_VERSION')) {
        wp_enqueue_script('aipa-upgrade-notice', AIPA_PLUGIN_URL . 'assets/js/upgrade-notice.js', array('jquery'), AIPA_VERSION, true);
        wp_localize_script('aipa-upgrade-notice', 'aipa_upgrade_notice_data', array(
            'ajaxurl' => admin_url('admin-ajax.php'),
            'nonce' => wp_create_nonce('aipa_dismiss_upgrade_notice')
        ));
    }
}
add_action('admin_footer', 'aipa_admin_upgrade_notice_js');

// AJAX handler for dismissing the notice
function aipa_dismiss_upgrade_notice() {
    check_ajax_referer('aipa_dismiss_upgrade_notice', 'nonce');
    
    // Check if user has permission to manage options
    if (!current_user_can('manage_options')) {
        wp_die(esc_html__('Insufficient permissions.', 'ai-product-assistant'));
    }
    
    update_user_meta(get_current_user_id(), 'aipa_upgrade_notice_dismissed', time());
    wp_die();
}
add_action('wp_ajax_aipa_dismiss_upgrade_notice', 'aipa_dismiss_upgrade_notice');

// API Factory class
class AIPA_API_Factory {
    private static $instance = null;
    private static $initialized = false;
    
    /**
     * Get the API implementation instance
     */
    public static function get_instance() {
        if (self::$instance === null && !self::$initialized) {
            // Check if Pro version API is available
            if (class_exists('AIPA_API_Pro')) {
                self::$instance = new AIPA_API_Pro();
                //AIPA_Debug::log(AIPA_Debug::LEVEL_DEBUG, 'AIPA_Factory', 'Created Pro instance');
            } else {
                self::$instance = new AIPA_API();
                //AIPA_Debug::log(AIPA_Debug::LEVEL_DEBUG, 'AIPA_Factory', 'Created Free instance');
            }
            self::$initialized = true;
        }
        
        //AIPA_Debug::log(AIPA_Debug::LEVEL_DEBUG, 'AIPA_Factory', 'Returning instance of ' . get_class(self::$instance));
        return self::$instance;
    }
    
    /**
     * Set a custom API implementation
     */
    public static function set_instance($api_instance) {
        if (!is_a($api_instance, 'AIPA_API')) {
            throw new Exception('API instance must extend AIPA_API class');
        }
        
        self::$instance = $api_instance;
        self::$initialized = true;
    }
    
    /**
     * Reset the instance (mainly for Pro version to override)
     */
    public static function reset() {
        self::$instance = null;
        self::$initialized = false;
    }
}

// Initialize the plugin
function aipa_init() {
    // Include required files
    require_once AIPA_PLUGIN_DIR . 'includes/class-aipa-debug.php';
    require_once AIPA_PLUGIN_DIR . 'includes/admin/class-aipa-admin-core.php';
    require_once AIPA_PLUGIN_DIR . 'includes/admin/class-aipa-admin-ai-provider.php';
    require_once AIPA_PLUGIN_DIR . 'includes/admin/class-aipa-admin-debug.php';
    require_once AIPA_PLUGIN_DIR . 'includes/admin/class-aipa-admin-tabs.php';
    require_once AIPA_PLUGIN_DIR . 'includes/admin/class-aipa-admin.php';
    require_once AIPA_PLUGIN_DIR . 'includes/admin/class-aipa-review-notice.php';

    require_once AIPA_PLUGIN_DIR . 'includes/class-aipa-api.php';
    require_once AIPA_PLUGIN_DIR . 'includes/class-aipa-product-search.php';
    require_once AIPA_PLUGIN_DIR . 'includes/class-aipa-cart-analysis.php';
    require_once AIPA_PLUGIN_DIR . 'includes/class-aipa-shortcodes.php';
    require_once AIPA_PLUGIN_DIR . 'includes/class-aipa-product-assistant.php';
    
    // Initialize admin
    $admin = new AIPA_Admin();
    $admin->init();
    
    // Delay API initialization until all plugins are loaded
    add_action('init', function() {
        $api = AIPA_API_Factory::get_instance();
        $api->init();
    }, 5);
    
    // Initialize product search
    $product_search = new AIPA_Product_Search();
    $product_search->init();
    
    // Initialize cart analysis
    $cart_analysis = new AIPA_Cart_Analysis();
    $cart_analysis->init();
    
    // Initialize shortcodes
    $shortcodes = new AIPA_Shortcodes();
    $shortcodes->init();

    // Initialize product assistant
    $product_assistant = new AIPA_Product_Assistant();
    $product_assistant->init();
    
    // Ensure install time is recorded
    if (!get_option('aipa_install_time')) {
        update_option('aipa_install_time', time());
    }
    
    // Add Pro submenu link that opens in a new tab
    add_action('admin_menu', 'aipa_add_upgrade_submenu', 99);
}

// Add direct upgrade to pro submenu that opens in a new tab
function aipa_add_upgrade_submenu() {
    // Only show if Pro version is not active
    if (!defined('AIPA_PRO_VERSION')) {
        add_submenu_page(
            'ai-product-assistant',
            __('Upgrade to Pro', 'ai-product-assistant'),
            '<span style="color:#fff; background:#FF5722; padding:3px 8px; border-radius:3px; display:block; text-align:center; font-weight:bold; margin:3px 0;">' . __('Upgrade to Pro', 'ai-product-assistant') . '</span>',
            'manage_options',
            'aipa_redirect_to_pro',
            '__return_false'
        );
    }
}

// Redirect for Pro submenu
function aipa_redirect_to_pro_page() {
    // Check if this is the redirect page and user has proper permissions
    // phpcs:ignore WordPress.Security.NonceVerification.Recommended
    if (isset($_GET['page']) && current_user_can('manage_options')) {
        // phpcs:ignore WordPress.Security.NonceVerification.Recommended
        $page = sanitize_text_field(wp_unslash($_GET['page']));
        if ($page === 'aipa_redirect_to_pro') {
            // This is a low-risk redirect operation for admin users
            wp_redirect(AIPA_PRO_URL);
            exit;
        }
    }
}
add_action('admin_init', 'aipa_redirect_to_pro_page');

// Check if WooCommerce is active before initializing the plugin
if (aipa_is_woocommerce_active()) {
    add_action('plugins_loaded', 'aipa_init');
} else {
    add_action('admin_notices', 'aipa_admin_notice_woocommerce_not_active');
}

/**
 * Plugin activation hook
 */
register_activation_hook(__FILE__, 'aipa_activate');
function aipa_activate() {
    // Record installation time for review notice
    if (!get_option('aipa_install_time')) {
        update_option('aipa_install_time', time());
    }
    
    // Create necessary database tables and default options
    add_option('aipa_settings', array(
        'api_provider' => 'openai',
        'api_key' => '',
        'api_url' => '',
        'api_model' => 'gpt-3.5-turbo',
        'format_template' => 'openai',
        'search_fields' => array(
            'title' => true,
            'short_description' => false,
            'content' => false,
            'categories' => false,
            'attributes' => false,
            'regular_price' => false,
            'sale_price' => false
        ),
        'show_out_of_stock' => false,
        'show_explanation' => true,
        'show_add_to_cart' => true,
        'enable_cart_analysis' => true,
        'use_native_template' => true,
        'prompt_prefix' => '',
        'prompt_suffix' => '',
        'search_batch_size' => 100,
        'enable_deep_search' => true
    ));
}

// Plugin deactivation hook
register_deactivation_hook(__FILE__, 'aipa_deactivate');
function aipa_deactivate() {
    // Cleanup if necessary
}

// Plugin uninstall hook (must be in a separate file)
// register_uninstall_hook(__FILE__, 'aipa_uninstall');

// Add at the bottom of the main plugin file
// Debug mode function
function aipa_debug_mode() {
    // Only display when debug mode is enabled and user has proper permissions
    // phpcs:ignore WordPress.Security.NonceVerification.Recommended
    if (isset($_GET['aipa_debug']) && current_user_can('manage_options')) {
        // phpcs:ignore WordPress.Security.NonceVerification.Recommended
        $debug_param = sanitize_text_field(wp_unslash($_GET['aipa_debug']));
        if ($debug_param) {
            // Additional security: only show for admin users and in specific contexts
            if (!current_user_can('manage_options')) {
                return;
            }
        }
        // Get settings
        $options = get_option('aipa_settings');
        
        echo '<div style="position: fixed; bottom: 80px; right: 20px; background: #fff; padding: 15px; border: 1px solid #ddd; z-index: 9999; max-height: 80vh; overflow-y: auto; width: 350px; font-size: 12px;">';
        echo '<h4 style="margin-top: 0;">AI Product Assistant Debug Information</h4>';
        echo '<p>Plugin Version: ' . esc_html(AIPA_VERSION) . '</p>';
        
        // API Settings
        echo '<div style="margin-bottom: 15px; border-bottom: 1px solid #eee; padding-bottom: 10px;">';
        echo '<strong>API Settings:</strong><br>';
        echo 'API Provider: ' . esc_html(isset($options['api_provider']) ? $options['api_provider'] : 'Not set') . '<br>';
        echo 'API URL: ' . esc_html(isset($options['api_url']) && !empty($options['api_url']) ? $options['api_url'] : 'Using default URL') . '<br>';
        echo 'API Model: ' . esc_html(isset($options['api_model']) ? $options['api_model'] : 'Not set') . '<br>';
        echo 'API Key Set: ' . esc_html(isset($options['api_key']) && !empty($options['api_key']) ? 'Yes' : 'No') . '<br>';
        if (isset($options['format_template'])) {
            echo 'Response Format Template: ' . esc_html($options['format_template']) . '<br>';
        }
        echo '</div>';
        
        // Search Settings
        echo '<div style="margin-bottom: 15px; border-bottom: 1px solid #eee; padding-bottom: 10px;">';
        echo '<strong>Search Settings:</strong><br>';
        
        // Search Fields
        echo 'Search Fields: ';
        $search_fields = isset($options['search_fields']) ? $options['search_fields'] : array();
        $enabled_fields = array();
        if (isset($search_fields['title']) && $search_fields['title']) $enabled_fields[] = 'Title';
        if (isset($search_fields['short_description']) && $search_fields['short_description']) $enabled_fields[] = 'Short Description';
        if (isset($search_fields['content']) && $search_fields['content']) $enabled_fields[] = 'Content';
        if (isset($search_fields['categories']) && $search_fields['categories']) $enabled_fields[] = 'Categories';
        if (isset($search_fields['attributes']) && $search_fields['attributes']) $enabled_fields[] = 'Attributes';
        if (isset($search_fields['regular_price']) && $search_fields['regular_price']) $enabled_fields[] = 'Regular Price';
        if (isset($search_fields['sale_price']) && $search_fields['sale_price']) $enabled_fields[] = 'Sale Price';

        echo esc_html(implode(', ', $enabled_fields)) . '<br>';
        
        // Search Button Settings
        echo 'Show Floating Search Button: ' . (isset($options['show_floating_button']) && $options['show_floating_button'] ? 'Yes' : 'No') . '<br>';
        echo 'Search Button Icon Type: ' . esc_html(isset($options['search_button_icon_type']) ? $options['search_button_icon_type'] : 'Default') . '<br>';
        echo 'Search Button Icon Text: ' . esc_html(isset($options['search_button_icon']) ? $options['search_button_icon'] : 'Not set') . '<br>';
        echo 'Search Button Icon Image: ' . (isset($options['search_button_icon_image']) && !empty($options['search_button_icon_image']) ? 'Set' : 'Not set') . '<br>';

        // Inline Search Button Settings
        echo 'Inline Button Icon Type: ' . esc_html(isset($options['button_icon_type']) ? $options['button_icon_type'] : 'Default') . '<br>';
        echo 'Inline Button Icon Text: ' . esc_html(isset($options['button_icon']) ? $options['button_icon'] : 'Not set') . '<br>';
        echo 'Inline Button Icon Image: ' . (isset($options['button_icon_image']) && !empty($options['button_icon_image']) ? 'Set' : 'Not set') . '<br>';

        // Display Options
        echo 'Show Out of Stock Products: ' . (isset($options['show_out_of_stock']) && $options['show_out_of_stock'] ? 'Yes' : 'No') . '<br>';
        echo 'Show AI Explanation: ' . (isset($options['show_explanation']) && $options['show_explanation'] ? 'Yes' : 'No') . '<br>';
        echo 'Show Add to Cart Button: ' . (isset($options['show_add_to_cart']) && $options['show_add_to_cart'] ? 'Yes' : 'No') . '<br>';
        echo 'Use Native Template: ' . (isset($options['use_native_template']) && $options['use_native_template'] ? 'Yes' : 'No') . '<br>';
        echo '</div>';
        
        // Cart Analysis Settings
        echo '<div style="margin-bottom: 15px; border-bottom: 1px solid #eee; padding-bottom: 10px;">';
        echo '<strong>Cart Analysis Settings:</strong><br>';
        echo 'Enable Cart Analysis: ' . (isset($options['enable_cart_analysis']) && $options['enable_cart_analysis'] ? 'Yes' : 'No') . '<br>';
        echo '</div>';
        
        // Performance Settings
        echo '<div style="margin-bottom: 15px; border-bottom: 1px solid #eee; padding-bottom: 10px;">';
        echo '<strong>Performance Settings:</strong><br>';
        echo 'Products Per Batch: ' . esc_html(isset($options['search_batch_size']) ? $options['search_batch_size'] : '100') . '<br>';
        
        // Calculate dynamic default for max batches
        $display_max_batches = 1; // fallback
        if (isset($options['max_batches'])) {
            $display_max_batches = $options['max_batches'];
        } else {
            $batch_size_for_calc = isset($options['search_batch_size']) ? intval($options['search_batch_size']) : 100;
            if ($product_count > 0) {
                $display_max_batches = ceil($product_count / $batch_size_for_calc);
            }
        }
        echo 'Max Batches for Deep Search: ' . esc_html($display_max_batches) . '<br>';
        // Description length limits
        echo 'Product Description Length Limit: ' . esc_html(isset($options['description_length_limit']) ? $options['description_length_limit'] : 'Not set') . '<br>';
        echo 'Short Description Length Limit: ' . esc_html(isset($options['short_description_length_limit']) ? $options['short_description_length_limit'] : 'Not set') . '<br>';

        // Product count information
        $product_count = (int) wp_count_posts('product')->publish;
        echo 'Total Products: ' . esc_html($product_count) . '<br>';
        
        // Calculate approximate token usage
        $batch_size = isset($options['search_batch_size']) ? intval($options['search_batch_size']) : 100;
        $token_estimate = $batch_size * 150 + 300; // Rough estimate
        echo 'Estimated Token Usage Per Batch: ~' . esc_html($token_estimate) . '<br>';
        
        // Use the same dynamic calculation for max batches
        echo 'Estimated Token Usage for Full Deep Search: ~' . esc_html($token_estimate * $display_max_batches) . '<br>';
        echo '</div>';
        
        // Script Loading Check
        echo '<div style="margin-bottom: 15px;">';
        echo '<strong>Script Loading Check:</strong><br>';
        echo 'Frontend JS Loaded: ' . (wp_script_is('aipa-frontend-js') ? 'Yes' : 'No') . '<br>';
        echo 'Frontend CSS Loaded: ' . (wp_style_is('aipa-frontend-css') ? 'Yes' : 'No') . '<br>';
        echo 'Admin JS Loaded: ' . (wp_script_is('aipa-admin-script') ? 'Yes' : 'No') . '<br>';
        echo 'Admin CSS Loaded: ' . (wp_style_is('aipa-admin-style') ? 'Yes' : 'No') . '<br>';
        echo '</div>';
        
        echo '</div>';
    }
}
add_action('wp_footer', 'aipa_debug_mode');
add_action('admin_footer', 'aipa_debug_mode'); // Also show in admin interface