<?php
/**
 * Frontend handler for UncertifyAI plugin
 *
 * @package UncertifyAI
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

/**
 * Frontend handler class
 */
class UncertifyAI_Frontend {

    /**
     * API handler instance
     *
     * @var UncertifyAI_API
     */
    private $api;

    /**
     * Database handler instance
     *
     * @var UncertifyAI_DB
     */
    private $db;

    /**
     * User data from API
     *
     * @var array
     */
    private $user_data;

    /**
     * Constructor
     *
     * @param UncertifyAI_API $api API handler.
     * @param UncertifyAI_DB  $db Database handler.
     */
    public function __construct( $api, $db ) {
        $this->api = $api;
        $this->db = $db;
        
        // Get user data if API is configured
        if ($this->api->is_configured()) {
            $this->user_data = $this->api->get_user_data();
        }
        
        $this->init_hooks();
    }

    /**
     * Initialize WordPress hooks
     */
    private function init_hooks() {
        add_action( 'wp_enqueue_scripts', array( $this, 'enqueue_scripts' ) );
        add_filter( 'the_content', array( $this, 'display_ai_check_results' ) );
    }

    /**
     * Check if user has free subscription
     *
     * @return bool
     */
    private function is_free_user() {
        return isset($this->user_data['subscription_plan']) && 
               $this->user_data['subscription_plan'] === 'free';
    }

    /**
     * Check if certificate should be shown
     *
     * @return bool
     */
    private function should_show_certificate() {
        $options = get_option( 'uncertifyai_settings', array() );
        
        // Free users always show certificate
        if ($this->is_free_user()) {
            return true;
        }
        
        // Premium users respect the settings
        return isset( $options['show_symbol'] ) && $options['show_symbol'];
    }

    /**
     * Get local image URL based on AI probability
     *
     * @param float $probability AI probability score.
     * @return string Local image URL.
     */
    private function get_local_image_url( $probability ) {
        $image_name = 'green.png'; // Default
        
        if ( $probability > 0.7 ) {
            $image_name = 'red.png';
        } elseif ( $probability > 0.4 ) {
            $image_name = 'orange.png';
        }
        
        return UNCERTIFYAI_PLUGIN_URL . 'assets/images/' . $image_name;
    }

    /**
     * Enqueue frontend scripts
     */
    public function enqueue_scripts() {
        if ($this->should_show_certificate()) {
            wp_enqueue_style( 'uncertifyai-frontend-style', UNCERTIFYAI_PLUGIN_URL . 'assets/frontend-style.css', array(), UNCERTIFYAI_VERSION );
        }
    }

    /**
     * Display AI check results after content
     *
     * @param string $content Post content.
     * @return string
     */
    public function display_ai_check_results( $content ) {
        $options = get_option( 'uncertifyai_settings', array() );
        
        // Only show results if enabled and we're on a single post/page
        if ( ! is_singular() || ! isset( $options['display_results'] ) || ! $options['display_results'] ) {
            return $content;
        }
        
        $post_id = get_the_ID();
        $ai_check = get_post_meta( $post_id, '_uncertifyai_ai_check', true );
        
        if ( $ai_check ) {
            $local_image_url = $this->get_local_image_url( $ai_check['ai_probability'] );
            
            $results_html = '<div class="uncertifyai-results-frontend">';
            $results_html .= ( $this->should_show_certificate() ? '<div><img src="'. esc_url( $local_image_url ) .'" alt="'. esc_attr__("AI Detection & AI Humanization By Uncertify AI", 'ai-humanizer-ai-detector-uncertify-ai').'" /></div>' : '');
            $results_html .= '<div>';
            $results_html .= '<h2>' . esc_html__( 'AI Content Analysis', 'ai-humanizer-ai-detector-uncertify-ai' ) . '</h2>';
            $results_html .= '<p>' . esc_html__( 'This content has been analyzed for AI generation:', 'ai-humanizer-ai-detector-uncertify-ai' ) . '</p>';
            $results_html .= '<ul>';
            $results_html .= '<li style="color:'.  ( $ai_check['ai_probability'] > 0.7 ? 'red' : ( $ai_check['ai_probability'] > 0.4 ? 'orange' : 'green' ) ).'">' . esc_html__( 'AI Probability:', 'ai-humanizer-ai-detector-uncertify-ai' ) . ' ' . esc_html( round($ai_check['ai_probability']*100,2) ) . '%</li>';
            $results_html .= '<li style="color:'.  ( $ai_check['ai_probability'] > 0.7 ? 'red' : ( $ai_check['ai_probability'] > 0.4 ? 'orange' : 'green' ) ).'">' . esc_html__( 'Confidence:', 'ai-humanizer-ai-detector-uncertify-ai' ) . ' ' . esc_html( $ai_check['confidence'] ) . '</li>';
            $results_html .= '<li>' . esc_html__( 'Last Checked:', 'ai-humanizer-ai-detector-uncertify-ai' ) . ' ' . esc_html( date_i18n( get_option( 'date_format' ), $ai_check['timestamp'] ) ) . '</li>';
            $results_html .= '</ul>';
            // Add note for free users
            if ($this->is_free_user()) {
                $results_html .= '<div class="uncertifyai-free-user-note">';
                $results_html .= '<h4>'.esc_html__('Certify By Uncertify AI | ', 'ai-humanizer-ai-detector-uncertify-ai').'<a href="https://uncertifyai.com">'.esc_html__('AI Detecotor', 'ai-humanizer-ai-detector-uncertify-ai').'</a> '.esc_html__(' & ', 'ai-humanizer-ai-detector-uncertify-ai').' <a href="https://uncertifyai.com">'.esc_html__('AI Humanizer', 'ai-humanizer-ai-detector-uncertify-ai').'</a></h4>';
                $results_html .= '</div>';
            }
            
            $results_html .= '</div></div>';
            
            $content .= $results_html;
        }
        
        return $content;
    }
}