<?php
/**
 * Database handler for UncertifyAI plugin
 *
 * @package UncertifyAI
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

/**
 * Database handler class
 */
class UncertifyAI_DB {

    /**
     * Create custom tables
     */
    public function create_tables() {
        global $wpdb;

        $charset_collate = $wpdb->get_charset_collate();
        $table_name      = $wpdb->prefix . sanitize_key( 'uncertifyai_checks' );

        $sql = "CREATE TABLE IF NOT EXISTS `$table_name` (
            id mediumint(9) NOT NULL AUTO_INCREMENT,
            post_id bigint(20) NOT NULL,
            ai_probability float NOT NULL,
            confidence varchar(100) NOT NULL,
            checked_by bigint(20) NOT NULL,
            checked_on datetime DEFAULT CURRENT_TIMESTAMP,
            revision_id bigint(20) DEFAULT 0,
            PRIMARY KEY (id),
            KEY post_id (post_id),
            KEY checked_by (checked_by),
            KEY revision_id (revision_id)
        ) $charset_collate;";

        require_once ABSPATH . 'wp-admin/includes/upgrade.php';
        dbDelta( $sql );

        // Set default options
        $default_options = array(
            'api_key'          => '',
            'check_posts'      => 1,
            'check_pages'      => 0,
            'check_categories' => array(),
            'display_results'  => 0,
            'show_symbol'      => 1,
            'auto_recheck'     => 0,
        );

        if ( false === get_option( 'uncertifyai_settings' ) ) {
            add_option( 'uncertifyai_settings', $default_options );
        }
    }

    /**
     * Get check history for a post
     */
    public function get_check_history( $post_id ) {
        global $wpdb;
        $table_name  = esc_sql($wpdb->prefix . sanitize_key( 'uncertifyai_checks' ));
        $cache_key   = "uncertifyai_check_history_{$post_id}";
        $cache_group = 'uncertifyai';

        $results = wp_cache_get( $cache_key, $cache_group );
        if ( false === $results ) {
            $sql = $wpdb->prepare(
                // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared
                "SELECT * FROM $table_name WHERE post_id = %d ORDER BY checked_on DESC LIMIT 10",
                $post_id
            );
            // phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared, WordPress.DB.DirectDatabaseQuery.DirectQuery
            $results = $wpdb->get_results( $sql );
            wp_cache_set( $cache_key, $results, $cache_group, MINUTE_IN_SECONDS * 5 );
        }

        return $results;
    }

    /**
     * Save check result
     */
    public function save_check_result( $post_id, $result, $revision_id = 0 ) {
        global $wpdb;
        $table_name = esc_sql($wpdb->prefix . sanitize_key( 'uncertifyai_checks' ));
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery
        $inserted = $wpdb->insert(
            $table_name,
            array(
                'post_id'        => $post_id,
                'ai_probability' => $result['ai_probability'],
                'confidence'     => $result['confidence'],
                'checked_by'     => get_current_user_id(),
                'checked_on'     => current_time( 'mysql' ),
                'revision_id'    => $revision_id,
            ),
            array( '%d', '%f', '%s', '%d', '%s', '%d' )
        );

        if ( $inserted ) {
            wp_cache_delete( "uncertifyai_check_history_{$post_id}", 'uncertifyai' );
            wp_cache_delete( "uncertifyai_recent_checks", 'uncertifyai' );
            wp_cache_delete( "uncertifyai_check_counts", 'uncertifyai' );
        }

        return $inserted;
    }

    /**
     * Get recent checks
     */
    public function get_recent_checks( $limit = 100 ) {
        global $wpdb;
        $table_name  = esc_sql($wpdb->prefix . sanitize_key( 'uncertifyai_checks' ));
        $cache_key   = "uncertifyai_recent_checks_{$limit}";
        $cache_group = 'uncertifyai';

        $results = wp_cache_get( $cache_key, $cache_group );
        if ( false === $results ) {
            $sql = $wpdb->prepare(
                // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared
                "SELECT c.*, p.post_title, u.user_nicename FROM $table_name c
                 LEFT JOIN `{$wpdb->posts}` p ON c.post_id = p.ID
                 LEFT JOIN `{$wpdb->users}` u ON c.checked_by = u.ID
                 ORDER BY c.checked_on DESC
                 LIMIT %d",
                $limit
            );
            // phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared, WordPress.DB.DirectDatabaseQuery.DirectQuery
            $results = $wpdb->get_results( $sql );
            wp_cache_set( $cache_key, $results, $cache_group, MINUTE_IN_SECONDS * 5 );
        }

        return $results;
    }

    /**
     * Get check counts
     */
    public function get_check_counts() {
        global $wpdb;
        $table_name  = esc_sql($wpdb->prefix . sanitize_key( 'uncertifyai_checks' ));
        $cache_key   = "uncertifyai_check_counts";
        $cache_group = 'uncertifyai';

        $counts = wp_cache_get( $cache_key, $cache_group );
        if ( false === $counts ) {
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery
            $total = $wpdb->get_var(
                // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared
                $wpdb->prepare("SELECT COUNT(*) FROM $table_name")
            );
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery
            $today = $wpdb->get_var(
                $wpdb->prepare(
                    // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared
                    "SELECT COUNT(*) FROM $table_name WHERE DATE(checked_on) = %s",
                    gmdate('Y-m-d')
                )
            );

            $counts = array(
                'total' => (int) $total,
                'today' => (int) $today,
            );

            wp_cache_set( $cache_key, $counts, $cache_group, MINUTE_IN_SECONDS * 5 );
        }

        return $counts;
    }
}