<?php
/**
 * API handler for UncertifyAI plugin
 *
 * @package UncertifyAI
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

/**
 * API handler class
 */
class UncertifyAI_API {

    /**
     * API base URL
     *
     * @var string
     */
    private $api_url = 'https://api.uncertifyai.com';

    /**
     * API key
     *
     * @var string
     */
    private $api_key;

    /**
     * Constructor
     */
    public function __construct() {
        $options = get_option( 'uncertifyai_settings', array() );
        $this->api_key = isset( $options['api_key'] ) ? $options['api_key'] : '';
    }

    /**
     * Make API request
     *
     * @param string $endpoint API endpoint.
     * @param array  $data Request data.
     * @param string $method HTTP method.
     * @return array|WP_Error
     */
    private function request( $endpoint, $data = array(), $method = 'POST' ) {
        $url = $this->api_url . $endpoint;
        
        $args = array(
            'headers' => array(
                'Content-Type' => 'application/json',
                'Authorization' => 'Bearer ' . $this->api_key,
            ),
            'timeout' => 30,
        );
        
        if ( 'POST' === $method ) {
            $args['body'] = wp_json_encode( $data );
        } elseif ( 'GET' === $method && ! empty( $data ) ) {
            $url = add_query_arg( $data, $url );
        }
        
        if ( 'GET' === $method ) {
            $response = wp_remote_get( $url, $args );
        } else {
            $response = wp_remote_post( $url, $args );
        }
        
        if ( is_wp_error( $response ) ) {
            return $response;
        }
        
        $body = wp_remote_retrieve_body( $response );
        $decoded = json_decode( $body, true );
        
        if ( json_last_error() !== JSON_ERROR_NONE ) {
            return new WP_Error( 'invalid_json', __( 'Invalid JSON response from API', 'ai-humanizer-ai-detector-uncertify-ai' ) );
        }
        
        return $decoded;
    }

    /**
     * Detect AI content
     *
     * @param string $content Content to analyze.
     * @return array|WP_Error
     */
    public function detect_ai_content( $content ) {
        return $this->request( '/api/detection/analyze', array(
            'text' => $content,
            'method' => 'ensemble'
        ) );
    }

    /**
     * Humanize content
     *
     * @param string $content Content to humanize.
     * @return array|WP_Error
     */
    public function humanize_content( $content ) {
        return $this->request( '/api/humanize/process', array(
            'text' => $content,
            'intensity' => 0.7
        ) );
    }

    /**
     * Get user data
     *
     * @return array|WP_Error
     */
    public function get_user_data() {
        $user_data = get_transient( 'uncertifyai_user_data' );
        
        if ( ! $user_data && ! empty( $this->api_key ) ) {
            $response = $this->request( '/api/users/me', array(), 'GET' );
            
            if ( ! is_wp_error( $response ) && isset( $response )) {
                $user_data = $response;
                set_transient( 'uncertifyai_user_data', $user_data, HOUR_IN_SECONDS );
            }
        }
        
        return $user_data;
    }

    /**
     * Get usage data
     *
     * @param bool $force_refresh Whether to force refresh the data.
     * @return array|WP_Error
     */
    public function get_usage_data( $force_refresh = false ) {
        if ( $force_refresh ) {
            delete_transient( 'uncertifyai_usage_data' );
        }
        
        $usage_data = get_transient( 'uncertifyai_usage_data' );
        
        if ( ! $usage_data && ! empty( $this->api_key ) ) {
            $response = $this->request( '/api/users/me/usage', array(), 'GET' );
            
            if ( ! is_wp_error( $response ) && isset( $response ) ) {
                $usage_data = $response;
                set_transient( 'uncertifyai_usage_data', $usage_data, 5 * MINUTE_IN_SECONDS );
            }
        }
        
        return $usage_data;
    }

    /**
     * Check if API is configured
     *
     * @return bool
     */
    public function is_configured() {
        return ! empty( $this->api_key );
    }
}