<?php
/**
 * Admin handler for UncertifyAI plugin
 *
 * @package UncertifyAI
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

/**
 * Admin handler class
 */
class UncertifyAI_Admin {

    /**
     * API handler instance
     *
     * @var UncertifyAI_API
     */
    private $api;

    /**
     * Database handler instance
     *
     * @var UncertifyAI_DB
     */
    private $db;

    /**
     * Constructor
     *
     * @param UncertifyAI_API $api API handler.
     * @param UncertifyAI_DB  $db Database handler.
     */
    public function __construct( $api, $db ) {
        $this->api = $api;
        $this->db = $db;
        
        $this->init_hooks();
    }

    /**
     * Initialize WordPress hooks
     */
    private function init_hooks() {
        add_action( 'admin_menu', array( $this, 'create_admin_menu' ) );
        add_action( 'admin_init', array( $this, 'register_settings' ) );
        add_action( 'add_meta_boxes', array( $this, 'add_meta_boxes' ) );
        add_action( 'wp_ajax_uncertifyai_process', array( $this, 'ajax_process_content' ) );
        add_action( 'wp_ajax_uncertifyai_get_usage', array( $this, 'ajax_get_usage' ) );
        add_action( 'admin_enqueue_scripts', array( $this, 'enqueue_scripts' ) );
        add_action( 'wp_dashboard_setup', array( $this, 'add_dashboard_widget' ) );
        add_filter( 'post_row_actions', array( $this, 'add_post_row_action' ), 10, 2 );
        add_filter( 'page_row_actions', array( $this, 'add_post_row_action' ), 10, 2 );
        add_action( 'admin_action_uncertifyai_recheck', array( $this, 'recheck_post' ) );
        add_action( 'admin_notices', array( $this, 'admin_recheck_notice' ) );
        add_action( 'save_post', array( $this, 'maybe_check_content_on_save' ), 10, 3 );
    }

    /**
     * Create admin menu
     */
    public function create_admin_menu() {
        add_menu_page(
            __( 'UncertifyAI Settings', 'ai-humanizer-ai-detector-uncertify-ai' ),
            'UncertifyAI',
            'manage_options',
            'uncertifyai-settings',
            array( $this, 'settings_page' ),
            'dashicons-edit'
        );
        
        add_submenu_page(
            'uncertifyai-settings',
            __( 'Check Results', 'ai-humanizer-ai-detector-uncertify-ai' ),
            __( 'Check Results', 'ai-humanizer-ai-detector-uncertify-ai' ),
            'manage_options',
            'uncertifyai-results',
            array( $this, 'results_page' )
        );
        
        add_submenu_page(
            'uncertifyai-settings',
            __( 'Usage Statistics', 'ai-humanizer-ai-detector-uncertify-ai' ),
            __( 'Usage Stats', 'ai-humanizer-ai-detector-uncertify-ai' ),
            'manage_options',
            'uncertifyai-usage',
            array( $this, 'usage_page' )
        );
    }

    /**
     * Register settings
     */
    public function register_settings() {
        register_setting( 'uncertifyai_settings', 'uncertifyai_settings', array( $this, 'sanitize_settings' ) );
        
        add_settings_section(
            'uncertifyai_main_section',
            __( 'API Configuration', 'ai-humanizer-ai-detector-uncertify-ai' ),
            array( $this, 'main_section_callback' ),
            'uncertifyai-settings'
        );
        
        add_settings_field(
            'api_key',
            __( 'API Key', 'ai-humanizer-ai-detector-uncertify-ai' ),
            array( $this, 'api_key_field_callback' ),
            'uncertifyai-settings',
            'uncertifyai_main_section'
        );
        
        add_settings_section(
            'uncertifyai_auto_check_section',
            __( 'Automatic Content Checking', 'ai-humanizer-ai-detector-uncertify-ai' ),
            array( $this, 'auto_check_section_callback' ),
            'uncertifyai-settings'
        );
        
        add_settings_field(
            'check_posts',
            __( 'Check Posts Automatically', 'ai-humanizer-ai-detector-uncertify-ai' ),
            array( $this, 'check_posts_field_callback' ),
            'uncertifyai-settings',
            'uncertifyai_auto_check_section'
        );
        
        add_settings_field(
            'check_pages',
            __( 'Check Pages Automatically', 'ai-humanizer-ai-detector-uncertify-ai' ),
            array( $this, 'check_pages_field_callback' ),
            'uncertifyai-settings',
            'uncertifyai_auto_check_section'
        );
        
        add_settings_field(
            'check_categories',
            __( 'Categories to Check', 'ai-humanizer-ai-detector-uncertify-ai' ),
            array( $this, 'check_categories_field_callback' ),
            'uncertifyai-settings',
            'uncertifyai_auto_check_section'
        );
        
        add_settings_field(
            'auto_recheck',
            __( 'Auto Recheck on Update', 'ai-humanizer-ai-detector-uncertify-ai' ),
            array( $this, 'auto_recheck_field_callback' ),
            'uncertifyai-settings',
            'uncertifyai_auto_check_section'
        );
        
        add_settings_section(
            'uncertifyai_display_section',
            __( 'Display Settings', 'ai-humanizer-ai-detector-uncertify-ai' ),
            array( $this, 'display_section_callback' ),
            'uncertifyai-settings'
        );
        
        add_settings_field(
            'display_results',
            __( 'Display Results to Visitors', 'ai-humanizer-ai-detector-uncertify-ai' ),
            array( $this, 'display_results_field_callback' ),
            'uncertifyai-settings',
            'uncertifyai_display_section'
        );
        
        add_settings_field(
            'show_symbol',
            __( 'Show AI Certification Symbol', 'ai-humanizer-ai-detector-uncertify-ai' ),
            array( $this, 'show_symbol_field_callback' ),
            'uncertifyai-settings',
            'uncertifyai_display_section'
        );
    }

    /**
     * Sanitize settings
     *
     * @param array $input Settings input.
     * @return array
     */
    public function sanitize_settings( $input ) {
        $sanitized = array();
        
        if ( isset( $input['api_key'] ) ) {
            $sanitized['api_key'] = sanitize_text_field( $input['api_key'] );
            
            // If API key changed, clear cached user and usage data
            $current_options = get_option( 'uncertifyai_settings', array() );
            if ( isset( $current_options['api_key'] ) && $sanitized['api_key'] !== $current_options['api_key'] ) {
                delete_transient( 'uncertifyai_user_data' );
                delete_transient( 'uncertifyai_usage_data' );
            }
        }
        
        $sanitized['check_posts'] = isset( $input['check_posts'] ) ? 1 : 0;
        $sanitized['check_pages'] = isset( $input['check_pages'] ) ? 1 : 0;
        $sanitized['auto_recheck'] = isset( $input['auto_recheck'] ) ? 1 : 0;
        
        if ( isset( $input['check_categories'] ) ) {
            $sanitized['check_categories'] = array_map( 'absint', $input['check_categories'] );
        } else {
            $sanitized['check_categories'] = array();
        }

        // --- Enforce display_results based on subscription ---
        $user_data    = $this->api->get_user_data();
        $is_free_user = isset( $user_data['subscription_plan'] ) && $user_data['subscription_plan'] === 'free';

        if ( $is_free_user ) {
            $sanitized['display_results'] = 1; // always true for free users
            $sanitized['show_symbol'] = 1;
        } else {
            $sanitized['display_results'] = isset( $input['display_results'] ) ? 1 : 0;
            $sanitized['show_symbol'] = isset( $input['show_symbol'] ) ? 1 : 0;
        }
        
        return $sanitized;
    }

    /**
     * Main section callback
     */
    public function main_section_callback() {
        echo '<p>' . esc_html__( 'Enter your UncertifyAI API credentials below. Get your API key from UncertifyAI.', 'ai-humanizer-ai-detector-uncertify-ai' ) . '</p>';
    }

    /**
     * Auto check section callback
     */
    public function auto_check_section_callback() {
        echo '<p>' . esc_html__( 'Configure which content should be automatically checked for AI generation.', 'ai-humanizer-ai-detector-uncertify-ai' ) . '</p>';
    }

    /**
     * Display section callback
     */
    public function display_section_callback() {
        echo '<p>' . esc_html__( 'Configure how AI check results are displayed on your site.', 'ai-humanizer-ai-detector-uncertify-ai' ) . '</p>';
    }

    /**
     * API key field callback
     */
    public function api_key_field_callback() {
        $options = get_option( 'uncertifyai_settings', array() );
        $value = isset( $options['api_key'] ) ? $options['api_key'] : '';
        echo '<input type="password" name="uncertifyai_settings[api_key]" value="' . esc_attr( $value ) . '" class="regular-text" />';
        echo '<p class="description">' . esc_html__( 'Your UncertifyAI API key (Bearer token)', 'ai-humanizer-ai-detector-uncertify-ai' ) . '</p>';
    }

    /**
     * Check posts field callback
     */
    public function check_posts_field_callback() {
        $options = get_option( 'uncertifyai_settings', array() );
        $checked = isset( $options['check_posts'] ) && $options['check_posts'] ? 'checked' : '';
        echo '<input type="checkbox" name="uncertifyai_settings[check_posts]" value="1" ' . esc_attr( $checked ) . ' />';
        echo '<p class="description">' . esc_html__( 'Automatically check all posts for AI content when saved', 'ai-humanizer-ai-detector-uncertify-ai' ) . '</p>';
    }

    /**
     * Check pages field callback
     */
    public function check_pages_field_callback() {
        $options = get_option( 'uncertifyai_settings', array() );
        $checked = isset( $options['check_pages'] ) && $options['check_pages'] ? 'checked' : '';
        echo '<input type="checkbox" name="uncertifyai_settings[check_pages]" value="1" ' . esc_attr( $checked ) . ' />';
        echo '<p class="description">' . esc_html__( 'Automatically check all pages for AI content when saved', 'ai-humanizer-ai-detector-uncertify-ai' ) . '</p>';
    }

    /**
     * Check categories field callback
     */
    public function check_categories_field_callback() {
        $options = get_option( 'uncertifyai_settings', array() );
        $selected_categories = isset( $options['check_categories'] ) ? $options['check_categories'] : array();
        $categories = get_categories( array( 'hide_empty' => false ) );
        
        echo '<select name="uncertifyai_settings[check_categories][]" multiple="multiple" style="width: 50%; min-height: 150px;">';
        foreach ( $categories as $category ) {
            $selected = in_array( $category->term_id, $selected_categories ) ? 'selected' : '';
            echo '<option value="' . esc_attr( $category->term_id ) . '" ' . selected( $selected, true, false ) . '>' . esc_html( $category->name ) . '</option>';
        }
        echo '</select>';
        echo '<p class="description">' . esc_html__( 'Hold Ctrl/Cmd to select multiple categories. Leave empty to check all categories.', 'ai-humanizer-ai-detector-uncertify-ai' ) . '</p>';
    }

    /**
     * Auto recheck field callback
     */
    public function auto_recheck_field_callback() {
        $options = get_option( 'uncertifyai_settings', array() );
        $checked = isset( $options['auto_recheck'] ) && $options['auto_recheck'] ? 'checked' : '';
        echo '<input type="checkbox" name="uncertifyai_settings[auto_recheck]" value="1" ' . esc_attr( $checked ) . ' />';
        echo '<p class="description">' . esc_html__( 'Automatically recheck content when it is updated', 'ai-humanizer-ai-detector-uncertify-ai' ) . '</p>';
    }

    /**
     * Display results field callback
     */
    public function display_results_field_callback() {
        $options = get_option( 'uncertifyai_settings', array() );
        $checked = isset( $options['display_results'] ) && $options['display_results'] ? 'checked' : '';
        // Get user data
        $user_data = $this->api->get_user_data();
        $is_free_user = isset($user_data['subscription_plan']) && $user_data['subscription_plan'] === 'free';
        if ($is_free_user) {
            echo '<input type="checkbox" name="uncertifyai_settings[display_results]" value="1" checked disabled />';
            echo '<p class="description">' . esc_html__( 'Free plan: Results always visible', 'ai-humanizer-ai-detector-uncertify-ai' ) . '</p>';
            echo '<p class="description">' . esc_html__( 'Show AI detection results to website visitors', 'ai-humanizer-ai-detector-uncertify-ai' ) . '</p>';
        }else{
            echo '<input type="checkbox" name="uncertifyai_settings[display_results]" value="1" ' . esc_attr( $checked ) . ' />';
            echo '<p class="description">' . esc_html__( 'Show AI detection results to website visitors', 'ai-humanizer-ai-detector-uncertify-ai' ) . '</p>';
        }
    }

    /**
     * Show symbol field callback
     */
    public function show_symbol_field_callback() {
        $options = get_option( 'uncertifyai_settings', array() );
        $checked = isset( $options['show_symbol'] ) && $options['show_symbol'] ? 'checked' : '';
        
        // Get user data
        $user_data = $this->api->get_user_data();
        $is_free_user = isset($user_data['subscription_plan']) && $user_data['subscription_plan'] === 'free';
        
        if ($is_free_user) {
            // Free users - always show certificate, disable setting
            echo '<input type="checkbox" name="uncertifyai_settings[show_symbol]" value="1" checked disabled />';
            echo '<p class="description">' . esc_html__( 'Free plan: Certificate always visible', 'ai-humanizer-ai-detector-uncertify-ai' ) . '</p>';
            echo '<input type="hidden" name="uncertifyai_settings[show_symbol]" value="1" />';
        } else {
            // Premium users - respect the setting
            echo '<input type="checkbox" name="uncertifyai_settings[show_symbol]" value="1" ' . esc_attr( $checked ) . ' />';
            echo '<p class="description">' . esc_html__( 'Show AI certification symbol on frontend', 'ai-humanizer-ai-detector-uncertify-ai' ) . '</p>';
        }
    }

    /**
     * Settings page
     */
    public function settings_page() {
        // Refresh usage data if needed
        $usage_data = $this->api->get_usage_data( true );
        ?>
        <div class="wrap">
            <h1><?php esc_html_e( 'UncertifyAI Settings', 'ai-humanizer-ai-detector-uncertify-ai' ); ?></h1>
            
            <?php if ( $this->api->is_configured() && $usage_data && isset( $usage_data['success'] ) && $usage_data['success'] ) : ?>
            <div class="card" style="max-width: 600px; margin-bottom: 20px;">
                <h2><?php esc_html_e( 'Usage Statistics', 'ai-humanizer-ai-detector-uncertify-ai' ); ?></h2>
                <p><?php esc_html_e( 'Detection used:', 'ai-humanizer-ai-detector-uncertify-ai' ); ?> <strong><?php echo esc_html( $usage_data['detection'] ); ?></strong> <?php esc_html_e( 'of', 'ai-humanizer-ai-detector-uncertify-ai' ); ?> <strong><?php echo esc_html( $usage_data['detection_limit'] ); ?></strong></p>
                <p><?php esc_html_e( 'Humanization used:', 'ai-humanizer-ai-detector-uncertify-ai' ); ?> <strong><?php echo esc_html( $usage_data['humanization'] ); ?></strong> <?php esc_html_e( 'of', 'ai-humanizer-ai-detector-uncertify-ai' ); ?> <strong><?php echo esc_html( $usage_data['humanization_limit'] ); ?></strong></p>
                <p><?php esc_html_e( 'Usage resets on:', 'ai-humanizer-ai-detector-uncertify-ai' ); ?> <strong><?php echo esc_html( date_i18n( get_option( 'date_format' ) . ' @ ' . get_option( 'time_format' ), strtotime( $usage_data['reset_time'] ) ) ); ?></strong></p>
            </div>
            <?php endif; ?>
            
            <form method="post" action="options.php">
                <?php
                settings_fields( 'uncertifyai_settings' );
                do_settings_sections( 'uncertifyai-settings' );
                submit_button();
                ?>
            </form>
            
            <div style="margin-top: 30px; padding: 20px; background: #f9f9f9; border: 1px solid #ddd;">
                <h2><?php esc_html_e( 'Manual Content Check', 'ai-humanizer-ai-detector-uncertify-ai' ); ?></h2>
                <p><?php esc_html_e( 'Use this form to manually check content without saving it to your site.', 'ai-humanizer-ai-detector-uncertify-ai' ); ?></p>
                <textarea id="uncertifyai-manual-check" style="width: 100%; height: 150px; margin-bottom: 10px;" placeholder="<?php esc_attr_e( 'Paste content to check...', 'ai-humanizer-ai-detector-uncertify-ai' ); ?>"></textarea>
                <div style="display: flex; gap: 10px;">
                    <button type="button" class="button button-primary" id="uncertifyai-detect-btn"><?php esc_html_e( 'Detect AI Content', 'ai-humanizer-ai-detector-uncertify-ai' ); ?></button>
                    <button type="button" class="button button-secondary" id="uncertifyai-humanize-btn"><?php esc_html_e( 'Humanize Content', 'ai-humanizer-ai-detector-uncertify-ai' ); ?></button>
                </div>
                <div id="uncertifyai-manual-results" style="margin-top: 15px; display: none;"></div>
            </div>
        </div>
        <?php
    }

    /**
     * Results page
     */
    public function results_page() {
        $results = $this->db->get_recent_checks( 100 );
        ?>
        <div class="wrap">
            <h1><?php esc_html_e( 'UncertifyAI Check Results', 'ai-humanizer-ai-detector-uncertify-ai' ); ?></h1>
            
            <table class="wp-list-table widefat fixed striped">
                <thead>
                    <tr>
                        <th><?php esc_html_e( 'Post/Page', 'ai-humanizer-ai-detector-uncertify-ai' ); ?></th>
                        <th><?php esc_html_e( 'AI Probability', 'ai-humanizer-ai-detector-uncertify-ai' ); ?></th>
                        <th><?php esc_html_e( 'Confidence', 'ai-humanizer-ai-detector-uncertify-ai' ); ?></th>
                        <th><?php esc_html_e( 'Checked By', 'ai-humanizer-ai-detector-uncertify-ai' ); ?></th>
                        <th><?php esc_html_e( 'Checked On', 'ai-humanizer-ai-detector-uncertify-ai' ); ?></th>
                        <th><?php esc_html_e( 'Revision', 'ai-humanizer-ai-detector-uncertify-ai' ); ?></th>
                    </tr>
                </thead>
                <tbody>
                    <?php if ( empty( $results ) ) : ?>
                    <tr>
                        <td colspan="7"><?php esc_html_e( 'No check results found.', 'ai-humanizer-ai-detector-uncertify-ai' ); ?></td>
                    </tr>
                    <?php else : ?>
                    <?php foreach ( $results as $result ) : ?>
                    <tr>
                        <td>
                            <a href="<?php echo esc_url( get_edit_post_link( $result->post_id ) ); ?>">
                                <?php echo esc_html( $result->post_title ? $result->post_title : __( '(No Title)', 'ai-humanizer-ai-detector-uncertify-ai' ) ); ?>
                            </a>
                        </td>
                        <td style="color:<?php echo ( $result->ai_probability > 0.7 ? 'red' : ( $result->ai_probability > 0.4 ? 'orange' : 'green' ) ); ?>">
                            <?php echo esc_html( $result->ai_probability * 100 ); ?>%
                        </td>
                        <td style="color:<?php echo ( $result->ai_probability > 0.7 ? 'red' : ( $result->ai_probability > 0.4 ? 'orange' : 'green' ) ); ?>">
                            <?php echo esc_html( $result->confidence ); ?>
                        </td>
                        <td><?php echo esc_html( $result->user_nicename ); ?></td>
                        <td><?php echo esc_html( date_i18n( get_option( 'date_format' ) . ' @ ' . get_option( 'time_format' ), strtotime( $result->checked_on ) ) ); ?></td>
                        <td>
                            <?php if ( $result->revision_id ) : ?>
                            <a href="<?php echo esc_url( get_edit_post_link( $result->revision_id ) ); ?>"><?php esc_html_e( 'View Revision', 'ai-humanizer-ai-detector-uncertify-ai' ); ?></a>
                            <?php else : ?>
                            <?php esc_html_e( 'Current', 'ai-humanizer-ai-detector-uncertify-ai' ); ?>
                            <?php endif; ?>
                        </td>
                    </tr>
                    <?php endforeach; ?>
                    <?php endif; ?>
                </tbody>
            </table>
        </div>
        <?php
    }

    /**
     * Usage page
     */
    public function usage_page() {
        if ( ! $this->api->is_configured() ) {
            echo '<div class="wrap"><div class="notice notice-error"><p>' . esc_html__( 'API key not configured.', 'ai-humanizer-ai-detector-uncertify-ai' ) . '</p></div></div>';
            return;
        }
        
        $usage_data = $this->api->get_usage_data( true );
        ?>
        <div class="wrap">
            <h1><?php esc_html_e( 'UncertifyAI Usage Statistics', 'ai-humanizer-ai-detector-uncertify-ai' ); ?></h1>
            
            <?php if ( $usage_data && isset( $usage_data ) ) : ?>
            <div class="uncertifyai-usage-stats">
                <div class="usage-card">
                    <h3><?php esc_html_e( 'AI Detection', 'ai-humanizer-ai-detector-uncertify-ai' ); ?></h3>
                    <div class="usage-progress">
                        <div class="progress-bar">
                            <div class="progress-fill" style="width: <?php echo esc_attr( ( $usage_data['detection'] / $usage_data['detection_limit'] ) * 100 ); ?>%"></div>
                        </div>
                        <div class="usage-numbers">
                            <span><?php echo esc_html( $usage_data['detection'] ); ?> / <?php echo esc_html( $usage_data['detection_limit'] ); ?></span>
                            <span><?php echo esc_html( round( ( $usage_data['detection'] / $usage_data['detection_limit'] ) * 100 ) ); ?>%</span>
                        </div>
                    </div>
                </div>
                
                <div class="usage-card">
                    <h3><?php esc_html_e( 'Text Humanization', 'ai-humanizer-ai-detector-uncertify-ai' ); ?></h3>
                    <div class="usage-progress">
                        <div class="progress-bar">
                            <div class="progress-fill" style="width: <?php echo esc_attr( ( $usage_data['humanization'] / $usage_data['humanization_limit'] ) * 100 ); ?>%"></div>
                        </div>
                        <div class="usage-numbers">
                            <span><?php echo esc_html( $usage_data['humanization'] ); ?> / <?php echo esc_html( $usage_data['humanization_limit'] ); ?></span>
                            <span><?php echo esc_html( round( ( $usage_data['humanization'] / $usage_data['humanization_limit'] ) * 100 ) ); ?>%</span>
                        </div>
                    </div>
                </div>
                
                <div class="usage-info">
                    <p><strong><?php esc_html_e( 'Reset Time:', 'ai-humanizer-ai-detector-uncertify-ai' ); ?></strong> <?php echo esc_html( date_i18n( get_option( 'date_format' ) . ' \a\t ' . get_option( 'time_format' ) . ' T', strtotime( $usage_data['reset_time'] ) ) ); ?></p>
                    <button type="button" class="button button-secondary" id="uncertifyai-refresh-usage"><?php esc_html_e( 'Refresh Usage Data', 'ai-humanizer-ai-detector-uncertify-ai' ); ?></button>
                </div>
            </div>
            <?php else : ?>
            <div class="notice notice-error">
                <p><?php esc_html_e( 'Unable to retrieve usage data. Please check your API key.', 'ai-humanizer-ai-detector-uncertify-ai' ); ?></p>
            </div>
            <?php endif; ?>
        </div>
        <?php
    }

    /**
     * Add meta boxes to post editor
     */
    public function add_meta_boxes() {
        $options = get_option( 'uncertifyai_settings', array() );
        $post_types = array();
        
        if ( isset( $options['check_posts'] ) && $options['check_posts'] ) {
            $post_types[] = 'post';
        }
        
        if ( isset( $options['check_pages'] ) && $options['check_pages'] ) {
            $post_types[] = 'page';
        }
        
        if ( ! empty( $post_types ) ) {
            add_meta_box(
                'uncertifyai_meta_box',
                __( 'UncertifyAI Tools', 'ai-humanizer-ai-detector-uncertify-ai' ),
                array( $this, 'render_meta_box' ),
                $post_types,
                'side',
                'high'
            );
        }
    }

    /**
     * Render meta box
     *
     * @param WP_Post $post Post object.
     */
    public function render_meta_box( $post ) {
        $ai_check = get_post_meta( $post->ID, '_uncertifyai_ai_check', true );
        $check_history = $this->db->get_check_history( $post->ID );
        ?>
        <div id="uncertifyai-metabox">
            <p>
                <button type="button" class="button button-primary" id="uncertifyai-detect">
                    <?php echo $ai_check ? esc_html__( 'Re-check for AI Content', 'ai-humanizer-ai-detector-uncertify-ai' ) : esc_html__( 'Check for AI Content', 'ai-humanizer-ai-detector-uncertify-ai' ); ?>
                </button>
            </p>
            
            <?php if ( $ai_check ) : ?>
            <div id="uncertifyai-results">
                <h4><?php esc_html_e( 'Last Check Results:', 'ai-humanizer-ai-detector-uncertify-ai' ); ?></h4>
                <p><?php esc_html_e( 'AI Probability:', 'ai-humanizer-ai-detector-uncertify-ai' ); ?> <span style="color:<?php echo ( $ai_check["ai_probability"] > 0.7 ? 'red' : ( $ai_check["ai_probability"] > 0.4 ? 'orange' : 'green' ) ); ?>"><?php echo esc_html( round($ai_check["ai_probability"] * 100, 2)  ); ?>% </span></p>
                <p><?php esc_html_e( 'Confidence:', 'ai-humanizer-ai-detector-uncertify-ai' ); ?> <span style="color:<?php echo ( $ai_check["ai_probability"] > 0.7 ? 'red' : ( $ai_check["ai_probability"] > 0.4 ? 'orange' : 'green' ) ); ?>"><?php echo esc_html( $ai_check['confidence'] ); ?></span></p>
                <p><?php esc_html_e( 'Checked on:', 'ai-humanizer-ai-detector-uncertify-ai' ); ?> <?php echo esc_html( date_i18n( get_option( 'date_format' ) . ' @ ' . get_option( 'time_format' ), $ai_check['timestamp'] ) ); ?></p>
            </div>
            <?php else : ?>
            <div id="uncertifyai-results" style="display: none;">
                <h4><?php esc_html_e( 'Results:', 'ai-humanizer-ai-detector-uncertify-ai' ); ?></h4>
                <div id="uncertifyai-results-content"></div>
            </div>
            <?php endif; ?>
            
            <?php if ( ! empty( $check_history ) ) : ?>
            <div style="margin-top: 15px;">
                <h4><?php esc_html_e( 'Check History:', 'ai-humanizer-ai-detector-uncertify-ai' ); ?></h4>
                <select id="uncertifyai-check-history" style="width: 100%;">
                    <option value=""><?php esc_html_e( 'Select a previous check...', 'ai-humanizer-ai-detector-uncertify-ai' ); ?></option>
                    <?php foreach ( $check_history as $check ) : ?>
                    <option value="<?php echo esc_attr( $check->id ); ?>" data-ai="<?php echo esc_attr( $check->ai_probability * 100 ); ?>" data-human="<?php echo esc_attr( $check->confidence ); ?>"  data-date="<?php echo esc_attr( date_i18n( get_option( 'date_format' ) . ' @ ' . get_option( 'time_format' ), strtotime( $check->checked_on ) ) ); ?>">
                        <?php echo esc_html( date_i18n( get_option( 'date_format' ) . ' @ ' . get_option( 'time_format' ), strtotime( $check->checked_on ) ) ); ?> - AI: <?php echo esc_html( $check->ai_probability ); ?>%
                    </option>
                    <?php endforeach; ?>
                </select>
                <div id="uncertifyai-history-results" style="display: none; margin-top: 10px; padding: 10px; background: #f6f7f7; border-radius: 4px;"></div>
            </div>
            <?php endif; ?>
        </div>
        <?php
    }

    /**
     * Enqueue admin scripts
     *
     * @param string $hook Current admin page.
     */
    public function enqueue_scripts( $hook ) {
        if ( in_array( $hook, array( 'post.php', 'post-new.php' ), true ) || strpos( $hook, 'uncertifyai' ) !== false ) {
            wp_enqueue_style( 'uncertifyai-admin-style', UNCERTIFYAI_PLUGIN_URL . 'assets/admin-style.css', array(), UNCERTIFYAI_VERSION );
            wp_enqueue_script( 'uncertifyai-admin-script', UNCERTIFYAI_PLUGIN_URL . 'assets/admin-script.js', array( 'jquery' ), UNCERTIFYAI_VERSION, true );
            
            wp_localize_script( 'uncertifyai-admin-script', 'uncertifyai_ajax', array(
                'ajax_url' => admin_url( 'admin-ajax.php' ),
                'nonce' => wp_create_nonce( 'uncertifyai_nonce' ),
                'checking_text' => __( 'Checking...', 'ai-humanizer-ai-detector-uncertify-ai' ),
                'error_text' => __( 'An error occurred while processing your request', 'ai-humanizer-ai-detector-uncertify-ai' ),
                'checked_on_text' => __( 'Last Check Date: ', 'ai-humanizer-ai-detector-uncertify-ai' ),
                'results_text' => __( 'Results: ', 'ai-humanizer-ai-detector-uncertify-ai' ),
                'humanized_content_text' => __( 'humanized Results: ', 'ai-humanizer-ai-detector-uncertify-ai' ),
                'processed_on_text' => __( 'Last humanized Date: ', 'ai-humanizer-ai-detector-uncertify-ai' ),
            ) );
        }
        
        if ( 'index.php' === $hook ) {
            wp_enqueue_style( 'uncertifyai-admin-style', UNCERTIFYAI_PLUGIN_URL . 'assets/admin-style.css', array(), UNCERTIFYAI_VERSION );
        }
    }

    /**
     * Process content via AJAX
     */
    public function ajax_process_content() {
        // Verify nonce
        if ( ! isset( $_POST['nonce'] ) || 
            ! wp_verify_nonce( sanitize_text_field( wp_unslash( $_POST['nonce'] ) ), 'uncertifyai_nonce' ) ) {
            wp_send_json_error( __( 'Security check failed', 'ai-humanizer-ai-detector-uncertify-ai' ) );
        }
        
        $action = '';
        if ( isset( $_POST['action_type'] ) ) {
            $action = sanitize_text_field( wp_unslash( $_POST['action_type'] ) );
        }
        $content = '';
        if ( isset( $_POST['content'] ) ) {
            $content = wp_kses_post( wp_unslash( $_POST['content'] ) );
        }
        $post_id = isset( $_POST['post_id'] ) ? intval( $_POST['post_id'] ) : 0;
        
        if ( ! $this->api->is_configured() ) {
            wp_send_json_error( __( 'API key not configured', 'ai-humanizer-ai-detector-uncertify-ai' ) );
        }
        
        if ( 'detect' === $action ) {
            $response = $this->api->detect_ai_content( $content );
        } else {
            $response = $this->api->humanize_content( $content );
        }
        
        if ( is_wp_error( $response ) ) {
            wp_send_json_error( $response->get_error_message() );
        }
        
        if ( $response && is_array( $response ) ) {
            if ( 'detect' === $action ) {
                // AI Detection response
                $result = array(
                    'ai_probability' => isset( $response['score'] ) ? $response['score'] : null,
                    'score'       => isset( $response['score'] ) ? $response['score'] : null,
                    'confidence'  => isset( $response['confidence'] ) ? $response['confidence'] : null,
                    'timestamp'   => time(),
                );

                if ( $post_id ) {
                    update_post_meta( $post_id, '_uncertifyai_ai_check', $result );
                    $this->db->save_check_result( $post_id, $result );
                }

                wp_send_json_success( $result );

            } else {
                // Humanization response
                $result = array(
                    'humanized_text'   => isset( $response['humanized_text'] ) ? wp_kses_post( $response['humanized_text'] ) : '',
                    'original_length'  => isset( $response['original_length'] ) ? intval( $response['original_length'] ) : 0,
                    'humanized_length' => isset( $response['humanized_length'] ) ? intval( $response['humanized_length'] ) : 0,
                    'changes_made'     => isset( $response['changes_made'] ) ? intval( $response['changes_made'] ) : 0,
                    'timestamp'        => time(),
                );

                wp_send_json_success( $result );
            }
        } else {
            $error = isset( $response['message'] ) ? $response['message'] : __( 'Unknown error occurred', 'ai-humanizer-ai-detector-uncertify-ai' );
            wp_send_json_error( $error );
        }
    }

    /**
     * Get usage data via AJAX
     */
    public function ajax_get_usage() {
        // Verify nonce
        if ( ! isset( $_POST['nonce'] ) ) {
            wp_send_json_error( __( 'Missing security nonce', 'ai-humanizer-ai-detector-uncertify-ai' ) );
        }

        $nonce = sanitize_text_field( wp_unslash( $_POST['nonce'] ) );

        if ( ! wp_verify_nonce( $nonce, 'uncertifyai_nonce' ) ) {
            wp_send_json_error( __( 'Security check failed', 'ai-humanizer-ai-detector-uncertify-ai' ) );
        }

        
        $usage_data = $this->api->get_usage_data( true );
        
        if ( $usage_data && isset( $usage_data)) {
            wp_send_json_success( $usage_data );
        } else {
            wp_send_json_error( __( 'Unable to retrieve usage data', 'ai-humanizer-ai-detector-uncertify-ai' ) );
        }
    }

    /**
     * Maybe check content on save
     *
     * @param int      $post_id Post ID.
     * @param WP_Post  $post Post object.
     * @param bool     $update Whether this is an update.
     */
    public function maybe_check_content_on_save( $post_id, $post, $update ) {
        // Skip if autosave, revisions, or user lacks permissions
        if ( defined( 'DOING_AUTOSAVE' ) && DOING_AUTOSAVE ) {
            return;
        }
        
        if ( wp_is_post_revision( $post_id ) || wp_is_post_autosave( $post_id ) ) {
            return;
        }
        
        if ( ! current_user_can( 'edit_post', $post_id ) ) {
            return;
        }
        
        $options = get_option( 'uncertifyai_settings', array() );
        
        // Check if we should process this content type
        $should_check = false;
        $is_recheck = false;
        
        // Check if this is an update and we should recheck
        $existing_check = get_post_meta( $post_id, '_uncertifyai_ai_check', true );
        if ( $update && $existing_check && isset( $options['auto_recheck'] ) && $options['auto_recheck'] ) {
            $is_recheck = true;
            $should_check = true;
        } 
        // Check if this is a new post and we should check based on settings
        elseif ( ! $update ) {
            if ( 'post' === $post->post_type && isset( $options['check_posts'] ) && $options['check_posts'] ) {
                // Check if we have category restrictions
                if ( isset( $options['check_categories'] ) && ! empty( $options['check_categories'] ) ) {
                    $post_categories = wp_get_post_categories( $post_id );
                    $intersect = array_intersect( $post_categories, $options['check_categories'] );
                    $should_check = ! empty( $intersect );
                } else {
                    $should_check = true;
                }
            } elseif ( 'page' === $post->post_type && isset( $options['check_pages'] ) && $options['check_pages'] ) {
                $should_check = true;
            }
        }
        
        if ( $should_check && $this->api->is_configured() ) {
            $content = $post->post_content;
            // Strip HTML tags for cleaner analysis
            $content = wp_strip_all_tags( $content );
            
            if ( ! empty( $content ) ) {
                $response = $this->api->detect_ai_content( $content );
                
                if ( ! is_wp_error( $response ) && $response && is_array( $response ) ) {
                    $result = array(
                        'ai_probability' => isset( $response['score'] ) ? $response['score'] : null,
                        'score'       => isset( $response['score'] ) ? $response['score'] : null,
                        'confidence'  => isset( $response['confidence'] ) ? $response['confidence'] : null,
                        'timestamp'   => time(),
                    );
                    
                    update_post_meta( $post_id, '_uncertifyai_ai_check', $result );
                    $this->db->save_check_result( $post_id, $result, $is_recheck ? $post_id : 0 );
                }
            }
        }
    }

    /**
     * Add dashboard widget
     */
    public function add_dashboard_widget() {
        if ( current_user_can( 'manage_options' ) ) {
            wp_add_dashboard_widget(
                'uncertifyai_dashboard_widget',
                __( 'UncertifyAI Statistics', 'ai-humanizer-ai-detector-uncertify-ai' ),
                array( $this, 'render_dashboard_widget' )
            );
        }
    }

    /**
     * Render dashboard widget
     */
    public function render_dashboard_widget() {
        $counts = $this->db->get_check_counts();
        $usage_data = $this->api->get_usage_data();
        ?>
        <div class="uncertifyai-dashboard-widget">
            <div class="uncertifyai-stat">
                <h3><?php esc_html_e( 'Total Checks', 'ai-humanizer-ai-detector-uncertify-ai' ); ?></h3>
                <p><?php echo esc_html( $counts['total'] ); ?></p>
            </div>
            
            <div class="uncertifyai-stat">
                <h3><?php esc_html_e( 'Today\'s Checks', 'ai-humanizer-ai-detector-uncertify-ai' ); ?></h3>
                <p><?php echo esc_html( $counts['today'] ); ?></p>
            </div>
            
            <?php if ( $usage_data && isset( $usage_data['success'] ) && $usage_data['success'] ) : ?>
            <div class="uncertifyai-stat">
                <h3><?php esc_html_e( 'Detection Usage', 'ai-humanizer-ai-detector-uncertify-ai' ); ?></h3>
                <p><?php echo esc_html( $usage_data['detection'] ); ?> / <?php echo esc_html( $usage_data['detection_limit'] ); ?></p>
            </div>
            
            <div class="uncertifyai-stat">
                <h3><?php esc_html_e( 'Humanization Usage', 'ai-humanizer-ai-detector-uncertify-ai' ); ?></h3>
                <p><?php echo esc_html( $usage_data['humanization'] ); ?> / <?php echo esc_html( $usage_data['humanization_limit'] ); ?></p>
            </div>
            <?php endif; ?>
            
            <div class="uncertifyai-actions">
                <a href="<?php echo esc_url( admin_url( 'admin.php?page=uncertifyai-settings' ) ); ?>" class="button button-primary"><?php esc_html_e( 'Settings', 'ai-humanizer-ai-detector-uncertify-ai' ); ?></a>
                <a href="<?php echo esc_url( admin_url( 'admin.php?page=uncertifyai-results' ) ); ?>" class="button"><?php esc_html_e( 'View Results', 'ai-humanizer-ai-detector-uncertify-ai' ); ?></a>
            </div>
        </div>
        <?php
    }

    /**
     * Add post row action
     *
     * @param array   $actions Existing actions.
     * @param WP_Post $post Post object.
     * @return array
     */
    public function add_post_row_action( $actions, $post ) {
        $options = get_option( 'uncertifyai_settings', array() );
        
        // Only add action for post types that can be checked
        if ( ( 'post' === $post->post_type && isset( $options['check_posts'] ) && $options['check_posts'] ) ||
            ( 'page' === $post->post_type && isset( $options['check_pages'] ) && $options['check_pages'] ) ) {
            
            $actions['uncertifyai_recheck'] = sprintf(
                '<a href="%s" title="%s">%s</a>',
                wp_nonce_url( admin_url( "admin.php?action=uncertifyai_recheck&post=$post->ID" ), 'uncertifyai_recheck_' . $post->ID ),
                __( 'Recheck for AI content', 'ai-humanizer-ai-detector-uncertify-ai' ),
                __( 'Recheck AI', 'ai-humanizer-ai-detector-uncertify-ai' )
            );
        }
        
        return $actions;
    }

    /**
     * Recheck post
     */
    public function recheck_post() {
        if ( ! isset( $_GET['post'] ) || ! isset( $_GET['_wpnonce'] ) ) {
            wp_die( esc_html__( 'Invalid request', 'ai-humanizer-ai-detector-uncertify-ai' ) );
        }
        
        $post_id = intval( $_GET['post'] );
        
        if ( ! isset( $_GET['_wpnonce'] ) || ! wp_verify_nonce( sanitize_text_field( wp_unslash( $_GET['_wpnonce'] ) ), 'uncertifyai_recheck_' . $post_id ) ) {
            wp_die( esc_html__( 'Security check failed', 'ai-humanizer-ai-detector-uncertify-ai' ) );
        }
        
        if ( ! current_user_can( 'edit_post', $post_id ) ) {
            wp_die( esc_html__( 'Insufficient permissions', 'ai-humanizer-ai-detector-uncertify-ai' ) );
        }
        
        $post = get_post( $post_id );
        $content = wp_strip_all_tags( $post->post_content );
        
        if ( ! empty( $content ) && $this->api->is_configured() ) {
            $response = $this->api->detect_ai_content( $content );
            
            if ( ! is_wp_error( $response ) && $response && is_array( $response ) ) {
                $result = array(
                    'ai_probability' => isset( $response['score'] ) ? $response['score'] : null,
                    'score'       => isset( $response['score'] ) ? $response['score'] : null,
                    'confidence'  => isset( $response['confidence'] ) ? $response['confidence'] : null,
                    'timestamp'   => time(),
                );
                
                update_post_meta( $post_id, '_uncertifyai_ai_check', $result );
                $this->db->save_check_result( $post_id, $result, $post_id );
                
                // Set transient for admin notice
                set_transient( 'uncertifyai_recheck_result_' . get_current_user_id(), array(
                    'success' => true,
                    'ai_probability' => $result['ai_probability'],
                    'confidence' => $result['confidence'],
                ), 30 );
            } else {
                $error = is_wp_error( $response ) ? $response->get_error_message() : ( isset( $response['message'] ) ? $response['message'] : __( 'Unknown error occurred', 'ai-humanizer-ai-detector-uncertify-ai' ) );
                set_transient( 'uncertifyai_recheck_result_' . get_current_user_id(), array(
                    'success' => false,
                    'error' => $error,
                ), 30 );
            }
        }
        
        wp_safe_redirect( admin_url( 'edit.php?post_type=' . $post->post_type ) );
        exit;
    }

    /**
     * Admin recheck notice
     */
    public function admin_recheck_notice() {
        $result = get_transient( 'uncertifyai_recheck_result_' . get_current_user_id() );
        
        if ( $result ) {
            if ( $result['success'] ) {
                echo '<div class="notice notice-success is-dismissible">';
                
                $message = sprintf(
                    /* translators: %1$s: AI Probability percentage, %2$s: Confidence percentage */
                    esc_html__( 'Content rechecked successfully. AI Probability: %1$s%%, Confidence: %2$s%%', 'ai-humanizer-ai-detector-uncertify-ai' ),
                    esc_html( $result['ai_probability'] * 100 ),
                    esc_html( $result['confidence'] )
                );
                echo '<p>' . esc_html( $message ) . '</p>';
                
                echo '</div>';
            } else {
                echo '<div class="notice notice-error is-dismissible">';
                
                $message = sprintf(
                    /* translators: %s: Error message returned from content recheck */
                    esc_html__( 'Error rechecking content: %s', 'ai-humanizer-ai-detector-uncertify-ai' ),
                    esc_html( $result['error'] )
                );
                echo '<p>' . esc_html( $message ) . '</p>';
                
                echo '</div>';
            }
            
            delete_transient( 'uncertifyai_recheck_result_' . get_current_user_id() );
        }
    }
}