<?php
if (!defined('ABSPATH')) die('Restricted Access');

/**
 * Handles all logic related to product data analysis.
 */
class AIFLT_Products_Logic {

    /**
     * Gets a list of low-performing products for the main dashboard.
     */
    public static function get_low_performing_products_for_dashboard($limit = 3) {
        global $wpdb;
        $funnel_table = $wpdb->prefix . 'aiflt_funnel_data';
        $posts_table = $wpdb->prefix . 'posts';
        $order_items_table = $wpdb->prefix . 'woocommerce_order_items';
        $order_itemmeta_table = $wpdb->prefix . 'woocommerce_order_itemmeta';
        
        // Use wc_order_stats to check the actual order status for sales data
        $order_stats_table = $wpdb->prefix . 'wc_order_stats'; 
        
        $date_threshold = gmdate('Y-m-d H:i:s', strtotime('-30 days'));

        // phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared -- Table names are safely prefixed internal constants and are not user-defined.
        $results = $wpdb->get_results($wpdb->prepare(
            "
            SELECT p.ID AS product_id, p.post_title AS product_name, views_query.views, COALESCE(sales_query.sales, 0) AS sales
            FROM {$posts_table} p
            JOIN (
                SELECT post_id, COUNT(DISTINCT session_id) AS views FROM {$funnel_table}
                WHERE page_name = 'product' AND timestamp >= %s GROUP BY post_id
            ) AS views_query ON p.ID = views_query.post_id
            LEFT JOIN (
                -- Sales query (CRITICAL FIX: constrained by confirmed order status)
                SELECT oim.meta_value AS product_id, COUNT(oim.order_item_id) AS sales 
                FROM {$order_itemmeta_table} oim
                JOIN {$order_items_table} oi ON oim.order_item_id = oi.order_item_id
                JOIN {$order_stats_table} os ON oi.order_id = os.order_id
                WHERE oim.meta_key = '_product_id' 
                AND os.date_created >= %s
                AND os.status NOT IN ('wc-cancelled', 'wc-failed', 'wc-refunded', 'wc-draft', 'wc-checkout-draft')
                GROUP BY oim.meta_value
            ) AS sales_query ON p.ID = sales_query.product_id
            WHERE p.post_type = 'product' AND p.post_status = 'publish'
            HAVING views > 20 AND sales < 2
            ORDER BY views DESC
            LIMIT %d
            ",
            // Arguments: [View Date Start], [Sales Date Start], [Limit]
            $date_threshold,
            $date_threshold,
            $limit
        ), ARRAY_A);

        return $results;
    }

    /**
     * --- REVISED: Now handles the case where $aiflt_products_per_page is -1 ---
     */
    public static function get_all_products_with_stats($aiflt_products_per_page, $aiflt_offset, $aiflt_start_date, $aiflt_end_date, $aiflt_status_filter, $orderby = 'views', $order = 'DESC') {
        global $wpdb;
        $posts_table = $wpdb->prefix . 'posts';
        $funnel_table = $wpdb->prefix . 'aiflt_funnel_data';
        $order_items_table = $wpdb->prefix . 'woocommerce_order_items';
        $order_itemmeta_table = $wpdb->prefix . 'woocommerce_order_itemmeta';

        // --- NEW LINE: Get Plugin Activation Time (as a MySQL-formatted timestamp) ---
        $aiflt_plugin_install_time = gmdate('Y-m-d H:i:s', get_option('aiflt_activation_time', strtotime('-1 year')));
        // Note: This relies on AIFLT_Funnel_Analyzer_Logic to calculate the store-wide sales total correctly.
        $aiflt_funnel_data = AIFLT_Funnel_Analyzer_Logic::get_funnel_data('all', $aiflt_start_date, $aiflt_end_date);
        $total_sessions = $aiflt_funnel_data['total_sessions'] > 0 ? (int)$aiflt_funnel_data['total_sessions'] : 1; // Prevent division by zero
        $store_acr = ($aiflt_funnel_data['sales'] / $total_sessions) * 100;
        
        // Set conversion thresholds based on ACR. Use a default minimum CR of 1.5% if ACR is 0 or low.
        $baseline_cr = max($store_acr, 1.5); // Baseline CR is either ACR or a minimum of 1.5%
        $low_performing_cr = $baseline_cr * 0.50; // Low: Converting at 50% of the baseline
        $critical_leak_cr = $baseline_cr * 0.20;  // Critical: Converting at 20% of the baseline
        $min_views_for_analysis = 25; // Minimum views required to classify a status
        // --- END NEW LOGIC ---

        // 1. Whitelist & Sanitize
        $valid_orderby = ['views', 'sales', 'name'];
        if (!in_array($orderby, $valid_orderby)) {
            $orderby = 'views';
        }
        $order = (strtoupper($order) === 'ASC') ? 'ASC' : 'DESC';

        // 2. Build Clauses and Arguments
        $where_clauses = ["p.post_type = 'product' AND p.post_status = 'publish'"];
        $args = []; 
        
        // Calculate the exclusive end date (1 second after the selected day ends)
        $exclusive_end_date = gmdate('Y-m-d', strtotime($aiflt_end_date . ' +1 day'));

        // --- A. Views Date Filter (Always Active) ---
        // CRITICAL FIX: Use the LATER of the two dates: the selected start date OR the install date.
        $min_query_date = max($aiflt_start_date . ' 00:00:00', $aiflt_plugin_install_time);
        
        $views_date_clause = "AND f.timestamp >= %s AND f.timestamp < %s";
        $args[] = $min_query_date; 
        $args[] = $exclusive_end_date;

        // --- B. Sales Date Filter (Always Active) ---
        // Change SQL from BETWEEN %s AND %s to >= %s AND < %s
        $sales_date_clause = "AND os.date_created >= %s AND os.date_created < %s";
        $args[] = $min_query_date; 
        $args[] = $exclusive_end_date; 

        // Sales Subquery Construction
        $sales_select_query = "
            (SELECT COUNT(DISTINCT oi.order_item_id)
             FROM {$order_items_table} oi
             JOIN {$order_itemmeta_table} oim ON oi.order_item_id = oim.order_item_id
             JOIN {$wpdb->prefix}wc_order_stats os ON oi.order_id = os.order_id
             WHERE oim.meta_key = '_product_id' AND oim.meta_value = p.ID
             AND os.status NOT IN ('wc-cancelled', 'wc-failed', 'wc-refunded', 'wc-draft', 'wc-checkout-draft')
             {$sales_date_clause}) as sales
        ";

        // 3. Build HAVING and LIMIT
        $having_clauses = [];
        
        // IMPORTANT: We only apply SQL filtering for statuses that rely only on views and a calculated CR.
        if (!empty($aiflt_status_filter)) {
            if ($aiflt_status_filter === 'Low-Performing') {
                // Low-Performing: Requires views > 25 AND CR below 50% of baseline
                $having_clauses[] = "COALESCE(views, 0) > {$min_views_for_analysis} AND (COALESCE(sales, 0) / views) * 100 < {$low_performing_cr}";
            } elseif ($aiflt_status_filter === 'Critical Leak') {
                // Critical Leak: Requires views > 25 AND CR below 20% of baseline
                $having_clauses[] = "COALESCE(views, 0) > {$min_views_for_analysis} AND (COALESCE(sales, 0) / views) * 100 < {$critical_leak_cr}";
            } elseif ($aiflt_status_filter === 'Good') {
                // Good: Requires views > 25 AND CR above 50% of baseline (Excludes 'Needs Data' products)
                $having_clauses[] = "COALESCE(views, 0) > {$min_views_for_analysis} AND (COALESCE(sales, 0) / views) * 100 >= {$low_performing_cr}";
            }
        }
        
        $where_sql = !empty($where_clauses) ? 'WHERE ' . implode(' AND ', $where_clauses) : '';
        $having_sql = !empty($having_clauses) ? 'HAVING ' . implode(' AND ', $having_clauses) : '';
        
        // Only add the LIMIT clause if $aiflt_products_per_page is not negative
        $limit_sql = '';
        $temp_args = $args; // Use temporary args array for splicing
        
        if ($aiflt_products_per_page > 0) {
            // If we are paginating, add the LIMIT clause with placeholders.
            $limit_sql = "LIMIT %d, %d";
            $temp_args[] = $aiflt_offset; // Argument 5
            $temp_args[] = $aiflt_products_per_page; // Argument 6
        }

        // 4. Construct the final query string and merge arguments
        $sql = "
            SELECT SQL_CALC_FOUND_ROWS 
                   p.ID AS product_id, 
                   p.post_title AS name, 
                   COALESCE(COUNT(DISTINCT f.session_id), 0) AS views, 
                   {$sales_select_query}
            FROM {$posts_table} p
            LEFT JOIN {$funnel_table} f 
                ON p.ID = f.post_id 
                AND f.page_name = 'product' 
                {$views_date_clause}
            {$where_sql}
            GROUP BY p.ID, p.post_title
            {$having_sql}
            ORDER BY {$orderby} {$order}
            {$limit_sql}
        ";

        // Prepare the query.
        $prepared_query = $wpdb->prepare( $sql, ...$temp_args );

        // Execute securely
        // phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared -- The query string is fully prepared in the preceding line and is safe for execution.
        $aiflt_products = $wpdb->get_results( $prepared_query, ARRAY_A );
        $aiflt_total_products = $wpdb->get_var("SELECT FOUND_ROWS()");

        // 6. Final Status Calculation (PHP side filtering for 'Needs Data')
        $filtered_products = [];
        foreach ($aiflt_products as &$aiflt_product) {
            $conversion_rate_raw = ($aiflt_product['views'] > 0) ? ($aiflt_product['sales'] / $aiflt_product['views']) * 100 : 0;
            $aiflt_product['conversion_rate'] = round($conversion_rate_raw, 2) . '%';
            
            // Re-evaluate status based on the dynamic CR thresholds
            if ($aiflt_product['views'] < $min_views_for_analysis) {
                $aiflt_product['status'] = __('Needs Data', 'ai-flash-tune');
            } elseif ($conversion_rate_raw < $critical_leak_cr) {
                $aiflt_product['status'] = __('Critical Leak', 'ai-flash-tune'); 
            } elseif ($conversion_rate_raw < $low_performing_cr) {
                $aiflt_product['status'] = __('Low-Performing', 'ai-flash-tune');
            } else {
                $aiflt_product['status'] = __('Good', 'ai-flash-tune');
            }
            
            $aiflt_product['edit_link'] = get_edit_post_link($aiflt_product['product_id']);

            // Post-query filtering for Needs Data
            if ($aiflt_status_filter === 'Needs Data') {
                if ($aiflt_product['status'] === 'Needs Data') {
                    $filtered_products[] = $aiflt_product;
                }
            } else {
                $filtered_products[] = $aiflt_product;
            }
        }
        
        // If filtering by Needs Data, we need to adjust the total count returned.
        if ($aiflt_status_filter === 'Needs Data') {
            $aiflt_total_products = count($filtered_products);
        }

        return ['products' => $filtered_products, 'total_products' => $aiflt_total_products];
    }

    /**
     * Retrieves detailed statistics for a single product.
     */
    public static function get_product_details($aiflt_product_id, $aiflt_start_date, $aiflt_end_date, $aiflt_device_type) {
        global $wpdb;
        $aiflt_product = wc_get_product($aiflt_product_id);
        if (!$aiflt_product) return null;

        $funnel_table = $wpdb->prefix . 'aiflt_funnel_data';
        $behavior_table = $wpdb->prefix . 'aiflt_behavior_data';
        $device_where = ($aiflt_device_type !== 'all') ? $wpdb->prepare("AND device_type = %s", $aiflt_device_type) : '';

        // --- Get Plugin Activation Time ---
        $aiflt_plugin_install_time = gmdate('Y-m-d H:i:s', get_option('aiflt_activation_time', strtotime('-1 year')));
        
        // This is the exclusive end boundary (e.g., Nov 4th 00:00:00).
        $exclusive_end_date = gmdate('Y-m-d', strtotime($aiflt_end_date . ' +1 day'));
        
        // CRITICAL: Determine the actual minimum date for the query
        $min_query_date = max($aiflt_start_date . ' 00:00:00', $aiflt_plugin_install_time);
        
        // Arguments array for the time/device filter
        $time_device_args = array_merge([$min_query_date, $exclusive_end_date], ($aiflt_device_type !== 'all' ? [$aiflt_device_type] : []));
        
        // --- 1. Product Sales Count (for the current product in the date range) ---
        $sales_sql = "
            SELECT COUNT(DISTINCT oim.order_item_id) 
            FROM {$wpdb->prefix}woocommerce_order_itemmeta oim
            JOIN {$wpdb->prefix}woocommerce_order_items oi ON oim.order_item_id = oi.order_item_id
            JOIN {$wpdb->prefix}wc_order_stats os ON oi.order_id = os.order_id
            WHERE oim.meta_key = %s AND oim.meta_value = %d
            AND os.status NOT IN ('wc-cancelled', 'wc-failed', 'wc-refunded', 'wc-draft', 'wc-checkout-draft')
            AND os.date_created >= %s AND os.date_created < %s
        ";
        
        // Arguments: ['_product_id', $aiflt_product_id, $min_query_date, $exclusive_end_date]
        $sales_args = ['_product_id', $aiflt_product_id, $min_query_date, $exclusive_end_date];
        $sales = (int) $wpdb->get_var($wpdb->prepare($sales_sql, $sales_args));

        // --- 2. Store Sales and Sessions (for benchmark) ---
        
        // Total Sessions (Store-wide unique sessions in the date range)
        // phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared -- Table name is safely prefixed.
        $total_store_sessions = (int) $wpdb->get_var($wpdb->prepare(
            "SELECT COUNT(DISTINCT session_id) 
             FROM {$funnel_table} 
             WHERE timestamp >= %s AND timestamp < %s",
            $min_query_date, $exclusive_end_date
        ));
        
        // Total Sales Orders (Store-wide unique sales in the date range)
        $total_store_sales = (int) $wpdb->get_var($wpdb->prepare("
            SELECT COUNT(DISTINCT order_id) 
            FROM {$wpdb->prefix}wc_order_stats 
            WHERE status NOT IN ('wc-cancelled', 'wc-failed', 'wc-refunded', 'wc-draft', 'wc-checkout-draft')
            AND date_created >= %s AND date_created < %s
        ", $min_query_date, $exclusive_end_date));
        
        $store_conversion_rate = ($total_store_sessions > 0) ? round(($total_store_sales / $total_store_sessions) * 100, 2) : 0;
        
        // --- 3. Traffic, Behavioral Data, and Page Speed (All use min_query_date) ---
        // Traffic (Views)
        $views_sql = "
            SELECT COUNT(DISTINCT session_id) 
            FROM {$funnel_table} 
            WHERE post_id = %d AND page_name = 'product' 
            AND timestamp >= %s AND timestamp < %s 
            {$device_where}
        ";
        $views_args = array_merge([$aiflt_product_id], $time_device_args);
        // phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared -- The query string uses safely prefixed table names and controlled internal strings for conditional filtering.
        $total_sessions = (int) $wpdb->get_var($wpdb->prepare($views_sql, $views_args));

        // Average Page Speed
        $avg_speed_sql = "
            SELECT AVG(page_speed) 
            FROM {$funnel_table} 
            WHERE post_id = %d AND page_name = 'product' 
            AND page_speed IS NOT NULL AND page_speed > 0
            AND timestamp >= %s AND timestamp < %s 
            {$device_where}
        ";
        $avg_speed_args = array_merge([$aiflt_product_id], $time_device_args);
        // phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared -- The query string uses safely prefixed table names and controlled internal strings for conditional filtering.
        $avg_page_speed_raw = $wpdb->get_var($wpdb->prepare($avg_speed_sql, $avg_speed_args));
        $avg_page_speed = $avg_page_speed_raw ? round($avg_page_speed_raw, 2) . 's' : __('N/A', 'ai-flash-tune');

        // Cart Adds
        $aiflt_cart_adds_sql = "
            SELECT COUNT(DISTINCT session_id) 
            FROM {$behavior_table} 
            WHERE product_id = %d AND behavior_type = 'add_to_cart' 
            AND timestamp >= %s AND timestamp < %s
        ";
        $aiflt_cart_adds_args = [$aiflt_product_id, $min_query_date, $exclusive_end_date];
        // phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared -- The query string uses safely prefixed table names and controlled internal strings for filtering.
        $aiflt_cart_adds = (int) $wpdb->get_var($wpdb->prepare($aiflt_cart_adds_sql, $aiflt_cart_adds_args));
        
        // Rage clicks, scroll stops, quick bounces logic (using $min_query_date)
        $behavior_base_args = [$aiflt_product_id, $min_query_date, $exclusive_end_date];

        $rage_clicks = (int) $wpdb->get_var($wpdb->prepare(
            "SELECT COUNT(*) FROM {$behavior_table} 
             WHERE product_id = %d AND behavior_type = 'rage_click' 
             AND timestamp >= %s AND timestamp < %s",
            ...$behavior_base_args
        ));

        $scroll_stops = (int) $wpdb->get_var($wpdb->prepare(
            "SELECT COUNT(*) FROM {$behavior_table} 
             WHERE product_id = %d AND behavior_type = 'scroll_stop' 
             AND timestamp >= %s AND timestamp < %s",
            ...$behavior_base_args
        ));

        $quick_bounces = (int) $wpdb->get_var($wpdb->prepare(
            "SELECT COUNT(*) FROM {$behavior_table} 
             WHERE product_id = %d AND behavior_type = 'quick_bounce' 
             AND timestamp >= %s AND timestamp < %s",
            ...$behavior_base_args
        ));

        return [
            'name' => $aiflt_product->get_name(),
            'views' => $total_sessions,
            'sales' => $sales,
            'conversion_rate' => ($total_sessions > 0) ? round(($sales / $total_sessions) * 100, 2) . '%' : '0%',
            'avg_page_speed' => $avg_page_speed,
            'store_conversion_rate' => $store_conversion_rate,
            'funnel' => ['views' => $total_sessions, 'cart_adds' => $aiflt_cart_adds, 'sales' => $sales],
            'behavioral_insights' => [
                'Rage Clicks' => $rage_clicks,
                'Quick Bounces' => $quick_bounces,
                'Scroll Stops' => $scroll_stops,
            ]
        ];
    }
    
    /**
     * Placeholder for getting an AI-generated copy suggestion.
     */
    public static function get_ai_copy_suggestion($aiflt_product_id) {
        $aiflt_product = wc_get_product($aiflt_product_id);
        if ($aiflt_product) {
            /* translators: %s: Product name */
            return __('AI suggests clarifying the main benefit of ', 'ai-flash-tune') . $aiflt_product->get_name() . ' ' . __('in the first sentence...', 'ai-flash-tune');
        }
        return __('No specific AI suggestion available.', 'ai-flash-tune');
    }

    /**
     * Retrieves the device breakdown for traffic.
     */
    public static function get_device_breakdown($aiflt_start_date, $aiflt_end_date) {
        global $wpdb;
        $funnel_table = $wpdb->prefix . 'aiflt_funnel_data';
        $results = $wpdb->get_results($wpdb->prepare("SELECT device_type, COUNT(DISTINCT session_id) as count FROM {$funnel_table} WHERE timestamp BETWEEN %s AND %s GROUP BY device_type", $aiflt_start_date, $aiflt_end_date), ARRAY_A);
        $total_sessions = array_sum(wp_list_pluck($results, 'count'));
        $breakdown = ['Desktop' => 0, 'Mobile' => 0];
        if ($total_sessions > 0) {
            foreach ($results as $row) {
                if (in_array($row['device_type'], ['desktop', 'mobile'])) {
                    $breakdown[ucfirst($row['device_type'])] = round(($row['count'] / $total_sessions) * 100);
                }
            }
        }
        return $breakdown;
    }

    /**
     * NEW: Analyzes a product page for visual noise (too many CTAs or images).
     *
     * @param int $aiflt_product_id The ID of the product to analyze.
     * @return array An array containing the analysis results.
     */
    public static function get_visual_noise_analysis($aiflt_product_id) {
        $url = get_permalink($aiflt_product_id);
        if (!$url) {
            return [
                'status' => 'Error',
                'message' => __('Could not find the product URL.', 'ai-flash-tune'),
                'css_class' => 'aiflashtune-text-gray-500',
                'icon_class' => 'fas fa-question-circle',
                'issues' => [],
            ];
        }

        // Use standard WordPress HTTP API without disabling SSL verification
        $response = wp_remote_get($url);

        if (is_wp_error($response) || wp_remote_retrieve_response_code($response) !== 200) {
            return [
                'status' => 'Error',
                'message' => __('Could not fetch the product page for analysis. This can happen on local development environments.', 'ai-flash-tune'),
                'css_class' => 'aiflashtune-text-gray-500',
                'icon_class' => 'fas fa-question-circle',
                'issues' => [],
            ];
        }

        $html = wp_remote_retrieve_body($response);
        if (empty($html)) {
            return [ 'status' => 'Error', 'message' => __('Product page content appears to be empty.', 'ai-flash-tune'), 'css_class' => 'aiflashtune-text-gray-500', 'icon_class' => 'fas fa-question-circle', 'issues' => []];
        }

        // Use DOMDocument to safely parse the HTML
        libxml_use_internal_errors(true);
        $dom = new DOMDocument();
        $dom->loadHTML($html);
        libxml_clear_errors();
        $xpath = new DOMXPath($dom);

        // Define what we consider a CTA or a primary image
        $cta_query = ".//a[contains(concat(' ', normalize-space(@class), ' '), ' button ')] | .//button[contains(concat(' ', normalize-space(@class), ' '), ' button ')] | .//button[contains(@name, 'add-to-cart')]";
        $image_query = ".//div[contains(concat(' ', normalize-space(@class), ' '), ' product ')]//img";

        $cta_count = $xpath->query($cta_query)->length;
        $image_count = $xpath->query($image_query)->length;

        // --- Heuristic Thresholds ---
        $cta_threshold = 5;
        $image_threshold = 10;
        $score = 0;
        $aiflt_issues = [];

        if ($cta_count > $cta_threshold) {
            $score++;
            $aiflt_issues[] = sprintf(
                /* translators: 1: Number of CTA buttons detected, 2: Recommended maximum number of CTA buttons */
                __( 'High number of CTA buttons detected: %1$d (Recommended: under %2$d)', 'ai-flash-tune' ),
                $cta_count,
                $cta_threshold
            );
        }
        if ($image_count > $image_threshold) {
            $score++;
            $aiflt_issues[] = sprintf(
                /* translators: 1: Number of images detected, 2: Recommended maximum number of images */
                __('High number of images detected: %1$d (Recommended: under %2$d)', 'ai-flash-tune'),
                $image_count,
                $image_threshold
            );
        }

        if ($score > 0) {
            return [
                'status' => 'High',
                'message' => __('High Visual Noise Detected', 'ai-flash-tune'),
                'description' => esc_html__( 'Too many competing elements can distract users from the main ', 'ai-flash-tune' ) . esc_html__( 'Add to Cart', 'ai-flash-tune' ) . esc_html__( ' action, potentially reducing conversions.', 'ai-flash-tune' ),
                'css_class' => 'aiflashtune-text-orange-500',
                'icon_class' => 'fas fa-exclamation-triangle',
                'issues' => $aiflt_issues,
            ];
        }

        return [
            'status' => 'Good',
            'message' => __('Optimal Noise Level', 'ai-flash-tune'),
            'description' => __('The page appears to have a clear and focused layout with a reasonable number of interactive elements.', 'ai-flash-tune'),
            'css_class' => 'aiflashtune-text-green-500',
            'icon_class' => 'fas fa-check-circle',
            'issues' => [],
        ];
    }

    public static function has_any_products() {
        global $wpdb;
        $count = (int) $wpdb->get_var("SELECT COUNT(ID) FROM {$wpdb->prefix}posts WHERE post_type = 'product' AND post_status = 'publish'");
        return $count > 0;
    }
}
