/**
 * Navigation buttons functionality for the Gutenberg editor
 *
 * @package AI_Editor_Post_Switcher
 */

wp.domReady(function () {
    // Global variable to store status counts
    let globalStatusCounts = null;

    // Store first/last posts data globally (they don't change during navigation)
    let navigationData = {
        first: null,
        last: null,
        firstTitle: null,
        lastTitle: null
    };

    // Search debounce timer
    let searchDebounceTimer = null;

    const loadSvgIcon = (iconName) => {
        return fetch(AI_Editor_HeaderData.plugin_url + 'assets/icons/' + iconName + '.svg')
            .then(response => {
                if (!response.ok) {
                    throw new Error('Failed to load SVG icon: ' + iconName);
                }
                return response.text();
            });
    };

    const createNavButton = (direction, url, title, disabled) => {
        const button = document.createElement('button');
        button.type = 'button';
        button.className = 'ai-nav-button';
        button.title = title;
        button.setAttribute('aria-label', title);
        button.setAttribute('data-direction', direction);

        if (disabled) {
            button.disabled = true;
            button.classList.add('is-disabled');
            button.setAttribute('aria-disabled', 'true');
        } else {
            button.addEventListener('click', () => {
                window.location.href = url;
            });
        }

        let iconName;
        switch (direction) {
            case 'first': iconName = 'icon-first'; break;
            case 'prev':  iconName = 'icon-previous'; break;
            case 'next':  iconName = 'icon-next'; break;
            case 'last':  iconName = 'icon-last'; break;
        }

        const iconContainer = document.createElement('span');
        iconContainer.className = 'ai-nav-icon';
        button.appendChild(iconContainer);
        iconContainer.innerHTML = '<span class="ai-loading-icon"></span>';
        loadSvgIcon(iconName)
            .then(svgContent => {
                iconContainer.innerHTML = svgContent;
                const svg = iconContainer.querySelector('svg');
                if (svg) {
                    svg.setAttribute('width', '16');
                    svg.setAttribute('height', '16');
                    svg.setAttribute('fill', 'currentColor');
                    svg.setAttribute('aria-hidden', 'true');
                    svg.setAttribute('focusable', 'false');
                }
            })
            .catch(() => {
                // These symbols are directional indicators, kept as-is as they're standard
                iconContainer.textContent = {
                    'first': '«', // Double left arrow for "first"
                    'prev': '‹',  // Single left arrow for "previous"
                    'next': '›',  // Single right arrow for "next"
                    'last': '»'   // Double right arrow for "last"
                }[direction] || '•';
            });
        return button;
    };

    // Create inline SVG icon helper
    const createInlineSvgIcon = (iconName, size = 'font') => {
        const span = document.createElement('span');
        span.className = 'ai-popup-item-icon';
        
        if (size === 'font') {
            span.style.width = '1em';
            span.style.height = '1em';
        } else {
            span.style.width = size + 'px';
            span.style.height = size + 'px';
        }
        
        loadSvgIcon(iconName)
            .then(svgContent => {
                span.innerHTML = svgContent;
                const svg = span.querySelector('svg');
                if (svg) {
                    svg.style.width = '100%';
                    svg.style.height = '100%';
                    svg.style.display = 'block';
                    svg.setAttribute('fill', 'currentColor');
                    svg.setAttribute('aria-hidden', 'true');
                    svg.setAttribute('focusable', 'false');
                }
            })
            .catch(() => {
                // Fallback symbols
                const fallbacks = {
                    'icon-first': '«',
                    'icon-last': '»',
                    'icon-previous': '‹',
                    'icon-next': '›',
                    'icon-current': '•'
                };
                span.textContent = fallbacks[iconName] || '•';
            });
        
        return span;
    };

    const addNavigationButtons = () => {
        if (
            typeof AI_Editor_HeaderData !== 'object' ||
            !Array.isArray(AI_Editor_HeaderData.allowed_post_types) ||
            typeof AI_Editor_HeaderData.current_post_type !== 'string' ||
            !AI_Editor_HeaderData.allowed_post_types.includes(AI_Editor_HeaderData.current_post_type) ||
            typeof AI_Editor_HeaderData.plugin_url !== 'string'
        ) {
            return;
        }
        const toolbar = document.querySelector('.editor-header__settings');
        if (!toolbar || toolbar.querySelector('.ai-editor-nav-buttons')) return;

        // Store first/last data on initial load (they won't change)
        if (!navigationData.first && AI_Editor_HeaderData.first) {
            navigationData.first = AI_Editor_HeaderData.first;
            navigationData.firstTitle = AI_Editor_HeaderData.first_title;
        }
        if (!navigationData.last && AI_Editor_HeaderData.last) {
            navigationData.last = AI_Editor_HeaderData.last;
            navigationData.lastTitle = AI_Editor_HeaderData.last_title;
        }

        const container = document.createElement('div');
        container.className = 'ai-editor-nav-buttons';

        if (AI_Editor_HeaderData.enable_first_last == 1 || AI_Editor_HeaderData.enable_first_last === true || AI_Editor_HeaderData.enable_first_last === "1") {
            const firstButton = createNavButton(
                'first',
                navigationData.first || AI_Editor_HeaderData.first,
                navigationData.firstTitle || AI_Editor_HeaderData.first_title || AI_Editor_HeaderData.labels.first,
                !navigationData.first && !AI_Editor_HeaderData.first
            );
            container.appendChild(firstButton);
        }
        
        const previousButton = createNavButton(
            'prev',
            AI_Editor_HeaderData.previous,
            AI_Editor_HeaderData.previous_title || AI_Editor_HeaderData.labels.previous,
            !AI_Editor_HeaderData.previous
        );
        container.appendChild(previousButton);
        
        const nextButton = createNavButton(
            'next',
            AI_Editor_HeaderData.next,
            AI_Editor_HeaderData.next_title || AI_Editor_HeaderData.labels.next,
            !AI_Editor_HeaderData.next
        );
        container.appendChild(nextButton);

        if (AI_Editor_HeaderData.enable_first_last == 1 || AI_Editor_HeaderData.enable_first_last === true || AI_Editor_HeaderData.enable_first_last === "1") {
            const lastButton = createNavButton(
                'last',
                navigationData.last || AI_Editor_HeaderData.last,
                navigationData.lastTitle || AI_Editor_HeaderData.last_title || AI_Editor_HeaderData.labels.last,
                !navigationData.last && !AI_Editor_HeaderData.last
            );
            container.appendChild(lastButton);
        }

        // Add search button
        const searchButton = document.createElement('button');
        searchButton.type = 'button';
        searchButton.className = 'ai-nav-button ai-search-button';
        searchButton.title = AI_Editor_HeaderData.labels.search || 'Search';
        searchButton.setAttribute('aria-label', AI_Editor_HeaderData.labels.search || 'Search');
        searchButton.innerHTML = '🔍'; // Magnifying glass emoji - kept as-is as it's a standard symbol
        searchButton.addEventListener('click', openSearchPopup);
        container.appendChild(searchButton);

        toolbar.prepend(container);
    };

    // Define status types and their labels (in display order)
    const STATUS_TYPES = {
        'publish': (AI_Editor_HeaderData.status_labels && AI_Editor_HeaderData.status_labels.publish) || 'PUBLISHED',
        'pending': (AI_Editor_HeaderData.status_labels && AI_Editor_HeaderData.status_labels.pending) || 'PENDING',
        'draft': (AI_Editor_HeaderData.status_labels && AI_Editor_HeaderData.status_labels.draft) || 'DRAFT',
        'future': (AI_Editor_HeaderData.status_labels && AI_Editor_HeaderData.status_labels.future) || 'SCHEDULED',
        'private': (AI_Editor_HeaderData.status_labels && AI_Editor_HeaderData.status_labels.private) || 'PRIVATE'
    };

    // Search popup functionality
    function openSearchPopup(event) {
        // Create popup if it doesn't exist
        let popup = document.getElementById('ai-posts-search-popup');
        if (!popup) {
            popup = document.createElement('div');
            popup.id = 'ai-posts-search-popup';
            popup.className = 'ai-posts-search-popup';
            
            // Create popup header with search input and close button
            const popupHeader = document.createElement('div');
            popupHeader.className = 'ai-popup-header';
            
            const searchInput = document.createElement('input');
            searchInput.id = 'ai-popup-search-input';
            searchInput.className = 'ai-popup-search-input';
            searchInput.type = 'text';
            searchInput.placeholder = AI_Editor_HeaderData.labels.search_placeholder || 'Type to search...';
            searchInput.addEventListener('input', function() {
                // Delay input handler to ensure data is loaded
                if (popup.hasAttribute('data-context-loaded')) {
                    handleSearchInput(popup, this.value);
                }
            });
            
            const closeButton = document.createElement('button');
            closeButton.innerHTML = '×';
            closeButton.className = 'ai-popup-close';
            closeButton.addEventListener('click', function() {
                popup.style.display = 'none';
            });
            
            popupHeader.appendChild(searchInput);
            popupHeader.appendChild(closeButton);
            
            // Position indicator
            const positionIndicator = document.createElement('div');
            positionIndicator.className = 'ai-popup-position-indicator';
            
            // Status filters section (always visible but empty initially)
            const statusFilters = document.createElement('div');
            statusFilters.className = 'ai-popup-status-filters';
            statusFilters.style.display = 'none'; // Hide initially until data loads
            
            // Create popup content for posts list
            const popupContent = document.createElement('div');
            popupContent.className = 'ai-popup-content';
            
            // Create loading indicator
            const loadingDiv = document.createElement('div');
            loadingDiv.className = 'ai-popup-loading';
            loadingDiv.textContent = AI_Editor_HeaderData.labels.loading || 'Loading...';
            popupContent.appendChild(loadingDiv);
            
            popup.appendChild(popupHeader);
            popup.appendChild(positionIndicator);
            popup.appendChild(statusFilters);
            popup.appendChild(popupContent);
            
            // Close when clicking outside
            document.addEventListener('click', function(event) {
                if (popup.style.display === 'block' && 
                    !popup.contains(event.target) && 
                    !event.target.classList.contains('ai-search-button')) {
                    popup.style.display = 'none';
                }
            });
            
            document.body.appendChild(popup);
        }
        
        // Show popup
        popup.style.display = 'block';
        
        // Position the popup
        const buttonRect = event.target.getBoundingClientRect();
        popup.style.top = (buttonRect.bottom + window.scrollY + 5) + 'px';
        popup.style.right = (window.innerWidth - buttonRect.right) + 'px';
        
        // Reset search input
        const searchInput = popup.querySelector('.ai-popup-search-input');
        searchInput.value = '';
        
        // Focus on search input after data loads
        setTimeout(() => {
            if (popup.hasAttribute('data-context-loaded')) {
                searchInput.focus();
            }
        }, 100);
        
        // Fetch context posts if not already loaded
        if (!popup.hasAttribute('data-context-loaded')) {
            fetchContextPosts(popup);
        } else {
            // Show context view again
            showContextView(popup);
        }
    }

    function fetchContextPosts(popup) {
        const popupContent = popup.querySelector('.ai-popup-content');
        const currentPostId = wp.data.select('core/editor').getCurrentPostId();
        const currentPostType = wp.data.select('core/editor').getCurrentPostType() || AI_Editor_HeaderData.current_post_type;
        
        // Get the nonce parameter name from the localized data
        const nonceParamName = AI_Editor_HeaderData.ajax_nonce_name || 'ai_editor_ajax_nonce';
        
        // Build URL with sort parameters from current page
        const urlParams = new URLSearchParams(window.location.search);
        const ajaxParams = new URLSearchParams({
            action: 'ai_editor_get_context_posts',
            post_type: currentPostType,
            current_id: currentPostId,
            [nonceParamName]: AI_Editor_HeaderData.nonce
        });
        
        // Pass along sort parameters
        if (urlParams.has('orderby')) {
            ajaxParams.set('orderby', urlParams.get('orderby'));
        }
        if (urlParams.has('order')) {
            ajaxParams.set('order', urlParams.get('order'));
        }
        if (urlParams.has('post_status')) {
            ajaxParams.set('post_status', urlParams.get('post_status'));
        }
        
        // Create AJAX request to fetch context posts
        const xhr = new XMLHttpRequest();
        xhr.open('GET', ajaxurl + '?' + ajaxParams.toString(), true);
        
        xhr.onload = function() {
            if (xhr.status === 200) {
                try {
                    const response = JSON.parse(xhr.responseText);
                    if (response.success && response.data) {
                        // Store the complete data including status counts
                        popup.setAttribute('data-context', JSON.stringify(response.data));
                        popup.setAttribute('data-context-loaded', 'true');
                        
                        // Store status counts globally and locally
                        if (response.data.status_counts) {
                            globalStatusCounts = response.data.status_counts;
                            popup.setAttribute('data-status-counts', JSON.stringify(response.data.status_counts));
                            popup._statusCounts = response.data.status_counts;
                        }
                        
                        // Render with the fresh data
                        renderContextPosts(popup, response.data);
                        updatePositionIndicator(popup, response.data);
                    } else {
                        popupContent.innerHTML = '<div class="ai-popup-error">' + 
                            (AI_Editor_HeaderData.labels.error_loading || 'Error loading posts') + '</div>';
                    }
                } catch (e) {
                    popupContent.innerHTML = '<div class="ai-popup-error">' + 
                        (AI_Editor_HeaderData.labels.error_parsing || 'Error parsing response') + '</div>';
                }
            } else {
                popupContent.innerHTML = '<div class="ai-popup-error">' + 
                    (AI_Editor_HeaderData.labels.error_loading || 'Error loading posts') + '</div>';
            }
        };
        
        xhr.onerror = function() {
            popupContent.innerHTML = '<div class="ai-popup-error">' + 
                (AI_Editor_HeaderData.labels.network_error || 'Network error') + '</div>';
        };
        
        xhr.send();
    }

    function renderContextPosts(popup, contextData) {
        const popupContent = popup.querySelector('.ai-popup-content');
        
        // Clear loading message
        popupContent.innerHTML = '';
        
        if (!contextData.posts || contextData.posts.length === 0) {
            popupContent.innerHTML = '<div class="ai-popup-no-posts">' + 
                (AI_Editor_HeaderData.labels.no_posts_found || 'No results') + '</div>';
            return;
        }
        
        // Create status filters for context posts (excluding prev_2 and next_2)
        const filteredPosts = contextData.posts.filter(post => 
            post.context_type !== 'prev_2' && post.context_type !== 'next_2'
        );
        createStatusFilters(popup, filteredPosts);
        
        // Create list element
        const list = document.createElement('ul');
        list.className = 'ai-popup-posts-list ai-popup-context-list';
        
        // Add each post to the list
        contextData.posts.forEach((post, index) => {
            // Skip prev_2 and next_2 posts
            if (post.context_type === 'prev_2' || post.context_type === 'next_2') {
                return;
            }
            
            const listItem = document.createElement('li');
            listItem.className = 'ai-popup-post-item ai-popup-post-with-icon';
            
            // Add context type classes
            if (post.context_type === 'current') {
                listItem.classList.add('ai-popup-current-post');
            } else if (post.context_type === 'first' || post.context_type === 'last') {
                listItem.classList.add('ai-popup-boundary-post');
            }
            
            const link = document.createElement('a');
            link.href = post.edit_url;
            
            // Determine which icon to use
            let iconName = null;
            if (post.context_type === 'first') {
                iconName = 'icon-first';
            } else if (post.context_type === 'last') {
                iconName = 'icon-last';
            } else if (post.context_type === 'prev_1') {
                iconName = 'icon-previous';
            } else if (post.context_type === 'next_1') {
                iconName = 'icon-next';
            } else if (post.context_type === 'current') {
                iconName = 'icon-current';
            }
            
            // Add icon if needed
            if (iconName) {
                link.appendChild(createInlineSvgIcon(iconName));
            }
            
            // Add title in a span
            const titleSpan = document.createElement('span');
            titleSpan.className = 'ai-popup-item-title';
            titleSpan.textContent = post.title || '(Untitled)';
            link.appendChild(titleSpan);
            
            // Add status indicator
            if (post.status) {
                const statusSpan = document.createElement('span');
                statusSpan.className = 'ai-popup-post-status ai-status-' + post.status;
                statusSpan.textContent = STATUS_TYPES[post.status] || post.status.toUpperCase();
                statusSpan.setAttribute('data-status', post.status);
                statusSpan.setAttribute('title', AI_Editor_HeaderData.labels.filter_by_status || 'Click to filter by this status');
                statusSpan.style.cursor = 'pointer';
                statusSpan.addEventListener('click', function(e) {
                    e.preventDefault();
                    e.stopPropagation();
                    filterByStatus(popup, post.status);
                });
                
                // Add password indicator to status badge if needed
                if (post.password_protected) {
                    // statusSpan.style.position = 'relative'; // Intentionally removed
                    const passwordIndicator = document.createElement('span');
                    passwordIndicator.className = 'ai-popup-password-indicator';
                    passwordIndicator.setAttribute('title', AI_Editor_HeaderData.labels.password_protected || 'Password Protected');
                    statusSpan.appendChild(passwordIndicator);
                }
                
                listItem.appendChild(statusSpan); // Append statusSpan to listItem directly
            }
            
            // Store data for filtering
            listItem.setAttribute('data-title', post.title || '');
            listItem.setAttribute('data-status', post.status || '');
            
            listItem.appendChild(link);
            list.appendChild(listItem);
        });
        
        popupContent.appendChild(list);
        
        // Apply current filter if any
        const activeStatus = popup.getAttribute('data-active-status');
        if (activeStatus && activeStatus !== 'all') {
            filterByStatus(popup, activeStatus);
        }
    }

    function updatePositionIndicator(popup, contextData) {
        const indicator = popup.querySelector('.ai-popup-position-indicator');
        if (indicator && contextData.current_position && contextData.total_posts) {
            // translators: %1$d = current page's rank, %2$d = total number of pages
            const positionFormat = AI_Editor_HeaderData.labels.position_format || 'Rank: %1$d of %2$d';

            // Format with HTML for visual emphasis
            const positionHtml = positionFormat
                .replace('%1$d', `<span class="ai-rank-current">${contextData.current_position}</span>`)
                .replace('%2$d', `<span class="ai-rank-total">${contextData.total_posts}</span>`);

            indicator.innerHTML = `<div class="ai-rank-indicator">${positionHtml}</div>`;
            indicator.style.display = 'block';

            // translators: %1$d = current page's rank, %2$d = total number of pages
            const positionText = positionFormat
                .replace('%1$d', contextData.current_position)
                .replace('%2$d', contextData.total_posts);
            indicator.setAttribute('title', positionText);
            // Or use: indicator.setAttribute('aria-label', positionText);
        }
    }

    function showContextView(popup) {
        const contextData = popup.getAttribute('data-context');
        if (contextData) {
            const data = JSON.parse(contextData);
            
            // Get stored status counts from multiple sources
            if (!data.status_counts) {
                // Try from backup
                if (popup._statusCounts) {
                    data.status_counts = popup._statusCounts;
                }
                // Try from data attribute
                else {
                    const statusCountsData = popup.getAttribute('data-status-counts');
                    if (statusCountsData) {
                        try {
                            data.status_counts = JSON.parse(statusCountsData);
                        } catch (e) {
                            console.error('Error parsing status counts in showContextView:', e);
                        }
                    }
                }
            }
            
            renderContextPosts(popup, data);
            updatePositionIndicator(popup, data);
        }
    }

    function handleSearchInput(popup, searchTerm) {
        // Clear existing timer
        if (searchDebounceTimer) {
            clearTimeout(searchDebounceTimer);
        }
        
        // If search is empty, show context view
        if (searchTerm.trim().length === 0) {
            showContextView(popup);
            return;
        }
        
        // If search is too short, don't search yet
        if (searchTerm.trim().length < 2) {
            // Still update status filters to show database counts
            let statusCounts = popup._statusCounts;
            if (!statusCounts) {
                const statusCountsData = popup.getAttribute('data-status-counts');
                if (statusCountsData) {
                    try {
                        statusCounts = JSON.parse(statusCountsData);
                    } catch (e) {
                        console.error('Error parsing status counts:', e);
                    }
                }
            }
            
            if (statusCounts) {
                createStatusFilters(popup, [], statusCounts);
            }
            return;
        }
        
        // Set new timer
        searchDebounceTimer = setTimeout(() => {
            performSearch(popup, searchTerm);
        }, 300);
    }

    function performSearch(popup, searchTerm) {
        const popupContent = popup.querySelector('.ai-popup-content');
        const currentPostId = wp.data.select('core/editor').getCurrentPostId();
        const currentPostType = AI_Editor_HeaderData.current_post_type;
        
        // Show loading state
        popupContent.innerHTML = '<div class="ai-popup-loading">' + 
            (AI_Editor_HeaderData.labels.loading || 'Searching...') + '</div>';
        
        // Hide position indicator during search
        const positionIndicator = popup.querySelector('.ai-popup-position-indicator');
        if (positionIndicator) {
            positionIndicator.style.display = 'none';
        }
        
        // Get the nonce parameter name
        const nonceParamName = AI_Editor_HeaderData.ajax_nonce_name || 'ai_editor_ajax_nonce';
        
        // Build search parameters
        const urlParams = new URLSearchParams(window.location.search);
        const ajaxParams = new URLSearchParams({
            action: 'ai_editor_search_posts',
            post_type: currentPostType,
            current_id: currentPostId,
            search: searchTerm,
            [nonceParamName]: AI_Editor_HeaderData.nonce
        });
        
        // Pass along sort parameters
        if (urlParams.has('orderby')) {
            ajaxParams.set('orderby', urlParams.get('orderby'));
        }
        if (urlParams.has('order')) {
            ajaxParams.set('order', urlParams.get('order'));
        }
        if (urlParams.has('post_status')) {
            ajaxParams.set('post_status', urlParams.get('post_status'));
        }
        
        // Create AJAX request
        const xhr = new XMLHttpRequest();
        xhr.open('GET', ajaxurl + '?' + ajaxParams.toString(), true);
        
        xhr.onload = function() {
            if (xhr.status === 200) {
                try {
                    const response = JSON.parse(xhr.responseText);
                    if (response.success && response.data) {
                        // Store status counts globally and locally
                        const statusCounts = response.data.status_counts || {
                            total: response.data.posts ? response.data.posts.length : 0,
                            publish: 0,
                            draft: 0,
                            pending: 0,
                            private: 0,
                            future: 0
                        };
                        
                        if (response.data.status_counts) {
                            globalStatusCounts = response.data.status_counts;
                        }
                        
                        popup.setAttribute('data-status-counts', JSON.stringify(statusCounts));
                        popup._statusCounts = statusCounts;
                        
                        renderSearchResults(popup, response.data);
                        createStatusFilters(popup, response.data.posts || [], statusCounts);
                    } else {
                        popupContent.innerHTML = '<div class="ai-popup-error">' + 
                            (response.data || AI_Editor_HeaderData.labels.error_loading || 'Error searching posts') + '</div>';
                    }
                } catch (e) {
                    popupContent.innerHTML = '<div class="ai-popup-error">' + 
                        (AI_Editor_HeaderData.labels.error_parsing || 'Error parsing response') + '</div>';
                }
            } else {
                popupContent.innerHTML = '<div class="ai-popup-error">' + 
                    (AI_Editor_HeaderData.labels.error_loading || 'Error searching posts') + '</div>';
            }
        };
        
        xhr.onerror = function() {
            popupContent.innerHTML = '<div class="ai-popup-error">' + 
                (AI_Editor_HeaderData.labels.network_error || 'Network error') + '</div>';
        };
        
        xhr.send();
    }

    function renderSearchResults(popup, searchData) {
        const popupContent = popup.querySelector('.ai-popup-content');
        
        // Clear content
        popupContent.innerHTML = '';
        
        if (!searchData.posts || searchData.posts.length === 0) {
            popupContent.innerHTML = '<div class="ai-popup-no-results">' + 
                (AI_Editor_HeaderData.labels.no_results || 'No results found') + '</div>';
            return;
        }
        
        // Show result count
        if (searchData.total_found > searchData.posts.length) {
            const resultInfo = document.createElement('div');
            resultInfo.className = 'ai-popup-search-info';
            
            // Use translatable format string
            const resultsFormat = AI_Editor_HeaderData.labels.search_results_format || 'Showing: %1$d of %2$d results';
            const resultsText = resultsFormat
                .replace('%1$d', searchData.posts.length)
                .replace('%2$d', searchData.total_found);
            resultInfo.textContent = resultsText;
            
            popupContent.appendChild(resultInfo);
        }
        
        // Create list element
        const list = document.createElement('ul');
        list.className = 'ai-popup-posts-list ai-popup-search-results';
        
        // Add each post to the list
        searchData.posts.forEach(post => {
            const listItem = document.createElement('li');
            listItem.className = 'ai-popup-post-item';
            
            if (post.is_current) {
                listItem.classList.add('ai-popup-current-post');
            }
            
            const link = document.createElement('a');
            link.href = post.edit_url;
            
            // Add title in a span
            const titleSpan = document.createElement('span');
            titleSpan.className = 'ai-popup-item-title'; // This should make title take remaining space
            titleSpan.textContent = post.title || '(Untitled)';
            link.appendChild(titleSpan);
            
            // Add status indicator
            if (post.status) {
                const statusSpan = document.createElement('span');
                statusSpan.className = 'ai-popup-post-status ai-status-' + post.status;
                statusSpan.textContent = STATUS_TYPES[post.status] || post.status.toUpperCase();
                statusSpan.setAttribute('data-status', post.status);
                statusSpan.setAttribute('title', AI_Editor_HeaderData.labels.filter_by_status || 'Click to filter by this status');
                statusSpan.addEventListener('click', function(e) {
                    e.preventDefault();
                    e.stopPropagation();
                    filterByStatus(popup, post.status);
                });
                
                // Add password indicator to status badge if needed
                if (post.password_protected) {
                    // statusSpan.style.position = 'relative'; // Intentionally removed
                    const passwordIndicator = document.createElement('span');
                    passwordIndicator.className = 'ai-popup-password-indicator';
                    passwordIndicator.setAttribute('title', AI_Editor_HeaderData.labels.password_protected || 'Password Protected');
                    statusSpan.appendChild(passwordIndicator);
                }
                
                listItem.appendChild(statusSpan); // Append statusSpan to listItem directly
            }
            
            listItem.appendChild(link); // Link should now contain title and icon (if applicable)
            list.appendChild(listItem);
            
            // Store data for filtering
            listItem.setAttribute('data-title', post.title || '');
            listItem.setAttribute('data-status', post.status || '');
        });
        
        popupContent.appendChild(list);
        
        // Store search data for filtering
        popup.setAttribute('data-search-results', JSON.stringify(searchData));
    }

    function createStatusFilters(popup, posts, statusCounts) {
        const statusFiltersContainer = popup.querySelector('.ai-popup-status-filters');
        if (!statusFiltersContainer) return;
        
        // Clear existing filters
        statusFiltersContainer.innerHTML = '';
        
        // Get current active status
        const currentActiveStatus = popup.getAttribute('data-active-status') || 'all';
        
        // Ensure we have a valid posts array
        posts = posts || [];
        
        // Use provided counts, or fallback to global, or defaults
        let dbCounts = statusCounts;
        
        if (!dbCounts || typeof dbCounts !== 'object') {
            // Try global counts
            if (globalStatusCounts) {
                dbCounts = globalStatusCounts;
            }
            // Try popup property
            else if (popup._statusCounts) {
                dbCounts = popup._statusCounts;
            }
            // Try data attribute
            else {
                const statusCountsData = popup.getAttribute('data-status-counts');
                if (statusCountsData) {
                    try {
                        dbCounts = JSON.parse(statusCountsData);
                    } catch (e) {
                        console.error('Error parsing status counts:', e);
                    }
                }
            }
        }
        
        // Final fallback to defaults
        if (!dbCounts || typeof dbCounts !== 'object') {
            dbCounts = {
                total: posts.length,
                publish: 0,
                draft: 0,
                pending: 0,
                private: 0,
                future: 0
            };
        }
        
        const totalCount = dbCounts.total || posts.length;
        
        // Create "All" filter button
        const allButton = document.createElement('button');
        allButton.className = 'ai-popup-status-filter ai-popup-status-filter-all';
        if (currentActiveStatus === 'all') {
            allButton.classList.add('active');
        }
        allButton.innerHTML = (AI_Editor_HeaderData.labels.status_all || 'ALL') + ' <span class="ai-filter-count">(' + totalCount + ')</span>';
        allButton.setAttribute('data-status', 'all');
        allButton.setAttribute('title', (AI_Editor_HeaderData.labels.status_all || 'All posts') + ': ' + totalCount);
        allButton.addEventListener('click', function() {
            filterByStatus(popup, 'all');
        });
        statusFiltersContainer.appendChild(allButton);
        
        // Create filter buttons for each status type in order
        const statusOrder = ['publish', 'pending', 'draft', 'future', 'private'];
        statusOrder.forEach(status => {
            if (STATUS_TYPES[status]) {
                const count = dbCounts[status] || 0;
                const button = document.createElement('button');
                button.className = 'ai-popup-status-filter ai-status-' + status;
                
                if (currentActiveStatus === status) {
                    button.classList.add('active');
                }
                
                // Get status label for tooltip
                const statusLabel = STATUS_TYPES[status] || status.toUpperCase();
                
                button.innerHTML = '<span class="ai-filter-count">' + count + '</span>';
                button.setAttribute('data-status', status);
                button.setAttribute('title', statusLabel + ': ' + count);
                
                // Only make clickable if there are posts
                if (count > 0) {
                    button.addEventListener('click', function() {
                        filterByStatus(popup, status);
                    });
                } else {
                    button.classList.add('ai-filter-disabled');
                    button.style.cursor = 'not-allowed';
                }
                
                statusFiltersContainer.appendChild(button);
            }
        });
        
        // Always show the filters
        statusFiltersContainer.style.display = 'flex';
    }

    function filterByStatus(popup, status) {
        // Update active filter button
        const filterButtons = popup.querySelectorAll('.ai-popup-status-filter');
        filterButtons.forEach(button => {
            if (button.getAttribute('data-status') === status) {
                button.classList.add('active');
            } else {
                button.classList.remove('active');
            }
        });
        
        // Set active status on the popup
        popup.setAttribute('data-active-status', status);
        
        // Filter posts
        const items = popup.querySelectorAll('.ai-popup-post-item');
        let foundItems = 0;
        
        items.forEach(item => {
            const postStatus = item.getAttribute('data-status');
            
            if (status === 'all' || postStatus === status) {
                item.style.display = 'block'; // Changed from 'flex' to 'block' for li
                foundItems++;
            } else {
                item.style.display = 'none';
            }
        });
        
        // Handle no results message
        const popupContent = popup.querySelector('.ai-popup-content');
        let noResults = popupContent.querySelector('.ai-popup-no-filter-results');
        
        if (foundItems === 0) {
            if (!noResults) {
                noResults = document.createElement('div');
                noResults.className = 'ai-popup-no-filter-results ai-popup-no-results';
                noResults.textContent = AI_Editor_HeaderData.labels.no_results || 'No results found';
            }
            // Insert at the beginning of content or after the list
            const list = popupContent.querySelector('.ai-popup-posts-list');
            if (list) {
                if (list.nextSibling) {
                    popupContent.insertBefore(noResults, list.nextSibling);
                } else {
                    popupContent.appendChild(noResults);
                }
            } else {
                popupContent.appendChild(noResults);
            }
            noResults.style.display = 'block';
        } else if (noResults) {
            noResults.style.display = 'none';
        }
    }

    const targetNode = document.body;
    const observerConfig = { childList: true, subtree: true };
    const observer = new MutationObserver(() => {
        const toolbar = document.querySelector('.editor-header__settings');
        if (toolbar && !toolbar.querySelector('.ai-editor-nav-buttons')) {
            addNavigationButtons();
        }
    });
    observer.observe(targetNode, observerConfig);
    setTimeout(addNavigationButtons, 100);
});