<?php
/**
 * Nonce Manager Class
 *
 * Centralized nonce management for the plugin
 *
 * @package AI_Editor_Post_Switcher
 * @since 1.0.8
 */

// Exit if accessed directly
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Class AI_Editor_Nonce_Manager
 *
 * Handles all nonce operations for the plugin
 */
class AI_Editor_Nonce_Manager {
	/**
	 * Nonce action constants
	 */
	const NAVIGATION_ACTION = 'ai_editor_navigation';
	const POST_TYPES_ACTION = 'ai_editor_post_types';
	const SETTINGS_ACTION = 'ai_editor_settings';
	const TAB_SWITCH_ACTION = 'ai_editor_tab_switch';
	const SORT_EDIT_ACTION = 'ai_editor_sort_edit';
	const AJAX_ACTION = 'ai_editor_ajax';
	const UNINSTALL_ACTION = 'ai_editor_uninstall_settings';

	/**
	 * Nonce field name constants
	 */
	const NAVIGATION_NONCE = 'ai_editor_navigation_nonce';
	const POST_TYPES_NONCE = 'ai_editor_post_types_nonce';
	const SETTINGS_NONCE = 'ai_editor_settings_nonce';
	const TAB_SWITCH_NONCE = 'ai_editor_tab_nonce';
	const SORT_EDIT_NONCE = 'ai_editor_sort_nonce';
	const AJAX_NONCE = 'ai_editor_ajax_nonce';

	/**
	 * Create a nonce
	 *
	 * @param string $action The nonce action.
	 * @return string The nonce.
	 */
	public static function create_nonce( $action ) {
		return wp_create_nonce( self::get_action_name( $action ) );
	}

	/**
	 * Verify a nonce
	 *
	 * @param string $nonce The nonce to verify.
	 * @param string $action The nonce action.
	 * @return bool|int False if invalid, 1 if valid and generated 0-12 hours ago, 2 if valid and generated 12-24 hours ago.
	 */
	public static function verify_nonce( $nonce, $action ) {
		return wp_verify_nonce( $nonce, self::get_action_name( $action ) );
	}

	/**
	 * Create a nonce field
	 *
	 * @param string $action The nonce action.
	 * @param string $name The nonce field name.
	 * @param bool   $referer Whether to create referer field.
	 * @param bool   $echo Whether to echo or return.
	 * @return string The nonce field HTML.
	 */
	public static function create_nonce_field( $action, $name, $referer = true, $echo = true ) {
		return wp_nonce_field( self::get_action_name( $action ), $name, $referer, $echo );
	}

	/**
	 * Get the full action name
	 *
	 * @param string $action The action constant.
	 * @return string The full action name.
	 */
	private static function get_action_name( $action ) {
		return $action;
	}

	/**
	 * Verify request nonce
	 *
	 * @param string $nonce_name The nonce field name.
	 * @param string $action The nonce action.
	 * @param string $request_method The request method (GET or POST).
	 * @return bool True if nonce is valid, false otherwise.
	 */
	public static function verify_request( $nonce_name, $action, $request_method = 'POST' ) {
		// phpcs:ignore WordPress.Security.NonceVerification.Missing, WordPress.Security.NonceVerification.Recommended -- This method IS the nonce verification
		$request = 'POST' === $request_method ? $_POST : $_GET;
		
		if ( ! isset( $request[ $nonce_name ] ) ) {
			return false;
		}

		$nonce = sanitize_text_field( wp_unslash( $request[ $nonce_name ] ) );
		return self::verify_nonce( $nonce, $action );
	}

	/**
	 * Add nonce to URL
	 *
	 * @param string $url The URL to add nonce to.
	 * @param string $action The nonce action.
	 * @param string $name The nonce parameter name.
	 * @return string The URL with nonce added.
	 */
	public static function add_nonce_to_url( $url, $action, $name ) {
		return wp_nonce_url( $url, self::get_action_name( $action ), $name );
	}

	/**
	 * Check admin referer with our action
	 *
	 * @param string $action The nonce action.
	 * @param string $query_arg The nonce query argument.
	 * @return bool|int False if invalid, 1 if valid and generated 0-12 hours ago, 2 if valid and generated 12-24 hours ago.
	 */
	public static function check_admin_referer( $action, $query_arg = '_wpnonce' ) {
		return check_admin_referer( self::get_action_name( $action ), $query_arg );
	}

	/**
	 * Check AJAX referer with our action
	 *
	 * @param string $action The nonce action.
	 * @param string $query_arg The nonce query argument.
	 * @param bool   $die Whether to die if invalid.
	 * @return bool|int False if invalid, 1 if valid and generated 0-12 hours ago, 2 if valid and generated 12-24 hours ago.
	 */
	public static function check_ajax_referer( $action, $query_arg = false, $die = true ) {
		return check_ajax_referer( self::get_action_name( $action ), $query_arg, $die );
	}
}