<?php
/**
 * Admin Page Main Class
 *
 * @package AI_Editor_Post_Switcher
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Admin Page Main Class
 */
class AI_Editor_Admin_Page {
	/**
	 * Settings tab instance
	 *
	 * @var AI_Editor_Admin_Settings_Tab
	 */
	private $settings_tab;

	/**
	 * Changelog tab instance
	 *
	 * @var AI_Editor_Admin_Changelog_Tab
	 */
	private $changelog_tab;

	/**
	 * Constructor
	 */
	public function __construct() {
		require_once AI_EDITOR_PATH . 'includes/admin/class-admin-helpers.php';
		require_once AI_EDITOR_PATH . 'includes/admin/class-admin-settings-tab.php';
		require_once AI_EDITOR_PATH . 'includes/admin/class-admin-changelog-tab.php';

		$this->settings_tab = new AI_Editor_Admin_Settings_Tab();
		$this->changelog_tab = new AI_Editor_Admin_Changelog_Tab();

		add_action( 'admin_menu', array( $this, 'add_admin_menu' ) );
		// Remove the WordPress options API registration to prevent default success messages
		// add_action( 'admin_init', array( $this->settings_tab, 'register_settings' ) );
		add_action( 'admin_init', array( $this->settings_tab, 'process_post_types_form' ) );
		add_action( 'admin_init', array( $this->settings_tab, 'process_navigation_options_form' ) );
		add_action( 'admin_init', array( $this->settings_tab, 'process_uninstall_settings_form' ) );
		add_action( 'admin_enqueue_scripts', array( $this, 'enqueue_admin_assets' ) );
	}

	/**
	 * Add admin menu
	 */
	public function add_admin_menu() {
		add_options_page(
			__( 'AI Editor Post Switcher', 'ai-editor-post-switcher' ),
			__( 'AI Editor Post Switcher', 'ai-editor-post-switcher' ),
			'manage_options',
			'ai-editor-post-switcher',
			array( $this, 'render_settings_page' )
		);
	}

	/**
	 * Enqueue admin assets
	 *
	 * @param string $hook Current admin page hook.
	 */
	public function enqueue_admin_assets( $hook ) {
		if ( 'settings_page_ai-editor-post-switcher' === $hook ) {
			// Enqueue admin styles
			wp_enqueue_style(
				'ai-editor-admin-page',
				AI_EDITOR_URL . 'css/admin-page.css',
				array(),
				AI_EDITOR_VERSION
			);

			// Enqueue changelog script (contains saved badge logic now)
			wp_enqueue_script(
				'ai-editor-changelog',
				AI_EDITOR_URL . 'js/admin-changelog.js',
				array( 'jquery' ),
				AI_EDITOR_VERSION,
				true
			);
		}
	}

	/**
	 * Render settings page
	 */
	public function render_settings_page() {
		if ( ! current_user_can( 'manage_options' ) ) {
			return;
		}

		$allowed_tabs = array( 'settings', 'changelog' );
		$default_tab = 'settings';
		$active_tab = $default_tab;

		// Handle tab switching with nonce verification
		// phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Initial check for tab parameter
		if ( isset( $_GET['tab'] ) && in_array( $_GET['tab'], $allowed_tabs, true ) ) {
			// Allow tab switching without nonce for direct links, but verify if nonce is present
			// phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Checking if nonce exists before verification
			if ( isset( $_GET[ AI_Editor_Nonce_Manager::TAB_SWITCH_NONCE ] ) ) {
				// phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Nonce is verified in the next line
				$nonce = sanitize_text_field( wp_unslash( $_GET[ AI_Editor_Nonce_Manager::TAB_SWITCH_NONCE ] ) );
				if ( AI_Editor_Nonce_Manager::verify_nonce( $nonce, AI_Editor_Nonce_Manager::TAB_SWITCH_ACTION ) ) {
					// phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Nonce already verified
					$active_tab = sanitize_key( $_GET['tab'] );
				}
			} else {
				// Allow direct tab access without nonce for bookmarking
				// phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Intentionally allowing direct access for bookmarking
				$active_tab = sanitize_key( $_GET['tab'] );
			}
		}

		// Create nonce for tab switching
		$tab_nonce = AI_Editor_Nonce_Manager::create_nonce( AI_Editor_Nonce_Manager::TAB_SWITCH_ACTION );
		?>
		<div class="wrap">
        <h1><?php esc_html_e( 'AI Editor Post Switcher', 'ai-editor-post-switcher' ); ?> <span class="ai-version-number">Version: <?php echo esc_html( AI_EDITOR_VERSION ); ?></span></h1>
        <div style="display: flex; align-items: center; gap: 10px;">
				<h2 class="nav-tab-wrapper">
					<a href="<?php echo esc_url( add_query_arg( array( 'tab' => 'settings', AI_Editor_Nonce_Manager::TAB_SWITCH_NONCE => $tab_nonce ), menu_page_url( 'ai-editor-post-switcher', false ) ) ); ?>" 
					   class="nav-tab <?php echo 'settings' === $active_tab ? 'nav-tab-active' : ''; ?>">
						<?php esc_html_e( 'Settings', 'ai-editor-post-switcher' ); ?>
					</a>
					<a href="<?php echo esc_url( add_query_arg( array( 'tab' => 'changelog', AI_Editor_Nonce_Manager::TAB_SWITCH_NONCE => $tab_nonce ), menu_page_url( 'ai-editor-post-switcher', false ) ) ); ?>" 
					   class="nav-tab <?php echo 'changelog' === $active_tab ? 'nav-tab-active' : ''; ?>">
						<?php esc_html_e( 'Changelog', 'ai-editor-post-switcher' ); ?>
					</a>
				</h2>
				<?php 
				// phpcs:ignore WordPress.Security.NonceVerification.Recommended -- This is just a success indicator, not form processing
				if ( isset( $_GET['form-updated'] ) && 'true' === $_GET['form-updated'] ) : 
				?>
					<span id="ai-editor-saved-badge" class="ai-editor-saved-badge"><?php esc_html_e( '✔ Saved', 'ai-editor-post-switcher' ); ?></span>
				<?php endif; ?>
			</div>
			<div>
				<?php
				if ( 'settings' === $active_tab ) {
					$this->settings_tab->render();
				} elseif ( 'changelog' === $active_tab ) {
					$this->changelog_tab->render();
				}
				?>
			</div>
			<div class="ai-legal-footer-sticky">
				<p class="ai-legal-text">
					<?php 
					printf(
						// translators: 1: HTML link to 4wp.ai, 2: Today's date formatted.
						esc_html__( 'Propulsed by %1$s - %2$s', 'ai-editor-post-switcher' ),
						'<a href="https://4wp.ai" target="_blank">4wp.ai</a>',
						esc_html( date_i18n( get_option( 'date_format' ) ) )
					); 
					?>
				</p>
			</div>
		</div>
		<?php
	}
}