<?php
/**
 * Admin Settings Tab
 *
 * @package AI_Editor_Post_Switcher
 */

// Exit if accessed directly
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Class AI_Editor_Admin_Settings_Tab
 * Handles settings tab functionality
 */
class AI_Editor_Admin_Settings_Tab {
	/**
	 * Register settings
	 */
	public function register_settings() {
		register_setting(
			'ai_editor_post_switcher_settings',
			'ai_editor_post_switcher_settings',
			array( $this, 'sanitize_settings' )
		);
	}
	
	/**
	 * Sanitize settings
	 *
	 * @param array $input Input settings.
	 * @return array Sanitized settings.
	 */
	public function sanitize_settings( $input ) {
		$output = array();
		$output['delete_on_uninstall'] = ! empty( $input['delete_on_uninstall'] ) ? 1 : 0;
		$output['enable_first_last'] = ! empty( $input['enable_first_last'] ) ? 1 : 0;
		
		return $output;
	}
	
	/**
	 * Process post types form
	 */
	public function process_post_types_form() {
		// Check page parameter first
		// phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Checking page context before nonce verification
		if ( ! isset( $_GET['page'] ) || 'ai-editor-post-switcher' !== $_GET['page'] ) {
			return;
		}

		// Check if form was submitted
		// phpcs:ignore WordPress.Security.NonceVerification.Missing -- Nonce is verified in the next condition
		if ( ! isset( $_POST['ai_editor_post_types'] ) ) {
			return;
		}

		// Verify user capability
		if ( ! current_user_can( 'manage_options' ) ) {
			return;
		}

		// Verify nonce
		if ( ! AI_Editor_Nonce_Manager::verify_request( AI_Editor_Nonce_Manager::POST_TYPES_NONCE, AI_Editor_Nonce_Manager::POST_TYPES_ACTION, 'POST' ) ) {
			return;
		}

		// Process the form - nonce has been verified
		// phpcs:ignore WordPress.Security.NonceVerification.Missing -- Nonce already verified above
		$selected = array_map( 'sanitize_text_field', wp_unslash( $_POST['ai_editor_post_types'] ) );
		update_option( 'ai_editor_post_types', $selected );
		
		// Delete transient cache
		delete_transient( 'ai_editor_allowed_post_types' );
		
		// Add success parameter to the redirect URL
		$redirect_url = add_query_arg(
			array(
				'form-updated' => 'true',
			),
			menu_page_url( 'ai-editor-post-switcher', false )
		);
		wp_safe_redirect( $redirect_url );
		exit;
	}
	
	/**
	 * Process navigation options form
	 */
	public function process_navigation_options_form() {
		// Check page parameter first
		// phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Checking page context before nonce verification
		if ( ! isset( $_GET['page'] ) || 'ai-editor-post-switcher' !== $_GET['page'] ) {
			return;
		}

		// Check if form was submitted
		// phpcs:ignore WordPress.Security.NonceVerification.Missing -- Nonce is verified in the next condition
		if ( ! isset( $_POST['ai_editor_navigation_options_submit'] ) ) {
			return;
		}

		// Verify user capability
		if ( ! current_user_can( 'manage_options' ) ) {
			return;
		}

		// Verify nonce
		if ( ! AI_Editor_Nonce_Manager::verify_request( AI_Editor_Nonce_Manager::SETTINGS_NONCE, AI_Editor_Nonce_Manager::SETTINGS_ACTION, 'POST' ) ) {
			return;
		}

		// Get current settings
		$settings = AI_Editor_Admin_Helpers::get_plugin_settings();
		
		// Update only the navigation option
		// phpcs:ignore WordPress.Security.NonceVerification.Missing -- Nonce already verified above
		$settings['enable_first_last'] = ! empty( $_POST['ai_editor_post_switcher_settings']['enable_first_last'] ) ? 1 : 0;
		
		// Save settings
		update_option( 'ai_editor_post_switcher_settings', $settings );
		
		// Add success parameter to the redirect URL
		$redirect_url = add_query_arg(
			array(
				'form-updated' => 'true',
			),
			menu_page_url( 'ai-editor-post-switcher', false )
		);
		wp_safe_redirect( $redirect_url );
		exit;
	}
	
	/**
	 * Process uninstall settings form
	 */
	public function process_uninstall_settings_form() {
		// Check page parameter first
		// phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Checking page context before nonce verification
		if ( ! isset( $_GET['page'] ) || 'ai-editor-post-switcher' !== $_GET['page'] ) {
			return;
		}

		// Check if form was submitted
		// phpcs:ignore WordPress.Security.NonceVerification.Missing -- Nonce is verified in the next condition
		if ( ! isset( $_POST['ai_editor_uninstall_settings_submit'] ) ) {
			return;
		}

		// Verify user capability
		if ( ! current_user_can( 'manage_options' ) ) {
			return;
		}

		// Verify nonce
		if ( ! AI_Editor_Nonce_Manager::verify_request( 'ai_editor_uninstall_nonce', AI_Editor_Nonce_Manager::UNINSTALL_ACTION, 'POST' ) ) {
			return;
		}

		// Get current settings
		$settings = AI_Editor_Admin_Helpers::get_plugin_settings();
		
		// Update only the uninstall option
		// phpcs:ignore WordPress.Security.NonceVerification.Missing -- Nonce already verified above
		$settings['delete_on_uninstall'] = ! empty( $_POST['ai_editor_post_switcher_settings']['delete_on_uninstall'] ) ? 1 : 0;
		
		// Save settings
		update_option( 'ai_editor_post_switcher_settings', $settings );
		
		// Add success parameter to the redirect URL
		$redirect_url = add_query_arg(
			array(
				'form-updated' => 'true',
			),
			menu_page_url( 'ai-editor-post-switcher', false )
		);
		wp_safe_redirect( $redirect_url );
		exit;
	}
	
	/**
	 * Render settings tab content
	 */
	public function render() {
		// Get selected post types
		$selected_post_types = get_option( 'ai_editor_post_types', array( 'post', 'page' ) );
		
		// Get all available post types
		$post_types = AI_Editor_Admin_Helpers::get_compatible_post_types();
		
		// Get plugin settings with defaults
		$settings = AI_Editor_Admin_Helpers::get_plugin_settings();
		$first_last_enabled = $settings['enable_first_last'];
		$delete_on_uninstall = $settings['delete_on_uninstall'];
		
		?>
		<div class="ai-settings-row">
			<div class="ai-frame ai-frame-main">
				<h2><?php 
					// translators: Title for post types selection section
					esc_html_e( 'Select Gutenberg-Compatible Post Types', 'ai-editor-post-switcher' );
				?></h2>
				<!-- translators: Description text for the post type selection section -->
				<p class="description"><?php esc_html_e( 'For selected post types, navigation buttons will appear in the Gutenberg editor header.', 'ai-editor-post-switcher' ); ?></p>
				<?php if ( empty( $post_types ) ) : ?>
					<p><?php 
						// translators: Message displayed when no compatible post types are found
						esc_html_e( 'No post types using the block editor found.', 'ai-editor-post-switcher' );
					?></p>
				<?php else : ?>
					<form method="post" action="">
						<?php AI_Editor_Nonce_Manager::create_nonce_field( AI_Editor_Nonce_Manager::POST_TYPES_ACTION, AI_Editor_Nonce_Manager::POST_TYPES_NONCE ); ?>
						<?php foreach ( $post_types as $post_type ) :
							$input_id = 'ai_post_type_' . esc_attr( $post_type->name );
						?>
							<div style="margin: 8px 0;">
								<input type="checkbox" id="<?php echo esc_attr( $input_id ); ?>" name="ai_editor_post_types[]" value="<?php echo esc_attr( $post_type->name ); ?>"
									<?php checked( in_array( $post_type->name, $selected_post_types, true ) ); ?> />
								<label for="<?php echo esc_attr( $input_id ); ?>">
									<?php echo esc_html( $post_type->labels->singular_name ); ?><span class="ai-post-type-slug"><?php echo esc_html( $post_type->name ); ?></span>
								</label>
							</div>
						<?php endforeach; ?>
						<p><button class="button button-primary" type="submit"><?php 
							// translators: Button label for saving post type selection
							esc_html_e( 'Save Selection', 'ai-editor-post-switcher' );
						?></button></p>
					</form>
				<?php endif; ?>
			</div>
			
			<div class="ai-frame ai-frame-side">
				<h2><?php 
					// translators: Section title for navigation options
					esc_html_e( 'Navigation Options', 'ai-editor-post-switcher' );
				?></h2>
				<!-- translators: Description text for the navigation options section -->
				<p class="description"><?php esc_html_e( 'Additional navigation settings.', 'ai-editor-post-switcher' ); ?></p>
				
				<form method="post" action="">
					<?php AI_Editor_Nonce_Manager::create_nonce_field( AI_Editor_Nonce_Manager::SETTINGS_ACTION, AI_Editor_Nonce_Manager::SETTINGS_NONCE ); ?>
					<div style="margin: 8px 0;">
						<input type="checkbox" name="ai_editor_post_switcher_settings[enable_first_last]" id="enable_first_last"
							value="1" <?php checked( $first_last_enabled ); ?> />
						<?php
						// Get default post type singular name
						$default_post_type = get_post_type_object( 'post' );
						$default_post_type_singular = $default_post_type ? strtolower( $default_post_type->labels->singular_name ) : __( 'post', 'ai-editor-post-switcher' );
						?>
						<label for="enable_first_last"><?php 
							// translators: %s is the post type singular name
							printf( esc_html__( 'Add buttons to first / last %s', 'ai-editor-post-switcher' ), esc_html( $default_post_type_singular ) );
						?></label>
					</div>
					
					<input type="hidden" name="ai_editor_navigation_options_submit" value="1" />
					<p><button class="button button-primary" type="submit"><?php 
						// translators: Button label for saving navigation options
						esc_html_e( 'Save Options', 'ai-editor-post-switcher' );
					?></button></p>
				</form>
			</div>
		</div>

		<div class="ai-frame">
			<h2><?php 
				// translators: Section title for uninstall options
				esc_html_e( 'Uninstall', 'ai-editor-post-switcher' );
			?></h2>
			<!-- translators: Description text for the uninstall options section -->
			<p class="description"><?php esc_html_e( 'If you check this option, all plugin data will be deleted when the plugin is uninstalled.', 'ai-editor-post-switcher' ); ?></p>

			<form method="post" action="">
				<?php AI_Editor_Nonce_Manager::create_nonce_field( AI_Editor_Nonce_Manager::UNINSTALL_ACTION, 'ai_editor_uninstall_nonce' ); ?>
				<input type="checkbox" name="ai_editor_post_switcher_settings[delete_on_uninstall]" id="delete_on_uninstall"
					value="1" <?php checked( $delete_on_uninstall ); ?> />
				<label for="delete_on_uninstall"><?php 
					// translators: Label for option to delete data on uninstall
					esc_html_e( 'Delete data on uninstall', 'ai-editor-post-switcher' );
				?></label>

				<input type="hidden" name="ai_editor_uninstall_settings_submit" value="1" />
				<p><button class="button button-primary" type="submit"><?php 
					// translators: Button label for saving settings
					esc_html_e( 'Save Settings', 'ai-editor-post-switcher' );
				?></button></p>
			</form>
		</div>
		<?php
	}
}