<?php
/**
 * Admin Changelog Tab
 *
 * @package AI_Editor_Post_Switcher
 */

// Exit if accessed directly
if (!defined('ABSPATH')) {
    exit;
}

/**
 * Class AI_Editor_Admin_Changelog_Tab
 * Handles changelog tab functionality
 */
class AI_Editor_Admin_Changelog_Tab {
    /**
     * Render changelog tab content
     */
    public function render() {
        // Get changelog data - récupérer toutes les entrées sans pagination
        $changelog_data = $this->get_changelog(9999, 1);
        ?>
        <div class="ai-frame">
            <h2><?php 
                // translators: Tab label for changelog section
                esc_html_e('Changelog', 'ai-editor-post-switcher');
            ?></h2>

            <div class="ai-changelog-wrapper">
                <!-- En-tête fixe du tableau -->
                <table class="ai-changelog-table ai-changelog-header">
                    <!-- translators: This is the caption for the changelog table -->
                    <p class="description"><?php esc_html_e('Plugin update history', 'ai-editor-post-switcher'); ?></p>
                    <thead>
                        <tr>
                            <th><?php 
                                // translators: Column header for version number in changelog
                                esc_html_e('Version', 'ai-editor-post-switcher');
                            ?></th>
                            <th><?php 
                                // translators: Column header for date in changelog
                                esc_html_e('Date', 'ai-editor-post-switcher');
                            ?></th>
                            <th><?php 
                                // translators: Column header for changes description in changelog
                                esc_html_e('Changes', 'ai-editor-post-switcher');
                            ?></th>
                        </tr>
                    </thead>
                </table>
                
                <!-- Corps du tableau avec ascenseur -->
                <div class="ai-changelog-container">
                    <table class="ai-changelog-table ai-changelog-body">
                        <tbody>
                        <?php
                        if (!empty($changelog_data['entries'])) {
                            foreach ($changelog_data['entries'] as $entry) {
                                echo '<tr>';
                                echo '<td>' . esc_html($entry[0]) . '</td>'; // Version
                                echo '<td>' . esc_html($entry[1]) . '</td>'; // Date
                                echo '<td>' . wp_kses_post($entry[2]) . '</td>'; // Changes (HTML allowed)
                                echo '</tr>';
                            }
                        } else {
                            echo '<tr><td colspan="3">' . 
                                // translators: Error message when changelog file cannot be found
                                esc_html__('Changelog file not found.', 'ai-editor-post-switcher') . 
                                '</td></tr>';
                        }
                        ?>
                        </tbody>
                    </table>
                </div>
            </div>
        </div>
        <?php
    }
    
    /**
     * Get changelog entries from readme.txt
     *
     * @param int $limit Entries per page
     * @param int $page Current page
     * @return array
     */
    private function get_changelog($limit = 10, $page = 1) {
        $readme_path = AI_EDITOR_PATH . 'readme.txt';
        $entries = array();
        $offset = ($page - 1) * $limit;
        $total = 0;
        
        if (file_exists($readme_path)) {
            $readme_content = file_get_contents($readme_path);
            
            // Find the changelog section
            if (preg_match('/== Changelog ==\s*(.*?)(?=\s*==|$)/s', $readme_content, $matches)) {
                $changelog_section = trim($matches[1]);
                
                // Extract all version blocks using regex - this will work for the first entry too
                preg_match_all('/= ([0-9.]+) =\s*([0-9]{4}-[0-9]{2}-[0-9]{2})(.+?)(?=\n= [0-9.]+|$)/s', $changelog_section, $version_blocks, PREG_SET_ORDER);
                
                $total = count($version_blocks);
                $all_entries = array();
                
                foreach ($version_blocks as $block) {
                    $version = trim($block[1]);
                    $date = trim($block[2]);
                    $content = trim($block[3]);
                    
                    // Format the changes as HTML list
                    $changes_html = '<ul>';
                    
                    // Split by lines and filter out empty ones
                    $lines = array_filter(array_map('trim', explode("\n", $content)), function($line) {
                        return !empty($line);
                    });
                    
                    foreach ($lines as $line) {
                        if (!empty($line)) {
                            $changes_html .= '<li>' . esc_html($line) . '</li>';
                        }
                    }
                    $changes_html .= '</ul>';
                    
                    $all_entries[] = array(
                        $version,
                        $date,
                        $changes_html
                    );
                }
                
                // Get slice of entries for current page
                $entries = array_slice($all_entries, $offset, $limit);
            }
        }
        
        return array(
            'entries' => $entries,
            'total' => $total,
            'pages' => ceil($total / $limit)
        );
    }
}