<?php
/**
 * Core plugin class.
 *
 * @package AIDF
 * @since   1.0.0
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Main plugin singleton.
 */
class AIDF_Plugin {

	/**
	 * Singleton instance.
	 *
	 * @var AIDF_Plugin|null
	 */
	private static $instance = null;

	/**
	 * Get singleton instance.
	 *
	 * @return AIDF_Plugin
	 */
	public static function get_instance() {
		if ( null === self::$instance ) {
			self::$instance = new self();
		}
		return self::$instance;
	}

	/**
	 * Constructor — registers hooks.
	 */
	private function __construct() {
		AIDF_Server::init();

		if ( is_admin() ) {
			require_once AIDF_PLUGIN_DIR . 'admin/class-admin.php';
			require_once AIDF_PLUGIN_DIR . 'admin/class-settings.php';
			AIDF_Admin::init();
			AIDF_Settings::init();
		}
	}

	/**
	 * Get default settings.
	 *
	 * @return array<string, mixed>
	 */
	public static function get_defaults() {
		static $defaults = null;

		if ( null !== $defaults ) {
			return $defaults;
		}

		$defaults = array(
			// Identity.
			'business_name'      => get_bloginfo( 'name' ),
			'legal_name'         => '',
			'tagline'            => get_bloginfo( 'description' ),
			'contact_email'      => get_option( 'admin_email' ),
			'contact_phone'      => '',
			'contact_url'        => self::find_page_url( array( 'contact', 'contact-us', 'get-in-touch' ) ),
			'address'            => '',
			'postcode'           => '',
			'location_city'      => '',
			'location_region'    => '',
			'location_country'   => '',
			'services'           => array(),
			'not_services'       => '',
			'industry'           => '',
			'founded_year'       => '',
			'employee_count'     => '',
			'business_type'      => '',
			'company_number'     => '',
			'company_jurisdiction' => '',
			'vat_registered'     => false,
			'vat_number'         => '',
			'customer_type'      => '',
			'offering_type'      => '',
			'is_ecommerce'       => false,
			'service_areas'      => '',
			'not_service_areas'  => '',
			'operating_hours_preset' => 'mon_fri_9_5',
			'operating_hours'    => self::get_default_operating_hours(),
			'social_links'       => '',
			'logo_url'           => self::get_site_logo_url(),
			'products_url'       => self::find_page_url( array( 'products', 'pricing', 'shop', 'store', 'services' ) ),
			'privacy_policy_url' => function_exists( 'get_privacy_policy_url' ) ? get_privacy_policy_url() : '',
			'terms_url'          => self::find_page_url( array( 'terms', 'terms-of-service', 'terms-and-conditions', 'terms-of-use', 'tos' ) ),

			// Permissions.
			'ai_usage'           => 'allow_with_attribution',
			'ai_training'        => 'restrict',
			'crawler_policy'     => 'allow_known',
			'content_licence'    => '',
			'citation_format'    => get_bloginfo( 'name' ) . ' (' . home_url( '/' ) . ')',

			// Content.
			'brand_alternates'    => '',
			'brand_never'         => '',
			'brand_pronunciation' => '',
			'brand_misspellings'  => '',
			'brand_voice'         => '',
			'brand_taglines'      => '',
			'brand_boilerplate'   => '',
			'key_people'          => array(),
			'faqs'                => array(),

			// Technical.
			'developer_notes'    => '',
			'api_info'           => '',
			'tech_stack'         => '',

			// Generation control.
			'active_files'       => array( 'llms-txt', 'ai-txt' ),
			'spec_attribution'   => false,

			// Directory verification.
			'verify_code'        => '',
		);

		return $defaults;
	}

	/**
	 * Get merged settings (defaults + saved).
	 *
	 * @return array<string, mixed>
	 */
	public static function get_settings() {
		$defaults = self::get_defaults();
		$saved    = get_option( 'aidf_settings', array() );

		if ( ! is_array( $saved ) ) {
			$saved = array();
		}

		return wp_parse_args( $saved, $defaults );
	}

	/**
	 * Get available industry options.
	 *
	 * @return array<string, string>
	 */
	public static function get_industries() {
		return array(
			'Accounting & Finance'       => __( 'Accounting & Finance', 'ai-discovery-files' ),
			'Agriculture'                => __( 'Agriculture', 'ai-discovery-files' ),
			'Architecture & Planning'    => __( 'Architecture & Planning', 'ai-discovery-files' ),
			'Arts & Entertainment'       => __( 'Arts & Entertainment', 'ai-discovery-files' ),
			'Automotive'                 => __( 'Automotive', 'ai-discovery-files' ),
			'Construction'               => __( 'Construction', 'ai-discovery-files' ),
			'Consulting'                 => __( 'Consulting', 'ai-discovery-files' ),
			'Digital Marketing'          => __( 'Digital Marketing', 'ai-discovery-files' ),
			'E-Commerce & Retail'        => __( 'E-Commerce & Retail', 'ai-discovery-files' ),
			'Education & Training'       => __( 'Education & Training', 'ai-discovery-files' ),
			'Energy & Utilities'         => __( 'Energy & Utilities', 'ai-discovery-files' ),
			'Engineering'                => __( 'Engineering', 'ai-discovery-files' ),
			'Fashion & Apparel'          => __( 'Fashion & Apparel', 'ai-discovery-files' ),
			'Food & Beverage'            => __( 'Food & Beverage', 'ai-discovery-files' ),
			'Government & Public Sector' => __( 'Government & Public Sector', 'ai-discovery-files' ),
			'Healthcare & Medical'       => __( 'Healthcare & Medical', 'ai-discovery-files' ),
			'Hospitality & Tourism'      => __( 'Hospitality & Tourism', 'ai-discovery-files' ),
			'Insurance'                  => __( 'Insurance', 'ai-discovery-files' ),
			'Legal Services'             => __( 'Legal Services', 'ai-discovery-files' ),
			'Manufacturing'              => __( 'Manufacturing', 'ai-discovery-files' ),
			'Media & Publishing'         => __( 'Media & Publishing', 'ai-discovery-files' ),
			'Non-Profit & Charity'       => __( 'Non-Profit & Charity', 'ai-discovery-files' ),
			'Property & Real Estate'     => __( 'Property & Real Estate', 'ai-discovery-files' ),
			'Recruitment & HR'           => __( 'Recruitment & HR', 'ai-discovery-files' ),
			'SaaS & Software'            => __( 'SaaS & Software', 'ai-discovery-files' ),
			'Security'                   => __( 'Security', 'ai-discovery-files' ),
			'Sports & Fitness'           => __( 'Sports & Fitness', 'ai-discovery-files' ),
			'Technology & IT'            => __( 'Technology & IT', 'ai-discovery-files' ),
			'Telecommunications'         => __( 'Telecommunications', 'ai-discovery-files' ),
			'Transport & Logistics'      => __( 'Transport & Logistics', 'ai-discovery-files' ),
			'Web Design & Development'   => __( 'Web Design & Development', 'ai-discovery-files' ),
			'Other'                      => __( 'Other', 'ai-discovery-files' ),
		);
	}

	/**
	 * List of all supported file types with metadata.
	 *
	 * @return array<string, array<string, string>>
	 */
	public static function get_file_types() {
		return array(
			'llms-txt'         => array(
				'code'         => 'ADF-001',
				'filename'     => 'llms.txt',
				'label'        => __( 'llms.txt', 'ai-discovery-files' ),
				'description'  => __( 'AI-readable business identity', 'ai-discovery-files' ),
				'content_type' => 'text/plain; charset=utf-8',
				'tier'         => 'essential',
			),
			'llm-txt'          => array(
				'code'         => 'ADF-002',
				'filename'     => 'llm.txt',
				'label'        => __( 'llm.txt', 'ai-discovery-files' ),
				'description'  => __( 'Compatibility alias for llms.txt', 'ai-discovery-files' ),
				'content_type' => 'text/plain; charset=utf-8',
				'tier'         => 'complete',
			),
			'llms-html'        => array(
				'code'         => 'ADF-003',
				'filename'     => 'llms.html',
				'label'        => __( 'llms.html', 'ai-discovery-files' ),
				'description'  => __( 'Human-readable reference version', 'ai-discovery-files' ),
				'content_type' => 'text/html; charset=utf-8',
				'tier'         => 'complete',
			),
			'ai-txt'           => array(
				'code'         => 'ADF-004',
				'filename'     => 'ai.txt',
				'label'        => __( 'ai.txt', 'ai-discovery-files' ),
				'description'  => __( 'AI usage permissions', 'ai-discovery-files' ),
				'content_type' => 'text/plain; charset=utf-8',
				'tier'         => 'essential',
			),
			'ai-json'          => array(
				'code'         => 'ADF-005',
				'filename'     => 'ai.json',
				'label'        => __( 'ai.json', 'ai-discovery-files' ),
				'description'  => __( 'Machine-parseable permissions', 'ai-discovery-files' ),
				'content_type' => 'application/json; charset=utf-8',
				'tier'         => 'recommended',
			),
			'identity-json'    => array(
				'code'         => 'ADF-006',
				'filename'     => 'identity.json',
				'label'        => __( 'identity.json', 'ai-discovery-files' ),
				'description'  => __( 'Structured business identity', 'ai-discovery-files' ),
				'content_type' => 'application/json; charset=utf-8',
				'tier'         => 'recommended',
			),
			'brand-txt'        => array(
				'code'         => 'ADF-007',
				'filename'     => 'brand.txt',
				'label'        => __( 'brand.txt', 'ai-discovery-files' ),
				'description'  => __( 'Brand naming and terminology', 'ai-discovery-files' ),
				'content_type' => 'text/plain; charset=utf-8',
				'tier'         => 'recommended',
			),
			'faq-ai-txt'       => array(
				'code'         => 'ADF-008',
				'filename'     => 'faq-ai.txt',
				'label'        => __( 'faq-ai.txt', 'ai-discovery-files' ),
				'description'  => __( 'FAQs formatted for AI', 'ai-discovery-files' ),
				'content_type' => 'text/plain; charset=utf-8',
				'tier'         => 'recommended',
			),
			'developer-ai-txt' => array(
				'code'         => 'ADF-009',
				'filename'     => 'developer-ai.txt',
				'label'        => __( 'developer-ai.txt', 'ai-discovery-files' ),
				'description'  => __( 'Technical context for developers', 'ai-discovery-files' ),
				'content_type' => 'text/plain; charset=utf-8',
				'tier'         => 'complete',
			),
			'robots-ai-txt'    => array(
				'code'         => 'ADF-010',
				'filename'     => 'robots-ai.txt',
				'label'        => __( 'robots-ai.txt', 'ai-discovery-files' ),
				'description'  => __( 'AI crawler-specific directives', 'ai-discovery-files' ),
				'content_type' => 'text/plain; charset=utf-8',
				'tier'         => 'complete',
			),
		);
	}

	/**
	 * Get the site custom logo URL if set.
	 *
	 * @return string Logo URL or empty string.
	 */
	public static function get_site_logo_url() {
		$custom_logo_id = get_theme_mod( 'custom_logo' );

		if ( ! $custom_logo_id ) {
			return '';
		}

		$url = wp_get_attachment_image_url( $custom_logo_id, 'full' );

		return $url ? $url : '';
	}

	/**
	 * Find a published page URL by trying common slugs.
	 *
	 * @param  array<string> $slugs List of post_name slugs to try, in priority order.
	 * @return string Page permalink or empty string.
	 */
	public static function find_page_url( $slugs ) {
		foreach ( $slugs as $slug ) {
			$page = get_page_by_path( $slug );

			if ( $page && 'publish' === $page->post_status ) {
				return get_permalink( $page );
			}
		}

		return '';
	}

	/**
	 * Build a sensible default for operating hours using the site's timezone.
	 *
	 * @return string e.g. "Mon-Fri 9am-5pm GMT" or "Mon-Fri 9am-5pm America/New_York".
	 */
	public static function get_default_operating_hours() {
		$tz_string = wp_timezone_string();

		return 'Mon-Fri 9am-5pm ' . $tz_string;
	}

	/**
	 * Get available operating hours presets.
	 *
	 * @return array<string, string>
	 */
	public static function get_operating_hours_presets() {
		$tz = wp_timezone_string();

		return array(
			'mon_fri_9_5'    => sprintf(
				/* translators: %s: timezone string */
				__( 'Mon-Fri 9am-5pm %s', 'ai-discovery-files' ),
				$tz
			),
			'mon_fri_8_6'    => sprintf(
				/* translators: %s: timezone string */
				__( 'Mon-Fri 8am-6pm %s', 'ai-discovery-files' ),
				$tz
			),
			'mon_sat_9_5'    => sprintf(
				/* translators: %s: timezone string */
				__( 'Mon-Sat 9am-5pm %s', 'ai-discovery-files' ),
				$tz
			),
			'mon_sat_8_6'    => sprintf(
				/* translators: %s: timezone string */
				__( 'Mon-Sat 8am-6pm %s', 'ai-discovery-files' ),
				$tz
			),
			'mon_sun_9_5'    => sprintf(
				/* translators: %s: timezone string */
				__( 'Mon-Sun 9am-5pm %s', 'ai-discovery-files' ),
				$tz
			),
			'24_7'           => __( '24/7', 'ai-discovery-files' ),
			'custom'         => __( 'Custom', 'ai-discovery-files' ),
		);
	}

	/**
	 * Get business types that require a company registration number.
	 *
	 * @return array<int, string>
	 */
	public static function get_registered_business_types() {
		return array(
			'Limited Company',
			'Limited Liability Partnership',
			'Public Limited Company',
			'Corporation',
			'LLC',
		);
	}

	/**
	 * Get tiers with metadata.
	 *
	 * @return array<string, array<string, string>>
	 */
	public static function get_tiers() {
		return array(
			'essential'   => array(
				'label'       => __( 'Essential', 'ai-discovery-files' ),
				'description' => __( 'Core AI discoverability — who you are and how AI may use your content.', 'ai-discovery-files' ),
			),
			'recommended' => array(
				'label'       => __( 'Recommended', 'ai-discovery-files' ),
				'description' => __( 'Machine-parseable identity, permissions, brand control, and FAQs.', 'ai-discovery-files' ),
			),
			'complete'    => array(
				'label'       => __( 'Complete', 'ai-discovery-files' ),
				'description' => __( 'Full coverage with human presentation, developer context, and crawler directives.', 'ai-discovery-files' ),
			),
		);
	}
}
