<?php
/**
 * File generator — renders AI Discovery Files from templates and data.
 *
 * @package AIDF
 * @since   1.0.0
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Generates AI Discovery File content using templates and collected data.
 */
class AIDF_Generator {

	/**
	 * Generate a specific file type.
	 *
	 * @param  string $file_slug The file type slug (e.g. 'llms-txt').
	 * @return string|false Generated content, or false on failure.
	 */
	public static function generate( $file_slug ) {
		$file_types = AIDF_Plugin::get_file_types();

		if ( ! isset( $file_types[ $file_slug ] ) ) {
			return false;
		}

		$data     = AIDF_Data_Collector::collect();
		$template = AIDF_PLUGIN_DIR . 'templates/' . $file_slug . '.php';

		if ( ! file_exists( $template ) ) {
			return false;
		}

		/**
		 * Filters the collected data before template rendering.
		 *
		 * @param array  $data      Collected site data.
		 * @param string $file_slug File type being generated.
		 */
		$data = apply_filters( 'aidf_template_data', $data, $file_slug );

		ob_start();
		include $template;
		$content = ob_get_clean();

		/**
		 * Filters the generated file content.
		 *
		 * @param string $content   Generated content.
		 * @param string $file_slug File type that was generated.
		 * @param array  $data      Data used for generation.
		 */
		return apply_filters( 'aidf_generated_content', $content, $file_slug, $data );
	}

	/**
	 * Generate all active files and return as an associative array.
	 *
	 * @return array<string, string> File slug => content.
	 */
	public static function generate_all() {
		$settings     = AIDF_Plugin::get_settings();
		$active_files = isset( $settings['active_files'] ) ? (array) $settings['active_files'] : array();
		$results      = array();

		foreach ( $active_files as $slug ) {
			$content = self::generate( $slug );

			if ( false !== $content ) {
				$results[ $slug ] = $content;
			}
		}

		return $results;
	}

	/**
	 * Get a formatted spec attribution block.
	 *
	 * @param  string $comment_char Comment character(s) for the file format.
	 * @param  string $file_slug    File type slug for spec-specific URL.
	 * @return string
	 */
	public static function get_attribution( $comment_char = '#', $file_slug = '' ) {
		$settings = AIDF_Plugin::get_settings();

		if ( empty( $settings['spec_attribution'] ) ) {
			return '';
		}

		$spec_url = self::get_spec_url( $file_slug );

		$lines = array(
			$comment_char . ' ' . self::get_spec_label( $file_slug ),
			$comment_char . ' ' . $spec_url,
			$comment_char . ' Generated by AI Discovery Files for WordPress v' . AIDF_VERSION,
		);

		return implode( "\n", $lines ) . "\n";
	}

	/**
	 * Get a footer attribution block for Markdown files.
	 *
	 * Markdown uses # for headings, not comments, so attribution is placed
	 * at the bottom of the file separated by a horizontal rule.
	 *
	 * @param  string $file_slug File type slug for spec-specific URL.
	 * @return string
	 */
	public static function get_footer_attribution( $file_slug = '' ) {
		$settings = AIDF_Plugin::get_settings();

		if ( empty( $settings['spec_attribution'] ) ) {
			return '';
		}

		$spec_url = self::get_spec_url( $file_slug );

		$lines = array(
			'',
			'---',
			self::get_spec_label( $file_slug ),
			$spec_url,
			'Generated by AI Discovery Files for WordPress v' . AIDF_VERSION,
		);

		return implode( "\n", $lines ) . "\n";
	}

	/**
	 * Get a JSON-formatted spec attribution comment.
	 *
	 * @param  string $file_slug File type slug for spec-specific URL.
	 * @return array<string, string> Key-value pair to include in JSON output.
	 */
	public static function get_json_attribution( $file_slug = '' ) {
		$settings = AIDF_Plugin::get_settings();

		if ( empty( $settings['spec_attribution'] ) ) {
			return array();
		}

		return array(
			'_specification' => self::get_spec_url( $file_slug ),
			'_generator'     => 'AI Discovery Files for WordPress v' . AIDF_VERSION,
		);
	}

	/**
	 * Get the specification URL for a specific file type.
	 *
	 * @param  string $file_slug File type slug.
	 * @return string
	 */
	private static function get_spec_url( $file_slug ) {
		$spec_paths = array(
			'llms-txt'         => 'llms-txt',
			'llm-txt'          => 'llm-txt',
			'llms-html'        => 'llms-html',
			'ai-txt'           => 'ai-txt',
			'ai-json'          => 'ai-json',
			'identity-json'    => 'identity-json',
			'brand-txt'        => 'brand-txt',
			'faq-ai-txt'       => 'faq-ai-txt',
			'developer-ai-txt' => 'developer-ai-txt',
			'robots-ai-txt'    => 'robots-ai-txt',
		);

		if ( ! empty( $file_slug ) && isset( $spec_paths[ $file_slug ] ) ) {
			return 'https://www.ai-visibility.org.uk/specifications/' . $spec_paths[ $file_slug ] . '/';
		}

		return 'https://www.ai-visibility.org.uk/specifications/';
	}

	/**
	 * Get the specification label for a specific file type.
	 *
	 * @param  string $file_slug File type slug.
	 * @return string
	 */
	private static function get_spec_label( $file_slug ) {
		$file_types = AIDF_Plugin::get_file_types();

		if ( ! empty( $file_slug ) && isset( $file_types[ $file_slug ] ) ) {
			return $file_types[ $file_slug ]['filename'] . ' Specification (' . $file_types[ $file_slug ]['code'] . ')';
		}

		return 'AI Discovery Files Specification';
	}

	/**
	 * Format a permission value for human-readable display.
	 *
	 * @param  string $value Permission value slug.
	 * @return string
	 */
	public static function format_permission( $value ) {
		$map = array(
			'allow'                 => 'Allowed',
			'allow_with_attribution' => 'Allowed with attribution',
			'restrict'              => 'Restricted',
			'prohibit'              => 'Prohibited',
			'allow_all'             => 'All crawlers allowed',
			'allow_known'           => 'Known AI crawlers allowed',
		);

		return isset( $map[ $value ] ) ? $map[ $value ] : $value;
	}

	/**
	 * Format a content licence value for human-readable display.
	 *
	 * @param  string $value Licence value slug.
	 * @return string
	 */
	public static function format_licence( $value ) {
		$map = array(
			'all_rights_reserved' => 'All Rights Reserved',
			'cc_by'              => 'CC BY 4.0',
			'cc_by_sa'           => 'CC BY-SA 4.0',
			'cc_by_nc'           => 'CC BY-NC 4.0',
			'cc_by_nd'           => 'CC BY-ND 4.0',
			'cc0'                => 'CC0 (Public Domain)',
		);

		return isset( $map[ $value ] ) ? $map[ $value ] : '';
	}

	/**
	 * Build the current ISO 8601 date.
	 *
	 * @return string
	 */
	public static function get_date() {
		return wp_date( 'c' );
	}
}
